package clinical.web.workflow.cbf;

import static clinical.web.CBFBIRNConstants.ANATOMICAL;
import static clinical.web.CBFBIRNConstants.CSF;
import static clinical.web.CBFBIRNConstants.FIELD_MAP;
import static clinical.web.CBFBIRNConstants.MINCON;
import static clinical.web.CBFBIRNConstants.MATLAB_EXEC;

import java.io.File;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;

import clinical.server.vo.Deriveddata;
import clinical.utils.Assertion;
import clinical.utils.Executor;
import clinical.utils.FileUtils;
import clinical.web.CBFBIRNConstants;
import clinical.web.common.UserInfo;
import clinical.web.exception.BaseException;
import clinical.web.helpers.CBFBirnHelper;
import clinical.web.vo.DerivedDataInfo;
import clinical.web.vo.JobProvenanceInfo;
import clinical.web.vo.upload.SegmentInfo;
import clinical.web.vo.upload.SeriesHeaderInfo;
import clinical.web.vo.upload.VisitInfo;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: SiemensCBFProcJobHelper.java 768 2013-02-09 00:29:51Z bozyurt $
 */
public final class SiemensCBFProcJobHelper extends AbstractCBFProcJobHelper {
	private SegmentInfo b0MagSI;
	private SegmentInfo b0PhaseSI;
	private Map<SegmentInfo, BrikWrapper> si2biMap = new HashMap<SegmentInfo, BrikWrapper>();
	private List<NIFTIWrapper> niftiList = new ArrayList<NIFTIWrapper>(4);
	public final static String B0_MAG_DIRNAME = "B0_mag1";
	public final static String B0_PHASE_DIRNAME = "B0_phase1";

	protected static Log _log = LogFactory
			.getLog(SiemensCBFProcJobHelper.class);

	public SiemensCBFProcJobHelper(UserInfo ui, CBFWFContext context,
			String templateDir) throws BaseException {
		super(ui, context, templateDir);
	}

	public VisitProcessInfo prepare4Job() throws Exception {
		File afniDir = new File(this.context.getCacheDir(),
				CBFBIRNConstants.AFNI_DIR_NAME);
		afniDir.mkdir();
		File outDir = new File(this.context.getCacheDir(), "matlab_out");
		outDir.mkdir();
		File fmDir = new File(this.context.getCacheDir(), "fm");
		fmDir.mkdir();
		File resultsDir = new File(this.context.getCacheDir(), "results");
		resultsDir.mkdir();
		File skullDir = new File(this.context.getCacheDir(), "skull_strip");
		skullDir.mkdir();

		File preprocessDir = new File(this.context.getCacheDir(), "preprocess");
		preprocessDir.mkdir();

		File workDir = new File(this.context.getCacheDir(), "work");
		workDir.mkdir();

		File intermediateDir = new File(this.context.getCacheDir(),
				"intermediate");
		intermediateDir.mkdir();
		List<SegmentInfo> si4AFNIList = getSegmentsNeedingAFNIConversion();
		checkIfProcessable(si4AFNIList);

		// handleAFNIConversions(si4AFNIList, afniDIR);
		handlePreprocessing(si4AFNIList, afniDir, workDir);

		for (BrikWrapper bi : this.brikInfoList) {
			si2biMap.put(bi.getSi(), bi);
		}
		for (BrikWrapper bi : this.existingBrikInfoList) {
			si2biMap.put(bi.getSi(), bi);
		}

		List<SiemensFieldMapInfo> fmList = null;
		if (this.context.isDoB0Correction()) {
			determineFieldMapDicomSeries();
			List<SegmentInfo> si4FMList = getSegmentsNeedingFieldMapCorrection();
			fmList = prepare4FieldMaps(si4FMList, fmDir);
		}

		List<BrikWrapper> cbfBIList = new ArrayList<BrikWrapper>();
		for (BrikWrapper bi : this.brikInfoList) {
			if (CBFBirnHelper.isCBFProcessable(bi.getSi().getProtocolId())) {
				cbfBIList.add(bi);
			}
		}

		for (BrikWrapper bi : this.existingBrikInfoList) {
			if (CBFBirnHelper.isCBFProcessable(bi.getSi().getProtocolId())) {
				File newBrikFile = copyBrikFilesTo(bi.getBrikFile(), afniDir);
				bi.brikFile = newBrikFile;
				cbfBIList.add(bi);
			} else {
				// still needs to copy them to working afni directory so that
				// matlab processes can find them
				copyBrikFilesTo(bi.getBrikFile(), afniDir);
			}
		}

		List<CBFProcessInfo> cpiList = prepare4CBFProcessing(cbfBIList, outDir);

		VisitProcessInfo vpi = new VisitProcessInfo(cpiList);
		vpi.setSiemensFmList(fmList);
		// FIXME orig (cleanupPreviousDerivedData)
		// cleanupPreviousDerivedData();
		return vpi;
	}

	protected void determineFieldMapDicomSeries() throws Exception {
		VisitInfo visitInfo = context.getViList().get(0);
		for (SegmentInfo si : visitInfo.getSiList()) {
			if (si.getShi().getImageType().equals(SeriesHeaderInfo.DICOM)
					&& si.getProtocolId().equals(FIELD_MAP)) {
				String dicomDir = si.getShi().getImages().get(0);
				String seriesName = getSeriesName(dicomDir);
				System.out.println("seriesName:" + seriesName);
				String dicomDirName = new File(dicomDir).getName();
				if (dicomDirName.equalsIgnoreCase(B0_MAG_DIRNAME)) {
					this.b0MagSI = si;
				} else if (dicomDirName.equalsIgnoreCase(B0_PHASE_DIRNAME)) {
					this.b0PhaseSI = si;
				} else {
					System.err.println("Not a recognized field map series "
							+ "description:" + seriesName + " dicom Dir:"
							+ dicomDir);
				}
			}
		}
	}

	public List<SiemensFieldMapInfo> prepare4FieldMaps(
			List<SegmentInfo> si4FMList, File outDir) {
		List<SiemensFieldMapInfo> fmiList = new ArrayList<SiemensFieldMapInfo>();
		if (b0MagSI == null || b0PhaseSI == null) {
			_log.warn("Both B0 magnitude and B0 phase images "
					+ "are needed for Siemens field map correction!");
			return fmiList;
		}
		// all Siemens CBF series are DICOMs
		for (SegmentInfo si : si4FMList) {
			NIFTIWrapper niftiWrapper = findMatching(si);
			Assertion.assertNotNull(niftiWrapper);
			File srcNiftiFile = niftiWrapper.nifti.niftiFile;
			String prefix = niftiWrapper.nifti.getBasename();
			File b0MagDicomDir = new File(this.b0MagSI.getShi().getImages()
					.get(0));
			File b0PhaseDicomDir = new File(this.b0PhaseSI.getShi().getImages()
					.get(0));
			SiemensFieldMapInfo fmi = new SiemensFieldMapInfo(srcNiftiFile,
					b0MagDicomDir, b0PhaseDicomDir, prefix, outDir, si);
			fmiList.add(fmi);
		}
		return fmiList;
	}

	protected NIFTIWrapper findMatching(SegmentInfo si) {
		for (NIFTIWrapper nw : this.niftiList) {
			String seriesName = nw.nifti.getBasename();
			if (matches(seriesName, si)) {
				return nw;
			}
		}
		return null;
	}

	public void handleFieldMapProcessing(SiemensFieldMapInfo fmi,
			boolean forcePrep) throws Exception {
		File workDir = new File(this.context.getCacheDir(), "work");
		workDir.mkdir();
		File destDir = fmi.getOutputDir();
		NIFTIWrapper series2Correct = new NIFTIWrapper(new NIFTIFile(
				fmi.getSrcNiftiFile()), fmi.getSrcSI());

		NIFTIAndBrikFiles nbFiles = doFieldMapCorrection(destDir, workDir,
				series2Correct, fmi.getB0MagDicomDir(),
				fmi.getB0PhaseDicomDir(), forcePrep);
		File afniDir = new File(this.context.getCacheDir(),
				CBFBIRNConstants.AFNI_DIR_NAME);
		Assertion.assertTrue(nbFiles.briks.size() == 1);
		AFNIBrik srcB0CorrectedBrik = nbFiles.briks.get(0);
		AFNIBrik b0CorrectedBrik = null;
		File brikFile = new File(afniDir, srcB0CorrectedBrik.getBrikFile()
				.getName());
		if (!clinical.utils.FileUtils.moveTo(srcB0CorrectedBrik.getBrikFile(),
				brikFile)) {
			throw new Exception("Cannot move file:"
					+ srcB0CorrectedBrik.getBrikFile().getName());
		}
		File headFile = new File(afniDir, srcB0CorrectedBrik.getHeaderFile()
				.getName());
		if (!clinical.utils.FileUtils.moveTo(
				srcB0CorrectedBrik.getHeaderFile(), headFile)) {
			throw new Exception("Cannot move file:"
					+ srcB0CorrectedBrik.getHeaderFile().getName());
		}
		b0CorrectedBrik = new AFNIBrik(headFile, brikFile);
		fmi.setFmOutBrik(b0CorrectedBrik);
	}

	public void updateCPIListWithFieldMapCorrection(VisitProcessInfo vpi)
			throws Exception {
		if (vpi.getSiemensFmList() == null || vpi.getSiemensFmList().isEmpty()) {
			return;
		}
		Map<String, SiemensFieldMapInfo> fmiMap = new HashMap<String, SiemensFieldMapInfo>(
				17);
		SiemensFieldMapInfo csfFMI = null;
		SiemensFieldMapInfo minconFMI = null;
		for (SiemensFieldMapInfo fmi : vpi.getSiemensFmList()) {
			if (fmi.getFmOutBrik() != null) {
				String prefix = fmi.getFmOutBrik().getSeriesName();
				int idx = prefix.indexOf("_B0");
				prefix = prefix.substring(0, idx);
				System.out.println("Siemens b0_corrected prefix:" + prefix);
				fmiMap.put(prefix, fmi);
				if (fmi.getSrcSI().getProtocolId().equals(MINCON)) {
					minconFMI = fmi;
				} else if (fmi.getSrcSI().getProtocolId().equals(CSF)) {
					csfFMI = fmi;
				}
			}
		}

		Set<SiemensFieldMapInfo> usedSet = new HashSet<SiemensFieldMapInfo>();
		for (CBFProcessInfo cpi : vpi.getCpiList()) {
			if (csfFMI != null) {
				cpi.csfBrik = csfFMI.getFmOutBrik().getBrikFile();
				cpi.csfSI = csfFMI.getSrcSI();
				usedSet.add(csfFMI);
			}
			if (minconFMI != null) {
				cpi.minconBrik = minconFMI.getFmOutBrik().getBrikFile();
				cpi.minconSI = minconFMI.getSrcSI();
				usedSet.add(minconFMI);
			}
			File cbfBrikFile = cpi.getCbfBrik();
			String prefix = getPrefix(cbfBrikFile);
			if (prefix.endsWith("+orig")) {
				prefix = prefix.replaceFirst("\\+orig$", "");
			}
			SiemensFieldMapInfo fmi = fmiMap.get(prefix);
			if (fmi != null) {
				cpi.cbfBrik = fmi.getFmOutBrik().getBrikFile();
				cpi.cbfSI = fmi.getSrcSI();
				usedSet.add(fmi);
			}
		}

		File outDir = new File(this.context.getCacheDir(), "matlab_out");
		for (SiemensFieldMapInfo fmi : fmiMap.values()) {
			if (usedSet.contains(fmi)) {
				continue;
			}
			if (CBFBirnHelper.isCBFProcessable(fmi.getSrcSI().getProtocolId())) {
				CBFProcessInfo cpi = prepareSiemensFMI4CBFProcessing(fmi,
						csfFMI, minconFMI, outDir);
				vpi.getCpiList().add(cpi);
			}
		}
	}

	@Override
	public void handleCBFProcessingAutomated(CBFProcessInfo cpi,
			File matlabDir, boolean doSkullStripping, double gmThreshold,
			boolean doPartialVolumeCorrection, int alignment, int csfMethod, 
			boolean useLTMethod, double smoothingParam)
			throws Exception {
		// FIXME handle useLTMethod, smoothingParam
		Properties p = new Properties();
		p.setProperty("file.resource.loader.path", templateDir);
		p.setProperty("runtime.log.logsystem.class",
				"org.apache.velocity.runtime.log.NullLogSystem");
		Velocity.init(p);
		VelocityContext ctx = new VelocityContext();
		File afniDir = new File(this.context.getCacheDir(),
				CBFBIRNConstants.AFNI_DIR_NAME);
		ctx.put("brikdir", afniDir.getAbsolutePath());
		ctx.put("cbf", getPrefix(cpi.getCbfBrik()));
		ctx.put("csf", getPrefix(cpi.getCsfBrik()));
		ctx.put("mincon", getPrefix(cpi.getMinconBrik()));
		if (cpi.getAnatBrik() != null) {
			ctx.put("anat", getPrefix(cpi.getAnatBrik()));
			ctx.put("hasAnat", Boolean.TRUE);
		} else {
			ctx.put("anat", "");
			ctx.put("hasAnat", Boolean.FALSE);
		}
		ctx.put("outdir", cpi.getOutputDir());
		ctx.put("matlabdir", matlabDir);

		String pvcStr = doPartialVolumeCorrection ? "1" : "0";
		ctx.put("pvc", pvcStr);
		ctx.put("align", alignment);
		ctx.put("csfMethod", csfMethod);

		String siteName = null;
		if (!context.getViList().isEmpty()) {
			siteName = context.getViList().get(0).getSiteName();
			if (siteName == null || siteName.trim().length() == 0) {
				siteName = "none";
			} else {
				siteName = siteName.toLowerCase();
			}
		}

		ctx.put("site", siteName);

		StringWriter sw = new StringWriter();
		Template template = Velocity.getTemplate("cbf.vm");
		template.merge(ctx, sw);

		System.out.println(sw.toString());
		File matlabDriverScriptFile = new File(cpi.getOutputDir(), "cbf.m");
		matlabDriverScriptFile.delete();
		FileUtils.save2File(sw.toString(),
				matlabDriverScriptFile.getAbsolutePath());

		Executor executor = new Executor(MATLAB_EXEC, true);
		String cmdLine = "-nodesktop -nosplash  < "
				+ matlabDriverScriptFile.getAbsolutePath();

		runProcess(executor, cmdLine);

		File processedDir = new File(cpi.getOutputDir(), "processed");
		Assertion.assertTrue(processedDir.isDirectory());
		File skullDir = new File(this.context.getCacheDir(), "skull_strip");

		String cbfPrefix = getPrefix(cpi.getCbfBrik());
		cbfPrefix = cbfPrefix.replaceFirst("\\+orig$", "");

		// make sure the necessary AFNI briks for skull stripping copied to
		// their expected locations
		for (File f : processedDir.listFiles()) {
			if (!f.isFile()) {
				continue;
			}
			String name = f.getName();
			if (hasAFNIExtension(name)) {
				File destFile = new File(skullDir, f.getName());
				if (name.startsWith("anat+reg+orig")
						|| name.startsWith("anat_ASL+reg+orig")) {
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
					cpi.addRegAnatImage(destFile);
				} else if (name.startsWith(cbfPrefix)
						&& name.indexOf("+reg+orig") != -1) {
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
					cpi.addCbfRegImage(destFile);
				}
			}
		}

		// write the GM CBF value to gm.txt file in the outputDir
		handleGMCBFValueExtraction(cpi, matlabDir, doSkullStripping,
				gmThreshold, smoothingParam);

		// handling of intermediate files
		File intermediateDir = new File(this.context.getCacheDir(),
				"intermediate");
		Assertion.assertTrue(intermediateDir.isDirectory());

		File derivedIntDir = createResultsDerivedDataDir(cpi, intermediateDir);

		File resultsDir = new File(this.context.getCacheDir(), "results");
		File derivedDir = createResultsDerivedDataDir(cpi, resultsDir);

		File[] files = processedDir.listFiles();

		for (File f : files) {
			String name = f.getName();
			if (name.equals("fmri_struct.mat")) {
				File resultFile = new File(derivedDir, name);
				boolean ok = FileUtils.moveTo(f, resultFile);
				if (!ok) {
					throw new Exception("Cannot move file:" + f.getName());
				}

				cpi.setMatFile(resultFile);
			} else if (name.endsWith(".png")) {
				if (name.equals("cbfmap.png") || name.equals("cbfhist.png")
						|| name.equals("motion.png")) {
					// image files summarizing the results
					File resultFile = new File(derivedDir, name);
					boolean ok = FileUtils.moveTo(f, resultFile);
					if (!ok) {
						throw new Exception("Cannot move file:" + f.getName());
					}
					cpi.addResultImage(resultFile);
				} else {
					// the segmentation images
					File destFile = new File(derivedIntDir, f.getName());
					if (destFile.exists()) {
						_log.warn("cbf: intermediate file already exists:"
								+ destFile);
					}
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
				}
			} else if (name.equals("gm.txt")
					|| name.equals("process_summary.txt")
					|| name.equals("cbfbirn_path1_summary.txt")) {
				File resultFile = new File(derivedDir, name);
				boolean ok = FileUtils.moveTo(f, resultFile);
				if (!ok) {
					throw new Exception("Cannot move file:" + f.getName());
				}
			} else {
				// other intermediate files
				if (f.isFile() && !f.getName().endsWith(".m")) {
					File destFile = new File(derivedIntDir, f.getName());
					if (destFile.exists()) {
						_log.warn("cbf: intermediate file already exists:"
								+ destFile);
					}
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
				}
			}
		}

		// result files can be also in output directory
		files = cpi.getOutputDir().listFiles();
		for (File f : files) {
			String name = f.getName();
			if (name.equals("cbfmap.png") || name.equals("cbfhist.png")
					|| name.equals("motion.png")) {
				// image files summarizing the results
				File resultFile = new File(derivedDir, name);
				boolean ok = FileUtils.moveTo(f, resultFile);
				if (!ok) {
					throw new Exception("Cannot move file:" + f.getName());
				}
				cpi.addResultImage(resultFile);
			} else if (name.equals("gm.txt")
					|| name.equals("process_summary.txt")
					|| name.equals("cbfbirn_path1_summary.txt")) {
				File resultFile = new File(derivedDir, name);
				boolean ok = FileUtils.moveTo(f, resultFile);
				if (!ok) {
					throw new Exception("Cannot move file:" + f.getName());
				}
			}
		}

		// register all intermediate files to their CBFProcessInfo object for
		// final sanctioned location move
		File[] intFiles = derivedIntDir.listFiles();

		Map<String, BrikInfo> brikInfoMap = new LinkedHashMap<String, BrikInfo>(
				17);
		for (File intFile : intFiles) {
			if (intFile.isFile()) {
				String basename = intFile.getName()
						.replaceFirst("\\.\\w+$", "");

				if (hasAFNIExtension(intFile.getName())) {
					BrikInfo bi = brikInfoMap.get(basename);
					if (bi == null) {
						bi = new BrikInfo(basename);
						brikInfoMap.put(basename, bi);
					}
					if (intFile.getName().endsWith(".HEAD")) {
						bi.setHeadFile(intFile);
					} else {
						bi.setBrikFile(intFile);
					}
				} else {
					SingleFileInfo sfi = new SingleFileInfo(intFile);
					cpi.addIntermediateFile(sfi);
				}
			}
		}
		for (BrikInfo bi : brikInfoMap.values()) {
			cpi.addIntermediateFile(bi);
		}

		// register CBF+orig, MPPCASL_Mask+orig briks and process_summary.txt
		File[] derivedDirFiles = derivedDir.listFiles();
		for (File derivedFile : derivedDirFiles) {
			String name = derivedFile.getName();
			if (derivedFile.isFile()
					&& (name.startsWith("CBF+orig.")
							|| name.startsWith("MPPCASL_Mask+orig.")
							|| name.equals("process_summary.txt") || name
								.equals("cbfbirn_path1_summary.txt"))) {
				SingleFileInfo sfi = new SingleFileInfo(derivedFile);
				cpi.addAdditionalResultFile(sfi);
			}
		}

		// directory cleanup
		if (!TEST_MODE) {
			FileUtils.deleteRecursively(cpi.getOutputDir());
			cpi.getOutputDir().mkdir();
		}
	}

	protected CBFProcessInfo prepareSiemensFMI4CBFProcessing(
			SiemensFieldMapInfo fmi, SiemensFieldMapInfo csfFMI,
			SiemensFieldMapInfo minconFMI, File outDir) throws Exception {
		boolean hasAnat = hasAnatomical();
		BrikWrapper anatBI = null;
		if (hasAnat) {
			anatBI = findBrikInfo(ANATOMICAL);
		}
		File csfBrikFile = null;
		File minconBrikFile = null;
		SegmentInfo csfSI = null;
		SegmentInfo minconSI = null;
		CBFProcessInfo cpi = null;
		BrikWrapper csfBI = findBrikInfo(CSF);
		if (csfFMI != null && csfBI == null) {
			csfBrikFile = csfFMI.getFmOutBrik().getBrikFile();
			csfSI = csfFMI.getSrcSI();
		} else {
			csfBrikFile = csfBI.getBrikFile();
			csfSI = csfBI.getSi();
		}

		BrikWrapper minconBI = findBrikInfo(MINCON);
		if (minconFMI != null && minconBI == null) {
			minconBrikFile = minconFMI.getFmOutBrik().getBrikFile();
			minconSI = minconFMI.getSrcSI();
		} else {
			// minconBI can only be null if there is a corresponding minconFMI
			minconBrikFile = minconBI.getBrikFile();
			minconSI = minconBI.getSi();
		}

		File cbfBrikFile = fmi.getFmOutBrik().getBrikFile();
		if (hasAnat) {
			cpi = new CBFProcessInfo(cbfBrikFile, csfBrikFile, minconBrikFile,
					anatBI.getBrikFile(), fmi.getSrcSI(), csfSI, minconSI,
					anatBI.getSi(), outDir);
		} else {
			cpi = new CBFProcessInfo(cbfBrikFile, csfBrikFile, minconBrikFile,
					fmi.getSrcSI(), csfSI, minconSI, outDir);
		}
		return cpi;
	}

	public List<Deriveddata> saveDerivedData(VisitProcessInfo vpi)
			throws Exception {
		List<File> fmBrikFilesAvail = new ArrayList<File>(10);
		List<DerivedDataInfo> ddiList = new ArrayList<DerivedDataInfo>();

		// orig
		/*
		 * File rdRootFile = getRawDataRoot4Visit();
		 * Assertion.assertNotNull(rdRootFile); File derivedDir = new
		 * File(rdRootFile, "derived");
		 * 
		 * derivedDir.mkdir();
		 */
		// multi-workflow support
		List<JobProvenanceInfo> jpiList = getExistingWFResults();
		File derivedDir = getNextDerivedDir(jpiList, true);

		VisitInfo vi = context.getViList().get(0);
		String subjectID = vi.getSubjectID();
		int expId = vi.getExpId();
		int visitId = vi.getVisitId();
		if (vpi.getSiemensFmList() != null) {
			for (SiemensFieldMapInfo fmi : vpi.getSiemensFmList()) {
				File[] files = new File[] { fmi.getFmOutBrik().getHeaderFile(),
						fmi.getFmOutBrik().getBrikFile() };
				for (File f : files) {
					File derivedFile = copy2DerivedDataLoc(f, derivedDir);
					DerivedDataInfo ddi = prepDerivedDataInfo(expId, visitId,
							subjectID, fmi.getSrcSI(), derivedFile);
					ddiList.add(ddi);
					fmBrikFilesAvail.add(derivedFile);
				}
			}
		}
		// also the supporting briks (if not existing) anat, mincon, csf,
		// fieldmap1 fieldmap2 briks
		boolean anatBrikHandled = false;
		boolean csfBrikHandled = false;
		boolean minconBrikHandled = false;

		if (!vpi.getCpiList().isEmpty()) {
			CBFProcessInfo cpi = vpi.getCpiList().get(0);
			if (!fmBrikFilesAvail.isEmpty()) {
				if (cpi.getAnatBrik() != null
						&& !isExistingBrik(cpi.getAnatBrik(), fmBrikFilesAvail)) {
					prepDDForSupportBrik(cpi.getAnatBrik(), cpi.getAnatSI(),
							derivedDir, subjectID, expId, visitId, ddiList);
					anatBrikHandled = true;
				}

				if (!isExistingBrik(cpi.getCsfBrik(), fmBrikFilesAvail)) {
					prepDDForSupportBrik(cpi.getCsfBrik(), cpi.getCsfSI(),
							derivedDir, subjectID, expId, visitId, ddiList);
					csfBrikHandled = true;
				}
				if (!isExistingBrik(cpi.getMinconBrik(), fmBrikFilesAvail)) {
					prepDDForSupportBrik(cpi.getMinconBrik(),
							cpi.getMinconSI(), derivedDir, subjectID, expId,
							visitId, ddiList);
					minconBrikHandled = true;
				}
			}
		}

		// CBF processing derived data
		for (CBFProcessInfo cpi : vpi.getCpiList()) {
			String srcPrefix = cpi.getCbfBrik().getName();
			srcPrefix = srcPrefix.replaceFirst("\\.\\w+$", "");
			File cbfDerivedDir = new File(derivedDir, srcPrefix);
			cbfDerivedDir.mkdir();

			prepDDForCBFBrik(cpi, cbfDerivedDir, subjectID, expId, visitId,
					ddiList);
			if (!csfBrikHandled && !isExistingBrik(cpi.getCsfBrik(), null)) {
				prepDDForSupportBrik(cpi.getCsfBrik(), cpi.getCsfSI(),
						derivedDir, subjectID, expId, visitId, ddiList);
			}
			if (!minconBrikHandled
					&& !isExistingBrik(cpi.getMinconBrik(), null)) {
				prepDDForSupportBrik(cpi.getMinconBrik(), cpi.getMinconSI(),
						derivedDir, subjectID, expId, visitId, ddiList);
			}
			if (!anatBrikHandled && cpi.getAnatBrik() != null
					&& !isExistingBrik(cpi.getAnatBrik(), null)) {
				prepDDForSupportBrik(cpi.getAnatBrik(), cpi.getAnatSI(),
						derivedDir, subjectID, expId, visitId, ddiList);
			}
		}

		return super.saveDerivedDataWithJobProvenance(ddiList, derivedDir);
		// orig
		// return didService.addDerivedData(ui, ddiList);
	}

	@Override
	protected List<SegmentInfo> getSegmentsNeedingAFNIConversion() {
		// all ASL and anatomical data needs to be in DICOM for Siemens
		List<SegmentInfo> si4AFNIList = new ArrayList<SegmentInfo>();
		VisitInfo visitInfo = context.getViList().get(0);
		for (SegmentInfo si : visitInfo.getSiList()) {
			if (si.getShi().getImageType().equals(SeriesHeaderInfo.DICOM)
					&& (si.getProtocolId().equals(MINCON)
							|| si.getProtocolId().equals(CSF)
							|| si.getProtocolId().equals(ANATOMICAL) || CBFBirnHelper
								.isCBFProcessable(si.getProtocolId()))) {
				si4AFNIList.add(si);
			}
		}
		return si4AFNIList;
	}

	protected void handlePreprocessing(List<SegmentInfo> si4AFNIList,
			File destDir, File workDir) throws Exception {

		SegmentInfo csfSI = findSegmentInfo(CSF, si4AFNIList);
		SegmentInfo minconSI = findSegmentInfo(MINCON, si4AFNIList);
		SegmentInfo anatSI = findSegmentInfo(ANATOMICAL, si4AFNIList);
		SegmentInfo aslSI = findASL(si4AFNIList);

		File csfSrcDir = getSeriesDir(csfSI);
		File minconSrcDir = getSeriesDir(minconSI);
		File aslSrcDir = getSeriesDir(aslSI);
		File anatSrcDir = getSeriesDir(anatSI);

		NIFTIAndBrikFiles nbFiles = preprocess(destDir, workDir, aslSrcDir,
				csfSrcDir, minconSrcDir, anatSrcDir);
		for (AFNIBrik brik : nbFiles.briks) {
			String seriesName = brik.getSeriesName();
			if (matches(seriesName, csfSI)) {
				brikInfoList.add(new BrikWrapper(brik.brikFile, csfSI));
			} else if (matches(seriesName, minconSI)) {
				brikInfoList.add(new BrikWrapper(brik.brikFile, minconSI));
			} else if (matches(seriesName, aslSI)) {
				brikInfoList.add(new BrikWrapper(brik.brikFile, aslSI));
			} else if (seriesName.equals("anat_ASL")) {
				brikInfoList.add(new BrikWrapper(brik.brikFile, anatSI));
			}
		}
		for (NIFTIFile nf : nbFiles.niftiFiles) {
			String seriesName = nf.getBasename();
			if (matches(seriesName, csfSI)) {
				niftiList.add(new NIFTIWrapper(nf, csfSI));
			} else if (matches(seriesName, minconSI)) {
				niftiList.add(new NIFTIWrapper(nf, minconSI));
			} else if (matches(seriesName, aslSI)) {
				niftiList.add(new NIFTIWrapper(nf, aslSI));
			} else if (seriesName.equals("anat_ASL")) {
				niftiList.add(new NIFTIWrapper(nf, anatSI));
			}
		}
	}

	protected File getSeriesDir(SegmentInfo si) {
		if (si == null) {
			return null;
		}
		File imageFile = new File(si.getShi().getImages().get(0));
		if (imageFile.isDirectory()) {
			return imageFile;
		}
		return imageFile.getParentFile();
	}

	protected List<CBFProcessInfo> prepare4CBFProcessing(
			List<BrikWrapper> biList, File outDir) {
		List<CBFProcessInfo> cpiList = new ArrayList<CBFProcessInfo>();
		boolean hasAnat = hasAnatomical();
		BrikWrapper minconBI = findBrikInfo(MINCON);
		BrikWrapper csfBI = findBrikInfo(CSF);
		BrikWrapper anatBI = null;
		if (hasAnat) {
			anatBI = findBrikInfo(ANATOMICAL);
		}
		for (BrikWrapper bi : biList) {
			SegmentInfo si = bi.getSi();
			if (CBFBirnHelper.isCBFProcessable(si.getProtocolId())) {
				CBFProcessInfo cpi = null;
				if (hasAnat) {
					cpi = new CBFProcessInfo(bi.getBrikFile(),
							csfBI.getBrikFile(), minconBI.getBrikFile(),
							anatBI.getBrikFile(), bi.getSi(), csfBI.getSi(),
							minconBI.getSi(), anatBI.getSi(), outDir);
				} else {
					cpi = new CBFProcessInfo(bi.getBrikFile(),
							csfBI.getBrikFile(), minconBI.getBrikFile(),
							bi.getSi(), csfBI.getSi(), minconBI.getSi(), outDir);
				}
				cpiList.add(cpi);
			}
		}

		return cpiList;
	}

	protected List<SegmentInfo> getSegmentsNeedingFieldMapCorrection() {
		List<SegmentInfo> si4FMList = new ArrayList<SegmentInfo>(5);
		VisitInfo visitInfo = context.getViList().get(0);
		for (SegmentInfo si : visitInfo.getSiList()) {
			if (si.getShi().getImageType().equals(SeriesHeaderInfo.DICOM)) {
				if (si.getProtocolId().equals(MINCON)
						|| si.getProtocolId().equals(CSF)
						|| CBFBirnHelper.isCBFProcessable(si.getProtocolId())) {
					si4FMList.add(si);
				}
			}
		}
		return si4FMList;
	}

	public NIFTIAndBrikFiles doFieldMapCorrection(File destDir, File workDir,
			NIFTIWrapper series2Correct, File magnitudeDir, File phaseDir,
			boolean forcePrep) throws CBFException {
		Executor executor = new Executor(
				"/usr/local/bin/do_B0_mosaic_cbfbirn.pl", true);
		executor.setErrorStreamReliable(false);
		StringBuilder sb = new StringBuilder(256);
		try {
			NIFTIAndBrikFiles nbFiles = new NIFTIAndBrikFiles();
			sb.append("--dest ").append(workDir.getCanonicalPath());
			sb.append(" --mag ").append(magnitudeDir.getCanonicalPath());
			sb.append(" --phase ").append(phaseDir.getCanonicalPath());
			sb.append(" --asl  ").append(
					series2Correct.nifti.niftiFile.getCanonicalPath());
			if (forcePrep) {
				sb.append(" --force-prep");
			}
			String cmdLine = sb.toString();
			_log.info("running  do_B0_mosaic_cbfbirn.pl " + cmdLine);

			runProcess(executor, cmdLine);

			String aslSeriesName = clinical.utils.FileUtils
					.getBasename(series2Correct.nifti.niftiFile.getName());
			File b0CorrectedDir = new File(workDir, "B0_corrected");
			Assertion.assertTrue(b0CorrectedDir.isDirectory());
			File[] files = b0CorrectedDir.listFiles();
			for (File f : files) {
				String name = f.getName();
				if (f.isFile() && name.startsWith(aslSeriesName)) {
					if (FileUtils.isAFNIBrick(name)
							|| FileUtils.isAFNIHeader(name)) {
						String seriesName = AFNIBrik.getSeriesName(name);
						AFNIBrik ab = nbFiles.getBrik(seriesName);
						File destFile = new File(destDir, f.getName());
						if (!clinical.utils.FileUtils.moveTo(f, destFile)) {
							throw new Exception("Cannot move file:"
									+ f.getName());
						}
						if (FileUtils.isAFNIBrick(name)) {
							if (ab == null) {
								ab = new AFNIBrik(null, destFile);
								nbFiles.addBrik(ab);
							} else {
								ab.brikFile = destFile;
							}
						} else {
							if (ab == null) {
								ab = new AFNIBrik(destFile, null);
								nbFiles.addBrik(ab);
							} else {
								ab.headerFile = destFile;
							}
						}
					} else if (FileUtils.isNiftiFile(name)) {
						File destFile = new File(destDir, name);
						if (!clinical.utils.FileUtils.moveTo(f, destFile)) {
							throw new Exception("Cannot move file:"
									+ f.getName());
						}
						nbFiles.addNIFTI(new NIFTIFile(destFile));
					}
				}
			} // for
				// return B0 corrected AFNI BRIK and NIFTI file info (all moved
				// to
				// their final destinations)
			return nbFiles;

		} catch (Exception ex) {
			ex.printStackTrace();
			throw new CBFException(ex);
		}
	}

	public NIFTIAndBrikFiles preprocess(File destDir, File workDir,
			File srcASLDicomDir, File srcCSFDicomDir, File srcMinConDicomDir,
			File srcT1DicomDir) throws CBFException {
		Executor executor = new Executor("/usr/local/bin/asl_prep_cbfbirn.pl",
				true);

		StringBuilder sb = new StringBuilder(256);
		try {
			sb.append("--dest ").append(workDir.getCanonicalPath());
			sb.append(" --asl-src ").append(srcASLDicomDir.getCanonicalPath());
			sb.append(" --csf-src ").append(srcCSFDicomDir.getCanonicalPath());
			sb.append(" --mincon-src ").append(
					srcMinConDicomDir.getCanonicalPath());
			if (srcT1DicomDir != null) {
				sb.append(" --t1-src ")
						.append(srcT1DicomDir.getCanonicalPath());
			}
			_log.info("running asl_prep_cbfbirn " + sb.toString());

			executor.addEnvParam("PATH", this.envPath);
			for (String ev : this.envMap.keySet()) {
				String value = this.envMap.get(ev);
				executor.addEnvParam(ev, value);
			}
			executor.execute(sb.toString());

			NIFTIAndBrikFiles nbFiles = new NIFTIAndBrikFiles();

			String t1SeriesName = null;
			if (srcT1DicomDir != null) {
				t1SeriesName = srcT1DicomDir.getName();
			}
			File[] files = workDir.listFiles();
			for (File f : files) {
				String name = f.getName();
				if (f.isFile()) {
					if (FileUtils.isAFNIBrick(name)
							|| FileUtils.isAFNIHeader(name)) {
						String seriesName = AFNIBrik.getSeriesName(name);
						AFNIBrik ab = nbFiles.getBrik(seriesName);
						File destFile = new File(destDir, f.getName());
						if (!clinical.utils.FileUtils.moveTo(f, destFile)) {
							throw new Exception("Cannot move file:"
									+ f.getName());
						}
						if (FileUtils.isAFNIBrick(name)) {
							if (ab == null) {
								ab = new AFNIBrik(null, destFile);
								nbFiles.addBrik(ab);
							} else {
								ab.brikFile = destFile;
							}
						} else {
							if (ab == null) {
								ab = new AFNIBrik(destFile, null);
								nbFiles.addBrik(ab);
							} else {
								ab.headerFile = destFile;
							}
						}
					} else if (FileUtils.isNiftiFile(name)) {
						// skip NIFTI file with T1 series name, since the final
						// anat NIFTI will be start with anat_ASL
						if (t1SeriesName != null
								&& name.startsWith(t1SeriesName)) {
							continue;
						}
						File destFile = new File(destDir, name);
						if (!clinical.utils.FileUtils.moveTo(f, destFile)) {
							throw new Exception("Cannot move file:"
									+ f.getName());
						}
						nbFiles.addNIFTI(new NIFTIFile(destFile));
					}
				}
			}

			return nbFiles;

		} catch (Exception ex) {
			ex.printStackTrace();
			throw new CBFException(ex);
		}
	}

	public static class NIFTIAndBrikFiles {
		List<AFNIBrik> briks = new ArrayList<AFNIBrik>(5);
		List<NIFTIFile> niftiFiles = new ArrayList<NIFTIFile>(5);
		Map<String, AFNIBrik> brikMap = new HashMap<String, AFNIBrik>(11);

		public void addBrik(AFNIBrik brik) {
			briks.add(brik);
			brikMap.put(brik.getSeriesName(), brik);
		}

		public void addNIFTI(NIFTIFile nf) {
			niftiFiles.add(nf);
		}

		public AFNIBrik getBrik(String brikName) {
			String key = brikName;
			int idx = brikName.indexOf('+');
			if (idx != -1) {
				key = brikName.substring(0, idx);
			}
			return brikMap.get(key);
		}

	}

	public static class NIFTIFile {
		File niftiFile;
		int type = NIFTI;
		public static int NIFTI = 1;
		public static int GZIPPED_NIFTI = 1;

		public NIFTIFile(File niftiFile) {
			this.niftiFile = niftiFile;
			String name = niftiFile.getName();
			if (name.endsWith(".nii.gz")) {
				type = GZIPPED_NIFTI;
			} else if (name.endsWith(".nii")) {
				type = NIFTI;
			} else {
				throw new RuntimeException(
						"Unknown NIFTI file suffix for file:" + niftiFile);
			}
		}

		public String getBasename() {
			String name = niftiFile.getName();
			int idx = name.indexOf('.');
			Assertion.assertTrue(idx != -1);
			return name.substring(0, idx);
		}

		public File getNiftiFile() {
			return niftiFile;
		}
	}// ;

	public static class NIFTIWrapper {
		NIFTIFile nifti;
		SegmentInfo si;

		public NIFTIWrapper(NIFTIFile nifti, SegmentInfo si) {
			super();
			this.nifti = nifti;
			this.si = si;
		}

		public NIFTIFile getNifti() {
			return nifti;
		}

		public SegmentInfo getSi() {
			return si;
		}
	}// ;

	protected boolean checkIfProcessable(List<SegmentInfo> siList)
			throws Exception {
		// there must be a single ASL , single CSF and single MinCon DICOM
		// series and an optional T1
		boolean hasASL = false;
		boolean hasCSF = false;
		boolean hasMincon = false;
		boolean hasAnat = false;
		for (SegmentInfo si : siList) {
			String protocol = si.getProtocolId();
			if (CBFBirnHelper.isCBFProcessable(protocol)) {
				if (hasASL) {
					throw new Exception(
							"Multiple ASL series are not supported for Siemens!");
				}
				hasASL = true;
			} else if (protocol.equals(ANATOMICAL)) {
				if (hasAnat) {
					throw new Exception(
							"Multiple Anatomical (T1) series are not supported for Siemens!");
				}
				hasAnat = true;
			} else if (protocol.equals(CSF)) {
				if (hasCSF) {
					throw new Exception(
							"Multiple CSF series are not supported for Siemens!");
				}
				hasCSF = true;
			} else if (protocol.equals(MINCON)) {
				if (hasMincon) {
					throw new Exception(
							"Multiple Mincon series are not supported for Siemens!");
				}
				hasMincon = true;
			}
		}
		if (!hasMincon || !hasCSF || !hasASL) {
			throw new Exception(
					"Not a complete data set! A processable Siemens data set has an ASL, "
							+ "a CSF and a Mincon DICOM series with optional anatomical T1 dicom series.");
		}

		return true;
	}

	protected SegmentInfo findSegmentInfo(String protocol,
			List<SegmentInfo> siList) {
		for (SegmentInfo si : siList) {
			if (si.getProtocolId().equals(protocol)) {
				return si;
			}
		}
		return null;
	}

	protected SegmentInfo findASL(List<SegmentInfo> siList) {
		for (SegmentInfo si : siList) {
			if (CBFBirnHelper.isCBFProcessable(si.getProtocolId())) {
				return si;
			}
		}
		return null;
	}

	protected boolean matches(String seriesName, SegmentInfo si) {
		File seriesDir = getSeriesDir(si);
		if (seriesDir != null) {
			return seriesName.equals(seriesDir.getName());
		}
		return false;
	}
}
