package clinical.web.workflow.cbf;

import static clinical.web.CBFBIRNConstants.ANATOMICAL;
import static clinical.web.CBFBIRNConstants.CSF;
import static clinical.web.CBFBIRNConstants.MATLAB_EXEC;

import java.io.File;
import java.io.IOException;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;

import clinical.server.vo.Deriveddata;
import clinical.utils.Assertion;
import clinical.utils.Executor;
import clinical.utils.FileUtils;
import clinical.web.CBFBIRNConstants;
import clinical.web.common.UserInfo;
import clinical.web.exception.BaseException;
import clinical.web.helpers.CBFBirnHelper;
import clinical.web.vo.DerivedDataInfo;
import clinical.web.vo.JobProvenanceInfo;
import clinical.web.vo.upload.SegmentInfo;
import clinical.web.vo.upload.VisitInfo;
import clinical.web.workflow.cbf.AbstractCBFProcJobHelper.BrikWrapper;

public class UCLACBFProcHelper extends AbstractCBFProcJobHelper {
	protected static Log _log = LogFactory
			.getLog(AbstractCBFProcJobHelper.class);
	protected List<AnalyzeWrapper> analyzeImgSetList = new ArrayList<AnalyzeWrapper>(
			3);
	protected static final boolean TEST_MODE = true;

	public UCLACBFProcHelper(UserInfo ui, CBFWFContext context,
			String templateDir) throws BaseException {
		super(ui, context, templateDir);
	}

	private void makeDir(File aDir) throws Exception {
		if (!aDir.mkdir()) {
			_log.error("cannot create dir:" + aDir);
			throw new Exception(
					"A unrecovervable problem occured during CBF job processing!");
		}
	}

	public VisitProcessInfo prepare4Job(File matlabDirFile) throws Exception {
		File analyzeDIR = new File(this.context.getCacheDir(),
				CBFBIRNConstants.ANALYZE_DIR_NAME);
		makeDir(analyzeDIR);
		File outDir = new File(this.context.getCacheDir(), "matlab_out");
		makeDir(outDir);
		File fmDir = new File(this.context.getCacheDir(), "fm");
		makeDir(fmDir);

		File resultsDir = new File(this.context.getCacheDir(), "results");
		makeDir(resultsDir);

		File senseDir = new File(this.context.getCacheDir(), "sense");
		makeDir(senseDir);

		File skullDir = new File(this.context.getCacheDir(), "skull_strip");
		makeDir(skullDir);

		File intermediateDir = new File(this.context.getCacheDir(),
				"intermediate");
		makeDir(intermediateDir);

		List<AnalyzeWrapper> cbfBIList = new ArrayList<AnalyzeWrapper>(3);
		VisitInfo visitInfo = context.getViList().get(0);
		for (SegmentInfo si : visitInfo.getSiList()) {
			if (si.getProtocolId().equals(CSF)
					|| si.getProtocolId().equals(ANATOMICAL)
					|| CBFBirnHelper.isCBFProcessable(si.getProtocolId())) {
				File analyzeImgFile = null;
				for (String imgFile : si.getShi().getImages()) {
					if (FileUtils.isAnalyzeImage(new File(imgFile))) {
						analyzeImgFile = new File(imgFile);
						break;
					}
				}
				Assertion.assertNotNull(
						analyzeImgFile,
						"Missing ANALYZE 7.5 img file for series:"
								+ si.getName());
				AnalyzeWrapper analyzeWrapper = new AnalyzeWrapper(
						analyzeImgFile, si);
				this.analyzeImgSetList.add(analyzeWrapper);
				if (CBFBirnHelper.isCBFProcessable(si.getProtocolId())) {
					cbfBIList.add(analyzeWrapper);
				}
				// make sure analyze files are copied to staging directory for
				// matlab
				copyAnalyzeFilesTo(analyzeWrapper.getAnalyzeImgFile(),
						analyzeDIR);
			}
		}

		List<CBFProcessInfo> cpiList = prepare4CBFProcessing(cbfBIList, outDir,
				matlabDirFile);

		VisitProcessInfo vpi = new VisitProcessInfo(cpiList);

		return vpi;
	}

	public void handleUCLACBF(CBFProcessInfo cpi, File matlabDir,
			boolean doSkullStripping, double gmThreshold, double smoothingParam)
			throws Exception {
		Assertion.assertNotNull(cpi.getAnatBrik());

		Properties p = new Properties();
		p.setProperty("file.resource.loader.path", templateDir);
		p.setProperty("runtime.log.logsystem.class",
				"org.apache.velocity.runtime.log.NullLogSystem");
		Velocity.init(p);
		VelocityContext ctx = new VelocityContext();
		File analyzeDIR = new File(this.context.getCacheDir(),
				CBFBIRNConstants.ANALYZE_DIR_NAME);

		ctx.put("inputpath", analyzeDIR.getAbsolutePath());
		ctx.put("cbf", getPrefix(cpi.getCbfBrik()));
		ctx.put("csf", getPrefix(cpi.getCsfBrik()));
		ctx.put("anat", getPrefix(cpi.getAnatBrik()));
		ctx.put("outdir", cpi.getOutputDir());
		ctx.put("matlabdir", matlabDir);

		StringWriter sw = new StringWriter();
		Template template = Velocity.getTemplate("cbfucla.vm");
		template.merge(ctx, sw);

		System.out.println(sw.toString());
		File matlabDriverScriptFile = new File(cpi.getOutputDir(), "cbfucla.m");
		matlabDriverScriptFile.delete();
		FileUtils.save2File(sw.toString(),
				matlabDriverScriptFile.getAbsolutePath());

		Executor executor = new Executor(MATLAB_EXEC, true);
		String cmdLine = "-nodesktop -nosplash  < "
				+ matlabDriverScriptFile.getAbsolutePath();

		runProcess(executor, cmdLine);

		File processedDir = new File(cpi.getOutputDir(), "processed");
		Assertion.assertTrue(processedDir.isDirectory());
		File skullDir = new File(this.context.getCacheDir(), "skull_strip");

		String cbfPrefix = getPrefix(cpi.getCbfBrik());
		cbfPrefix = cbfPrefix.replaceFirst("\\+orig$", "");

		// make sure the necessary AFNI briks for skull stripping copied to
		// their expected locations
		for (File f : processedDir.listFiles()) {
			if (!f.isFile()) {
				continue;
			}
			String name = f.getName();
			if (hasAFNIExtension(name)) {
				File destFile = new File(skullDir, f.getName());
				if (name.startsWith("anat+reg+orig")) {
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
					cpi.addRegAnatImage(destFile);
				} else if (name.startsWith("cbf+reg+orig")
						|| (name.startsWith(cbfPrefix) && name
								.indexOf("+reg+orig") != -1)) {
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
					cpi.addCbfRegImage(destFile);
				}
			}
		}

		// write the GM CBF value to gm.txt file in the outputDir
		handleGMCBFValueExtraction(cpi, matlabDir, doSkullStripping,
				gmThreshold, smoothingParam);

		// handling of intermediate files
		File intermediateDir = new File(this.context.getCacheDir(),
				"intermediate");
		Assertion.assertTrue(intermediateDir.isDirectory());

		File derivedIntDir = createResultsDerivedDataDir(cpi, intermediateDir);

		File resultsDir = new File(this.context.getCacheDir(), "results");
		File derivedDir = createResultsDerivedDataDir(cpi, resultsDir);

		File[] files = processedDir.listFiles();

		for (File f : files) {
			String name = f.getName();
			if (name.equals("fmri_struct.mat")) {
				File resultFile = new File(derivedDir, name);
				boolean ok = FileUtils.moveTo(f, resultFile);
				if (!ok) {
					throw new Exception("Cannot move file:" + f.getName());
				}

				cpi.setMatFile(resultFile);
			} else if (name.endsWith(".png")) {
				if (name.equals("cbfmap.png") || name.equals("cbfhist.png")
						|| name.equals("motion.png")) {
					// image files summarizing the results
					File resultFile = new File(derivedDir, name);
					boolean ok = FileUtils.moveTo(f, resultFile);
					if (!ok) {
						throw new Exception("Cannot move file:" + f.getName());
					}
					cpi.addResultImage(resultFile);
				} else {
					File destFile = new File(derivedIntDir, f.getName());
					if (destFile.exists()) {
						_log.warn("cbf: intermediate file already exists:"
								+ destFile);
					}
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
				}
			} else if (name.equals("gm.txt")
					|| name.equals("process_summary.txt")
					|| name.equals("cbfbirn_path1_summary.txt")) {
				File resultFile = new File(derivedDir, name);
				boolean ok = FileUtils.moveTo(f, resultFile);
				if (!ok) {
					throw new Exception("Cannot move file:" + f.getName());
				}
			} else if (name.startsWith("CBF+orig.")
					|| name.startsWith("MPPCASL_Mask+orig.")) {
				File resultFile = new File(derivedDir, name);
				boolean ok = FileUtils.moveTo(f, resultFile);
				if (!ok) {
					throw new Exception("Cannot move brik file:" + f.getName());
				}
			} else {
				// other intermediate files
				if (f.isFile() && !f.getName().endsWith(".m")) {
					if (f.getName().equals("process_summary.txt")
							|| f.getName().equals("cbfbirn_path1_summary.txt")) {
						File resultFile = new File(derivedDir, f.getName());
						FileUtils.copyFile(f.getAbsolutePath(),
								resultFile.getAbsolutePath());
						continue;
					}
					File destFile = new File(derivedIntDir, f.getName());
					if (destFile.exists()) {
						_log.warn("cbf: intermediate file already exists:"
								+ destFile);
					}
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
				}
			}
		}

		// result files can be anywhere (seems to keep changing from version to
		// version of the MATLAB code) 5/18/2012
		File[] outDirFiles = cpi.getOutputDir().listFiles();
		for (File f : outDirFiles) {
			String name = f.getName();
			if (name.equals("cbfmap.png") || name.equals("cbfhist.png")
					|| name.equals("motion.png")) {
				// image files summarizing the results
				File resultFile = new File(derivedDir, name);
				boolean ok = FileUtils.moveTo(f, resultFile);
				if (!ok) {
					throw new Exception("Cannot move file:" + f.getName());
				}
				cpi.addResultImage(resultFile);
			} else if (name.equals("gm.txt")
					|| name.equals("process_summary.txt")
					|| name.equals("cbfbirn_path1_summary.txt")) {
				File resultFile = new File(derivedDir, name);
				boolean ok = FileUtils.moveTo(f, resultFile);
				if (!ok) {
					throw new Exception("Cannot move file:" + f.getName());
				}
			}
		}

		// register all intermediate files to their CBFProcessInfo object for
		// final sanctioned location move
		File[] intFiles = derivedIntDir.listFiles();

		Map<String, BrikInfo> brikInfoMap = new LinkedHashMap<String, BrikInfo>(
				17);
		for (File intFile : intFiles) {
			if (intFile.isFile()) {
				String basename = intFile.getName()
						.replaceFirst("\\.\\w+$", "");

				if (hasAFNIExtension(intFile.getName())) {
					BrikInfo bi = brikInfoMap.get(basename);
					if (bi == null) {
						bi = new BrikInfo(basename);
						brikInfoMap.put(basename, bi);
					}
					if (intFile.getName().endsWith(".HEAD")) {
						bi.setHeadFile(intFile);
					} else {
						bi.setBrikFile(intFile);
					}
				} else {
					SingleFileInfo sfi = new SingleFileInfo(intFile);
					cpi.addIntermediateFile(sfi);
				}
			}
		}
		for (BrikInfo bi : brikInfoMap.values()) {
			cpi.addIntermediateFile(bi);
		}

		// register CBF+orig, MPPCASL_Mask+orig briks and process_summary.txt
		// 5/18/2012
		File[] derivedDirFiles = derivedDir.listFiles();
		for (File derivedFile : derivedDirFiles) {
			String name = derivedFile.getName();
			if (derivedFile.isFile()
					&& (name.startsWith("CBF+orig.")
							|| name.startsWith("MPPCASL_Mask+orig.")
							|| name.equals("process_summary.txt") || name
								.equals("cbfbirn_path1_summary.txt"))) {
				SingleFileInfo sfi = new SingleFileInfo(derivedFile);
				cpi.addAdditionalResultFile(sfi);
			}
		}

		// directory cleanup
		if (!TEST_MODE) {
			FileUtils.deleteRecursively(cpi.getOutputDir());
			cpi.getOutputDir().mkdir();
		}
	}

	public List<Deriveddata> saveDerivedData(VisitProcessInfo vpi)
			throws Exception {

		List<DerivedDataInfo> ddiList = new ArrayList<DerivedDataInfo>();
		List<JobProvenanceInfo> jpiList = getExistingWFResults();

		File derivedDir = getNextDerivedDir(jpiList, true);

		VisitInfo vi = context.getViList().get(0);
		String subjectID = vi.getSubjectID();
		int expId = vi.getExpId();
		int visitId = vi.getVisitId();

		// CBF processing derived data
		for (CBFProcessInfo cpi : vpi.getCpiList()) {
			String srcPrefix = cpi.getCbfBrik().getName();
			srcPrefix = srcPrefix.replaceFirst("\\.\\w+$", "");
			File cbfDerivedDir = new File(derivedDir, srcPrefix);
			cbfDerivedDir.mkdir();

			prepDDForCBFAnalyzeSeries(cpi, cbfDerivedDir, subjectID, expId,
					visitId, ddiList);

			prepDDForSupportAnalyzeSeries(cpi.getCsfBrik(), cpi.getCsfSI(),
					derivedDir, subjectID, expId, visitId, ddiList);

			prepDDForSupportAnalyzeSeries(cpi.getAnatBrik(), cpi.getAnatSI(),
					derivedDir, subjectID, expId, visitId, ddiList);

		}

		return super.saveDerivedDataWithJobProvenance(ddiList, derivedDir);
	}

	protected void prepDDForCBFAnalyzeSeries(CBFProcessInfo cpi,
			File derivedDir, String subjectID, int expId, int visitId,
			List<DerivedDataInfo> ddiList) throws IOException {
		File matFile = cpi.getMatFile();
		File derivedFile = copy2DerivedDataLoc(matFile, derivedDir);
		DerivedDataInfo ddi = prepDerivedDataInfo(expId, visitId, subjectID,
				cpi.getCbfSI(), derivedFile);
		ddiList.add(ddi);

		// handle gm.txt
		File gmFile = new File(matFile.getParent(), "gm.txt");
		if (gmFile.exists()) {
			copy2DerivedDataLoc(gmFile, derivedDir);
		}

		for (File resultImgFile : cpi.getResultImageFiles()) {
			File derivedImgFile = copy2DerivedDataLoc(resultImgFile, derivedDir);
			DerivedDataInfo iddi = prepDerivedDataInfo(expId, visitId,
					subjectID, cpi.getCbfSI(), derivedImgFile);
			ddiList.add(iddi);
		}
		// also add brik file if it is not an existing brik file as derived data
		prepDDForSupportAnalyzeSeries(cpi.getCbfBrik(), cpi.getCbfSI(),
				derivedDir, subjectID, expId, visitId, ddiList);

		// include intermediate files generated by CBF process also (04/12/2011)
		if (!cpi.getIntermediateFiles().isEmpty()) {
			File intDir = new File(derivedDir, "intermediate");
			intDir.mkdir();
			Assertion.assertTrue(intDir.isDirectory());
			for (IFileInfo ifi : cpi.getIntermediateFiles()) {
				List<File> files = ifi.getFiles();
				for (File f : files) {
					File derivedDataFile = copy2DerivedDataLoc(f, intDir);
					DerivedDataInfo addi = prepDerivedDataInfo(expId, visitId,
							subjectID, cpi.getCbfSI(), derivedDataFile);
					ddiList.add(addi);
				}
			}
		}

		// include additional result files generated by the CBF process
		// (5/18/2012)
		if (!cpi.getAdditionalResultImageFiles().isEmpty()) {
			for (IFileInfo ifi : cpi.getAdditionalResultImageFiles()) {
				List<File> files = ifi.getFiles();
				for (File f : files) {
					File derivedDataFile = copy2DerivedDataLoc(f, derivedDir);
					DerivedDataInfo addi = prepDerivedDataInfo(expId, visitId,
							subjectID, cpi.getCbfSI(), derivedDataFile);
					ddiList.add(addi);
				}
			}
		}

	}

	protected void prepDDForSupportAnalyzeSeries(File analyzeFile,
			SegmentInfo si, File derivedDir, String subjectID, int expId,
			int visitId, List<DerivedDataInfo> ddiList) throws IOException {
		String prefix = analyzeFile.getName().replaceFirst("\\.\\w+$", "");
		File[] files = analyzeFile.getParentFile().listFiles();
		List<File> analyzeFiles = new ArrayList<File>(2);
		for (File f : files) {
			if (f.getName().startsWith(prefix)
					&& hasAnalyzeExtension(f.getName())) {
				analyzeFiles.add(f);
			}
		}
		Assertion.assertTrue(analyzeFiles.size() == 2);
		for (File f : analyzeFiles) {
			File derivedFile = copy2DerivedDataLoc(f, derivedDir);
			DerivedDataInfo ddi = prepDerivedDataInfo(expId, visitId,
					subjectID, si, derivedFile);
			ddiList.add(ddi);
		}
	}

	public static boolean hasAnalyzeExtension(String filename) {
		filename = filename.toLowerCase();
		return (filename.endsWith(".img") || filename.endsWith(".hdr"));
	}

	protected File copyAnalyzeFilesTo(File analyzeImgFile, File analyzeDIR)
			throws IOException {
		File newAnalyzeFile = new File(analyzeDIR, analyzeImgFile.getName());
		FileUtils.copyFile(analyzeImgFile.getAbsolutePath(),
				newAnalyzeFile.getAbsolutePath());
		String headName = analyzeImgFile.getName();
		headName = headName.replaceFirst("\\.img$", ".hdr");
		File headFile = new File(analyzeImgFile.getParent(), headName);
		Assertion.assertTrue(headFile.exists());
		File newHeadFile = new File(analyzeDIR, headName);
		FileUtils.copyFile(headFile.getAbsolutePath(),
				newHeadFile.getAbsolutePath());

		return newAnalyzeFile;
	}

	private List<CBFProcessInfo> prepare4CBFProcessing(
			List<AnalyzeWrapper> biList, File outDir, File matlabDirFile)
			throws Exception {
		List<CBFProcessInfo> cpiList = new ArrayList<CBFProcessInfo>();
		boolean hasAnat = hasAnatomical();
		AnalyzeWrapper csfBI = findAnalyzeInfo(CBFBIRNConstants.CSF);
		AnalyzeWrapper anatBI = findAnalyzeInfo(CBFBIRNConstants.ANATOMICAL);
		for (AnalyzeWrapper bi : biList) {
			SegmentInfo si = bi.getSi();
			if (CBFBirnHelper.isCBFProcessable(si.getProtocolId())) {
				CBFProcessInfo cpi = null;
				if (hasAnat) {
					cpi = new CBFProcessInfo(bi.getAnalyzeImgFile(),
							csfBI.getAnalyzeImgFile(), null,
							anatBI.getAnalyzeImgFile(), bi.getSi(),
							csfBI.getSi(), null, anatBI.getSi(), outDir);
				} else {
					cpi = new CBFProcessInfo(bi.getAnalyzeImgFile(),
							csfBI.getAnalyzeImgFile(), null, bi.getSi(),
							csfBI.getSi(), null, outDir);
				}
				cpiList.add(cpi);
			}
		}

		return cpiList;
	}

	protected AnalyzeWrapper findAnalyzeInfo(String protocol) {
		for (AnalyzeWrapper bi : this.analyzeImgSetList) {
			if (bi.getSi().getProtocolId().equals(protocol)) {
				return bi;
			}
		}
		return null;
	}

}
