package clinical.web.workflow.cbf.group;

import java.io.File;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;
import org.json.JSONException;
import org.json.JSONObject;

import clinical.server.vo.GroupAnalysisData;
import clinical.utils.Assertion;
import clinical.utils.Executor;
import clinical.utils.FileUtils;
import clinical.utils.GenUtils;
import clinical.web.CBFBIRNConstants;
import clinical.web.Constants;
import clinical.web.IAppConfigService;
import clinical.web.ServiceFactory;
import clinical.web.common.UserInfo;
import clinical.web.common.vo.Factor;
import clinical.web.download.FileBundleConfig;
import clinical.web.download.Packager;
import clinical.web.download.PathWrapper;
import clinical.web.exception.BaseException;
import clinical.web.helpers.EmailHelper;
import clinical.web.scheduler.IJob;
import clinical.web.scheduler.IJobEventListener;
import clinical.web.scheduler.IJobFactory;
import clinical.web.scheduler.JobException;
import clinical.web.scheduler.JobVisitContext;
import clinical.web.scheduler.remote.WFESLocation;
import clinical.web.services.ICBFGroupAnalysisService;
import clinical.web.vo.GroupDerivedDataInfo;
import clinical.web.workflow.cbf.CleanupUtils;

public class BaselineGroupAnalysisJob implements IJob {
	private String id;
	private UserInfo ui;
	private String description;
	private boolean canceled = false;
	private String resultTarFile;
	private String emailTo;
	private String status;
	private GroupAnalysisContext context;
	private File derivedDir;
	protected List<IJobEventListener> listeners = new ArrayList<IJobEventListener>(
			1);
	private Map<String, String> envMap = new HashMap<String, String>(17);
	private String envPath;

	protected Log log = LogFactory.getLog(BaselineGroupAnalysisJob.class);

	public BaselineGroupAnalysisJob(String id, UserInfo ui, String description,
			GroupAnalysisContext context) throws BaseException {
		super();
		this.id = id;
		this.ui = ui;
		this.description = description;
		this.context = context;
		this.emailTo = context.getEmailTo();

		IAppConfigService configService = ServiceFactory.getAppConfigService();
		String dataRoot = configService
				.getParamValue(Constants.CBFBIRN_DATA_ROOT_KEY);
		this.derivedDir = new File(dataRoot, "analysis");
		this.derivedDir = new File(this.derivedDir, "baseline_" + id);

		String bundleName = prepBundleName();
		File f = new File(context.getCacheDir(), bundleName + ".tar.gz");
		this.resultTarFile = f.getAbsolutePath();

	}

	@Override
	public String getID() {
		return this.id;
	}

	@Override
	public String getUser() {
		return this.ui.getPerceivedName();
	}

	@Override
	public String getType() {
		return CBFBIRNConstants.CBF_BASELINE_GROUP_ANALYSIS;
	}

	@Override
	public String getDescription() {
		return this.description;
	}

	@Override
	public int getNumberOfStages() {
		return 1;
	}

	@Override
	public void execute(int stageId) throws JobException {
		try {
			IAppConfigService configService = ServiceFactory
					.getAppConfigService();
			prepareEnvironment(configService);

			ICBFGroupAnalysisService gaService = ServiceFactory
					.getCBFGroupAnalysisService(context.getDbID());
			File augmentedCSVFile = gaService.prepareBaselineAugmentedCSVFile(
					ui, new File(context.getCacheDir()), context.getJiList());
			List<Factor> selectedFactors = context.getSelectedFactors();

			Properties props = new Properties();
			props.setProperty("file.resource.loader.path",
					context.getTemplateDir());
			props.setProperty("runtime.log.logsystem.class",
					"org.apache.velocity.runtime.log.NullLogSystem");
			Velocity.init(props);
			VelocityContext ctx = new VelocityContext();

			File outputDir = new File(this.context.getCacheDir(), "output");
			if (!outputDir.isDirectory()) {
				outputDir.mkdir();
			}

			// TODO specific treatment for sDEMOG_BP (Demographics)

			ctx.put("inputCsv", augmentedCSVFile.getAbsolutePath());
			ctx.put("outdir", outputDir.getAbsolutePath());
			ctx.put("matlabdir", context.getMatlabDir());
			ctx.put("pathType", new Integer(1)); // for baseline
			ctx.put("repeated", context.isRepeatedMeasures() ? new Integer(1)
					: new Integer(0));
			ctx.put("factors", Factor.toFactorsArgString(selectedFactors));
			ctx.put("levels", Factor.toLevelsArgString(selectedFactors));

			StringWriter sw = new StringWriter();
			Template template = Velocity.getTemplate("do_genstats.vm");
			template.merge(ctx, sw);

			System.out.println(sw.toString());
			File matlabDriverScriptFile = new File(outputDir, "do_genstats.m");
			matlabDriverScriptFile.delete();
			FileUtils.save2File(sw.toString(),
					matlabDriverScriptFile.getAbsolutePath());

			Executor executor = new Executor(CBFBIRNConstants.MATLAB_EXEC, true);
			String cmdLine = "-nodesktop -nosplash < "
					+ matlabDriverScriptFile.getAbsolutePath();

			runProcess(executor, cmdLine);

			if (!derivedDir.mkdirs()) {
				log.error("cannot create directory " + this.derivedDir);
				throw new Exception(
						"Cannot create the directory to save the results in!");
			}

			File[] outFiles = outputDir.listFiles();
			List<File> destFiles = new ArrayList<File>(10);
			for (File outFile : outFiles) {
				if (outFile.getName().endsWith(".png")
						|| outFile.getName().endsWith(".txt")) {
					File destFile = new File(derivedDir, outFile.getName());
					FileUtils.copyFile(outFile.getAbsolutePath(),
							destFile.getAbsolutePath());
					destFiles.add(destFile);
				}
			}
			// copy input CSV file to the final output
			File destCSVFile = new File(derivedDir, augmentedCSVFile.getName());
			FileUtils.copyFile(augmentedCSVFile.getAbsolutePath(),
					destCSVFile.getAbsolutePath());
			destFiles.add(destCSVFile);

			postprocess(destFiles);

		} catch (Throwable t) {
			log.error("execute", t);
			cleanupFailedJobDerivedDir();
			throw new JobException(t);
		}
	}

	protected void cleanupFailedJobDerivedDir() {
		if (this.derivedDir != null && this.derivedDir.isDirectory()) {
			FileUtils.deleteRecursively(this.derivedDir);
		}
	}

	protected void postprocess(List<File> destFiles) throws Exception {
		if (isCanceled()) {
			return;
		}
		List<GroupDerivedDataInfo> gddiList = new ArrayList<GroupDerivedDataInfo>(
				9);
		for (File destFile : destFiles) {
			GroupDerivedDataInfo gddi = new GroupDerivedDataInfo(this.id,
					destFile.getAbsolutePath());
			gddiList.add(gddi);
		}
		GroupAnalysisHelper helper = new GroupAnalysisHelper(ui, this.context);

		List<GroupAnalysisData> gadList = helper
				.saveDerivedDataWithJobProvenance(gddiList, this.derivedDir);

		// bundle results for download
		List<GroupAnalysisData> downloadGADList = new ArrayList<GroupAnalysisData>(
				1);
		downloadGADList.addAll(gadList);
		String bundleName = prepBundleName();
		List<PathWrapper> allFiles = new ArrayList<PathWrapper>(1);
		String srcRootDir = this.derivedDir.getAbsolutePath();
		for (GroupAnalysisData gad : downloadGADList) {
			String datauURI = gad.getDatauri();
			String relativePath = FileUtils.getRelativePath(srcRootDir,
					datauURI);
			PathWrapper pw = new PathWrapper(relativePath, srcRootDir);
			allFiles.add(pw);
		}
		// final packaging
		Packager p = new Packager(context.getCacheDir(), bundleName,
				FileBundleConfig.GZIPPED);
		p.includeFiles(allFiles);
		p.pack();
		if (isCanceled()) {
			return;
		}
		handleEmail();
	}

	@Override
	public void saveResults(File file) throws JobException {
		// no op
	}

	@Override
	public String[] getResultsFiles() {
		String[] arr = new String[1];
		arr[0] = this.resultTarFile;
		return arr;
	}

	@Override
	public synchronized void cancel() {
		this.canceled = true;
	}

	@Override
	public synchronized boolean isCanceled() {
		return canceled;
	}

	@Override
	public String getDbID() {
		return this.context.getDbID();
	}

	@Override
	public UserInfo getUserInfo() {
		return this.ui;
	}

	@Override
	public int getDurationType() {
		return SHORT;
	}

	@Override
	public void addJobEventListener(IJobEventListener listener) {
		this.listeners.add(listener);
	}

	@Override
	public void shutdown() {
		this.listeners = null;
	}

	@Override
	public List<JobVisitContext> getJobVisitContextList() {
		// no job visit context list for this type of job
		return new ArrayList<JobVisitContext>(0);
	}

	@Override
	public String getContextAsJSON() {
		try {
			JSONObject js = context.toJSON();
			return js.toString();
		} catch (JSONException e) {
			log.error("getContextAsJSON", e);
			return null;
		}
	}

	@Override
	public IJobFactory getJobFactory() {
		return new GroupAnalysisJob.GroupAnalysisJobFactory();
	}

	@Override
	public synchronized void setStatus(String status) {
		this.status = status;
	}

	@Override
	public synchronized String getStatus() {
		return this.status;
	}

	@Override
	public void cleanup() {
		log.info("job cleanup");
		CleanupUtils.cleanupWorkingDir(getStatus(), context.getCacheDir(), log);
	}

	@Override
	public boolean setWFESLocation(WFESLocation location) {
		return false;
	}

	protected String prepBundleName() {
		return "Baseline_Analysis_" + id;
	}

	protected void runProcess(Executor executor, String cmdLine)
			throws Exception {
		if (isCanceled()) {
			return;
		}
		executor.addEnvParam("PATH", this.envPath);
		for (String ev : this.envMap.keySet()) {
			String value = envMap.get(ev);
			executor.addEnvParam(ev, value);
		}
		setMatlabPrefDirEnvVar(executor);
		executor.showOutput(true);
		executor.execute(cmdLine);

		String output = executor.getOutput();
		System.out.println(output);
	}

	private void setMatlabPrefDirEnvVar(Executor executor) {
		File prefDir = new File(this.context.getCacheDir(), ".matlab");
		if (!prefDir.exists()) {
			prefDir.mkdirs();
		}
		Assertion.assertTrue(prefDir.exists() && prefDir.isDirectory());
		executor.addEnvParam("MATLAB_PREFDIR", prefDir.getAbsolutePath());
	}

	public void setEnvPath(String envPath) {
		this.envPath = envPath;
	}

	public void addEnvironmentVar(String name, String value) {
		envMap.put(name, value);
	}

	protected void prepareEnvironment(IAppConfigService configService) {
		String envPath = configService.getParamValue("cbfbirn.env.path");
		setEnvPath(envPath);
		addEnvironmentVar("FSLDIR", configService.getParamValue("FSLDIR"));
		addEnvironmentVar("FSLOUTPUTTYPE",
				configService.getParamValue("FSLOUTPUTTYPE"));
		addEnvironmentVar("AFNI_PLUGINPATH",
				configService.getParamValue("AFNI_PLUGINPATH"));
		addEnvironmentVar("FSLMULTIFILEQUIT",
				configService.getParamValue("FSLMULTIFILEQUIT"));
		addEnvironmentVar("FSLCONFDIR",
				configService.getParamValue("FSLCONFDIR"));
		addEnvironmentVar("FSLMACHTYPE",
				configService.getParamValue("FSLMACHTYPE"));
	}

	protected void handleEmail() throws Exception {
		System.out.println("handling email...");
		IAppConfigService configService = ServiceFactory.getAppConfigService();

		String emailHost = configService.getParamValue("email.host");
		String emailUser = configService.getParamValue("email.user");
		String emailPwd = configService.getParamValue("email.pwd");
		String emailFrom = configService.getParamValue("email.from");
		boolean canSendEmail = GenUtils.hasEnoughInfoForEmail(emailHost,
				emailFrom);

		if (!canSendEmail) {
			return;
		}
		EmailHelper helper = new EmailHelper(emailHost, emailUser, emailPwd,
				emailFrom);
		String toEmail = this.emailTo;
		System.out.println("handleEmail:: toEmail:" + toEmail);
		if (toEmail == null) {
			log.error("Cannot send email to:" + toEmail);
			return;
		}
		String subject = "Your Baseline Group Analyis with Job ID '" + this.id
				+ "' has finished";
		Map<String, String> paramMap = new HashMap<String, String>(7);
		paramMap.put("email", toEmail);
		paramMap.put("username", ui.getName());
		paramMap.put("jobId", this.id);
		try {
			helper.sendEmail(toEmail, subject, paramMap,
					this.context.getTemplateDir(), "blga_wf_finished.vm", false);
		} catch (Throwable t) {
			log.error("sendEmail", t);
		}
	}
}
