package clinical.web.workflow.cbf.group;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.json.JSONException;
import org.json.JSONObject;

import clinical.server.vo.GroupAnalysisData;
import clinical.utils.FileUtils;
import clinical.utils.GenUtils;
import clinical.web.CBFBIRNConstants;
import clinical.web.Constants;
import clinical.web.IAppConfigService;
import clinical.web.ServiceFactory;
import clinical.web.common.UserInfo;
import clinical.web.download.FileBundleConfig;
import clinical.web.download.Packager;
import clinical.web.download.PathWrapper;
import clinical.web.exception.BaseException;
import clinical.web.helpers.EmailHelper;
import clinical.web.scheduler.IJob;
import clinical.web.scheduler.IJobEventListener;
import clinical.web.scheduler.IJobFactory;
import clinical.web.scheduler.JobException;
import clinical.web.scheduler.JobRecord;
import clinical.web.scheduler.JobVisitContext;
import clinical.web.scheduler.remote.WFESLocation;
import clinical.web.vo.GroupDerivedDataInfo;
import clinical.web.workflow.cbf.AFNIBrik;
import clinical.web.workflow.cbf.CleanupUtils;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */
public class GroupAnalysisJob implements IJob {
	private String id;
	private UserInfo ui;
	private String description;
	private boolean canceled = false;
	private String resultTarFile;
	private String emailTo;
	private String status;
	private GroupAnalysisContext context;
	private GroupAnalysisHelper helper;
	private String dataRoot;
	private File derivedDir;
	private SSGAProcessInfo ssgapi;
	protected List<IJobEventListener> listeners = new ArrayList<IJobEventListener>(
			1);
	protected Log log = LogFactory.getLog(GroupROIAnalysisJob.class);

	public GroupAnalysisJob(String id, UserInfo ui, String description,
			GroupAnalysisContext context, String emailTo) throws BaseException {
		this.id = id;
		this.ui = ui;
		this.description = description;
		this.emailTo = emailTo;
		this.context = context;
		this.helper = new GroupAnalysisHelper(ui, context);

		IAppConfigService configService = ServiceFactory.getAppConfigService();

		this.dataRoot = configService
				.getParamValue(Constants.CBFBIRN_DATA_ROOT_KEY);

		this.derivedDir = new File(dataRoot, "analysis");
		this.derivedDir = new File(this.derivedDir, "standard_space_" + id);
		if (!derivedDir.mkdirs()) {
			log.error("cannot create directory " + this.derivedDir);
			throw new BaseException(
					"Cannot create the directory to save the results in!");
		}

		String bundleName = "Standard_Space_Analysis_" + id;
		File f = new File(context.getCacheDir(), bundleName + ".tar.gz");
		this.resultTarFile = f.getAbsolutePath();
	}

	@Override
	public String getID() {
		return this.id;
	}

	@Override
	public String getUser() {
		return this.ui.getPerceivedName();
	}

	@Override
	public String getType() {
		return CBFBIRNConstants.CBF_STANDARD_SPACE_ANALYSIS;
	}

	@Override
	public String getDescription() {
		return this.description;
	}

	@Override
	public int getNumberOfStages() {
		return 1;
	}

	@Override
	public void execute(int stageId) throws JobException {
		try {
			IAppConfigService configService = ServiceFactory
					.getAppConfigService();

			prepareEnvironment(configService);
			File inputRootDir = helper.prepareInputData();
			File inputCSVFile = new File(inputRootDir, "input.csv");
			File assessmentCSVFile = new File(inputRootDir, "assessment.csv");
			File outCSVFile = new File(this.derivedDir,
					"standard_space_summary_" + this.id + ".csv");

			this.ssgapi = new SSGAProcessInfo(inputCSVFile, outCSVFile,
					assessmentCSVFile);

			helper.handleStandardSpaceAnalysis(this.ssgapi,
					new File(context.getMatlabDir()), this.derivedDir);
			postprocess();

		} catch (Exception e) {
			cleanupFailedJobDerivedDir();
			throw new JobException(e);
		}
	}

	protected void cleanupFailedJobDerivedDir() {
		if (this.derivedDir != null && this.derivedDir.isDirectory()) {
			FileUtils.deleteRecursively(this.derivedDir);
		}
	}

	protected void postprocess() throws Exception {
		if (isCanceled()) {
			return;
		}
		List<GroupDerivedDataInfo> gddiList = new ArrayList<GroupDerivedDataInfo>(
				9);
		GroupDerivedDataInfo gddi = new GroupDerivedDataInfo(this.id, ssgapi
				.getOutCSVFile().getAbsolutePath());
		gddiList.add(gddi);
		for (AFNIBrik brik : ssgapi.getResultBriks()) {
			gddiList.add(new GroupDerivedDataInfo(this.id, brik.getBrikFile()
					.getAbsolutePath()));
			gddiList.add(new GroupDerivedDataInfo(this.id, brik.getHeaderFile()
					.getAbsolutePath()));
		}
		for (File resultFile : ssgapi.getResultFiles()) {
			gddiList.add(new GroupDerivedDataInfo(this.id, resultFile
					.getAbsolutePath()));
		}

		for (AFNIBrik brik : ssgapi.getIntermediateTLRCBriks()) {
			gddiList.add(new GroupDerivedDataInfo(this.id, brik.getBrikFile()
					.getAbsolutePath()));
			gddiList.add(new GroupDerivedDataInfo(this.id, brik.getHeaderFile()
					.getAbsolutePath()));
		}
		// save group analysis data
		List<GroupAnalysisData> gadList = helper
				.saveDerivedDataWithJobProvenance(gddiList, this.derivedDir);
		// bundle results for download
		List<GroupAnalysisData> downloadGADList = new ArrayList<GroupAnalysisData>(
				1);
		downloadGADList.addAll(gadList);

		String bundleName = "Standard_Space_Analysis_" + id;
		List<PathWrapper> allFiles = new ArrayList<PathWrapper>(1);
		String srcRootDir = this.derivedDir.getAbsolutePath();
		for (GroupAnalysisData gad : downloadGADList) {
			String datauURI = gad.getDatauri();
			String relativePath = FileUtils.getRelativePath(srcRootDir,
					datauURI);
			PathWrapper pw = new PathWrapper(relativePath, srcRootDir);
			allFiles.add(pw);
		}

		// final packaging
		Packager p = new Packager(context.getCacheDir(), bundleName,
				FileBundleConfig.GZIPPED);
		p.includeFiles(allFiles);
		p.pack();
		if (isCanceled()) {
			return;
		}
		handleEmail();

	}

	@Override
	public void saveResults(File file) throws JobException {
		// no op
	}

	@Override
	public String[] getResultsFiles() {
		String[] arr = new String[1];
		arr[0] = this.resultTarFile;
		return arr;
	}

	@Override
	public synchronized void cancel() {
		this.canceled = true;
	}

	@Override
	public synchronized boolean isCanceled() {
		return canceled;
	}

	@Override
	public String getDbID() {
		return this.context.getDbID();
	}

	@Override
	public UserInfo getUserInfo() {
		return this.ui;
	}

	@Override
	public int getDurationType() {
		return LONG;
	}

	@Override
	public void addJobEventListener(IJobEventListener listener) {
		this.listeners.add(listener);
	}

	@Override
	public void shutdown() {
		this.listeners = null;
	}

	@Override
	public List<JobVisitContext> getJobVisitContextList() {
		// no job visit context list for this type of job
		return new ArrayList<JobVisitContext>(0);
	}

	@Override
	public String getContextAsJSON() {
		try {
			JSONObject js = context.toJSON();
			return js.toString();
		} catch (JSONException e) {
			log.error("getContextAsJSON", e);
			return null;
		}
	}

	@Override
	public IJobFactory getJobFactory() {
		return new GroupAnalysisJobFactory();
	}

	@Override
	public synchronized void setStatus(String status) {
		this.status = status;
	}

	@Override
	public synchronized String getStatus() {
		return this.status;
	}

	@Override
	public void cleanup() {
		log.info("job cleanup");
		CleanupUtils.cleanupWorkingDir(getStatus(), context.getCacheDir(), log);
	}

	@Override
	public boolean setWFESLocation(WFESLocation location) {
		return false;
	}

	protected void prepareEnvironment(IAppConfigService configService) {
		String envPath = configService.getParamValue("cbfbirn.env.path");
		helper.setEnvPath(envPath);
		helper.addEnvironmentVar("FSLDIR",
				configService.getParamValue("FSLDIR"));
		helper.addEnvironmentVar("FSLOUTPUTTYPE",
				configService.getParamValue("FSLOUTPUTTYPE"));
		helper.addEnvironmentVar("AFNI_PLUGINPATH",
				configService.getParamValue("AFNI_PLUGINPATH"));
		helper.addEnvironmentVar("FSLMULTIFILEQUIT",
				configService.getParamValue("FSLMULTIFILEQUIT"));
		helper.addEnvironmentVar("FSLCONFDIR",
				configService.getParamValue("FSLCONFDIR"));
		helper.addEnvironmentVar("FSLMACHTYPE",
				configService.getParamValue("FSLMACHTYPE"));
	}

	protected void handleEmail() throws Exception {
		System.out.println("handling email...");
		IAppConfigService configService = ServiceFactory.getAppConfigService();

		String emailHost = configService.getParamValue("email.host");
		String emailUser = configService.getParamValue("email.user");
		String emailPwd = configService.getParamValue("email.pwd");
		String emailFrom = configService.getParamValue("email.from");
		boolean canSendEmail = GenUtils.hasEnoughInfoForEmail(emailHost, emailFrom);
		
		if (!canSendEmail) {
			return;
		}
		EmailHelper helper = new EmailHelper(emailHost, emailUser, emailPwd,
				emailFrom);
		String toEmail = this.emailTo;
		System.out.println("handleEmail:: toEmail:" + toEmail);
		if (toEmail == null) {
			log.error("Cannot send email to:" + toEmail);
			return;
		}
		String subject = "Your Standard Space Group Analyis with Job ID '"
				+ this.id + "' has finished";
		Map<String, String> paramMap = new HashMap<String, String>(7);
		paramMap.put("email", toEmail);
		paramMap.put("username", ui.getName());
		paramMap.put("jobId", this.id);
		try {
			helper.sendEmail(toEmail, subject, paramMap,
					this.context.getTemplateDir(), "ssga_wf_finished.vm", false);
		} catch (Throwable t) {
			log.error("sendEmail", t);
		}
	}

	public static class GroupAnalysisJobFactory implements IJobFactory {

		@Override
		public IJob create(JobRecord jr) throws JobException {
			if (jr == null) {
				return null;
			}
			String ctxStr = jr.getContext();
			if (ctxStr == null || ctxStr.trim().length() == 0) {
				return null;
			}

			try {
				JSONObject js = new JSONObject(jr.getContext());
				if (js.has("jiList")) {
					GroupAnalysisContext ctx = GroupAnalysisContext
							.fromJSON(js);
					UserInfo jobUI = new UserInfo(jr.getUser(), null, null);
					GroupAnalysisJob job = new GroupAnalysisJob(jr.getJobID(),
							jobUI, jr.getDescription(), ctx, ctx.getEmailTo());
					return job;
				}
			} catch (Throwable t) {
				throw new JobException(t);
			}
			return null;
		}

	}

}
