package clinical.web.workflow.cbf.group;

import static clinical.web.CBFBIRNConstants.MATLAB_EXEC;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;

import clinical.server.vo.Experiment;
import clinical.server.vo.GroupAnalysisData;
import clinical.server.vo.Jobs;
import clinical.utils.Assertion;
import clinical.utils.DateTimeUtils;
import clinical.utils.Executor;
import clinical.utils.FileUtils;
import clinical.utils.GenUtils;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.UserInfo;
import clinical.web.common.vo.Factor;
import clinical.web.exception.BaseException;
import clinical.web.services.BatchAssessmentServiceImpl.CSVTable;
import clinical.web.services.BatchAssessmentServiceImpl.CSVTableRow;
import clinical.web.services.CBFROILookupService;
import clinical.web.services.IBatchAssessmentService;
import clinical.web.services.ICBFROIGroupAnalysisService;
import clinical.web.services.IJobProvenanceService;
import clinical.web.vo.CBFROIJobAssociation;
import clinical.web.vo.CBFROIJobAssociation.JobVisitInfo;
import clinical.web.vo.GroupDerivedDataInfo;
import clinical.web.vo.JobProvenanceInfo;
import clinical.web.vo.JobProvenanceInfo.JobProvenanceParamInfo;
import clinical.web.vo.SubjectAsScoreValueSummary;
import clinical.web.workflow.common.WFGenUtils;

public class GroupROIAnalysisJobHelper {
	protected UserInfo ui;
	protected GroupROIAnalysisContext context;
	protected String templateDir;
	protected String envPath;
	protected Map<String, String> envMap = new HashMap<String, String>(17);
	private boolean canceled = false;
	// protected static final String MATLAB_EXEC =
	// "/apps/matlabr2011a/bin/matlab";
	protected IJobProvenanceService jpmService;
	protected ICBFROIGroupAnalysisService crgaService;
	IDBCache dbCache;

	protected static Log _log = LogFactory
			.getLog(GroupROIAnalysisJobHelper.class);

	public GroupROIAnalysisJobHelper(UserInfo ui,
			GroupROIAnalysisContext context, String templateDir)
			throws BaseException {
		super();
		this.ui = ui;
		this.context = context;
		this.templateDir = templateDir;
		jpmService = ServiceFactory.getJobProvenanceService(context.getDbID());
		crgaService = ServiceFactory.getCBFROIGroupAnalysisService(context
				.getDbID());
		dbCache = ServiceFactory.getDBCache(context.getDbID());
	}

	public String getEnvPath() {
		return envPath;
	}

	public void setEnvPath(String envPath) {
		this.envPath = envPath;
		_log.info("envPath set:" + envPath);
	}

	public void addEnvironmentVar(String name, String value) {
		envMap.put(name, value);
	}

	public void setCanceled(boolean canceled) {
		this.canceled = canceled;
	}

	public boolean isCanceled() {
		return canceled;
	}

	public File prepareInputData() throws Exception {
		File inputRootDir = new File(context.getCacheDir(), "input");
		inputRootDir.mkdir();
		crgaService.prepareROIProcessInputData(ui, inputRootDir,
				context.getAssocList());
		return inputRootDir;
	}

	public void handleROIGroupAnalysis(ROIProcessInfo rpi, File matlabDir)
			throws Exception {

		File outputDir = new File(this.context.getCacheDir(), "output");
		if (!outputDir.isDirectory()) {
			outputDir.mkdir();
		}

		Properties p = new Properties();
		p.setProperty("file.resource.loader.path", this.templateDir);
		p.setProperty("runtime.log.logsystem.class",
				"org.apache.velocity.runtime.log.NullLogSystem");
		Velocity.init(p);
		VelocityContext ctx = new VelocityContext();

		ctx.put("inputCsv", rpi.getInputCSVFile().getAbsolutePath());
		ctx.put("roiLabelMatFile", rpi.getRoiMatFile().getAbsolutePath());
		ctx.put("outdir", outputDir.getAbsolutePath());
		ctx.put("matlabdir", matlabDir);

		StringWriter sw = new StringWriter();
		Template template = Velocity.getTemplate("genroistats.vm");
		template.merge(ctx, sw);

		System.out.println(sw.toString());
		File matlabDriverScriptFile = new File(outputDir, "genroistats.m");
		matlabDriverScriptFile.delete();
		FileUtils.save2File(sw.toString(),
				matlabDriverScriptFile.getAbsolutePath());

		Executor executor = new Executor(MATLAB_EXEC, true);
		String cmdLine = "-nodesktop -nosplash < "
				+ matlabDriverScriptFile.getAbsolutePath();

		runProcess(executor, cmdLine);

		File[] outFiles = outputDir.listFiles();
		File roiCSVFile = null;
		for (File outFile : outFiles) {
			if (outFile.getName().equals("ROI_Summary.csv")) {
				roiCSVFile = outFile;
				break;
			}
		}
		Assertion.assertNotNull(roiCSVFile);

		// combine with clinical assessment data
		// replace roi variable names
		augmentROIStatsCSVFile(roiCSVFile, rpi.getOutCSVFile());

		File tmpOutCSVFile = new File(outputDir, rpi.getOutCSVFile().getName());
		if (!rpi.getOutCSVFile().equals(tmpOutCSVFile)) {
			FileUtils.copyFile(rpi.getOutCSVFile().getAbsolutePath(),
					tmpOutCSVFile.getAbsolutePath());
		}

		// now run genstats path 2

		List<Factor> selectedFactors = context.getSelectedFactors();

		Velocity.init(p);
		ctx = new VelocityContext();

		ctx.put("inputCsv", rpi.getOutCSVFile().getAbsolutePath());
		ctx.put("roiLabelMatFile", rpi.getRoiMatFile().getAbsolutePath());
		ctx.put("outdir", outputDir.getAbsolutePath());
		ctx.put("matlabdir", matlabDir);
		ctx.put("pathType", new Integer(2)); // for native space ROI path
		ctx.put("repeated", context.isRepeatedMeasures() ? new Integer(1)
				: new Integer(0));
		ctx.put("factors", Factor.toFactorsArgString(selectedFactors));
		ctx.put("levels", Factor.toLevelsArgString(selectedFactors));

		sw = new StringWriter();
		template = Velocity.getTemplate("do_genstats.vm");
		template.merge(ctx, sw);

		System.out.println(sw.toString());
		matlabDriverScriptFile = new File(outputDir, "do_genstats.m");
		matlabDriverScriptFile.delete();
		FileUtils.save2File(sw.toString(),
				matlabDriverScriptFile.getAbsolutePath());

		executor = new Executor(MATLAB_EXEC, true);
		cmdLine = "-nodesktop -nosplash < "
				+ matlabDriverScriptFile.getAbsolutePath();

		runProcess(executor, cmdLine);

		File derivedDir = rpi.getOutCSVFile().getParentFile();
		outFiles = outputDir.listFiles();
		for (File outFile : outFiles) {
			String fname = outFile.getName();
			if (fname.equals("path2_summary.csv")) {
				File derivedFile = new File(derivedDir, fname);
				FileUtils.copyFile(outFile.getAbsolutePath(),
						derivedFile.getAbsolutePath());
				rpi.setTableCSVFile(derivedFile);
			} else if (fname.endsWith(".png")) {
				File derivedFile = new File(derivedDir, fname);
				FileUtils.copyFile(outFile.getAbsolutePath(),
						derivedFile.getAbsolutePath());
				rpi.addResultImageFile(derivedFile);
			} else if (fname.equals("summary_report.txt")) {
				File derivedFile = new File(derivedDir, fname);
				FileUtils.copyFile(outFile.getAbsolutePath(),
						derivedFile.getAbsolutePath());
				rpi.setReportFile(derivedFile);
			}
		}

	}

	public void handleROIStats(ROIProcessInfo rpi, File matlabDir)
			throws Exception {
		Properties p = new Properties();
		p.setProperty("file.resource.loader.path", this.templateDir);
		p.setProperty("runtime.log.logsystem.class",
				"org.apache.velocity.runtime.log.NullLogSystem");
		Velocity.init(p);
		VelocityContext ctx = new VelocityContext();

		File outputDir = new File(this.context.getCacheDir(), "output");
		if (!outputDir.isDirectory()) {
			outputDir.mkdir();
		}

		// copy roi mat file to the input/work directory
		// FileUtils.copyFile(rpi.getRoiMatFile().getAbsolutePath(), new
		// File(rpi
		// .getInputCSVFile().getParentFile(), rpi.getRoiMatFile()
		// .getName()).getAbsolutePath());

		ctx.put("inputCsv", rpi.getInputCSVFile().getAbsolutePath());
		ctx.put("roiLabelMatFile", rpi.getRoiMatFile().getAbsolutePath());
		ctx.put("outdir", outputDir.getAbsolutePath());
		ctx.put("matlabdir", matlabDir);

		StringWriter sw = new StringWriter();
		Template template = Velocity.getTemplate("genroistats.vm");
		template.merge(ctx, sw);

		System.out.println(sw.toString());
		File matlabDriverScriptFile = new File(outputDir, "genroistats.m");
		matlabDriverScriptFile.delete();
		FileUtils.save2File(sw.toString(),
				matlabDriverScriptFile.getAbsolutePath());

		Executor executor = new Executor(MATLAB_EXEC, true);
		String cmdLine = "-nodesktop -nosplash < "
				+ matlabDriverScriptFile.getAbsolutePath();

		runProcess(executor, cmdLine);

		File[] outFiles = outputDir.listFiles();
		File roiCSVFile = null;
		for (File outFile : outFiles) {
			if (outFile.getName().equals("ROI_Summary.csv")) {
				roiCSVFile = outFile;
				break;
			}
		}
		Assertion.assertNotNull(roiCSVFile);

		// combine with clinical assessment data
		// replace roi variable names

		augmentROIStatsCSVFile(roiCSVFile, rpi.getOutCSVFile());
	}

	public void augmentROIStatsCSVFile(File roiCSVFile, File outROICSVFile)
			throws Exception {
		CSVTable asCSVTable = loadAssessmentData();

		List<List<String>> rows = WFGenUtils.loadCSV(roiCSVFile);

		List<CSVColumnMapping> headerMappings = prepHeaderMappings(rows.get(0),
				asCSVTable);

		Map<String, JobVisitInfo> jviMap = new HashMap<String, JobVisitInfo>();
		List<CBFROIJobAssociation> assocList = this.context.getAssocList();
		for (CBFROIJobAssociation assoc : assocList) {
			for (JobVisitInfo jvi : assoc.getCandidates()) {
				String key = jvi.getSubjectID() + ":" + jvi.getJobID();
				jviMap.put(key, jvi);
			}
		}

		List<String> augHeadRow = new ArrayList<String>(headerMappings.size());
		for (CSVColumnMapping mapping : headerMappings) {
			augHeadRow.add(mapping.destKey);
		}

		List<List<String>> augRows = new ArrayList<List<String>>(rows.size());

		augRows.add(augHeadRow);

		for (int i = 1; i < rows.size(); i++) {
			List<String> row = rows.get(i);
			String subjectID = row.get(0);
			String jobID = row.get(1);
			String key = subjectID + ":" + jobID;
			JobVisitInfo jvi = jviMap.get(key);
			Assertion.assertNotNull(jvi);
			List<String> augRow = new ArrayList<String>(headerMappings.size());
			augRows.add(augRow);

			String expName = jvi.getExpName();
			int visitID = jvi.getVisitID();
			@SuppressWarnings("unused")
			String visitDate = jvi.getVisitDate();
			String asKey = subjectID + ":" + expName + ":" + visitID;
			CSVTableRow tableRow = asCSVTable.getRowMap().get(asKey);
			for (CSVColumnMapping mapping : headerMappings) {
				if (mapping.srcType == CSVColumnMapping.OTHER) {
					if (mapping.srcKey.equals("SubjectID")) {
						augRow.add(subjectID);
					} else if (mapping.srcKey.equals("JobID")) {
						augRow.add(jobID);
					} else if (mapping.srcKey.equals("Project")) {
						augRow.add(expName);
					} else if (mapping.srcKey.equals("VisitID")) {
						augRow.add(String.valueOf(visitID));
					}
				} else if (mapping.srcType == CSVColumnMapping.ROI) {
					String srcValue = row.get(mapping.srcIdx);
					srcValue = srcValue.equals("NaN") ? "." : srcValue;
					augRow.add(srcValue);
				} else {
					// CSVColumnMapping.ASSESSMENT
					if (tableRow != null) {
						String value = tableRow.getValue(mapping.srcKey);
						value = WFGenUtils.replaceCommas(value, ";");
						augRow.add(value != null ? value : ".");
					} else {
						// missing value
						augRow.add(".");
					}
				}
			}

		}// i

		BufferedWriter out = null;
		try {
			out = new BufferedWriter(new FileWriter(outROICSVFile));
			for (List<String> augRow : augRows) {
				String csvRow = GenUtils.join(augRow, ",");
				out.write(csvRow);
				out.newLine();
			}
		} finally {
			FileUtils.close(out);
		}
	}

	protected List<CSVColumnMapping> prepHeaderMappings(List<String> headerRow,
			CSVTable asCSVTable) {
		// SubjectID, JobID, Exp Name, Visit ID, <subject Information>, <ROI
		// variables>, <rest of the assessments>

		List<CSVColumnMapping> mappings = new ArrayList<CSVColumnMapping>(
				headerRow.size()
						+ asCSVTable.getHeader().getScoreHeaders().size() + 2);
		CBFROILookupService crlService = CBFROILookupService.getInstance();
		mappings.add(new CSVColumnMapping(headerRow.get(0), headerRow.get(0),
				0, 0, CSVColumnMapping.OTHER));
		mappings.add(new CSVColumnMapping(headerRow.get(1), headerRow.get(1),
				1, 1, CSVColumnMapping.OTHER));
		mappings.add(new CSVColumnMapping("Project", "Project", 2, 2,
				CSVColumnMapping.OTHER));
		mappings.add(new CSVColumnMapping("VisitID", "VisitID", 3, 3,
				CSVColumnMapping.OTHER));
		List<String> scoreHeaders = asCSVTable.getHeader().getScoreHeaders();
		int startIdx = -1;
		int i = 0;
		for (String scoreHeader : scoreHeaders) {
			if (scoreHeader.startsWith("Subject_Information__")) {
				startIdx = i;
				break;
			}
			i++;
		}
		int destOffset = 4;
		if (startIdx != -1) {
			int j = 0;
			for (i = startIdx; i < scoreHeaders.size(); i++) {
				String scoreHeader = scoreHeaders.get(i);
				if (!scoreHeader.startsWith("Subject_Information__")) {
					destOffset += j;
					break;
				}
				mappings.add(new CSVColumnMapping(scoreHeader, scoreHeader, i,
						destOffset + j, CSVColumnMapping.ASSESSMENT));
				j++;
			}
		}

		// map the ROIs
		int j = 0;
		for (i = 2; i < headerRow.size(); i++) {
			String roiHeader = headerRow.get(i);
			int idx = roiHeader.lastIndexOf('_');
			String roiNoStr = roiHeader
					.substring(roiHeader.lastIndexOf('_') + 1);
			int roiNo = GenUtils.toInt(roiNoStr, -1);
			Assertion.assertTrue(roiNo != -1);
			String label = crlService.getLabel(CBFROILookupService.FREESURFER,
					roiNo);
			if (label != null) {
				label = label.replaceAll("\\s+", "_");
				String destKey = roiHeader.substring(0, idx) + "__" + label;
				mappings.add(new CSVColumnMapping(roiHeader, destKey, i,
						destOffset + j, CSVColumnMapping.ROI));
			} else {
				mappings.add(new CSVColumnMapping(roiHeader, roiHeader, i,
						destOffset + j, CSVColumnMapping.ROI));
			}
			j++;
		}
		destOffset += j;
		// now the rest of the assessments
		j = 0;
		for (i = 0; i < scoreHeaders.size(); i++) {
			String scoreHeader = scoreHeaders.get(i);
			if (!scoreHeader.startsWith("Subject_Information__")) {
				mappings.add(new CSVColumnMapping(scoreHeader, scoreHeader, i,
						destOffset + j, CSVColumnMapping.ASSESSMENT));
				j++;
			}
		}

		return mappings;
	}

	public static class CSVColumnMapping {
		String srcKey;
		String destKey;
		/** column index in the source */
		int srcIdx;
		int destIdx;
		int srcType = OTHER;
		public static final int ROI = 1;
		public static final int ASSESSMENT = 2;
		public static final int OTHER = 3;

		public CSVColumnMapping(String srcKey, String destKey, int srcIdx,
				int destIdx, int srcType) {
			super();
			this.srcKey = srcKey;
			this.destKey = destKey;
			this.srcIdx = srcIdx;
			this.destIdx = destIdx;
			this.srcType = srcType;
		}

	}

	protected CSVTable loadAssessmentData() throws Exception {
		IBatchAssessmentService bas = ServiceFactory
				.getBatchAssessmentService(this.context.getDbID());
		List<CBFROIJobAssociation> assocList = this.context.getAssocList();
		List<Experiment> experiments = dbCache.getExperiments(ui, false);
		Map<String, Experiment> expMap = new HashMap<String, Experiment>();
		for (Experiment exp : experiments) {
			expMap.put(exp.getName(), exp);
		}

		Map<Integer, List<String>> expSubjectIDListMap = new HashMap<Integer, List<String>>();
		Set<Integer> seenExpIdSet = new HashSet<Integer>();
		for (CBFROIJobAssociation assoc : assocList) {
			String subjectId = assoc.getSubjectId();
			seenExpIdSet.clear();
			for (JobVisitInfo jvi : assoc.getCandidates()) {
				Experiment exp = expMap.get(jvi.getExpName());
				if (exp != null) {
					Integer expId = new Integer(exp.getUniqueid().intValue());
					List<String> list = expSubjectIDListMap.get(expId);
					if (list == null) {
						list = new LinkedList<String>();
						expSubjectIDListMap.put(expId, list);
					}
					if (!seenExpIdSet.contains(expId)) {
						list.add(subjectId);
						seenExpIdSet.add(expId);
					}
				}
			}
		}

		List<SubjectAsScoreValueSummary> sasvsList = bas
				.getAllAssessmentValues(ui, expSubjectIDListMap);

		CSVTable csvTable = bas.prepCSVTable(ui, sasvsList);
		return csvTable;
	}

	protected List<GroupAnalysisData> saveDerivedDataWithJobProvenance(
			List<GroupDerivedDataInfo> gddiList, File derivedDir)
			throws Exception {
		String gaName = context.getGaName() != null ? context.getGaName() : "";
		JobProvenanceInfo jpi = new JobProvenanceInfo(gaName, "");
		jpi.setDataURI(derivedDir.getAbsolutePath());
		Jobs theJob = this.jpmService.getTheJob(ui, context.getJobID(),
				ui.getPerceivedName());
		jpi.setJob(theJob);
		// add Finish Date also as provenance data
		String value = DateTimeUtils.formatDate(theJob.getJobenddate());
		JobProvenanceParamInfo jppi = new JobProvenanceParamInfo("Finish Date",
				value);
		jpi.addParam(jppi);

		List<GroupAnalysisData> gadList = crgaService
				.addDerivedDataWithJobProvenance(ui, gddiList, jpi);
		return gadList;
	}

	protected void runProcess(Executor executor, String cmdLine)
			throws Exception {
		if (isCanceled()) {
			return;
		}
		executor.addEnvParam("PATH", this.envPath);
		for (String ev : this.envMap.keySet()) {
			String value = envMap.get(ev);
			executor.addEnvParam(ev, value);
		}
		setMatlabPrefDirEnvVar(executor);
		executor.showOutput(true);
		executor.execute(cmdLine);

		String output = executor.getOutput();
		System.out.println(output);
	}

	private void setMatlabPrefDirEnvVar(Executor executor) {
		File prefDir = new File(this.context.getCacheDir(), ".matlab");
		if (!prefDir.exists()) {
			prefDir.mkdirs();
		}
		Assertion.assertTrue(prefDir.exists() && prefDir.isDirectory());
		executor.addEnvParam("MATLAB_PREFDIR", prefDir.getAbsolutePath());
	}
}
