package clinical.test.framework;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import junit.framework.Assert;

import org.dbunit.dataset.Column;
import org.dbunit.dataset.DataSetException;
import org.dbunit.dataset.ITable;
import org.dbunit.dataset.ITableMetaData;
import org.dbunit.dataset.datatype.DataType;
import org.dbunit.dataset.datatype.UnknownDataType;

/**
 * @author I. Burak Ozyurt
 * @version $Id: DBAssertion.java 62 2009-05-29 23:54:50Z bozyurt $
 */
public class DBAssertion {
	protected static ColumnComparator COLUMN_COMPARATOR = new ColumnComparator();

	public static void assertFullyContains(ITable expectedTable,
			ITable actualTable) throws DataSetException {
		if (expectedTable == actualTable)
			return;

		ITableMetaData expectedMetaData = expectedTable.getTableMetaData();
		ITableMetaData actualMetaData = actualTable.getTableMetaData();
		String expectedTableName = expectedMetaData.getTableName();

		// Verify columns
		Column[] expectedColumns = getSortedColumns(expectedMetaData);
		Column[] actualColumns = getSortedColumns(actualMetaData);
		Assert.assertEquals("column count (table=" + expectedTableName + ")",
				expectedColumns.length, actualColumns.length);
		int idx = 0;
		for (Column ec : expectedColumns) {
			String expectedName = ec.getColumnName();
			String actualName = actualColumns[idx].getColumnName();
			if (!expectedName.equalsIgnoreCase(actualName)) {
				Assert.fail("expected columns "
						+ getColumnNamesAsString(expectedColumns) + " but was "
						+ getColumnNamesAsString(actualColumns) + " (table="
						+ expectedName + ")");
			}
			++idx;
		}

		// make sure actual table has >= rows than the expected table
		Assert.assertTrue("Actual table row count (table=" + expectedTableName
				+ ") was less than expected",
				expectedTable.getRowCount() <= actualTable.getRowCount());

		Map<String, Map<String, List<Integer>>> actualTableMap = new HashMap<String, Map<String, List<Integer>>>();
		for (Column ec : expectedColumns) {
			Map<String, List<Integer>> atColValueMap = actualTableMap.get(ec
					.getColumnName());
			if (atColValueMap == null) {
				atColValueMap = new HashMap<String, List<Integer>>();
				actualTableMap.put(ec.getColumnName(), atColValueMap);
			}
			for (int i = 0; i < actualTable.getRowCount(); i++) {
				Object value = actualTable.getValue(i, ec.getColumnName());
				String actualValueStr = value != null ? value.toString() : "nil";

				List<Integer> idxs = atColValueMap.get(actualValueStr);
				if (idxs == null) {
					idxs = new ArrayList<Integer>(1);
					atColValueMap.put(actualValueStr, idxs);
				}
				idxs.add(i);
			}
		}
		// values as strings
		for (int i = 0; i < expectedTable.getRowCount(); i++) {
			boolean first = true;
			int actualRowIdx = -1;
			for (int j = 0; j < expectedColumns.length; j++) {
				Column expectedColumn = expectedColumns[j];
				Column actualColumn = actualColumns[j];

				String columnName = expectedColumn.getColumnName();
				Object expectedValue = expectedTable.getValue(i, columnName);

				Map<String, List<Integer>> atColValueMap = actualTableMap
						.get(columnName);
				List<Integer> idxList = atColValueMap.get(expectedValue);
				if (idxList == null) {
					Assert.fail("value (table=" + expectedTableName + ", row="
							+ i + ", col=" + columnName + "): <"
							+ expectedValue + "> not found!");
				} else {
					if (first) {
						actualRowIdx = idxList.get(0);
						first = false;
					}
				}

				Object actualValue = actualTable.getValue(actualRowIdx,
						columnName);
				DataType dataType = getComparisonDataType(expectedTableName,
						expectedColumn, actualColumn);
				if (dataType.compare(expectedValue, actualValue) != 0) {
					Assert
							.fail("value (table=" + expectedTableName
									+ ", row=" + i + ", col=" + columnName
									+ "): expected:<" + expectedValue
									+ "> but was:<" + actualValue + ">");
				}
			}

		}
	}

	public static DataType getComparisonDataType(String tableName,
			Column expectedColumn, Column actualColumn) {
		DataType expectedDataType = expectedColumn.getDataType();
		DataType actualDataType = actualColumn.getDataType();

		// The two columns have different data type
		if (!expectedDataType.getClass().isInstance(actualDataType)) {
			// Expected column data type is unknown, use actual column data type
			if (expectedDataType instanceof UnknownDataType) {
				return actualDataType;
			}

			// Actual column data type is unknown, use expected column data type
			if (actualDataType instanceof UnknownDataType) {
				return expectedDataType;
			}

			// Impossible to determine which data type to use
			Assert.fail("Incompatible data types: " + expectedDataType + ", "
					+ actualDataType + " (table=" + tableName + ", col="
					+ expectedColumn.getColumnName() + ")");
		}

		// Both columns have same data type, return any one of them
		return expectedDataType;
	}

	private static String getColumnNamesAsString(Column[] expectedColumns) {
		StringBuffer buf = new StringBuffer();
		for (int i = 0; i < expectedColumns.length; i++) {
			buf.append(expectedColumns[i].getColumnName());
			if ((i + 1) > expectedColumns.length)
				buf.append(',');
		}
		return buf.toString();
	}

	public static Column[] getSortedColumns(ITableMetaData metaData)
			throws DataSetException {
		Column[] columns = metaData.getColumns();
		Arrays.sort(columns, COLUMN_COMPARATOR);
		return columns;
	}

	public static class ColumnComparator implements Comparator<Column> {
		public int compare(Column column1, Column column2) {
			String columnName1 = column1.getColumnName();
			String columnName2 = column2.getColumnName();
			return columnName1.compareToIgnoreCase(columnName2);
		}
	}
}
