package clinical.tools;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.Statement;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.input.SAXBuilder;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

/**
 * @author I. Burak Ozyurt
 * @version $Id: SimpleMediatorMapGen.java 62 2009-05-29 23:54:50Z bozyurt $
 */
public class SimpleMediatorMapGen {
	protected BufferedReader in;
	protected DBInfo firstDB;
	protected DBInfo secondDB;
	protected Map<String, AssessmentInfo> firstAsMap;
	protected Map<String, AssessmentInfo> secondAsMap;

	public SimpleMediatorMapGen() {
		in = new BufferedReader(new InputStreamReader(System.in));
	}

	public String getInput(String prompt) throws IOException {
		System.out.print(prompt);
		return in.readLine().trim();

	}

	public String getUserInput(String prompt, String[] posValues)
			throws IOException {
		StringBuffer buf = new StringBuffer(128);
		buf.append(prompt);
		if (posValues != null) {
			buf.append(" (");
			for (int i = 0; i < posValues.length; i++) {
				buf.append(posValues[i]);
				if ((i + 1) < posValues.length)
					buf.append(",");
			}
			buf.append(" ) ");
		}
		buf.append(" >> ");

		String answer = getInput(buf.toString());
		// System.out.print( buf.toString() );
		// String answer = in.readLine().trim();
		boolean ok = true;
		do {
			if (answer.length() > 0) {
				if (posValues != null && validAnswer(answer, posValues)) {
					break;
				} else if (posValues == null) {
					// if there are no choices return the first nonempty answer
					break;
				}
			} else {
				if (posValues != null) {
					// return the default answer
					return posValues[0];
				}
			}

			answer = getInput(buf.toString());
			// System.out.print( buf.toString() );
			// answer = in.readLine().trim();
		} while (!ok);

		return answer;
	}

	public static boolean validAnswer(String answer, String[] posValues) {
		for (int i = 0; i < posValues.length; i++) {
			if (answer.equals(posValues[i]))
				return true;
		}
		return false;
	}

	public void getDBInfos() {

	}

	public Map<String, AssessmentInfo> loadAssessmentInfo(DBInfo dbInfo) throws Exception {
		Statement st = null;
		Connection con = null;
		Map<String, AssessmentInfo> asMap = new LinkedHashMap<String, AssessmentInfo>();
		try {
			con = dbInfo.connect();
			st = con.createStatement();
			ResultSet rs = st
					.executeQuery("select a.ASSESSMENTID, a.name, b.scorename, b.scoretype from nc_assessment a, nc_assessmentscore b "
							+ "where a.ASSESSMENTID=b.ASSESSMENTID order by  a.ASSESSMENTID");
			while (rs.next()) {
				String asName = rs.getString(2);
				AssessmentInfo ai = asMap.get(asName);
				if (ai == null) {
					ai = new AssessmentInfo();
					asMap.put(asName, ai);
					ai.name = asName;
					ai.asID = rs.getInt(1);
				}

				ScoreInfo si = new ScoreInfo();
				si.name = rs.getString(3);
				si.type = rs.getString(4);

				ai.addScoreInfo(si);
			}
			rs.close();
			return asMap;

		} finally {
			dbInfo.shutdown();
		}
	}

	ScoreInfo findScore(String name, List<ScoreInfo> scoreInfos) {
		for (Iterator<ScoreInfo> iter = scoreInfos.iterator(); iter.hasNext();) {
			ScoreInfo si = iter.next();
			if (si.name.equals(name))
				return si;
		}
		return null;
	}

	// for identical clinical assessment mapping
	public void createMapXML(Element rootElem) {
		for (Iterator<AssessmentInfo> iter = firstAsMap.values().iterator(); iter.hasNext();) {
			AssessmentInfo ai = iter.next();
			AssessmentInfo secAi = secondAsMap.get(ai.name);
			if (secAi == null)
				continue;
			Element ae = new Element("as-map");
			rootElem.addContent(ae);

			ae.setAttribute("name", ai.name);
			ae.setAttribute("id", String.valueOf(ai.asID));
			ae.setAttribute("first-db", firstDB.dbID);
			ae.setAttribute("second-db", secondDB.dbID);
			Element sme = new Element("score-maps");
			ae.addContent(sme);
			for (Iterator<ScoreInfo> it = ai.scoreInfos.iterator(); it.hasNext();) {
				ScoreInfo si = it.next();
				ScoreInfo secSi = findScore(si.name, secAi.scoreInfos);
				if (secSi == null)
					continue;
				Element se = new Element("score");
				se.setAttribute("name", si.name);
				se.setAttribute("map", secSi.name);
				se.setAttribute("map-as-id", String.valueOf(secAi.asID));

				sme.addContent(se);
			}
		}
	}

	public void createMap(String xmlFile) throws Exception {
		File f = new File(xmlFile);
		Element rootElem = null;
		Document doc = null;
		if (f.exists()) {
			SAXBuilder builder = new SAXBuilder(false); // no validation
			doc = builder.build(xmlFile);
			rootElem = doc.getRootElement();
		} else {
			doc = new Document();
			doc.setRootElement(rootElem = new Element("assessment-mappings"));
		}

		createMapXML(rootElem);

		PrintWriter out = null;

		try {
			out = new PrintWriter(new FileWriter(xmlFile));
			XMLOutputter xmlOut = new XMLOutputter(Format.getPrettyFormat());
			// xmlOut.setNewlines(true);

			xmlOut.output(rootElem, out);

		} finally {
			if (out != null)
				try {
					out.close();
				} catch (Exception x) {
				}
		}

	}

	public void test() throws Exception {
		firstDB = new DBInfo("jdbc:oracle:thin:@fmri-gpop:1521:public1");
		firstDB.dbID = "site1_mbirn";
		firstDB.dbUser = "adsite1";
		firstDB.dbPwd = "";

		secondDB = new DBInfo("jdbc:oracle:thin:@fmri-gpop:1521:public1");
		secondDB.dbID = "site2_mbirn";
		secondDB.dbUser = "adsite2";
		secondDB.dbPwd = "";

		firstAsMap = loadAssessmentInfo(firstDB);
		secondAsMap = loadAssessmentInfo(secondDB);

		createMap("/home/bozyurt/dev/java/clinical/conf/as_var_map.xml");
	}

	public void test2() throws Exception {
		firstDB = new DBInfo("jdbc:oracle:thin:@fmri-gpop:1521:public1");
		firstDB.dbID = "site2_mbirn";
		firstDB.dbUser = "adsite2";
		firstDB.dbPwd = "";

		secondDB = new DBInfo("jdbc:oracle:thin:@fmri-gpop:1521:public1");
		secondDB.dbID = "site1_mbirn";
		secondDB.dbUser = "adsite1";
		secondDB.dbPwd = "";

		firstAsMap = loadAssessmentInfo(firstDB);
		secondAsMap = loadAssessmentInfo(secondDB);

		createMap("/home/bozyurt/dev/java/clinical/conf/as_var_map.xml");
	}

	public static class AssessmentInfo {
		String name;
		int asID;
		List<ScoreInfo> scoreInfos = new LinkedList<ScoreInfo>();

		public void addScoreInfo(ScoreInfo si) {
			scoreInfos.add(si);
		}

	}

	public static class ScoreInfo {
		String name;
		String type;
	}

	public static class DBInfo {
		String dbURL;
		String dbUser;
		String dbPwd;
		String dbID;
		Connection con;
		String driverClass = "oracle.jdbc.driver.OracleDriver";

		public DBInfo() {
		}

		public DBInfo(String dbURL) {
			this.dbURL = dbURL;
		}

		public Connection connect() throws Exception {
			Class.forName(driverClass);
			con = DriverManager.getConnection(dbURL, dbUser, dbPwd);
			return con;
		}

		public void shutdown() {
			if (con != null) {
				try {
					con.close();
				} catch (Exception ex) {
					ex.printStackTrace();
				}
			}
		}
	}

	public static void main(String[] args) {
		SimpleMediatorMapGen gen = null;
		try {
			gen = new SimpleMediatorMapGen();

			gen.test2();
		} catch (Exception ex) {
			ex.printStackTrace();
		}

	}

}