package clinical.tools.maintenance;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.web.Constants;
import clinical.web.common.IDBPoolService;
import clinical.web.common.UserInfo;
import clinical.web.common.security.DBConfig;
import clinical.web.exception.BaseException;
import clinical.web.exception.DBPoolServiceException;
import clinical.web.services.DBPoolService;
import clinical.web.services.SimpleSecurityService;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: AbstractHelper.java 62 2009-05-29 23:54:50Z bozyurt $
 */
public abstract class AbstractHelper {
	protected String dbID;
	protected static IDBPoolService dbPoolService;
	protected static Connection connection;
	protected UserInfo ui;
	protected DBHelper dbHelper;
	protected static Log log = LogFactory.getLog(AbstractHelper.class);

	public AbstractHelper(String dbID, String usersFile) throws BaseException {
		this.dbID = dbID;

		SimpleSecurityService secService = SimpleSecurityService
				.getInstance(usersFile);

		synchronized (AbstractHelper.class) {
			if (dbPoolService == null) {
				DBPoolService.getInstance(secService, secService
						.getDBConfigMap());

				// startup all database connection pools
				for (Object element : secService.getDBConfigMap().values()) {
					DBConfig dbConfig = (DBConfig) element;

					DBPoolService poolService = DBPoolService
							.getInstance(dbConfig.getId());
					if (this.dbID.equals(dbConfig.getId())) {
						AbstractHelper.dbPoolService = poolService;
					}
					poolService.startup();
				}
			}
		}
		// Map userMap = secService.getAllUsers(this.dbID);
		// User user = (User) userMap.get("admin");
		this.ui = new UserInfo(Constants.ADMIN_USER, null, null);
		this.dbHelper = new DBHelper(this.dbID, this.ui);
	}

	public UserInfo getUserInfo() {
		return ui;
	}

	public Connection getConnection() throws DBPoolServiceException,
			SQLException {
		return AbstractHelper.getConnection(ui);
	}

	public static synchronized Connection getConnection(UserInfo ui)
			throws DBPoolServiceException, SQLException {
		if (connection == null) {
			connection = dbPoolService.getConnection(ui.getName());
			connection.setAutoCommit(false);
		}
		return connection;
	}

	public List<?> find(String voName, String[] criteriaArgs) throws Exception {
		String voClassName = "clinical.server.vo." + voName;
		String daoClassName = "clinical.server.dao." + voName + "DAO";

		Map<String, PropertyDescriptor> voPDMap = getPropertyDescriptorMap(voClassName);
		// Map daoPDMap = getPropertyDescriptorMap(daoClassName);

		Object criteriaBean = Class.forName(voClassName).newInstance();
		Object daoBean = Class.forName(daoClassName).newInstance();

		if (criteriaArgs != null) {
			int idx = 0;
			while (idx < criteriaArgs.length) {
				String propertyName = criteriaArgs[idx];
				String propertyValue = criteriaArgs[idx + 1];
				PropertyDescriptor pd = (PropertyDescriptor) voPDMap
						.get(propertyName);
				setProperty(criteriaBean, propertyValue, pd);

				idx += 2;
			}
		}
		// System.out.println("criteria bean - " + criteriaBean.toString() );
		return find(criteriaBean, daoBean);
	}

	public void update(String voName, String[] updateArgs, String[] criteriaArgs)
			throws Exception {
		String voClassName = "clinical.server.vo." + voName;
		String daoClassName = "clinical.server.dao." + voName + "DAO";

		Map<String, PropertyDescriptor> voPDMap = getPropertyDescriptorMap(voClassName);
		// Map daoPDMap = getPropertyDescriptorMap(daoClassName);
		Object criteriaBean = Class.forName(voClassName).newInstance();
		Object daoBean = Class.forName(daoClassName).newInstance();

		Object updateBean = Class.forName(voClassName).newInstance();
		if (criteriaArgs != null) {
			int idx = 0;
			while (idx < criteriaArgs.length) {
				String propertyName = criteriaArgs[idx];
				String propertyValue = criteriaArgs[idx + 1];
				PropertyDescriptor pd = (PropertyDescriptor) voPDMap
						.get(propertyName);
				setProperty(criteriaBean, propertyValue, pd);

				idx += 2;
			}
		}

		if (updateArgs != null) {
			int idx = 0;
			while (idx < updateArgs.length) {
				String propertyName = updateArgs[idx];
				String propertyValue = updateArgs[idx + 1];
				PropertyDescriptor pd = (PropertyDescriptor) voPDMap
						.get(propertyName);
				setProperty(updateBean, propertyValue, pd);
				idx += 2;
			}
		}
		update(criteriaBean, updateBean, daoBean);
	}

	protected List<?> find(Object criteriaBean, Object daoBean)
			throws SQLException, DBPoolServiceException, SecurityException,
			NoSuchMethodException, InvocationTargetException,
			IllegalArgumentException, IllegalAccessException {
		Object[] args = { getConnection(), criteriaBean };
		Class<?> clazz = daoBean.getClass();
		Method m = clazz.getMethod("find", new Class[] { Connection.class,
				criteriaBean.getClass() });
		return (List<?>) m.invoke(daoBean, args);
	}

	protected void update(Object criteriaBean, Object updateBean, Object daoBean)
			throws SQLException, DBPoolServiceException, SecurityException,
			NoSuchMethodException, InvocationTargetException,
			IllegalArgumentException, IllegalAccessException {
		Object[] args = { getConnection(), updateBean, criteriaBean };
		Class<?> clazz = daoBean.getClass();
		Method m = clazz.getMethod("update", new Class[] { Connection.class,
				updateBean.getClass(), criteriaBean.getClass() });
		m.invoke(daoBean, args);
	}

	protected Map<String, PropertyDescriptor> getPropertyDescriptorMap(
			String className) throws ClassNotFoundException,
			IntrospectionException {
		Class<?> clazz = Class.forName(className);
		BeanInfo bi = Introspector.getBeanInfo(clazz);
		PropertyDescriptor[] descriptors = (PropertyDescriptor[]) bi
				.getPropertyDescriptors().clone();
		Map<String, PropertyDescriptor> descriptorsMap = new HashMap<String, PropertyDescriptor>(
				19);
		for (int i = 0; i < descriptors.length; i++) {
			descriptorsMap.put(descriptors[i].getName(), descriptors[i]);
		}
		return descriptorsMap;
	}

	protected void setProperty(Object object, String value,
			PropertyDescriptor pd) throws IntrospectionException,
			IllegalAccessException, InvocationTargetException {

		Object convertedValue = value;
		// type conversion
		if (pd.getPropertyType() == BigDecimal.class) {
			convertedValue = new BigDecimal(value);
		} else if (pd.getPropertyType() == Double.class) {
			convertedValue = new Double(value);
		} else if (pd.getPropertyType() == Boolean.class) {
			convertedValue = new Boolean(value);
		}

		Method m = pd.getWriteMethod();
		m.invoke(object, new Object[] { convertedValue });
	}

	protected Object getProperty(Object object, PropertyDescriptor pd)
			throws IntrospectionException, IllegalAccessException,
			InvocationTargetException {
		Method m = pd.getReadMethod();
		return m.invoke(object, new Object[0]);
	}

	public static synchronized void releaseConnection(UserInfo ui) {
		if (connection == null)
			return;
		try {
			dbPoolService.releaseConnection(ui.getName(), connection);
			connection = null;
		} catch (DBPoolServiceException x) {
			log.error("Cannot release connection for user " + ui.getName(), x);
		}
	}

	public static BigDecimal toBigDecimal(int value) {
		return new BigDecimal(String.valueOf(value));
	}

	public static synchronized void commit(UserInfo ui)
			throws DBPoolServiceException, SQLException {
		getConnection(ui).commit();
	}

	public static synchronized void rollback(UserInfo ui) {

		try {
			getConnection(ui).rollback();
		} catch (DBPoolServiceException ex) {
		} catch (SQLException ex) {
		}
	}

	public static synchronized void shutdown() {
		if (dbPoolService != null) {
			try {
				dbPoolService.shutdown();
			} catch (Exception ex) {
				ex.printStackTrace();
			}
		}
	}

}