package clinical.utils;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.LinkedList;
import java.util.List;
import java.util.StringTokenizer;

/**
 * given an external program/script and arguments it runs the external program
 * It captures the output and checks for errors
 * 
 * @author: I. Burak Ozyurt
 * @version: $Id: Executor.java 62 2009-05-29 23:54:50Z bozyurt $
 */

public class Executor {
	protected String cmd;
	protected List<String> args;
	protected List<String> envps;
	protected boolean isErrorStreamReliable = true;
	protected boolean showOutput = false;
	protected String output;

	public Executor(String cmd) {
		this(cmd, false);
	}

	public Executor(String cmd, boolean showOutput) {
		this.cmd = cmd;
		envps = new LinkedList<String>();
		args = new LinkedList<String>();
		this.showOutput = showOutput;
	}

	public void showOutput(boolean value) {
		showOutput = value;
	}

	public void addEnvParam(String envParamName, String value) {
		if (envParamName != null && value != null)
			envps.add(envParamName + "=" + value);
	}

	public void setErrorStreamReliable(boolean value) {
		isErrorStreamReliable = value;
	}

	public int execute(String argList) throws Exception {

		String[] args = parseArgs(argList);
		String[] cmdArray = new String[1 + args.length];
		cmdArray[0] = cmd;
		for (int i = 0; i < args.length; ++i) {
			cmdArray[i + 1] = args[i];
		}

		String[] envpsArr = getEnvParams();
		Process p = Runtime.getRuntime().exec(cmdArray, envpsArr);
		StreamConsumer outConsumer = null;
		StreamConsumer errConsumer = null;
		if (showOutput) {
			outConsumer = new StreamConsumer(p.getInputStream(), "output", true);
			errConsumer = new StreamConsumer(p.getErrorStream(), "error", true);
		} else {
			outConsumer = new StreamConsumer(p.getInputStream(), "output",
					false);
			errConsumer = new StreamConsumer(p.getErrorStream(), "error", false);
		}
		outConsumer.start();
		errConsumer.start();

		int rc = p.waitFor();
		String errorMsg = errConsumer.getStreamData();
		if (errorMsg.trim().length() > 0) {
			// guess if there is any indication of error
			if (isErrorStreamReliable)
				throw new Exception(errorMsg);
		}
		System.out.println(errorMsg);
		if (showOutput) {
			String outMsg = outConsumer.getStreamData();
			System.out.println(outMsg);
		}
		this.output = outConsumer.getStreamData();
		return rc;
	}

	public String getOutput() {
		return output;
	}

	public static class StreamConsumer extends Thread {
		StringBuffer buf = new StringBuffer();
		String type;
		InputStream in;
		boolean showOutput;

		public StreamConsumer(InputStream in, String type, boolean showOutput) {
			this.in = in;
			this.type = type;
			this.showOutput = showOutput;
		}

		public void run() {
			String line = null;
			BufferedReader bin = new BufferedReader(new InputStreamReader(in));
			try {
				while ((line = bin.readLine()) != null) {
					buf.append(line).append("\n");
					System.out.println(line);
				}
			} catch (IOException iox) {

			} finally {
				if (in != null)
					try {
						in.close();
					} catch (Exception x) {/* ignore */
					}
			}
		}

		String getStreamData() {
			return buf.toString();
		}

	}

	String[] parseArgs(String argList) {
		StringTokenizer stok = new StringTokenizer(argList, " ");
		String[] args = new String[stok.countTokens()];
		int i = 0;
		while (stok.hasMoreTokens())
			args[i++] = stok.nextToken();
		return args;
	}

	String[] getEnvParams() {
		if (envps.isEmpty())
			return new String[0];
		String[] eps = new String[envps.size()];
		envps.toArray(eps);
		return eps;
	}

	public static void test() throws Exception {
		Executor ex = new Executor("mri_converter.pl", true);
		ex.addEnvParam(
						"PATH",
						".:/home/bozyurt/bin:/usr/bin:/bin:/usr/local/AFNI/bin:/data1/opt/srb-2.1.1/bin:/usr/local/dicom/bin");
		ex.addEnvParam("PERL5LIB", "/home/bozyurt/dev/perl/modules");
		ex.addEnvParam("HOME", "/home/bozyurt");
		ex.addEnvParam("PWD", "/home/bozyurt");
		ex.addEnvParam("DCMDICTPATH", "/usr/local/dicom/lib/birn_dicom.dic");
		ex.execute("");

	}

	public static void main(String[] args) {
		try {
			Executor.test();
		} catch (Exception x) {
			x.printStackTrace();
		}
	}

}