package clinical.utils;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.math.BigDecimal;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;

import org.jdom.Element;

import clinical.web.Constants;

/**
 * A collection of static general utility methods.
 * 
 * @version $Id: GenUtils.java 73 2009-06-05 21:58:05Z bozyurt $
 * @author I. Burak Ozyurt
 */

public class GenUtils {
	protected static Map<String, String> siteIDMap = Collections
			.synchronizedMap(new HashMap<String, String>(17));

	protected GenUtils() {}

	public static synchronized void populateSiteIDMap(Properties props) {
		Enumeration<?> enumeration = props.propertyNames();
		while (enumeration.hasMoreElements()) {
			String name = (String) enumeration.nextElement();
			String value = props.getProperty(name);
			siteIDMap.put(name, value);
		}
	}

	public static String getSiteName(String numericSiteID) {
		return siteIDMap.get(numericSiteID);
	}

	public static boolean isScanVisitType(String visitType) {
		return visitType.equalsIgnoreCase(Constants.VISIT_TYPE_SCAN)
				|| visitType.equalsIgnoreCase("fmri scan")
				|| visitType.equalsIgnoreCase("mri scan");
	}

	/**
	 * 
	 * @param propsFilename
	 *           properties file to load the properties from. The file must be
	 *           accessible via the CLASSPATH
	 * @return a java.util.Properties object holding the properties loaded from
	 *         the file
	 * @throws IOException
	 *            if there is an error finding or reading from the properties
	 *            file
	 */
	public static Properties loadProperties(String propsFilename)
			throws IOException {
		InputStream is = GenUtils.class.getClassLoader().getResourceAsStream(
				propsFilename);
		Properties props = new Properties();
		props.load(is);

		return props;
	}

	/**
	 * Returns name value pair list from the given properties. The name-value
	 * pairs are defined using the following scheme in the properties;
	 * 
	 * <pre>
	 *  &lt;nameProperty&gt;.1=
	 *  &lt;valueProperty&gt;.1=
	 *  ...
	 *  &lt;nameProperty&gt;.N=
	 *  &lt;valueProperty&gt;.N=
	 * </pre>
	 * 
	 * @param props
	 * @param nameProperty
	 * @param valueProperty
	 * @return
	 */
	public static LinkedHashMap<String, String> extractNameValueMapFromProps(
			Properties props, String nameProperty, String valueProperty) {
		LinkedHashMap<String, String> map = new LinkedHashMap<String, String>(7);
		int idx = 1;
		while (true) {
			String nameKey = nameProperty + "." + idx;
			String name = props.getProperty(nameKey);
			if (name == null) {
				break;
			}
			String valueKey = valueProperty + "." + idx;
			String value = props.getProperty(valueKey);
			map.put(name, value);
			++idx;
		}
		return map;
	}

	public static String[] toList(Properties props, String propertyName,
			String delimiter) {
		String value = props.getProperty(propertyName);
		if (value == null)
			return new String[0];
		String regex = "\\s*" + delimiter + "\\s*";
		return value.split(regex);
	}

	/**
	 * Given a string, capitilizes every word in the string.
	 * 
	 * @param s
	 *           the string which needs to be converted into title case
	 * @return the title cased string
	 */
	public static String toTitleCase(String s) {
		StringBuffer buf = new StringBuffer(s);
		if (s.indexOf(' ') != -1) {
			for (int i = 0; i < buf.length(); i++) {
				if (i == 0
						|| (Character.isWhitespace(buf.charAt(i - 1)) && Character
								.isLetter(buf.charAt(i)))) {
					buf.setCharAt(i, Character.toUpperCase(buf.charAt(i)));
				}
			}
		} else {
			buf.setCharAt(0, Character.toUpperCase(buf.charAt(0)));
		}
		return buf.toString();
	}

	public static boolean toBoolean(String value, boolean defaultVal) {
		if (value == null) {
			return defaultVal;
		}
		if (value.equalsIgnoreCase("true") || value.equalsIgnoreCase("yes")
				|| value == "1") {
			return true;
		} else if (value.equalsIgnoreCase("false")
				|| value.equalsIgnoreCase("no") || value == "0") {
			return false;
		} else {
			return defaultVal;
		}
	}

	public static int toInt(String value, int defaultVal) {
		if (value == null) {
			return defaultVal;
		}
		try {
			return Integer.parseInt(value);
		} catch (NumberFormatException nfe) {
			return defaultVal;
		}
	}

	public static long toLong(String value, long defaultVal) {
		if (value == null) {
			return defaultVal;
		}
		try {
			return Long.parseLong(value);
		} catch (NumberFormatException nfe) {
			return defaultVal;
		}
	}

	public static BigDecimal toBigDecimal(int value) {
		return new BigDecimal(String.valueOf(value));
	}

	public static boolean isInteger(String value) {
		try {
			Integer.parseInt(value);
			return true;
		} catch (NumberFormatException nfe) {
			return false;
		}
	}

	public static boolean isBoolean(String value) {
		if (value.equalsIgnoreCase("true") || value.equalsIgnoreCase("false")
				|| value.equalsIgnoreCase("yes") || value.equalsIgnoreCase("no")
				|| value == "1" || value == "0") {
			return true;
		} else {
			return false;
		}
	}

	public static String getAttributeValueAsString(Element elem, String attrName) {
		if (elem.getAttribute(attrName) != null) {
			return elem.getAttributeValue(attrName);
		}
		return null;
	}

	public static String getAttributeValueAsString(Element elem,
			String attrName, String defVal) {
		if (elem.getAttribute(attrName) != null) {
			return elem.getAttributeValue(attrName);
		}
		return defVal;
	}

	public static int getAttributeValueAsInt(Element elem, String attrName,
			int defaultVal) {
		if (elem.getAttribute(attrName) != null) {
			return toInt(elem.getAttributeValue(attrName), defaultVal);
		}
		return defaultVal;
	}

	/**
	 * 
	 * Given a string, if a number the corresponding <code>Integer</code> or
	 * <code>Float</code> object.
	 * 
	 * @param value
	 *           a string to be converted to a <code>Number</code> object if
	 *           possible.
	 * @return a Number object or null if not a number
	 * @see Number
	 */
	public static Number toNumber(String value) {
		Number num = null;
		try {
			num = Integer.valueOf(value);
		} catch (NumberFormatException nfe) {
			try {
				num = Float.valueOf(value);
			} catch (NumberFormatException e) {
				num = null;
			}
		}
		return num;
	}

	public static boolean isDecimalNumber(String value) {
		return value.matches("[-+]?\\d+\\.\\d+");
	}

	/**
	 * Given a full path to a file returns a unique filename if there is a file
	 * with the given name. The algorithm used is based on adding a counter to
	 * the original filename and checking the existence of the new filename and
	 * continuing by incrementing the counter till the file name is unique.
	 * 
	 * @param filename
	 *           the filename checked for uniqueness and if not as the root of
	 *           the unique filename
	 * @return the unique filename in the format of
	 *         <code>filename.&lt;counter&gt;</code>
	 */
	public static String getUniqueFile(String filename) {
		File f = new File(filename);
		int count = 0;
		String newFilename = filename;
		while (f.exists()) {
			++count;
			newFilename = filename + "." + count;
			f = new File(newFilename);
		}
		return newFilename;
	}

	/**
	 * Given a (composite) object, and a filename, serialize the state of the
	 * object to the given filename.
	 * 
	 * @param o
	 *           the object to serialize
	 * @param filename
	 *           the filename to which the serialized object is written
	 * @throws IOException
	 */
	public static void serialize(Object o, String filename) throws IOException {
		assert (o != null);
		ObjectOutputStream out = null;
		try {

			String newFilename = getUniqueFile(filename);
			if (!newFilename.equals(filename)) {
				File f = new File(filename);
				boolean ok = f.renameTo(new File(newFilename));
				if (!ok) {
					System.err.println("Cannot rename " + filename + " to "
							+ newFilename + "!");
				}
			}
			out = new ObjectOutputStream(new BufferedOutputStream(
					new FileOutputStream(filename), 4096));
			out.writeObject(o);
		} finally {
			if (out != null) {
				try {
					out.close();
				} catch (Exception x) {}
			}
		}
	}

	/**
	 * Resurrects a serialized object.
	 * 
	 * @param filename
	 *           the filename from which the serialized object is read
	 * @return the deserialized object.
	 * @throws IOException
	 * @throws java.lang.ClassNotFoundException
	 */
	public static Object deserialize(String filename) throws IOException,
			ClassNotFoundException {
		assert (filename != null && new File(filename).isFile());
		ObjectInputStream in = null;
		try {
			in = new ObjectInputStream(new BufferedInputStream(
					new FileInputStream(filename), 4096));
			Object o = in.readObject();
			return o;
		} finally {
			if (in != null) {
				try {
					in.close();
				} catch (Exception x) {}
			}
		}
	}

	/**
	 * Given a filename, returns its base removing the suffix and the dot.
	 * 
	 * @param filename
	 *           the filename to extract its base
	 * @return the base part of the filename namely the part before the last dot
	 */
	public static String getBasePart(String filename) {
		if (filename == null) {
			return null;
		}
		int idx = filename.lastIndexOf('.');
		if (idx == -1) {
			return filename;
		}
		return filename.substring(0, idx);
	}

	/**
	 * Creates a temporary file full path using the specified temp directory. The
	 * format of the temp file is
	 * <code>temp_&lt;milliseconds from epoch&gt;</code>.
	 * 
	 * @param tempDir
	 *           the temporary directory
	 * @param extension
	 *           filename extension
	 * @return the full path to the temporary file
	 */
	public static synchronized String createTempFileName(String tempDir,
			String extension) {
		if (!extension.startsWith(".")) {
			extension = "." + extension;
		}
		StringBuffer buf = new StringBuffer(tempDir.length() + 40);
		buf.append(tempDir).append(File.separator);
		buf.append("temp_").append(System.currentTimeMillis()).append(extension);
		return buf.toString();
	}

	/**
	 * Given a database ID finds and returns corresponding siteID.
	 * 
	 * @param dbID
	 * @return
	 */
	public static String findSiteID(String dbID) {
		int idx = dbID.lastIndexOf('_');
		if (idx == -1) {
			return null;
		}
		String theSiteName = dbID.substring(0, idx);
		for (Map.Entry<String, String> entry : siteIDMap.entrySet()) {
			String siteName = (String) entry.getValue();
			if (siteName.equals(theSiteName)) {
				return (String) entry.getKey();
			}
		}
		return null;
	}

	public static String findSiteIDFromSiteName(String siteName) {
		siteName = siteName.toLowerCase();
		for (Map.Entry<String, String> entry : siteIDMap.entrySet()) {
			String sn = (String) entry.getValue();
			if (sn.equals(siteName)) {
				return (String) entry.getKey();
			}
		}
		return null;
	}

	public static String leftPad(String s, char padChar, int padLength) {
		StringBuilder buf = new StringBuilder();
		for (int i = 0; i < padLength; i++) {
			buf.append(padChar);
		}
		buf.append(s);
		return buf.toString();
	}

	/**
	 * 
	 * @param scheme
	 *           the name of the scheme (http , https)
	 * @param serverName
	 *           the hostname
	 * @param port
	 *           the numeric port
	 * @param uri
	 *           server relative URI
	 * @return the absolute URI
	 */
	public static String createServerURI(String scheme, String serverName,
			int port, String uri) {
		StringBuffer sb = new StringBuffer();
		if (port < 0) {
			port = 80;
		}
		sb.append(scheme).append("://").append(serverName);
		if ((scheme.equals("http") && port != 80)
				|| (scheme.equals("https") && port != 443)) {
			sb.append(':').append(port);
		}
		sb.append(uri);

		return sb.toString();
	}

	/**
	 * Given a list of objects , dumps their content if they have a
	 * <code>toString()</code> method defined
	 * 
	 * @param aList
	 * @param _log
	 */
	public static String dumpList(List<?> aList) {
		StringBuffer buf = new StringBuffer(200);
		buf.append("----------------\n");
		for (Iterator<?> iter = aList.iterator(); iter.hasNext();) {
			Object item = (Object) iter.next();
			buf.append(item.toString()).append('\n');
		}
		buf.append("----------------");
		return buf.toString();
	}

	public static StackTraceElement[] prepareStackTrace(Throwable t,
			String qnPrefix) {
		StackTraceElement[] stElems = t.getStackTrace();
		if (qnPrefix == null) {
			return stElems;
		}
		List<StackTraceElement> relevantTraces = new LinkedList<StackTraceElement>();
		for (int i = 0; i < stElems.length; i++) {
			if (stElems[i].getClassName().startsWith(qnPrefix)) {
				relevantTraces.add(stElems[i]);
			}
		}
		stElems = new StackTraceElement[relevantTraces.size()];
		relevantTraces.toArray(stElems);
		return stElems;
	}

	/**
	 * creates an acessor method name based on Java Beans spec
	 * 
	 * @param varName
	 * @param prefix
	 * @return
	 */
	public static String createAccessorName(String varName, String prefix) {
		StringBuffer buf = new StringBuffer();
		buf.append(prefix);
		buf.append(Character.toUpperCase(varName.charAt(0)));
		buf.append(varName.substring(1));
		return buf.toString();
	}

	/**
	 * creates the property name for the questionID and answer index
	 * 
	 * @param questionID
	 *           the ID of the question
	 * @param idx
	 *           the answer index (1 based), use a negative number to create a
	 *           property name for a single valued question.
	 * 
	 * @return the generated property name
	 */
	public static String createHiddenNotesVariable(int questionID, int idx) {
		StringBuffer buf = new StringBuffer();
		buf.append("q").append(questionID);
		if (idx >= 0) {
			buf.append("_ans").append(idx);
		}
		buf.append("_notes");
		return buf.toString();
	}

	/**
	 * creates the variable name for the hidden form field representing the
	 * missing value reason note for that particular score value on the online
	 * form. The format of the variable name is as follows
	 * 
	 * <pre>
	 *    q&lt;question-id&gt;[_ans&lt;idx&gt;[_&lt;score-id&gt;]]_notes
	 * </pre>
	 * 
	 * For example for question 10, for the score with CALM given id say 2, for
	 * the 4th score value of the score with id 2
	 * 
	 * <pre>
	 * q10_ans3_2_notes
	 * </pre>
	 * 
	 * @param questionID
	 *           the id of the question as given by CALM
	 * @param scoreID
	 *           if nonnegative indicates the id (key) of the score in a
	 *           multiscore question
	 * @param idx
	 *           if nonnegative, for multiple valued scores represents the order
	 *           index of the score value in the list of score values
	 * @return the missing value reason note variable name for the corresponding
	 *         hidden form field
	 */
	public static String createHiddenNotesVariable(int questionID, int scoreID,
			int idx) {
		StringBuffer buf = new StringBuffer();
		buf.append("q").append(questionID);
		if (idx >= 0) {
			buf.append("_ans").append(idx);
			if (scoreID >= 0) {
				buf.append('_').append(scoreID);
			}
		}
		buf.append("_notes");
		return buf.toString();
	}

	@SuppressWarnings("unchecked")
	public static List<Comparable> extractList(String property,
			String delimiter, Class typeClazz) {
		List<Comparable> list = new ArrayList<Comparable>();
		StringTokenizer stok = new StringTokenizer(property, delimiter);
		while (stok.hasMoreTokens()) {
			String token = stok.nextToken();
			if (typeClazz == Integer.class) {
				list.add(new Integer(token));
			} else if (typeClazz == String.class) {
				list.add(token);
			} else {
				throw new RuntimeException("Unsupported type to convert:"
						+ typeClazz);
			}
		}
		return list;
	}

	public static String generateKey(String subjectID, String siteID, int expID,
			int visitID, int segmentID) {
		StringBuffer buf = new StringBuffer();
		buf.append(subjectID).append('_').append(siteID).append('_');
		buf.append(expID).append('_').append(visitID).append('_');
		buf.append(segmentID);
		return buf.toString();
	}

	public static String formatNumber(double num, int fracDigits) {
		if (num == Double.NaN) {
			return "N/A";
		}
		NumberFormat numFormatter = NumberFormat.getInstance();
		if (fracDigits > 0) {
			numFormatter.setMaximumFractionDigits(fracDigits);
		}
		return numFormatter.format(num);
	}

	public static String applyWordWrapping(String line) {
		StringBuffer buf = new StringBuffer(line.length() + 10);
		if (line.length() < 80) {
			return line;
		}
		int maxLen = 80;
		int idx = 0;

		int len = line.length();
		while (idx < len) {
			if ((idx + maxLen) < len) {
				int locIdx = idx + maxLen;
				if (Character.isWhitespace(line.charAt(locIdx))) {
					buf.append(line.substring(idx, locIdx)).append("\n");
					idx += maxLen;
				} else {
					int endIdx = findFirstWhitespaceBefore(line, locIdx, idx);
					if (endIdx == idx) {
						endIdx = findFirstWhitespaceAfter(line, locIdx);
					}
					buf.append(line.substring(idx, endIdx)).append("\n");
					idx = endIdx;
				}
			} else {
				buf.append(line.substring(idx));
				idx = len;
			}
		}

		return buf.toString();
	}

	public static int findFirstWhitespaceAfter(String s, int startIdx) {
		int len = s.length();
		int idx = startIdx;
		while (idx < len) {
			if (Character.isWhitespace(s.charAt(idx))) {
				return idx;
			}
			idx++;
		}
		return idx;
	}

	public static int findFirstWhitespaceBefore(String s, int startIx,
			int lineStartIdx) {
		int idx = startIx;
		while (idx > lineStartIdx) {
			if (Character.isWhitespace(s.charAt(idx))) {
				return idx;
			}
			idx--;
		}
		return idx;
	}

	public static String join(int[] arr, String delimiter) {
		StringBuffer buf = new StringBuffer();
		for (int i = 0; i < arr.length; i++) {
			buf.append(arr[i]);
			if ((i + 1) < arr.length) {
				buf.append(delimiter);
			}
		}
		return buf.toString();
	}

	public static String join(List<?> list, String delimiter) {
		StringBuffer buf = new StringBuffer();
		for (Iterator<?> iter = list.iterator(); iter.hasNext();) {
			Object element = (Object) iter.next();
			buf.append(element.toString());
			if (iter.hasNext())
				buf.append(delimiter);
		}
		return buf.toString();
	}

	public static String join(List<?> list, String delimiter,
			String surroundingChar) {
		StringBuffer buf = new StringBuffer();
		for (Iterator<?> iter = list.iterator(); iter.hasNext();) {
			Object element = (Object) iter.next();
			if (surroundingChar != null) {
				buf.append(surroundingChar).append(element.toString()).append(
						surroundingChar);
			} else {
				buf.append(element.toString());
			}
			if (iter.hasNext())
				buf.append(delimiter);
		}
		return buf.toString();
	}

	public static String join(String[] arr, String delimiter,
			String surroundingChar) {
		StringBuffer buf = new StringBuffer();
		for (int i = 0; i < arr.length; i++) {
			if (surroundingChar != null) {
				buf.append(surroundingChar).append(arr[i]).append(surroundingChar);
			} else {
				buf.append(arr[i]);
			}
			if ((i + 1) < arr.length) {
				buf.append(delimiter);
			}
		}
		return buf.toString();
	}

	public static String join(String[] arr, String delimiter, int startIdx,
			int endIdx) {
		StringBuffer buf = new StringBuffer();
		for (int i = startIdx; i < endIdx; i++) {
			buf.append(arr[i]);
			if ((i + 1) < endIdx) {
				buf.append(delimiter);
			}
		}
		return buf.toString();
	}

	public static boolean arrayEquals(String[] arr1, String[] arr2) {
		if (arr1.length != arr2.length)
			return false;
		for (int i = 0; i < arr1.length; i++) {
			if (!arr1[i].equals(arr2[i])) {
				return false;
			}
		}
		return true;
	}

	public static String formatFileSize(long jobSize) {
		StringBuffer buf = new StringBuffer();
		if (jobSize < 1024) {
			buf.append(jobSize).append('B');
		} else if (jobSize >= 1024 && jobSize < 1048576) {
			buf.append(Math.round(jobSize / 1024.0)).append("Kb");
		} else if (jobSize >= 1048576 && jobSize < 1073741824) {
			buf.append(Math.round(jobSize / 1048576.0)).append("Mb");
		} else if (jobSize >= 1073741824) {
			NumberFormat numFormatter = NumberFormat.getInstance();
			numFormatter.setMaximumFractionDigits(2);
			double value = jobSize / 1073741824.0;
			buf.append(numFormatter.format(value)).append("Gb");
		}

		return buf.toString();
	}

	public static boolean isNotEmpty(String s) {
		if (s == null)
			return false;
		return s.trim().length() > 0;
	}

	public static boolean isValidEmail(String emailStr) {
		return emailStr.matches("^.+@[^\\.].*\\.[a-z]{2,}$");
	}

	public static String[] split(String s, String delimeters) {
		StringTokenizer stok = new StringTokenizer(s, delimeters);
		String[] toks = new String[stok.countTokens()];
		int i = 0;
		while (stok.hasMoreTokens()) {
			toks[i++] = stok.nextToken();
		}
		return toks;
	}

	public static double toDouble(String value, double defaultVal) {
		if (value == null)
			return defaultVal;
		try {
			return Double.parseDouble(value);
		} catch (NumberFormatException nfe) {
			return defaultVal;
		}
	}

	public static String toHex(byte[] bytes) {
		StringBuilder buf = new StringBuilder();
		for (int i = 0; i < bytes.length; ++i) {
			int byteVal = (bytes[i] < 0) ? bytes[i] + 256 : bytes[i];
			String s = Integer.toHexString(byteVal);
			s = (s.length() == 1) ? "0" + s : s;
			buf.append(s);
		}
		return buf.toString();
	}

	public static byte[] hexToBytes(String s) {
		char[] carr = s.toCharArray();
		byte[] bytes = new byte[carr.length / 2];

		StringBuilder buf = new StringBuilder(3);
		for (int i = 0; i < bytes.length; ++i) {
			buf.setLength(0);
			buf.append(carr[2 * i]);
			buf.append(carr[2 * i + 1]);
			int val = Integer.parseInt(buf.toString(), 16);
			bytes[i] = (byte) val;
		}
		return bytes;
	}

	public static String getMD5Digest(String text)
			throws NoSuchAlgorithmException {
		MessageDigest md = MessageDigest.getInstance("MD5");
		md.update(text.getBytes());
		byte[] digest = md.digest();
		return toHex(digest);
	}

	public static String clipString(String str, int maxLen) {
		if (str.length() <= maxLen)
			return str;
		StringBuilder sb = new StringBuilder();
		sb.append(str.subSequence(0, maxLen));
		sb.append("...");
		return sb.toString();
	}

}
