package clinical.web;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;

import clinical.server.vo.Assessment;
import clinical.server.vo.Assessmentscore;
import clinical.server.vo.Storedquery;
import clinical.web.common.AssessmentMapping;
import clinical.web.common.UserInfo;
import clinical.web.common.query.Operator;
import clinical.web.common.vo.AsScoreInfo;
import clinical.web.common.vo.AssessmentSelectionInfo;
import clinical.web.download.SubjectImageDataBundle;
import clinical.web.helpers.QueryWizardHelper;
import clinical.web.helpers.SubCorticalVarInfo;
import clinical.web.services.BatchQueryResult;
import clinical.web.vo.AssessmentInfo;
import clinical.web.vo.QuerySummary;
import clinical.web.vo.RangeInfo;
import clinical.web.vo.SubjectAsScoreValueSummary;

/**
 * Session facade interface for assessment query related services.
 *
 * @version $Id: IAssessmentService.java,v 1.19.2.1 2008/07/10 23:55:59 bozyurt
 *          Exp $
 * @author I. Burak Ozyurt
 */

public interface IAssessmentService {

	/**
	 * returns all assessments available from persistent store.
	 *
	 * @param userInfo
	 *            User Information used to use the correct named database user
	 *            connection from the connection pool
	 * @return a list of {@link clinical.server.vo.Assessment} value objects
	 * @throws Exception
	 */
	public List<Assessment> getAllAssessments(UserInfo userInfo)
			throws Exception;

	/**
	 * returns a list of Assessmentscore objects for Assessment.
	 *
	 * @param userInfo
	 *            User Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param assessment
	 *            a value object of type {@link clinical.server.vo.Assessment}
	 *            used as search criteria
	 * @return a list of {@link clinical.server.vo.Assessmentscore} objects
	 * @throws Exception
	 * @see clinical.server.vo.Assessmentscore
	 */
	public List<Assessmentscore> getScoresForAssessment(UserInfo userInfo,
			Assessment assessment) throws Exception;

	/**
	 *
	 * @param userInfo
	 *            userInfo User Information used to use the correct named
	 *            database user connection from the connection pool
	 * @param assessmentID
	 * @return an {@link clinical.web.vo.AssessmentInfo} containing type
	 *         information for the assessment and its scores
	 * @throws java.lang.Exception
	 */
	public AssessmentInfo getAssessmentWithScores(UserInfo userInfo,
			int assessmentID) throws Exception;

	/**
	 *
	 * @param ui
	 *            User Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param asiList
	 *            a list of {@link AssessmentInfo} objects containing only
	 *            assessment type info
	 *
	 * @return the same list of {@link AssessmentInfo} objects passed in now
	 *         also containing score type info without hierarchy adjustment. You
	 *         need to call
	 *         {@link QueryWizardHelper#createTree(AssessmentSelectionInfo)} on
	 *         each element of this list.
	 *
	 * @throws Exception
	 */
	public List<AssessmentSelectionInfo> getAssessmentsWithScores(UserInfo ui,
			List<AssessmentSelectionInfo> asiList) throws Exception;

	// search related interface methods
	/**
	 * returns a list of {@link clinical.web.vo.SubjectAsScoreValueSummary}
	 * objects.
	 *
	 * @param userInfo
	 *            User Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param op
	 * @param asiList
	 * @param queryScope
	 * @param primarySiteID
	 * @return list of {@link clinical.web.vo.SubjectAsScoreValueSummary}
	 *         objects
	 *
	 * @throws Exception
	 * @see clinical.web.vo.SubjectAsScoreValueSummary
	 */
	public List<SubjectAsScoreValueSummary> queryForScores(UserInfo userInfo,
			Operator op, List<AssessmentSelectionInfo> asiList,
			String queryScope, String primarySiteID) throws Exception;

	/**
	 * returns a list of {@link clinical.web.vo.SubjectAsScoreValueSummary}
	 * objects.
	 *
	 * @param userInfo
	 *            User Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param op
	 * @param asiList
	 * @param experimentIDs
	 *            a list of experiment ids as <code>Integer</code> objects.
	 * @param queryScope
	 * @param primarySiteID
	 * @return list of {@link clinical.web.vo.SubjectAsScoreValueSummary}
	 *         objects
	 *
	 * @throws Exception
	 * @see clinical.web.vo.SubjectAsScoreValueSummary
	 */
	public List<SubjectAsScoreValueSummary> queryForScores(UserInfo userInfo,
			Operator op, List<AssessmentSelectionInfo> asiList,
			List<Integer> experimentIDs, String queryScope, String primarySiteID)
			throws Exception;

	/**
	 *
	 * Returns a list of {@link clinical.web.vo.SubjectAsScoreValueSummary}
	 * objects from the database specified by <code>dbID</code>.
	 *
	 * @param userInfo
	 *            User Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param dbID
	 * @param firstDBID
	 * @param siteAsMap
	 * @param op
	 * @param asiList
	 *            list of <code>AssessmentSelectionInfo</code> objects
	 * @param experimentIDs
	 *            a list of experiment ids as <code>Integer</code> objects.
	 * @return
	 * @throws java.lang.Exception
	 *
	 * @see clinical.web.vo.SubjectAsScoreValueSummary
	 */
	public List<SubjectAsScoreValueSummary> queryForScores(UserInfo userInfo,
			String dbID, String firstDBID,
			Map<Integer, List<AssessmentMapping>> siteAsMap, Operator op,
			List<AssessmentSelectionInfo> asiList, List<Integer> experimentIDs)
			throws Exception;

	/**
	 * Queries mediator and returns a list of SubjectAsScoreValueSummary
	 * objects.
	 *
	 * @param op
	 * @param asiList
	 * @param siteAsMap
	 * @return list of <code>SubjectAsScoreValueSummary</code> objects
	 *
	 * @throws Exception
	 * @see clinical.web.vo.SubjectAsScoreValueSummary
	 */
	public List<SubjectAsScoreValueSummary> queryMediatorForScores(Operator op,
			List<AssessmentSelectionInfo> asiList,
			Map<Integer, List<AssessmentMapping>> siteAsMap) throws Exception;

	/**
	 * This method retrieves the assessments from mediated sources and creates
	 * an association table. It assumes there are only two mediated sources.
	 *
	 * @param initiatingSrcName
	 * @param otherSrcName
	 * @param assessments
	 * @return a map of Assessment IDs (as Integer objects) keyed by the
	 *         sourceName + assessmentName combination
	 * @throws java.lang.Exception
	 */
	public Map<String, Integer> getAssessmentsFromMediator(
			String initiatingSrcName, String otherSrcName,
			List<Assessment> assessments) throws Exception;

	/**
	 *
	 * This method retrieves the assessments from mediated sources and creates
	 * an association table. It assumes there are only two mediated sources.
	 *
	 * @param initiatingSrcName
	 * @param otherSrcName
	 * @param assessments
	 * @return a map of score names as <code>String</code></code> objects
	 *         keyed by the following;
	 *
	 * <pre>
	 *  &lt;sourceName&gt;_&lt;assessment-id&gt;
	 * </pre>
	 *
	 * @throws java.lang.Exception
	 */
	public Map<String, String> getAssessmentScoresFromMediator(
			String initiatingSrcName, String otherSrcName,
			List<Assessment> assessments) throws Exception;

	/**
	 *
	 * @param userInfo
	 *            User Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param scoreInfos
	 *            a list of {@link clinical.web.common.vo.AsScoreInfo} objects
	 * @param subCorticalVars
	 *            can be null
	 * @return a list of {@link clinical.web.vo.RangeInfo} objects for each
	 */
	public List<RangeInfo> getRangeInfos(UserInfo userInfo,
			List<AsScoreInfo> scoreInfos,
			List<SubCorticalVarInfo> subCorticalVars) throws Exception;

	/**
	 * Returns a list of previously saved user query in XML form from the
	 * database. Please note that for efficiency reasons, only descriptin and
	 * uniqueid fields are retrieved.
	 *
	 * @param userInfo
	 *            user Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param userName
	 *            the name of the user for which the stored query will be
	 *            retrieved
	 * @return a list of {@link clinical.server.vo.Storedquery} value objects (
	 *         or empty list if none exists) Please note that for efficiency
	 *         reasons, only descriptin and uniqueid fields are retrieved.
	 *
	 * @throws java.lang.Exception
	 */
	public List<Storedquery> getAvailableStoredQueries(UserInfo userInfo,
			String userName) throws Exception;

	/**
	 * returns a particular previously saved user query information from the
	 * database.
	 *
	 * @param userInfo
	 *            user Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param queryID
	 *            unique query ID
	 * @param userName
	 *            the name of the user for which the stored query will be
	 *            retrieved
	 * @return a {@link clinical.server.vo.Storedquery} value object
	 * @throws java.lang.Exception
	 */
	public Storedquery getFullStoredQuery(UserInfo userInfo,
			BigDecimal queryID, String userName) throws Exception;

	/**
	 * Saves the XML representation of an user query to the database.
	 *
	 * @param userInfo
	 *            user Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param queryState
	 *            the XML representation of the user query
	 * @param description
	 *            a short description (or label) for the persisted query
	 * @throws java.lang.Exception
	 */
	public void saveUserQuery(UserInfo userInfo, String queryState,
			String description) throws Exception;

	/**
	 *
	 * returns all score values for the provided assessments given the search
	 * results of an assessment query.
	 *
	 * @param userInfo
	 *            user Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param qsListt
	 *            list of {@link clinical.web.vo.QuerySummary} objects
	 * @param asiList
	 *            list of {@link clinical.web.common.vo.AssessmentSelectionInfo}
	 *            objects
	 * @return a {@link clinical.web.services.BatchQueryResult} wrapper object
	 *         containing a list of
	 *         {@link clinical.web.vo.SubjectAsScoreValueSummary} objects for
	 *         full assessments
	 * @throws Exception
	 */
	public BatchQueryResult doBatchQuery(UserInfo userInfo,
			List<QuerySummary> qsList, List<AssessmentSelectionInfo> asiList)
			throws Exception;

	/**
	 *
	 * @param userInfo
	 * @param sidbList
	 *            a list of {@link clinical.web.download.SubjectImageDataBundle}
	 *            objects
	 * @param asiList
	 *            list of {@link clinical.web.common.vo.AssessmentSelectionInfo}
	 *            objects
	 * @return a {@link clinical.web.services.BatchQueryResult} wrapper object
	 *         containing a list of
	 *         {@link clinical.web.vo.SubjectAsScoreValueSummary} objects for
	 *         full assessments
	 * @throws Exception
	 */
	public BatchQueryResult getFullAssessments(UserInfo userInfo,
			List<SubjectImageDataBundle> sidbList,
			List<AssessmentSelectionInfo> asiList) throws Exception;
}
