package clinical.web.common.vo;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import clinical.utils.GenUtils;
import clinical.web.Constants;
import clinical.web.common.query.BinaryOperator;
import clinical.web.common.query.Operator;
import clinical.web.common.query.QueryUtils;
import clinical.web.common.query.SearchPredicate;
import clinical.web.common.query.UnaryOperator;
import clinical.web.common.query.SearchPredicate.DateRange;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: AnalysisResultSummary.java,v 1.1.2.1 2008/07/03 21:46:56
 *          bozyurt Exp $
 */
public class AnalysisResultSummary implements Serializable {
	private static final long serialVersionUID = 1L;
	private String subjectID;
	private String expID;
	private String expName = "unknown";
	private String siteID;
	private String siteName;
	private boolean isRemote;
	private String timeStamp;
	private Map<String, VisitSegAnalysisResultValues> vsarvMap = new LinkedHashMap<String, VisitSegAnalysisResultValues>(
			3);
	private Set<String> uniqSiteIDs = new TreeSet<String>();
	private Map<String, String> paramMap = new HashMap<String, String>(3);

	public AnalysisResultSummary(String subjectID, String expID,
			String expName, String siteID, String timeStamp) {
		super();
		this.subjectID = subjectID;
		this.expID = expID;
		this.expName = expName;
		this.siteID = siteID;
		this.timeStamp = timeStamp;
	}

	public String getKey() {
		StringBuilder sb = new StringBuilder();
		sb.append(subjectID).append(":").append(expID);
		sb.append(':').append(siteID);
		return sb.toString();
	}

	public static String getKey(String subjectID, int expID, String siteID) {
		StringBuilder sb = new StringBuilder();
		sb.append(subjectID).append(":").append(expID);
		sb.append(':').append(siteID);
		return sb.toString();
	}

	public VisitSegAnalysisResultValues addVisitSegAnalysisResultValues(
			int visitID, int segmentID, int expID, String siteID) {
		int eid = GenUtils.toInt(this.expID, -1);
		if (expID != eid) {
			throw new RuntimeException("Invalid expID:" + expID + " Expected:"
					+ this.expID);
		}
		String key = getKey(visitID, segmentID, expID, siteID);
		VisitSegAnalysisResultValues vsarv = new VisitSegAnalysisResultValues(
				visitID, segmentID, expID, siteID);
		vsarvMap.put(key, vsarv);
		if (siteID != null)
			uniqSiteIDs.add(siteID);
		return vsarv;
	}

	public VisitSegAnalysisResultValues getVisitSegAnalysisResultValues(
			int visitID, int segmentID, int expID, String siteID) {
		String key = getKey(visitID, segmentID, expID, siteID);
		VisitSegAnalysisResultValues vsarv = vsarvMap.get(key);
		return vsarv;
	}

	public VisitSegAnalysisResultValues getOrPutVisitSegAnalysisResultValues(
			int visitID, int segmentID, int expID, String siteID) {
		String key = getKey(visitID, segmentID, expID, siteID);
		VisitSegAnalysisResultValues vsarv = vsarvMap.get(key);
		if (vsarv == null) {
			vsarv = new VisitSegAnalysisResultValues(visitID, segmentID, expID,
					siteID);
			vsarvMap.put(key, vsarv);
			if (siteID != null)
				uniqSiteIDs.add(siteID);
		}
		return vsarv;
	}

	public static boolean needsFiltering(Operator root) {
		ResultWrapper rw = new ResultWrapper(false);
		needsFiltering(root, rw);
		return rw.result;
	}

	protected static void needsFiltering(Operator op, ResultWrapper rw) {
		if (op instanceof UnaryOperator) {
			UnaryOperator uop = (UnaryOperator) op;
			SearchPredicate sp = uop.getPredicate();
			if (!sp.getValue().equals("*")) {
				rw.result = true;
				return;
			}
		} else if (op instanceof BinaryOperator) {
			BinaryOperator bop = (BinaryOperator) op;
			needsFiltering(bop.getLhs(), rw);
			if (rw.result)
				return;
			needsFiltering(bop.getRhs(), rw);
			if (rw.result)
				return;
		}
	}

	/**
	 * 
	 * @param op
	 * @return true if every {@link VisitSegAnalysisResultValues} object
	 *         contained is filtered and hence this AnalysisResultSummary object
	 *         can be dropped.
	 */
	public boolean filter(Operator op, String queryScope) {
		if (queryScope.equals(Constants.SEGMENT_SCOPE)) {
			List<VisitSegAnalysisResultValues> vsarvList = new ArrayList<VisitSegAnalysisResultValues>(
					vsarvMap.values());
			for (VisitSegAnalysisResultValues vsarv : vsarvList) {
				if (isFiltered(op, vsarv)) {
					String key = getKey(vsarv.getVisitID(), vsarv
							.getSegmentID(), vsarv.getExpID(), siteID);
					vsarvMap.remove(key);
				}
			}
		} else if (queryScope.equals(Constants.VISIT_SCOPE)) {
			Map<Integer, List<VisitSegAnalysisResultValues>> map = new HashMap<Integer, List<VisitSegAnalysisResultValues>>(
					5);
			for (VisitSegAnalysisResultValues vsarv : vsarvMap.values()) {
				List<VisitSegAnalysisResultValues> list = map.get(vsarv
						.getVisitID());
				if (list == null) {
					list = new ArrayList<VisitSegAnalysisResultValues>(2);
					map.put(vsarv.getVisitID(), list);
				}
				list.add(vsarv);
			}
			for (Iterator<List<VisitSegAnalysisResultValues>> it = map.values()
					.iterator(); it.hasNext();) {
				List<VisitSegAnalysisResultValues> visitList = it.next();
				if (isFiltered(op, visitList)) {
					for (VisitSegAnalysisResultValues vsarv : visitList) {
						String key = getKey(vsarv.getVisitID(), vsarv
								.getSegmentID(), vsarv.getExpID(), siteID);
						vsarvMap.remove(key);
					}
				}
			}
		}
		return (vsarvMap.isEmpty());
	}

	protected boolean isFiltered(Operator op, VisitSegAnalysisResultValues vsarv) {
		ResultWrapper result = new ResultWrapper();
		checkCondition(op, vsarv, result);
		return !result.result;
	}

	protected boolean isFiltered(Operator op,
			List<VisitSegAnalysisResultValues> vsarvList) {
		ResultWrapper result = new ResultWrapper();
		checkCondition(op, vsarvList, result);
		return !result.result;
	}

	protected void checkCondition(Operator op,
			List<VisitSegAnalysisResultValues> vsarvList, ResultWrapper result) {
		if (op instanceof UnaryOperator) {
			UnaryOperator uop = (UnaryOperator) op;
			SearchPredicate sp = uop.getPredicate();
			TupleQueryInfo tqi = (TupleQueryInfo) sp.getAttribute();
			String mgName = tqi.getMeasGroupName();
			String observationName = tqi.getCtdi().getColumnName();
			Object value = null;
			// TODO assumption only one matching mgName + observationName
			// combination is checked within a visit
			for (VisitSegAnalysisResultValues vsarv : vsarvList) {
				value = vsarv.getValue(mgName, observationName);
				if (value != null)
					break;
			}
			if (value != null) {
				checkPredicateCondition(result, sp, value);
			} else {
				// missing value does not satisfy condition by default
				result.result = false;
			}
		} else if (op instanceof BinaryOperator) {
			BinaryOperator bop = (BinaryOperator) op;
			Operator lhs = bop.getLhs();
			int connective = bop.getLogicalOp();
			checkCondition(lhs, vsarvList, result);
			boolean satisfied = result.result;
			if (connective == Operator.AND && !satisfied) {
				// short circuit
				return;
			}
			Operator rhs = bop.getRhs();
			checkCondition(rhs, vsarvList, result);
			if (connective == Operator.AND) {
				result.result = satisfied && result.result;
			} else {
				result.result = satisfied || result.result;
			}
		}
	}

	protected void checkPredicateCondition(ResultWrapper result,
			SearchPredicate sp, Object value) {
		String value2Check = value.toString();
		if (sp.getValue() instanceof SearchPredicate.Range) {
			SearchPredicate.Range range = (SearchPredicate.Range) sp.getValue();
			boolean satisfied = QueryUtils.condSatisfied(range, value2Check);
			result.result = satisfied;
		} else if (sp.getValue() instanceof SearchPredicate.DateRange) {
			SearchPredicate.DateRange range = (DateRange) sp.getValue();
			boolean satisfied = QueryUtils.condSatisfied(range, value2Check);
			result.result = satisfied;
		} else {
			boolean satisfied = QueryUtils.condSatisfied(sp.getValue(), sp
					.getOperator(), sp.getType(), value2Check);
			result.result = satisfied;
		}
	}

	protected void checkCondition(Operator op,
			VisitSegAnalysisResultValues vsarv, ResultWrapper result) {
		if (op instanceof UnaryOperator) {
			UnaryOperator uop = (UnaryOperator) op;
			SearchPredicate sp = uop.getPredicate();
			TupleQueryInfo tqi = (TupleQueryInfo) sp.getAttribute();
			String mgName = tqi.getMeasGroupName();
			String observationName = tqi.getCtdi().getColumnName();
			Object value = vsarv.getValue(mgName, observationName);
			if (value != null) {
				checkPredicateCondition(result, sp, value);
			} else {
				// missing value does not satisfy condition by default
				result.result = false;
			}
		} else if (op instanceof BinaryOperator) {
			BinaryOperator bop = (BinaryOperator) op;
			Operator lhs = bop.getLhs();
			int connective = bop.getLogicalOp();
			checkCondition(lhs, vsarv, result);
			boolean satisfied = result.result;
			if (connective == Operator.AND && !satisfied) {
				// short circuit
				return;
			}
			Operator rhs = bop.getRhs();
			checkCondition(rhs, vsarv, result);
			if (connective == Operator.AND) {
				result.result = satisfied && result.result;
			} else {
				result.result = satisfied || result.result;
			}
		}
	}

	static class ResultWrapper {
		boolean result = true;

		public ResultWrapper() {
		}

		public ResultWrapper(boolean result) {
			this.result = result;
		}
	}// ;

	public Map<String, String> getParamMap() {
		if (paramMap.isEmpty()) {
			paramMap.put("sid", getSubjectID());
			paramMap.put("siteid", getSiteID());
		}
		return paramMap;
	}

	public String getParamJSON() {
		StringBuilder buf = new StringBuilder();
		buf.append("['subjectID=").append(subjectID).append("',");
		buf.append("'siteID=").append(siteID);
		buf.append("','expID=").append(expID).append("']");
		return buf.toString();
	}

	public String getAllSiteIDs() {
		StringBuilder buf = new StringBuilder();
		for (Iterator<String> it = uniqSiteIDs.iterator(); it.hasNext();) {
			String sid = it.next();
			buf.append(sid);
			if (it.hasNext())
				buf.append(',');
		}
		return buf.toString();
	}

	public static String getKey(int visitID, int segmentID, int expID,
			String siteID) {
		StringBuilder sb = new StringBuilder();
		sb.append(visitID).append(':').append(segmentID).append(':').append(
				expID);
		sb.append(':').append(siteID);
		return sb.toString();
	}

	public String getSubjectID() {
		return subjectID;
	}

	public String getExpID() {
		return expID;
	}

	public String getExpName() {
		return expName;
	}

	public String getSiteID() {
		System.out.println(this);
		return siteID;
	}

	public String getTimeStamp() {
		return timeStamp;
	}

	public Map<String, VisitSegAnalysisResultValues> getVsarvMap() {
		return vsarvMap;
	}

	public List<VisitSegAnalysisResultValues> getVsarvList() {
		List<VisitSegAnalysisResultValues> list = new ArrayList<VisitSegAnalysisResultValues>(
				vsarvMap.values());
		return list;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((expID == null) ? 0 : expID.hashCode());
		result = prime * result + ((siteID == null) ? 0 : siteID.hashCode());
		result = prime * result
				+ ((subjectID == null) ? 0 : subjectID.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		final AnalysisResultSummary other = (AnalysisResultSummary) obj;
		if (expID == null) {
			if (other.expID != null)
				return false;
		} else if (!expID.equals(other.expID))
			return false;
		if (siteID == null) {
			if (other.siteID != null)
				return false;
		} else if (!siteID.equals(other.siteID))
			return false;
		if (subjectID == null) {
			if (other.subjectID != null)
				return false;
		} else if (!subjectID.equals(other.subjectID))
			return false;
		return true;
	}

	/**
	 * toString method: creates a String representation of the object
	 * 
	 * @return the String representation
	 * @author
	 */
	public String toString() {
		StringBuilder buffer = new StringBuilder();
		buffer.append("AnalysisResultSummary::[");
		buffer.append("expID = ").append(expID);
		buffer.append(", siteID = ").append(siteID);
		buffer.append(", subjectID = ").append(subjectID);
		buffer.append(", isremote = ").append(isRemote);
		for (VisitSegAnalysisResultValues vsarv : vsarvMap.values()) {
			buffer.append("\n\t").append(vsarv.toString());
		}
		buffer.append("]");
		return buffer.toString();
	}

	public Set<String> getUniqSiteIDs() {
		return uniqSiteIDs;
	}

	public String getSiteName() {
		return siteName;
	}

	public void setSiteName(String siteName) {
		this.siteName = siteName;
	}

	public boolean isRemote() {
		return isRemote;
	}

	public void setRemote(boolean isRemote) {
		this.isRemote = isRemote;
	}

	public void setSiteID(String siteID) {
		this.siteID = siteID;
		if (siteID != null)
			uniqSiteIDs.add(siteID);
	}

}
