package clinical.web.dd;

import java.io.File;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import clinical.utils.Assertion;
import clinical.utils.GenUtils;
import clinical.utils.XCEDEUtils;
import clinical.web.ConnectionSupportMixin;
import clinical.web.IProvenanceService;
import clinical.web.IXCEDEDerivedDataService;
import clinical.web.MinimalServiceFactory;
import clinical.web.ServiceFactory;
import clinical.web.services.SecurityService;
import clinical.xml.xcede2.AnalysisT;
import clinical.xml.xcede2.LevelDataRefsT;
import clinical.xml.xcede2.XCEDE;

/**
 * @author I. Burak Ozyurt
 * @version $Id: XCEDE2AnalysisImporter.java,v 1.3 2008/03/27 00:25:46 bozyurt
 *          Exp $
 */
public class XCEDE2AnalysisImporter {
	protected ConnectionSupportMixin csm;

	public XCEDE2AnalysisImporter(String propsFile) throws Exception {
		csm = new ConnectionSupportMixin(propsFile, true);
		// MinimalServiceFactory.setMimimalOpMode(true);
		ServiceFactory.setMimimalOpMode(false);
		csm.startup();

		SecurityService ss = SecurityService.getInstance(
				csm.getDbPoolService(), csm.getDbID(), csm.getDbType());
		ss.startup();
	}

	public XCEDE2AnalysisImporter(File usersXMLFile) throws Exception {
		csm = new ConnectionSupportMixin(usersXMLFile.getName());
		MinimalServiceFactory.setMimimalOpMode(false);
		ServiceFactory.setMimimalOpMode(false);
		csm.startup();
		SecurityService ss = SecurityService.getInstance(
				csm.getDbPoolService(), csm.getDbID(), csm.getDbType());
		ss.startup();
	}


	public XCEDE2AnalysisImporter(ConnectionSupportMixin csm) throws Exception {
		this.csm = csm;
		ServiceFactory.setMimimalOpMode(false);
		SecurityService ss = SecurityService.getInstance(
				csm.getDbPoolService(), csm.getDbID(), csm.getDbType());
		ss.startup();
	}

	public void shutdown() {
		if (csm != null)
			try {
				csm.shutdown();
			} catch (Exception e) {
				e.printStackTrace();
			}
	}

	public List<AnalysisT> loadMergedFSXCEDE(String mergedXcedeFile)
			throws Exception {
		XCEDE xcede = XCEDEUtils.unmarshal(mergedXcedeFile);
		List<Object> aList = xcede.getAnnotationListOrRevisionListOrProject();
		List<AnalysisT> analysisTypes = new ArrayList<AnalysisT>(5);
		if (!aList.isEmpty() && aList.get(0) instanceof AnalysisT) {
			for (Iterator<?> it = aList.iterator(); it.hasNext();) {
				AnalysisT a = (AnalysisT) it.next();
				analysisTypes.add(a);
			}
		}
		return analysisTypes;
	}

	protected AnalysisT findProvenanceAnalysis(List<AnalysisT> analysisTypeList) {
		for (AnalysisT a : analysisTypeList) {
			if (!a.getInput().isEmpty() && !a.getProvenance().isEmpty()) {
				return a;
			}
		}
		return null;
	}

	public void importFreesurferAnalyses(String mergedXcedeFile,
			String shortAnalysisName, String snapshotID) throws Exception {
		List<AnalysisT> atList = loadMergedFSXCEDE(mergedXcedeFile);
		AnalysisT provenanceAT = findProvenanceAnalysis(atList);
		List<AnalysisT> remAtList = new ArrayList<AnalysisT>(atList.size() - 1);
		for (AnalysisT at : atList) {
			if (at != provenanceAT) {
				remAtList.add(at);
			}
		}
		LevelDataRefsT ldr = provenanceAT.getInput().get(0);
		String subjectID = ldr.getSubjectID();
		String expName = ldr.getProjectID();
		String visitStr = ldr.getVisitID();
		String[] toks = visitStr.split("__");
		int visitID = GenUtils.toInt(toks[1], -1);

		String segmentName = ldr.getEpisodeID();

		Assertion.assertTrue(visitID != -1);
		IXCEDEDerivedDataService xdds = ServiceFactory
				.getXCEDEDerivedDataService(csm.getDbID());

		xdds.addAnalysesForSubject(csm.getUi(), subjectID, expName, visitID,
				segmentName, provenanceAT, remAtList, shortAnalysisName,
				snapshotID);
	}

	public void removeFreesurferAnalyses(String mergedXcedeFile)
			throws Exception {
		List<AnalysisT> atList = loadMergedFSXCEDE(mergedXcedeFile);
		AnalysisT provenanceAT = findProvenanceAnalysis(atList);

		IXCEDEDerivedDataService xdds = ServiceFactory
				.getXCEDEDerivedDataService(csm.getDbID());

		xdds.deleteAnalysesForSubject(csm.getUi(), provenanceAT);
	}

	public void addProvenanceForAnalysisInstance(String mergedXcedeFile,
			String shortAnalysisName, String snapshotID) throws Exception {
		List<AnalysisT> atList = loadMergedFSXCEDE(mergedXcedeFile);
		AnalysisT provenanceAT = findProvenanceAnalysis(atList);
		Assertion.assertNotNull(provenanceAT);
		IProvenanceService prvs = ServiceFactory.getProvenanceService(csm
				.getDbID());

		prvs.addProvenanceForAnalysisInstance(csm.getUi(), provenanceAT,
				shortAnalysisName, snapshotID);
	}

	public void deleteProvenanceForAnalysisInstance(String mergedXcedeFile)
			throws Exception {
		List<AnalysisT> atList = loadMergedFSXCEDE(mergedXcedeFile);
		AnalysisT provenanceAT = findProvenanceAnalysis(atList);
		Assertion.assertNotNull(provenanceAT);
		IProvenanceService prvs = ServiceFactory.getProvenanceService(csm
				.getDbID());

		prvs.deleteProvenanceForAnalysisInstance(csm.getUi(), provenanceAT);
	}

	public static void usage() {
		System.err
				.println("Usage: XCEDE2AnalysisImporter <usersXMLFile> <xcedeDir> -c <command>");
		System.exit(-1);
	}

	public static void main(String[] args) throws Exception {
		if (args.length != 4) {
			usage();
		}
		String usersXMLFile = args[0];
		String xcedeDir = args[1];
		if (!args[2].equals("-c"))
			usage();
		String cmd = args[3];
		if (!cmd.equals("delete"))
			usage();

		System.out.println("usersXMLFile=" + usersXMLFile + "\nxcedeDir="
				+ xcedeDir);

		XCEDE2AnalysisMerger merger = new XCEDE2AnalysisMerger();
		List<File> xcedeFiles = merger.findXCEDEFiles(new File(xcedeDir));
		File mergedFile = new File(xcedeDir, "merged.xml");
		XCEDE xcede = merger.merge(xcedeFiles);

		merger.saveMerged(xcede, mergedFile);

		XCEDE2AnalysisImporter importer = null;
		try {
			importer = new XCEDE2AnalysisImporter(new File(usersXMLFile));

			importer.removeFreesurferAnalyses(mergedFile.getAbsolutePath());

		} finally {
			if (importer != null)
				importer.shutdown();
		}

		// testDriver();

	}

	@SuppressWarnings("unused")
   private static void testDriver() throws Exception {
		String rootDir = "/home/bozyurt/work/fsanalysis/fBIRNPhaseII__0010/Data/001029291693/scanVisit__0010__0002/MRI__0001/t1/Analysis/FreeSurfer__0010__0001";
		XCEDE2AnalysisImporter importer = null;
		try {
			importer = new XCEDE2AnalysisImporter("test.properties");
			String mergedXcedeFile = rootDir + "/merged.xml";

			// importer.addProvenanceForAnalysisInstance(mergedXcedeFile,
			// "FreeSurfer", "0001");

			// importer.deleteProvenanceForAnalysisInstance(mergedXcedeFile);

			// importer.importFreesurferAnalyses(mergedXcedeFile, "FreeSurfer",
			// "0001");

			importer.removeFreesurferAnalyses(mergedXcedeFile);
		} finally {
			if (importer != null)
				importer.shutdown();
		}
	}
}
