package clinical.web.helpers;

import java.awt.Color;
import java.io.File;
import java.io.IOException;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.math.stat.regression.SimpleRegression;
import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;
import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartRenderingInfo;
import org.jfree.chart.ChartUtilities;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.entity.ChartEntity;
import org.jfree.chart.entity.EntityCollection;
import org.jfree.chart.entity.StandardEntityCollection;
import org.jfree.chart.imagemap.ToolTipTagFragmentGenerator;
import org.jfree.chart.imagemap.URLTagFragmentGenerator;
import org.jfree.chart.plot.PlotOrientation;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.renderer.xy.XYLineAndShapeRenderer;
import org.jfree.data.Range;
import org.jfree.data.category.DefaultCategoryDataset;
import org.jfree.data.function.Function2D;
import org.jfree.data.function.LineFunction2D;
import org.jfree.data.general.DatasetUtilities;
import org.jfree.data.statistics.Regression;
import org.jfree.data.xy.XYDataItem;
import org.jfree.data.xy.XYDataset;
import org.jfree.data.xy.XYSeries;
import org.jfree.data.xy.XYSeriesCollection;
import org.jfree.util.StringUtils;

import clinical.utils.FileUtils;
import clinical.utils.GenUtils;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: ChartingHelper.java 62 2009-05-29 23:54:50Z bozyurt $
 */

public class ChartingHelper {

	public ChartingHelper() {
	}

	public void prepareBarChart(
			Map<String, Map<Object, List<Double>>> seriesCategoryDataMap,
			String title, String xAxisLabel, String yAxisLabel,
			boolean showLegend, String imageFilename) throws IOException {
		DefaultCategoryDataset dataSet = prepareCategoryDataSet(seriesCategoryDataMap);

		JFreeChart chart = ChartFactory.createBarChart(title, xAxisLabel,
				yAxisLabel, dataSet, PlotOrientation.VERTICAL, showLegend,
				false, // no tooltips
				false // no URLS
				);
		ChartUtilities.saveChartAsPNG(new File(imageFilename), chart, 600, 500);

	}

	private DefaultCategoryDataset prepareCategoryDataSet(
			Map<String, Map<Object, List<Double>>> seriesCategoryDataMap) {
		DefaultCategoryDataset dataSet = new DefaultCategoryDataset();
		for (Map.Entry<String, Map<Object, List<Double>>> entry : seriesCategoryDataMap
				.entrySet()) {
			String seriesName = (String) entry.getKey();
			Map<Object, List<Double>> categoryMap = entry.getValue();
			for (Map.Entry<Object, List<Double>> cmEntry : categoryMap
					.entrySet()) {
				String categoryName = (String) cmEntry.getKey();
				List<Double> catData = cmEntry.getValue();
				for (Double dataValue : catData) {
					dataSet.addValue(dataValue.doubleValue(), seriesName,
							categoryName);
				}
			}
		}
		return dataSet;
	}

	private XYSeriesCollection prepareXYDataSet(
			LinkedHashMap<String, List<DataPoint>> seriesDataMap) {
		XYSeriesCollection dataSet = new XYSeriesCollection();
		for (Map.Entry<String,List<DataPoint>> entry : seriesDataMap.entrySet()) {
			String seriesName = entry.getKey();
			List<DataPoint> seriesData = entry.getValue();
			XYSeries series = new XYSeries(seriesName);
			for (Object element : seriesData) {
				DataPoint dp = (DataPoint) element;
				series.add(dp.getX(), dp.getY());
			}
			dataSet.addSeries(series);
		}
		return dataSet;
	}

	@SuppressWarnings("unused")
   private Map<Comparable<?>, double[]> prepareSeriesOLSCoeffMap(
			XYSeriesCollection dataSet) {
		Map<Comparable<?>, double[]> coeffMap = new HashMap<Comparable<?>, double[]>(
				7);
		for (Iterator<?> iter = dataSet.getSeries().iterator(); iter.hasNext();) {
			XYSeries series = (XYSeries) iter.next();
			double[] coeffs = Regression.getOLSRegression(
					new XYSeriesCollection(series), 0);
			coeffMap.put(series.getKey(), coeffs);
		}
		return coeffMap;
	}

	private Map<Comparable<?>, double[]> prepareSeriesOLSCoeffMap2(
			XYSeriesCollection dataSet,
			Map<Comparable<?>, RegressionInfo> regMap) {
		Map<Comparable<?>, double[]> coeffMap = new HashMap<Comparable<?>, double[]>(
				7);
		for (Iterator<?> iter = dataSet.getSeries().iterator(); iter.hasNext();) {
			XYSeries series = (XYSeries) iter.next();
			SimpleRegression reg = new SimpleRegression();
			for (Iterator<?> it = series.getItems().iterator(); it.hasNext();) {
				XYDataItem xyData = (XYDataItem) it.next();
				reg.addData(xyData.getX().doubleValue(), xyData.getY()
						.doubleValue());
			}
			double[] coeffs = new double[2];
			coeffs[0] = reg.getIntercept();
			coeffs[1] = reg.getSlope();
			coeffMap.put(series.getKey(), coeffs);
			regMap.put(series.getKey(), new RegressionInfo(reg, (String) series
					.getKey()));
		}
		return coeffMap;
	}

	public ChartRenderingInfo prepareScatterPlotWithRegressionLines(
			LinkedHashMap<String, List<DataPoint>> seriesDataMap,
			String xAxisLabel, String yAxisLabel, String title,
			boolean showLegend, String imageFilename,
			Map<Comparable<?>, RegressionInfo> regMap,
			List<JSURLGenerator> urlGenerators) throws IOException {
		XYSeriesCollection dataSet = prepareXYDataSet(seriesDataMap);
		Map<Comparable<?>, double[]> coeffMap = prepareSeriesOLSCoeffMap2(
				dataSet, regMap);

		NumberAxis xAxis = new NumberAxis(xAxisLabel);
		NumberAxis yAxis = new NumberAxis(yAxisLabel);
		xAxis.setAutoRangeIncludesZero(false);
		yAxis.setAutoRangeIncludesZero(false);

		Range domainRange = DatasetUtilities.findDomainBounds(dataSet);

		XYLineAndShapeRenderer r1 = null;
		XYLineAndShapeRenderer r2 = null;

		XYPlot plot = null;
		Color[] seriesColors = { Color.BLUE, Color.RED, Color.GREEN,
				Color.MAGENTA, Color.DARK_GRAY, Color.ORANGE, Color.BLACK };
		int seriesCount = dataSet.getSeriesCount();

		for (int i = 0; i < seriesCount; i++) {
			r1 = new XYLineAndShapeRenderer(false, true);
			r2 = new XYLineAndShapeRenderer(true, false);
			JSURLGenerator jug = new JSURLGenerator(i);
			urlGenerators.add(jug);
			r1.setURLGenerator(jug);
			if (i < seriesColors.length) {
				r1.setSeriesPaint(0, seriesColors[i]);
			}
			XYSeries series = dataSet.getSeries(i);
			if (i == 0) {
				plot = new XYPlot(new XYSeriesCollection(series), xAxis, yAxis,
						r1);
			} else {
				plot.setDataset(2 * i, new XYSeriesCollection(series));
				plot.setRenderer(2 * i, r1);
			}

			// now the regression line
			String seriesName = (String) dataSet.getSeriesKey(i);
			double[] coeffs = coeffMap.get(seriesName);
			Function2D line = new LineFunction2D(coeffs[0], coeffs[1]);

			XYDataset regData = DatasetUtilities.sampleFunction2D(line,
					domainRange.getLowerBound(), domainRange.getUpperBound(),
					100, "Regression Line for " + seriesName);
			plot.setDataset(2 * i + 1, regData);
			plot.setRenderer(2 * i + 1, r2);

			// set the regression line same color as the scatter data
			r2.setSeriesPaint(0, r1.getSeriesPaint(0));
		}
		JFreeChart chart = new JFreeChart(title, JFreeChart.DEFAULT_TITLE_FONT,
				plot, showLegend);

		ChartRenderingInfo info = new ChartRenderingInfo(
				new StandardEntityCollection());
		ChartUtilities.saveChartAsPNG(new File(imageFilename), chart, 650, 850,
				info);
		return info;
	}

	public static String getImageMap(String name, ChartRenderingInfo info) {
		return getImageMap(name, info, null, new JSURLTagFragmentGenerator());
	}

	public static String getImageMap(String name, ChartRenderingInfo info,
			ToolTipTagFragmentGenerator toolTipTagFragmentGenerator,
			URLTagFragmentGenerator urlTagFragmentGenerator) {

		StringBuffer sb = new StringBuffer();
		sb.append("<map id=\"" + name + "\" name=\"" + name + "\">");
		sb.append(StringUtils.getLineSeparator());
		EntityCollection entities = info.getEntityCollection();
		if (entities != null) {
			int count = entities.getEntityCount();
			for (int i = 0; i < count; i++) {

				ChartEntity entity = entities.getEntity(i);
				if (entity.getToolTipText() != null
						|| entity.getURLText() != null) {
					String area = entity.getImageMapAreaTag(
							toolTipTagFragmentGenerator,
							urlTagFragmentGenerator);
					if (area.length() > 0) {
						sb.append("<span>");
						sb.append(area).append("</span>");
						sb.append(StringUtils.getLineSeparator());
					}
				}
			}
		}
		sb.append("</map>");
		return sb.toString();

	}

	public static class JSURLTagFragmentGenerator implements
			URLTagFragmentGenerator {

		public String generateURLFragment(String urlText) {
			return " " + urlText;
		}

	}// ;

	/**
	 * 
	 * @param seriesDataMap
	 *            a hash table keyed by series name , containing series XY data
	 *            as a list of <code>DataPoint</code> objects per series.
	 * @param xAxisLabel
	 * @param yAxisLabel
	 * @param title
	 * @param showLegend
	 * @param imageFilename
	 *            the full path of the PNG file the graph will be saved
	 * @throws IOException
	 */
	public void prepareScatterPlot(
			LinkedHashMap<String, List<DataPoint>> seriesDataMap,
			String xAxisLabel, String yAxisLabel, String title,
			boolean showLegend, String imageFilename) throws IOException {
		XYSeriesCollection dataSet = prepareXYDataSet(seriesDataMap);
		JFreeChart chart = ChartFactory.createScatterPlot(title, xAxisLabel,
				yAxisLabel, dataSet, PlotOrientation.VERTICAL, showLegend,
				false, false);
		ChartUtilities.saveChartAsPNG(new File(imageFilename), chart, 600, 500);
	}

	public static String prepareScatterDataInfoJSON(
			LinkedHashMap<String, List<DataPoint>> seriesDataMap,
			List<JSURLGenerator> urlGenerators) {
		class DPInfo {
			int series;
			int item;
			DataPoint dp;

			public DPInfo(int series, int item, DataPoint dp) {
				this.series = series;
				this.item = item;
				this.dp = dp;
			}
		}// ;
		StringBuffer buf = new StringBuffer(3000);
		int i = 0;
		buf.append("{");
		for (Iterator<List<DataPoint>> iter = seriesDataMap.values().iterator(); iter
				.hasNext();) {
			List<DataPoint> seriesData = iter.next();
			if (!seriesData.isEmpty()) {
				buf.append(" series_").append(i).append(": { x: '");
				buf.append("x', y: '");
				buf.append("x'},\n");
			} else {
				buf.append(" series_").append(i).append(
						": { x: null, y: null},\n");
			}
			i++;
		}
		Map<Integer, Map<String, DataPoint>> sdMap = new HashMap<Integer, Map<String, DataPoint>>(
				7);
		i = 0;
		for (Iterator<List<DataPoint>> iter = seriesDataMap.values().iterator(); iter
				.hasNext();) {
			List<DataPoint> seriesData = iter.next();
			Map<String, DataPoint> dpMap = new HashMap<String, DataPoint>();
			sdMap.put(new Integer(i), dpMap);
			for (DataPoint dp : seriesData) {
				StringBuffer keyBuf = new StringBuffer();
				keyBuf.append(dp.getX()).append('_').append(dp.getY());
				dpMap.put(keyBuf.toString(), dp);
			}
			i++;
		}

		List<DPInfo> dpList = new ArrayList<DPInfo>();
		for (Iterator<JSURLGenerator> it = urlGenerators.iterator(); it
				.hasNext();) {
			JSURLGenerator jug = it.next();
			Integer seriesKey = new Integer(jug.getActualSeries());
			LinkedHashMap<String, String> indexMap = jug.getIndexMap();
			for (Map.Entry<String,String> entry : indexMap.entrySet()) {
				String idxKey =  entry.getKey();
				String idxValue = entry.getValue();
				String[] tokens = GenUtils.split(idxKey, "_");
				Map<String,DataPoint> dpMap = sdMap.get(seriesKey);
				DataPoint dp = (DataPoint) dpMap.get(idxValue);

				DPInfo dpi = new DPInfo(seriesKey.intValue(), GenUtils.toInt(
						tokens[1], -1), dp);
				dpList.add(dpi);
			}
		}

		buf.append("map: { ");
		i = 0;
		for (Iterator<DPInfo> it = dpList.iterator(); it.hasNext();) {
			DPInfo dpi = it.next();
			DataPoint dp = dpi.dp;
			buf.append('k').append(dpi.series).append('_');
			buf.append(dpi.item).append(":[");
			buf.append(dp.getX()).append(',').append(dp.getY()).append("]");
			if (it.hasNext()) {
				buf.append(',');
			}
			buf.append("\n");
		}

		buf.append("}\n");
		buf.append("};\n");

		return buf.toString();
	}

	public static void testScatterPlot() throws Exception {
		LinkedHashMap<String, List<DataPoint>> seriesMap = new LinkedHashMap<String, List<DataPoint>>();
		// Random r = new Random();
		int meanShift = 1;
		for (int sIdx = 0; sIdx < 3; ++sIdx) {
			List<DataPoint> series = new ArrayList<DataPoint>();
			for (int i = 0; i < 100; i++) {
				double x = i + meanShift;
				double y = 0.2 * i + meanShift;
				DataPoint dp = new DataPoint(x, y);
				series.add(dp);
			}
			seriesMap.put("Series " + sIdx, series);
			meanShift += 2;
		}

		ChartingHelper ch = new ChartingHelper();

		Map<Comparable<?>, RegressionInfo> regMap = new HashMap<Comparable<?>, RegressionInfo>(
				3);
		String templateDir = "/home/bozyurt/dev/java/BIRN/clinical/conf";
		Properties p = new Properties();
		p.setProperty("file.resource.loader.path", templateDir);
		Velocity.init(p);
		VelocityContext ctx = new VelocityContext();

		List<JSURLGenerator> urlGenerators = new ArrayList<JSURLGenerator>(10);
		String imageFilename = "/tmp/scatter_test.png";
		ChartRenderingInfo info = ch.prepareScatterPlotWithRegressionLines(
				seriesMap, "x", "y", "Dummy lines", true, imageFilename,
				regMap, urlGenerators);
		String imageMapName = "chart";
		String imageMap = ChartingHelper.getImageMap(imageMapName, info);

		System.out.println(imageMap);

		ctx.put("imageMapInfo", imageMap);
		String json = prepareScatterDataInfoJSON(seriesMap, urlGenerators);
		ctx.put("scatterDataInfoJSON", json);
		StringWriter sw = new StringWriter();
		Template template = Velocity.getTemplate("im.vm");
		template.merge(ctx, sw);
		System.out.println(sw.toString());
		FileUtils.save2File(sw.toString(), "/tmp/scatter_test.html");

	}

	public static void main(String[] args) throws Exception {
		ChartingHelper.testScatterPlot();
	}
}
