package clinical.web.helpers;

import gnu.trove.TIntObjectHashMap;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.math.stat.descriptive.DescriptiveStatistics;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.input.SAXBuilder;
import org.jfree.chart.ChartRenderingInfo;

import clinical.utils.Assertion;
import clinical.utils.Executor;
import clinical.utils.FileUtils;
import clinical.utils.GenUtils;
import clinical.utils.ProcessUtils;
import clinical.web.common.vo.AsScoreInfo;
import clinical.web.common.vo.VisitSegmentScoreValues;
import clinical.web.forms.AsQueryBuilderForm;
import clinical.web.forms.StatisticsForm;
import clinical.web.vo.AssessmentResultSummary;
import clinical.web.vo.QuerySummary;
import clinical.web.vo.ScoreMetaData;

/*******************************************************************************
 * @author I. Burak Ozyurt
 * @version $Id: StatsHelper.java 62 2009-05-29 23:54:50Z bozyurt $
 */
public class StatsHelper {
	public static final String TEMP_DIR = "/tmp";
	public static final String REL_IMAGES_PATH = "/mri_images/";

	public StatsHelper() {
	}


	public static Set<ScoreMetaData> getScoreMetaData(
			AsQueryBuilderForm queryForm) {
		Set<ScoreMetaData> uniqSet = new HashSet<ScoreMetaData>(23);

		for (QuerySummary qs : queryForm.getQsList()) {
			AssessmentResultSummary ars = qs.getArs();
			if (ars != null) {
				uniqSet.addAll(ars.getScoreMetadata());
			}
		}
		return uniqSet;
	}

	public static Map<String, String> getScoreMap(
			Set<ScoreMetaData> scoreMetaDataSet, boolean discrete) {
		Map<String, String> scoreMap = new HashMap<String, String>(7);

		if (!scoreMetaDataSet.isEmpty()) {
			for (ScoreMetaData smd : scoreMetaDataSet) {
				if (discrete) {
					if (smd.getScoreType().equals("varchar")) {
						// TODO need assessment name also
						scoreMap.put(smd.getScoreName(), smd.getScoreName());
					}
				} else {
					if (smd.getScoreType().equals("integer")
							|| smd.getScoreType().equals("float")) {
						// TODO need assessment name also
						scoreMap.put(smd.getScoreName(), smd.getScoreName());
					}
				}
			}
		}
		return scoreMap;
	}

	public static void prepareAndWriteInfoFile(AsQueryBuilderForm queryForm,
			String filename, char sepChar) throws IOException {
		StringBuffer infoBuf = new StringBuffer(256);
		StringBuffer buf = new StringBuffer(256);
		buf.append("SubjectID").append(sepChar);
		infoBuf.append("NA").append(sepChar);

		// site ID
		buf.append("SiteID").append(sepChar);
		infoBuf.append("DS").append(sepChar);

		for (Iterator<HeaderInfo> iter = queryForm.getHeaders().iterator(); iter
				.hasNext();) {
			HeaderInfo hi = iter.next();
			for (Iterator<String> it = hi.getSubHeaders().iterator(); it
					.hasNext();) {
				String subHeader = it.next();
				subHeader = subHeader.replace(' ', '_');
				String variableName = hi.getTitle().replace(' ', '_') + "_"
						+ subHeader;
				buf.append(variableName);
				if (it.hasNext()) {
					buf.append(sepChar);
				}
			}
			if (iter.hasNext()) {
				buf.append(sepChar);
			}
		}

		// prepare data type line
		for (Iterator<HeaderInfo> iter = queryForm.getHeaders().iterator(); iter
				.hasNext();) {
			HeaderInfo hi = iter.next();
			for (Iterator<String> it = hi.getDataTypes().iterator(); it
					.hasNext();) {
				String subHeaderDataType = it.next();
				infoBuf.append(subHeaderDataType);
				if (it.hasNext()) {
					infoBuf.append(sepChar);
				}
			}
			if (iter.hasNext()) {
				infoBuf.append(sepChar);
			}
		}

		BufferedWriter out = null;
		try {
			out = new BufferedWriter(new FileWriter(filename), 4096);
			out.write(buf.toString());
			out.write('\n');
			out.write(infoBuf.toString());
			out.write('\n');
		} finally {
			FileUtils.close(out);
		}
	}


	public static LinkedHashMap<String, List<DataPoint>> prepareSeriesDataMap(
			AsQueryBuilderForm queryForm, ScoreMetaData groupVar,
			ScoreMetaData indepVar, ScoreMetaData depVar,
			Map<String, String> varNameMap, boolean useSite,
			Map<String, String> siteNameMap) {
		boolean first = true;
		String indepVarname = null;
		String depVarname = null;
		String groupVarname = null;
		String seriesNameStatic = null;
		String seriesName = null;
		LinkedHashMap<String, List<DataPoint>> seriesMap = new LinkedHashMap<String, List<DataPoint>>(
				7);

		List<ScoreMetaData> varList = new ArrayList<ScoreMetaData>(2);
		varList.add(indepVar);
		varList.add(depVar);
		if (groupVar != null)
			varList.add(groupVar);
		for (QuerySummary qs : queryForm.getQsList()) {
			AssessmentResultSummary ars = qs.getArs();
			if (ars == null)
				continue;
			if (first) {
				first = false;
				indepVarname = indepVar.getScoreName();
				depVarname = depVar.getScoreName();
				varNameMap.put("x", indepVarname);
				varNameMap.put("y", depVarname);
				if (groupVar != null)
					groupVarname = groupVar.getScoreName();
				seriesNameStatic = indepVarname + " vs " + depVarname;
				seriesName = seriesNameStatic;
			}
			Map<String, String> valueMap = ars.getMatchingValues(varList);
			Object xValue = getValue(valueMap, indepVar);
			Object yValue = getValue(valueMap, depVar);
			if (xValue != null && yValue != null) {
				if (groupVarname != null) {
					Object gvValue = getValue(valueMap, groupVar);
					if (gvValue != null) {
						// TODO assumption: all discrete variables are assumed
						// to be strings
						StringBuilder buf = new StringBuilder();
						buf.append(seriesNameStatic);
						buf.append(" (").append(groupVarname);
						buf.append(" - ").append(gvValue).append(')');

						seriesName = buf.toString();
					} else {
						// special case site
						if (!useSite) {
							continue;
						}
					}
				} else {
					// special case site
					if (useSite) {
						StringBuffer buf = new StringBuffer();
						buf.append(seriesNameStatic);
						buf.append(" ( Site");

						// FIXME assuming there will be no multisite assessments
						// for the same subject
						String siteName = (String) siteNameMap.get(ars
								.getSiteID());
						buf.append(" ").append(siteName).append(')');
						seriesName = buf.toString();
					} else {
						seriesName = seriesNameStatic;
					}
				}
				List<DataPoint> seriesData = seriesMap.get(seriesName);
				if (seriesData == null) {
					seriesData = new ArrayList<DataPoint>();
					seriesMap.put(seriesName, seriesData);
				}
				try {
					double x = Double.parseDouble(xValue.toString());
					double y = Double.parseDouble(yValue.toString());

					String subjectID = ars.getSubjectID();
					String siteID = ars.getSiteID();
					SubjectDataPoint dp = new SubjectDataPoint(x, y, subjectID,
							siteID, indepVarname, depVarname);

					seriesData.add((DataPoint) dp);
				} catch (NumberFormatException nfe) {
					// log error
					continue;
				}
			}
		}
		return seriesMap;
	}

	public static String getValue(Map<String, String> valueMap,
			ScoreMetaData smd) {
		for (String key : valueMap.keySet()) {
			String[] toks = key.split(":");
			if (toks[0].equals(smd.getScoreName()))
				return valueMap.get(key);
		}
		return null;
	}


	public static Map<String, Map<Object, List<Double>>> prepareSeriesCategoryDataMap(
			AsQueryBuilderForm queryForm, ScoreMetaData categoryVar,
			List<ScoreMetaData> contVarList) {
		Map<String, Map<Object, List<Double>>> seriesCategoryDataMap = new HashMap<String, Map<Object, List<Double>>>(
				17);
		for (QuerySummary qs : queryForm.getQsList()) {
			AssessmentResultSummary ars = qs.getArs();
			if (ars == null)
				continue;
			List<ScoreMetaData> varList = new ArrayList<ScoreMetaData>(
					contVarList);
			varList.add(categoryVar);
			Map<String, String> valueMap = ars.getMatchingValues(varList);
			String categoryLabel = getValue(valueMap, categoryVar);
			for (ScoreMetaData contVar : contVarList) {
				String value = getValue(valueMap, contVar);
				if (value != null) {
					String seriesName = contVar.getScoreName();
					Map<Object, List<Double>> categoryMap = seriesCategoryDataMap
							.get(seriesName);
					if (categoryMap == null) {
						categoryMap = new HashMap<Object, List<Double>>(7);
						seriesCategoryDataMap.put(seriesName, categoryMap);
					}
					List<Double> seriesData = categoryMap.get(categoryLabel);
					if (seriesData == null) {
						seriesData = new ArrayList<Double>();
						categoryMap.put(categoryLabel, seriesData);
					}
					seriesData.add(new Double(value.toString()));
				}
			}
		}


		return seriesCategoryDataMap;
	}



	static TIntObjectHashMap prepHeaderMetaDataIdxMap(List<HeaderInfo> headers) {
		TIntObjectHashMap map = new TIntObjectHashMap(17);
		int idx = 0;
		for (HeaderInfo hi : headers) {
			for (Object metaData : hi.getSubHeaderMetaDataList()) {
				map.put(idx, metaData);
				++idx;
			}
		}
		return map;
	}


	public static void prepareAndWriteSRBCVSFile(AsQueryBuilderForm queryForm,
			OutputStream outStream) throws IOException {
		List<ScoreValueSummary> sumList = queryForm.getSummaryList();
		StringBuffer buf = new StringBuffer(256);
		buf.append("SubjectID,");
		buf.append("Site ID,");

		prepHeaderLine(queryForm, buf);

		BufferedWriter out = null;
		out = new BufferedWriter(new OutputStreamWriter(outStream), 4096);
		// write header
		out.write(buf.toString());
		out.write('\n');

		List<String[]> visitScoresRows = new LinkedList<String[]>();
		for (ScoreValueSummary summary : sumList) {

			VisitSegmentScoreValues vssv = (VisitSegmentScoreValues) summary
					.getVisitSegmentScoreValues().iterator().next();
			List<Object> values = vssv.getValues();
			String[] row = new String[values.size() + 2];

			row[0] = summary.getSubjectID();
			row[1] = vssv.getSiteID();
			int offset = 2;
			int idx = offset;
			for (Object item : values) {
				row[idx++] = item.toString().trim();
			}

			// combine assessments from different segments only if the visits'
			// assessments don't overlap
			boolean first = true;
			for (VisitSegmentScoreValues vss : summary
					.getVisitSegmentScoreValues()) {
				if (first) {
					first = false;
					continue;
				}
				idx = offset;
				for (Object item : vss.getValues()) {
					String value = item.toString().trim();
					if (value.length() > 0 && row[idx].length() == 0) {
						row[idx] = value;
					}
					++idx;
				}
			}
			visitScoresRows.add(row);
		}

		writeVisitScoresRows(out, visitScoresRows);
	}

	static void prepHeaderLine(AsQueryBuilderForm queryForm, StringBuffer buf) {
		for (Iterator<HeaderInfo> iter = queryForm.getHeaders().iterator(); iter
				.hasNext();) {
			HeaderInfo hi = iter.next();
			for (Iterator<String> it = hi.getSubHeaders().iterator(); it
					.hasNext();) {
				String subHeader = it.next();
				subHeader = subHeader.replace(' ', '_');
				String variableName = hi.getTitle().replace(' ', '_') + "_"
						+ subHeader;
				buf.append(variableName);
				if (it.hasNext()) {
					buf.append(',');
				}
			}
			if (iter.hasNext()) {
				buf.append(',');
			}
		}
	}

	static void writeVisitScoresRows(BufferedWriter out,
			List<String[]> visitScoresRows) throws IOException {
		for (Iterator<String[]> iter = visitScoresRows.iterator(); iter
				.hasNext();) {
			String[] row = iter.next();
			for (int i = 0; i < row.length; i++) {
				String value = row[i];
				if (row[i].length() == 0) {
					value = "."; // for SPSS/SAS
				}
				out.write(value);
				if ((i + 1) < row.length) {
					out.write(',');
				}
			}
			out.write('\n');
		}
		out.flush();
	}

	public static void plotBivariateOLSChart(AsQueryBuilderForm queryForm,
			StatisticsForm statsForm, String imagesDir,
			Map<String, String> siteNameMap) throws Exception {

		Map<String, String> varnameMap = new HashMap<String, String>(3);
		CombinedScoreSelector bv1Selector = statsForm.getBv1ScoreSelector();
		CombinedScoreSelector bv2Selector = statsForm.getBv2ScoreSelector();
		SingleScoreSelector groupVarSelector = statsForm
				.getGroupByVarSelector();

		boolean siteAsGroupVar = false;
		Map<String, ScoreMetaData> smdMap = statsForm.getSmdMap();

		ScoreMetaData groupVar = null, indepVar = null, depVar = null;
		if (groupVarSelector != null) {
			String groupVarName = groupVarSelector.getSelectedScore();
			if (groupVarName.equalsIgnoreCase("site")) {
				siteAsGroupVar = true;
			} else {
				groupVar = smdMap.get(groupVarName);
				Assertion.assertNotNull(groupVar);
			}
		}
		String selectedX = bv1Selector.getSelectedScore();
		indepVar = smdMap.get(selectedX);
		Assertion.assertNotNull(indepVar);

		String selectedY = bv2Selector.getSelectedScore();
		depVar = smdMap.get(selectedY);
		Assertion.assertNotNull(depVar);

		LinkedHashMap<String, List<DataPoint>> seriesMap = prepareSeriesDataMap(
				queryForm, groupVar, indepVar, depVar, varnameMap,
				siteAsGroupVar, siteNameMap);

		String imageFileName = GenUtils.createTempFileName(imagesDir, ".png");

		ChartingHelper ch = new ChartingHelper();
		String title = "Scatter plot with regression lines";
		String xAxisLabel = varnameMap.get("x");
		String yAxisLabel = varnameMap.get("y");
		Map<Comparable<?>, RegressionInfo> regMap = new LinkedHashMap<Comparable<?>, RegressionInfo>(
				7);
		List<JSURLGenerator> urlGenerators = new ArrayList<JSURLGenerator>(10);
		ChartRenderingInfo info = ch.prepareScatterPlotWithRegressionLines(
				seriesMap, xAxisLabel, yAxisLabel, title, true, imageFileName,
				regMap, urlGenerators);

		String relImagePath = REL_IMAGES_PATH
				+ new File(imageFileName).getName();

		String imageMapName = "chart";
		String imageMap = ChartingHelper.getImageMap(imageMapName, info);

		statsForm.setImageMapInfo(imageMap);

		statsForm.setScatterDataInfoJSON(prepareScatterDataInfoJSON(seriesMap,
				urlGenerators));

		statsForm.setOlsChartImageFile(relImagePath);
		if (!regMap.isEmpty()) {
			List<RegressionInfo> regInfos = new ArrayList<RegressionInfo>(
					regMap.values());
			statsForm.setRegressionInfos(regInfos);
		}

	}

	public static String prepareScatterDataInfoJSON(
			LinkedHashMap<String, List<DataPoint>> seriesDataMap,
			List<JSURLGenerator> urlGenerators) {
		class DPInfo {
			int series;
			int item;
			SubjectDataPoint dp;

			public DPInfo(int series, int item, SubjectDataPoint dp) {
				this.series = series;
				this.item = item;
				this.dp = dp;
			}
		}// ;

		Map<Integer, Map<String, DataPoint>> sdMap = new HashMap<Integer, Map<String, DataPoint>>(
				7);
		int i = 0;
		for (List<? extends DataPoint> seriesData : seriesDataMap.values()) {
			Map<String, DataPoint> dpMap = new HashMap<String, DataPoint>();
			sdMap.put(new Integer(i), dpMap);
			for (DataPoint dp : seriesData) {
				StringBuffer keyBuf = new StringBuffer();
				keyBuf.append(dp.getX()).append('_').append(dp.getY());
				dpMap.put(keyBuf.toString(), dp);
			}
			i++;
		}

		List<DPInfo> dpList = new ArrayList<DPInfo>();
		for (JSURLGenerator jug : urlGenerators) {
			Integer seriesKey = new Integer(jug.getActualSeries());
			LinkedHashMap<String, String> indexMap = jug.getIndexMap();
			for (Map.Entry<String, String> entry : indexMap.entrySet()) {
				String idxKey = entry.getKey();
				String idxValue = entry.getValue();
				String[] tokens = GenUtils.split(idxKey, "_");
				Map<String, DataPoint> dpMap = sdMap.get(seriesKey);
				SubjectDataPoint dp = (SubjectDataPoint) dpMap.get(idxValue);

				DPInfo dpi = new DPInfo(seriesKey.intValue(), GenUtils.toInt(
						tokens[1], -1), dp);
				dpList.add(dpi);
			}
		}

		StringBuffer buf = new StringBuffer(3000);
		i = 0;
		buf.append("{");
		for (List<? extends DataPoint> seriesData : seriesDataMap.values()) {
			if (!seriesData.isEmpty()) {
				buf.append(" series_").append(i).append(": { x: '");
				SubjectDataPoint dp = (SubjectDataPoint) seriesData.iterator()
						.next();
				buf.append(dp.getXVarName()).append("', y: '");
				buf.append(dp.getYVarName()).append("'},\n");
			} else {
				buf.append(" series_").append(i).append(
						": { x: null, y: null},\n");
			}
			i++;
		}

		buf.append("map: { ");
		for (Iterator<DPInfo> it = dpList.iterator(); it.hasNext();) {
			DPInfo dpi = it.next();
			SubjectDataPoint dp = dpi.dp;
			buf.append('k').append(dpi.series).append('_');
			buf.append(dpi.item).append(":['");
			buf.append(dp.getSubjectID()).append("','");
			buf.append(dp.getSite()).append("',");
			buf.append(dp.getX()).append(',').append(dp.getY()).append("]");
			if (it.hasNext()) {
				buf.append(',');
			}
			buf.append("\n");
		}

		buf.append("}\n");
		buf.append("};\n");

		return buf.toString();
	}

	public static List<DescriptiveStats> prepDescriptiveStats(
			AsQueryBuilderForm queryForm, StatisticsForm statsForm)
			throws Exception {
		ScoresSelector csSelector = statsForm.getContScoresSelector();
		String[] selectedCsArr = csSelector.getSelectedScores();
		List<ScoreMetaData> varList = new ArrayList<ScoreMetaData>(5);
		if (selectedCsArr != null && selectedCsArr.length > 0) {
			for (int i = 0; i < selectedCsArr.length; i++) {
				ScoreMetaData smd = statsForm.getSmdMap().get(selectedCsArr[i]);
				Assertion.assertNotNull(smd);
				varList.add(smd);
			}
		}
		Map<String, DescriptiveStatistics> statsMap = new HashMap<String, DescriptiveStatistics>(
				7);
		List<QuerySummary> qsList = queryForm.getQsList();
		String theExpName = null;
		for (QuerySummary qs : qsList) {
			if (qs.getArs() == null)
				continue;
			AssessmentResultSummary ars = qs.getArs();
			if (theExpName == null) {
				theExpName = ars.getExpName();
			}
			Map<String, String> valueMap = ars.getMatchingValues(varList);
			for (String key : valueMap.keySet()) {
				String[] toks = key.split(":");
				String scoreName = toks[0];
				DescriptiveStatistics stats = statsMap.get(scoreName);
				if (stats == null) {
					stats = DescriptiveStatistics.newInstance();
					statsMap.put(scoreName, stats);
				}
				stats.addValue(Double.parseDouble(valueMap.get(key)));
			}
		}

		List<DescriptiveStats> statsList = new ArrayList<DescriptiveStats>(
				statsMap.size());
		for (Iterator<String> iter = statsMap.keySet().iterator(); iter
				.hasNext();) {
			String varName = iter.next();
			DescriptiveStatistics stats = statsMap.get(varName);
			DescriptiveStats ds = new DescriptiveStats(varName);
			ds.setMin(stats.getMin());
			ds.setMax(stats.getMax());
			ds.setMean(stats.getMean());
			ds.setSD(stats.getStandardDeviation());
			ds.setVariance(stats.getVariance());
			ds.setKurtosis(stats.getKurtosis());
			ds.setN((int) stats.getN());
			statsList.add(ds);
		}

		return statsList;
	}

	public static void plotUnivariateBarChart(AsQueryBuilderForm queryForm,
			StatisticsForm statsForm, String imagesDir) throws Exception {
		ScoresSelector csSelector = statsForm.getContScoresSelector();
		SingleScoreSelector dsSelector = statsForm.getDiscreteScoreSelector();

		Map<String, ScoreMetaData> smdMap = statsForm.getSmdMap();
		String categoryVarName = dsSelector.getSelectedScore();
		Assertion.assertNotNull(categoryVarName);
		ScoreMetaData categoryVar = smdMap.get(categoryVarName);
		Assertion.assertNotNull(categoryVar);

		String[] selectedCsArr = csSelector.getSelectedScores();
		List<ScoreMetaData> varList = new ArrayList<ScoreMetaData>(5);
		if (selectedCsArr != null && selectedCsArr.length > 0) {
			for (int i = 0; i < selectedCsArr.length; i++) {
				ScoreMetaData smd = smdMap.get(selectedCsArr[i]);
				Assertion.assertNotNull(smd);
				varList.add(smd);
			}
		}

		Map<String, Map<Object, List<Double>>> seriesCatDataMap = prepareSeriesCategoryDataMap(
				queryForm, categoryVar, varList);
		String imageFileName = GenUtils.createTempFileName(imagesDir, ".png");

		ChartingHelper ch = new ChartingHelper();
		String title = "Scores grouped by Category '" + categoryVarName + "'";
		String yAxisLabel = "Scores";
		ch.prepareBarChart(seriesCatDataMap, title, categoryVarName,
				yAxisLabel, true, imageFileName);

		String relImagePath = REL_IMAGES_PATH
				+ new File(imageFileName).getName();

		statsForm.setBarChartImageFile(relImagePath);
	}

	public static String executeUnivariateStatsScript(Properties props,
			AsQueryBuilderForm queryForm, StatisticsForm statsForm,
			String inputDataFile) throws Exception {
		StringBuffer buf = new StringBuffer();
		boolean first = true;
		if (statsForm.isMsd()) {
			first = prepCLArg(buf, "a", first);
		}
		if (statsForm.isNormTest()) {
			first = prepCLArg(buf, "s", first);
		}
		if (statsForm.isPlotHistogram()) {
			first = prepCLArg(buf, "h", first);
		}
		if (statsForm.isPlotQQ()) {
			first = prepCLArg(buf, "q", first);
		}

		String paramsFile = GenUtils.createTempFileName(TEMP_DIR, ".par");
		String xmlFile = GenUtils.createTempFileName(TEMP_DIR, ".xml");
		prepareParamsFile(paramsFile, queryForm, statsForm);

		if (buf.length() == 0) {
			return null;
		}

		buf.append(" -p ").append(paramsFile);
		if (inputDataFile != null) {
			buf.append(" -i ").append(inputDataFile);
		}

		buf.append(" -x ").append(xmlFile);
		Executor executor = ProcessUtils.prepareExecutor(props);
		String cmdLine = ProcessUtils.prepareScript(props, "univariate-r.pl");
		cmdLine += " " + buf.toString();
		System.out.println("cmdLine: " + cmdLine);
		int rc = 0;
		// executor.showOutput(true);
		// executor.setErrorStreamReliable(true);

		try {
			rc = executor.execute(cmdLine);
			if (rc == 0) {
				String xmlOutput = FileUtils.loadTextFile(xmlFile);
				System.out.println(">> xmlOutput: " + xmlOutput);
				System.out.println("--------------");
				return xmlOutput;
			} else {
				System.out.println("error : " + executor.getOutput());
			}
		} catch (Exception x) {
			x.printStackTrace();
			throw x;
		}
		return null;
	}

	public static void updateStatsFormFromXML(String rStatsXMLStr,
			StatisticsForm statsForm) throws Exception {
		if (rStatsXMLStr == null) {
			return;
		}
		FileUtils.save2File(rStatsXMLStr, "/tmp/xml.dump");

		StringReader reader = new StringReader(rStatsXMLStr);
		SAXBuilder builder = new SAXBuilder(false); // no validation
		Document doc = builder.build(reader);
		Element statsElem = doc.getRootElement();

		statsForm.getStatResultsDataList().clear();
		List<?> children = statsElem.getChildren("var");
		for (Iterator<?> iter = children.iterator(); iter.hasNext();) {
			Element varElem = (Element) iter.next();
			String varName = varElem.getAttributeValue("name");
			Element preElem = varElem.getChild("pre");

			String prepText = preElem.getText();

			String histImgPath = null;
			if (varElem.getChild("hist") != null) {
				String histImg = varElem.getChild("hist").getAttributeValue(
						"img");

				File f = new File(histImg);
				histImgPath = REL_IMAGES_PATH + f.getName();
			}
			String qqImgPath = null;
			if (varElem.getChild("qq") != null) {
				String qqImg = varElem.getChild("qq").getAttributeValue("img");
				qqImgPath = REL_IMAGES_PATH + new File(qqImg).getName();
			}

			StatsResultsData srd = new StatsResultsData(varName, histImgPath,
					prepText, qqImgPath);
			statsForm.addStatResultsData(srd);
		}
	}

	private static boolean prepCLArg(StringBuffer buf, String argStr,
			boolean first) {
		if (first) {
			buf.append('-');
		}
		buf.append(argStr);
		return false;
	}

	protected static void prepareParamsFile(String fileName,
			AsQueryBuilderForm queryForm, StatisticsForm statsForm)
			throws IOException {
		List<String> factors = new ArrayList<String>();
		List<String> regions = new ArrayList<String>();
		ScoresSelector csSelector = statsForm.getContScoresSelector();

		for (HeaderInfo hi : queryForm.getHeaders()) {
			Iterator<Object> it2 = hi.getSubHeaderMetaDataList().iterator();
			for (String subHeader : hi.getSubHeaders()) {
				Object metaData = it2.next();
				subHeader = subHeader.replace(' ', '.');
				String variableName = hi.getTitle().replace(' ', '.') + "."
						+ subHeader;
				if (metaData instanceof AsScoreInfo) {
					AsScoreInfo si = (AsScoreInfo) metaData;
					if (csSelector.indexOf(si.getName()) != -1) {
						factors.add(variableName);
					}
					/** @todo discrete var */
				} else if (metaData instanceof SubCorticalVarInfo) {

				}//
			}// it
		}// iter
		BufferedWriter out = null;
		try {
			out = new BufferedWriter(new FileWriter(fileName));
			if (!factors.isEmpty()) {
				StringBuffer buf = new StringBuffer();
				buf.append("factors=");
				for (Iterator<String> iter = factors.iterator(); iter.hasNext();) {
					String factor = iter.next();
					buf.append(factor);
					if (iter.hasNext()) {
						buf.append(',');
					}
				}
				out.write(buf.toString());
				out.newLine();
			}
			if (!regions.isEmpty()) {
				StringBuffer buf = new StringBuffer();
				buf.append("structure=");
				for (Iterator<String> iter = regions.iterator(); iter.hasNext();) {
					String regionName = iter.next();
					buf.append(regionName);
					if (iter.hasNext()) {
						buf.append(',');
					}
				}
				out.write(buf.toString());
				out.newLine();
			}

		} finally {
			FileUtils.close(out);
		}
	}

	public static void main(String[] args) throws Exception {
		StatisticsForm form = new StatisticsForm();
		String xmlStr = FileUtils.loadTextFile("/tmp/test_results.xml");
		StatsHelper.updateStatsFormFromXML(xmlStr, form);
	}

}
