package clinical.web.services;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.sql.Connection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;

import clinical.server.dao.HumansubjectDAO;
import clinical.server.vo.Humansubject;
import clinical.web.Constants;
import clinical.web.DAOFactory;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBPoolService;

/**
 * Generates and maintains BIRN ids. A BIRN ID is a 4 digit zero-padded
 * institute ID followed by 8 digit zero padded secure random number.
 * 
 * @author I. Burak Ozyurt
 * @version $Id: BIRNIDGenerator.java 62 2009-05-29 23:54:50Z bozyurt $
 */
public class BIRNIDGenerator {
	private Random rng;
	private MessageDigest md;
	private IDBPoolService dbPoolService;
	private String dbID;
	private Map<String, String> birnIDsMap = Collections
			.synchronizedMap(new LinkedHashMap<String, String>());

	private static BIRNIDGenerator instance = null;
	public final static int NUMBER_LEN = 8;

	private BIRNIDGenerator(String dbID) throws NoSuchAlgorithmException {
		dbPoolService = ServiceFactory.getPoolService(dbID);
		this.dbID = dbID;
		// loadBIRNIDs(birnIDFilename);
		prepare();
	}

	public static synchronized BIRNIDGenerator getInstance(String dbID)
			throws Exception {
		if (instance == null) {
			instance = new BIRNIDGenerator(dbID);
		}
		return instance;
	}

	public static synchronized BIRNIDGenerator getInstance() {
		if (instance == null) {
			throw new RuntimeException(
					"BIRNIDGenerator not initialized properly!");
		}
		return instance;
	}

	private void prepare() throws NoSuchAlgorithmException {
		rng = SecureRandom.getInstance("SHA1PRNG");
		((SecureRandom) rng).setSeed(SecureRandom.getSeed(128));
	}

	public String createBIRNID(String prefix) throws Exception {
		return createBIRNID(prefix, NUMBER_LEN);
	}

	public String createBIRNID(String prefix, int numDigits) throws Exception {
		String birnID = prefix + createRandomID(numDigits);
		loadBIRNIDs();

		while (birnIDsMap.get(birnID) != null) {
			birnID = prefix + createRandomID(numDigits);
		}
		birnIDsMap.put(birnID, birnID);
		return birnID;
	}

	private String createRandomID(int length) {
		long maxNumber = 1;
		int i = length;
		while (i-- > 0) {
			maxNumber *= 10;
			if (maxNumber > Integer.MAX_VALUE) {
				break;
			}
		}

		StringBuffer buf = new StringBuffer(length);
		int number = rng.nextInt((int) maxNumber);
		buf.append(formatNumber(length, number));
		return buf.toString();
	}

	private String formatNumber(int noDigits, int number) {
		String s = Integer.toString(number);
		if (noDigits > s.length()) {
			// zero padding
			StringBuffer buf = new StringBuffer(noDigits);
			int diff = noDigits - s.length();
			for (int i = 0; i < diff; ++i)
				buf.append('0');
			buf.append(s);
			return buf.toString();
		} else
			return s;
	}

	protected byte[] createMessageDigest(String clinicalID) {
		md.update(clinicalID.getBytes());
		byte[] bytes = md.digest();
		return bytes;
	}

	protected static String toHex(byte[] bytes) {
		StringBuffer buf = new StringBuffer();
		for (int i = 0; i < bytes.length; ++i) {
			int byteVal = (bytes[i] < 0) ? bytes[i] + 256 : bytes[i];
			// System.out.println(""+ byteVal + ", hex="+ Integer.toHexString(
			// byteVal) );
			String s = Integer.toHexString(byteVal);
			s = (s.length() == 1) ? "0" + s : s;
			buf.append(s);
		}
		return buf.toString();
	}

	protected static byte[] hexToBytes(String s) {
		char[] carr = s.toLowerCase().toCharArray();
		byte[] bytes = new byte[carr.length / 2];

		for (int i = 0; i < bytes.length; ++i) {
			StringBuffer buf = new StringBuffer(3);
			buf.append(carr[2 * i]);
			buf.append(carr[2 * i + 1]);
			int val = Integer.parseInt(buf.toString(), 16);
			bytes[i] = (byte) val;
		}
		return bytes;
	}

	private void loadBIRNIDs() throws Exception {
		Connection con = null;
		try {
			con = dbPoolService.getConnection(Constants.ADMIN_USER);
			HumansubjectDAO dao = DAOFactory.createHumansubjectDAO(this.dbID);
			List<Humansubject> hsList = dao.find(con, new Humansubject());
			for (Humansubject hs : hsList) {
				if (birnIDsMap.get(hs.getSubjectid()) == null) {
					birnIDsMap.put(hs.getSubjectid(), hs.getSubjectid());
				}
			}
		} finally {
			if (con != null)
				dbPoolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}
}