package clinical.web.vo;

import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * A value object holding score values for a particular assessment done for a
 * subject for a specific experiment, at a specific segment of a specific visit.
 * 
 * @author I. Burak Ozyurt
 * @version $Id: AssessmentScoreValues.java,v 1.19 2006/01/27 21:44:16 bozyurt
 *          Exp $
 */
public class AssessmentScoreValues {
	protected AssessmentInfo asi;
	protected List<ScoreValue> scoreValues;
	protected Map<String, ScoreValue> scoreValueMap;
	protected int visitID;
	protected int segmentID;
	protected int experimentID;
	protected String subjectID;
	protected Date timeStamp;
	protected int storedAsID;
	protected String informantID;
	protected String informantRelation;
	protected String clinicalRater;

	protected static NumberFormat numberFormat = NumberFormat.getInstance();
	private static Log log = LogFactory.getLog(AssessmentScoreValues.class);

	static {
		numberFormat.setMinimumFractionDigits(2);
		numberFormat.setMaximumFractionDigits(2);
		numberFormat.setGroupingUsed(false);
	}

	public AssessmentScoreValues(AssessmentInfo asi) {
		this.asi = asi;
	}

	public int getAssessmentID() {
		return asi.getAssessmentID();
	}

	public String getAssessmentName() {
		return asi.getName();
	}

	public void addScoreValue(ScoreValue sv) {
		if (scoreValues == null)
			scoreValues = new ArrayList<ScoreValue>();
		scoreValues.add(sv);
	}

	public ScoreValue getScoreValue(int idx) {
		return scoreValues.get(idx);
	}

	public void orderScoreValues() {
		Collections.sort(scoreValues, new Comparator<ScoreValue>() {
			public int compare(ScoreValue sc1, ScoreValue sc2) {
				int diff = asi.findScore(sc1.getName()).getScoreSequence()
						- asi.findScore(sc2.getName()).getScoreSequence();
				return diff;
			}
		});
	}

	public void dumpScoreInfos() {
		System.out.println("dumping AssessmentInfo.ScoreInfo objects from asi");
		for (Iterator<AssessmentInfo.ScoreInfo> iter = asi.getScores()
				.iterator(); iter.hasNext();) {
			AssessmentInfo.ScoreInfo si = (AssessmentInfo.ScoreInfo) iter
					.next();
			System.out.println("\t" + si.toString());
		}
	}

	public List<ScoreValue> getScoreValues() {
		return scoreValues;
	}

	protected String createScoreValueKey(String scoreName, int scoreOrder) {
		StringBuffer buf = new StringBuffer();
		buf.append(scoreName).append(':').append(scoreOrder);
		return buf.toString();
	}

	/**
	 * 
	 * @param scoreName
	 *            the name of the score
	 * @return a <code>ScoreValue</code> object
	 */
	public ScoreValue getScoreValue(String scoreName, int scoreOrder) {
		if (scoreValueMap == null) {
			prepareScoreValueMap();
		}
		String key = createScoreValueKey(scoreName, scoreOrder);
		return scoreValueMap.get(key);
	}

	public int getScoreValueCount(String scoreName) {
		int count = 0;
		for (Iterator<ScoreValue> iter = scoreValues.iterator(); iter.hasNext();) {
			ScoreValue sv = iter.next();
			if (sv.getName().equals(scoreName)) {
				count++;
			}
		}
		return count;
	}

	private void prepareScoreValueMap() {
		Map<ScoreValue, ScoreValue> parentScoreValueMap = new HashMap<ScoreValue, ScoreValue>(
				17);
		scoreValueMap = new HashMap<String, ScoreValue>(17);
		log.info("for assessment " + getAssessmentName()
				+ " scoreValues.size()=" + scoreValues.size());
		for (Iterator<ScoreValue> iter = scoreValues.iterator(); iter.hasNext();) {
			ScoreValue sv = iter.next();
			String key = createScoreValueKey(sv.getName(), sv.getScoreOrder());
			scoreValueMap.put(key, sv);
			if (sv.getScoreOrder() == 1) {
				parentScoreValueMap.put(sv, sv);
			}

			// float type scores need special formatting
			AssessmentInfo.ScoreInfo si = asi.findScore(sv.getName());
			if (si == null) {
				log
						.error("*** [AssessmentScoreValues] Cannot find scoreInfo for "
								+ sv.getName());
			} else {
				if (si.getType().equalsIgnoreCase("float")) {
					sv.floatType = true;
				}
			}
		}

		for (Iterator<ScoreValue> iter = parentScoreValueMap.keySet()
				.iterator(); iter.hasNext();) {
			ScoreValue parentSv = iter.next();
			int so = 2;
			ScoreValue sv = null;
			do {
				String key = createScoreValueKey(parentSv.getName(), so);
				sv = scoreValueMap.get(key);
				if (sv != null) {
					ScoreValue loc = parentSv;
					while (loc.getNext() != null)
						loc = loc.getNext();
					loc.setNext(sv);
				} else {
					break;
				}
				so++;
			} while (sv != null);
		}
	}

	public String getScoreName(int idx) {
		return scoreValues.get(idx).getName();
	}

	public AssessmentInfo.ScoreInfo findScoreInfo(String scoreName) {
		for (Iterator<AssessmentInfo.ScoreInfo> iter = asi.getScores()
				.iterator(); iter.hasNext();) {
			AssessmentInfo.ScoreInfo si = iter.next();
			if (si.getName().equals(scoreName))
				return si;
		}
		return null;
	}

	// ---------------------- setters --------------
	public void setVisitID(int newVisitID) {
		this.visitID = newVisitID;
	}

	public void setSegmentID(int newSegmentID) {
		this.segmentID = newSegmentID;
	}

	public void setExperimentID(int newExperimentID) {
		this.experimentID = newExperimentID;
	}

	public void setSubjectID(String newSubjectID) {
		this.subjectID = newSubjectID;
	}

	public void setTimeStamp(Date newTimeStamp) {
		this.timeStamp = newTimeStamp;
	}

	public void setStoredAsID(int newStoredAsID) {
		this.storedAsID = newStoredAsID;
	}

	public void setInformantID(String newInformantID) {
		this.informantID = newInformantID;
	}

	public void setInformantRelation(String newInformantRelation) {
		this.informantRelation = newInformantRelation;
	}

	public void setClinicalRater(String newClinicalRater) {
		this.clinicalRater = newClinicalRater;
	}

	// ---------------------- getters --------------
	public int getVisitID() {
		return this.visitID;
	}

	public int getSegmentID() {
		return this.segmentID;
	}

	public int getExperimentID() {
		return this.experimentID;
	}

	public String getSubjectID() {
		return this.subjectID;
	}

	public Date getTimeStamp() {
		return this.timeStamp;
	}

	public int getStoredAsID() {
		return this.storedAsID;
	}

	public String getInformantID() {
		return this.informantID;
	}

	public String getInformantRelation() {
		return this.informantRelation;
	}

	public String getClinicalRater() {
		return this.clinicalRater;
	}

	public String toString() {
		StringBuffer buf = new StringBuffer(272);
		buf.append("AssessmentScoreValues::[");
		buf.append("asi=").append(asi);
		buf.append(", visitID=").append(visitID);
		buf.append(", segmentID=").append(segmentID);
		buf.append(", experimentID=").append(experimentID);
		buf.append(", subjectID=").append(subjectID);
		buf.append(", timeStamp=").append(timeStamp);
		buf.append(", storedAsID=").append(storedAsID);
		buf.append(", informantID=").append(informantID);
		buf.append(", informantRelation=").append(informantRelation);
		buf.append(", clinicalRater=").append(clinicalRater);
		if (scoreValues != null) {
			for (Iterator<ScoreValue> iter = scoreValues.iterator(); iter
					.hasNext();) {
				ScoreValue sv = iter.next();
				buf.append("\n\t").append(sv.toString());
			}
		}
		buf.append(']');
		return buf.toString();
	}

}
