package clinical.xml.gui;

import guilib.AppFrame;

import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;
import java.io.StringWriter;
import java.util.LinkedHashMap;
import java.util.Map;

import javax.swing.JFileChooser;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTree;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.TreePath;

import clinical.xml.XMLUtils;
import clinical.xml.export.ExportClient;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: ExportImportClient.java 62 2009-05-29 23:54:50Z bozyurt $
 */
public class ExportImportClient extends AppFrame {
	private static final long serialVersionUID = -583110607843047205L;
	JTree expSubjectsTree;
	DefaultMutableTreeNode root;
	XmlViewPane xmlPane;
	JSplitPane splitPane;
	ConfigInfo configInfo = new ConfigInfo();
	ExportClient ec = new ExportClient();

	public ExportImportClient(int width, int height, boolean useHTTPS)
			throws Exception {
		super(width, height);
		setTitle("XCEDE Export Client");
		init(useHTTPS);
	}

	protected void init(boolean useHTTPS) throws Exception {
		if (useHTTPS) {
			configInfo
					.setEndPoint("https://localhost:8443/clinical/services/XCEDEService");
		} else {
			configInfo
					.setEndPoint("http://localhost:8080/clinical/services/XCEDEService");
		}
		ParamsDialog dlg = new ParamsDialog(this,
				"Web Service EndPoint Params", configInfo);
		int rc = dlg.showDialog();
		root = new DefaultMutableTreeNode("<Experiments - Subjects>");

		if (rc == ParamsDialog.OK_PRESSED) {
			// connect to the web service and get experiment and subject list
			Object[] exps = ec.getAllExperiments(configInfo.getEndPoint(),
					configInfo.getUser(), configInfo.getPwd());
			int[] expIDs = (int[]) exps[0];
			Map<Integer, String[]> exp2SubjectsMap = new LinkedHashMap<Integer, String[]>();
			for (int i = 0; i < expIDs.length; i++) {
				String[] subjects = ec.getAllSubjects(configInfo.getEndPoint(),
						configInfo.getUser(), configInfo.getPwd(), expIDs[i]);
				if (subjects != null) {
					exp2SubjectsMap.put(new Integer(expIDs[i]), subjects);
				}
			}

			populateTree(root, exps, exp2SubjectsMap);
		}

		JScrollPane esPane = new JScrollPane(expSubjectsTree = new JTree(root));

		expSubjectsTree.addMouseListener(new TreeMouseAdapter(expSubjectsTree));

		xmlPane = new XmlViewPane();
		JScrollPane xmlScroll = new JScrollPane(xmlPane);
		xmlScroll.setPreferredSize(new Dimension(400, 400));

		splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, esPane, xmlPane);
		splitPane.setOneTouchExpandable(true);
		splitPane.setDividerLocation(220);
		getContentPane().add(splitPane);

	}

	protected void populateTree(DefaultMutableTreeNode root, Object[] exps,
			Map<Integer, String[]> exp2SubjectsMap) {
		int[] expIDs = (int[]) exps[0];
		String[] expNames = (String[]) exps[1];
		for (int i = 0; i < expIDs.length; i++) {
			DefaultMutableTreeNode child = new DefaultMutableTreeNode(
					new ExperimentInfo(expIDs[i], expNames[i]));
			root.add(child);
			String[] subjects = exp2SubjectsMap.get(new Integer(expIDs[i]));
			if (subjects != null) {
				for (int j = 0; j < subjects.length; j++) {
					DefaultMutableTreeNode child2 = new DefaultMutableTreeNode(
							new SubjectInfo(subjects[j], expIDs[i]));
					child.add(child2);
				}
			}
		}
	}

	// overide file menu creation method from the parent
	protected JMenu createFileMenu() {
		super.fileMenu = new JMenu("File");
		fileMenu.setMnemonic(KeyEvent.VK_F);

		fileMenu.addSeparator();
		JMenuItem item = new JMenuItem("Quit", KeyEvent.VK_Q);
		fileMenu.add(item);
		item.addActionListener(new CloseAdapter());

		return fileMenu;
	}

	public void handleExport(SubjectInfo si, String fileName) {

		try {
			String xmlStr = ec.getSubjectData(configInfo.getEndPoint(),
					configInfo.getUser(), configInfo.getPwd(), si.subjectID,
					si.expID);
			StringWriter out = new StringWriter();
			XMLUtils.prettyPrint(xmlStr, out);
			xmlPane.setXMLContent(out.toString());
			if (fileName != null) {
				saveContent(fileName, out.toString());
			}

		} catch (Exception ex) {
			ex.printStackTrace();
		}
	}

	private void saveContent(String filename, String content)
			throws IOException {
		BufferedWriter out = null;
		try {
			out = new BufferedWriter(new FileWriter(filename));
			out.write(content);

		} finally {
			if (out != null) {
				try {
					out.close();
				} catch (Exception x) {
				}
			}
		}
	}

	public void handleExperimentExport(int expID, String fileName) {
		try {
			String xmlStr = ec.getExperimentData(configInfo.getEndPoint(),
					configInfo.getUser(), configInfo.getPwd(), expID);
			StringWriter out = new StringWriter();
			XMLUtils.prettyPrint(xmlStr, out);
			xmlPane.setXMLContent(out.toString());
			if (fileName != null) {
				saveContent(fileName, out.toString());
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	class CloseAdapter implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			onClose();
		}
	}

	public static class SubjectInfo {
		String subjectID;
		int expID;

		public SubjectInfo(String subjectID, int expID) {
			this.subjectID = subjectID;
			this.expID = expID;
		}

		public String toString() {
			return subjectID;
		}
	}

	public static class ExperimentInfo {
		String expName;
		int expID;

		public ExperimentInfo(int expid, String name) {
			expID = expid;
			expName = name;
		}

		public String toString() {
			return expName;
		}
	}

	class TreeMouseAdapter extends MouseAdapter implements ActionListener {
		JTree tree;
		JPopupMenu popupMenu;
		JMenuItem[] menuItems;
		JFileChooser chooser = new JFileChooser();

		public TreeMouseAdapter(JTree tree) {
			this.tree = tree;
			preparePopupMenu();
		}

		protected JMenuItem prepMenuItem(String label) {
			JMenuItem menuItem = new JMenuItem(label);
			menuItem.addActionListener(this);
			popupMenu.add(menuItem);
			return menuItem;
		}

		public void preparePopupMenu() {
			menuItems = new JMenuItem[10];
			popupMenu = new JPopupMenu("");
			menuItems[0] = prepMenuItem("Export");
		}

		public void mouseReleased(MouseEvent me) {
			showPopup(me);
		}

		public void mousePressed(MouseEvent me) {
			showPopup(me);
		}

		protected void showPopup(MouseEvent me) {
			if (me.isPopupTrigger()) {
				TreePath path = tree.getPathForLocation(me.getX(), me.getY());
				if (path != null) {
					DefaultMutableTreeNode node = (DefaultMutableTreeNode) path
							.getLastPathComponent();
					Object userObj = node.getUserObject();
					if (userObj instanceof SubjectInfo
							|| userObj instanceof ExperimentInfo) {
						popupMenu.show(me.getComponent(), me.getX(), me.getY());
					}
				}
			}
		}

		protected DefaultMutableTreeNode checkSelection() {
			DefaultMutableTreeNode selectedNode = (DefaultMutableTreeNode) tree
					.getLastSelectedPathComponent();
			if (selectedNode == null) {
				JOptionPane.showMessageDialog(tree,
						"You need to select a node first!", "Error",
						JOptionPane.ERROR_MESSAGE);
				return null;
			}
			/*
			 * if (!(selectedNode.getUserObject() instanceof SubjectInfo)) {
			 * JOptionPane.showMessageDialog(tree, "You need to select a subject
			 * tree node!", "Error", JOptionPane.ERROR_MESSAGE); return null; }
			 */
			return selectedNode;
		}

		public void actionPerformed(ActionEvent e) {
			if (!(e.getSource() instanceof JMenuItem)) {
				return;
			}
			JMenuItem item = (JMenuItem) e.getSource();
			if (item.getText().equals("Export")) {
				DefaultMutableTreeNode selectedNode = checkSelection();
				if (selectedNode != null) {
					String fileName = null;
					int rc = JOptionPane.showConfirmDialog(
							ExportImportClient.this,
							"Do you want to save the exported results?",
							"Save Results", JOptionPane.YES_NO_OPTION);
					if (rc == JOptionPane.YES_OPTION) {
						int rv = chooser
								.showSaveDialog(ExportImportClient.this);
						if (rv == JFileChooser.APPROVE_OPTION) {
							fileName = chooser.getSelectedFile()
									.getAbsolutePath();
						}
					}
					if (selectedNode.getUserObject() instanceof SubjectInfo) {
						handleExport(
								(SubjectInfo) selectedNode.getUserObject(),
								fileName);
					} else if (selectedNode.getUserObject() instanceof ExperimentInfo) {
						ExperimentInfo ei = (ExperimentInfo) selectedNode
								.getUserObject();
						handleExperimentExport(ei.expID, fileName);
					}
				}
			}
		}

	}

	public static void main(String[] args) {
		ExportImportClient eic = null;
		boolean useHTTPS = false;
		if (args.length == 1) {
			useHTTPS = (args[0].indexOf("https") != -1);
		}
		try {
			eic = new ExportImportClient(650, 450, useHTTPS);
			eic.startup();
		} catch (Exception x) {
			x.printStackTrace();
		}

	}

}
