package clinical.xml.ws;

import java.io.IOException;
import java.io.StreamTokenizer;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;

import javax.jws.WebService;

import org.jdom.DocType;
import org.jdom.Document;
import org.jdom.Element;

import clinical.server.vo.Deriveddata;
import clinical.server.vo.Experiment;
import clinical.server.vo.Subjexperiment;
import clinical.utils.FileUtils;
import clinical.web.IAppConfigService;
import clinical.web.IDerivedImageDataService;
import clinical.web.ISubjectVisitManagement;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.UserInfo;
import clinical.web.services.AppConfigService;
import clinical.xml.ApplicationUtils;

@WebService(serviceName = "DataServices", endpointInterface = "clinical.xml.ws.DataServices")
public class DataServicesImpl implements DataServices {

	@Override
	public String getExperimentSubjects(String webUser, String pwd)
			throws Exception {
		String dbID = getDBID();
		UserInfo ui = ApplicationUtils.authenticate(webUser, pwd, dbID);
		System.out.println("ui=" + ui);
		ISubjectVisitManagement svm = ServiceFactory
				.getSubjectVisitManagement(dbID);
		Map<Experiment, List<Subjexperiment>> map = svm
				.getExperimentSubjects(ui);
		List<Experiment> sortedExpList = new ArrayList<Experiment>(map.keySet());

		Collections.sort(sortedExpList, new Comparator<Experiment>() {
			@Override
			public int compare(Experiment o1, Experiment o2) {
				return o1.getName().compareTo(o2.getName());
			}
		});

		Document doc = createSkeletonXHTML();
		Element body = doc.getRootElement().getChild("body");
		Element ul = new Element("ul");
		ul.setAttribute("class", "experiment");
		body.addContent(ul);
		for (Experiment exp : sortedExpList) {
			List<Subjexperiment> slist = map.get(exp);
			Element li = new Element("li");
			ul.addContent(li);
			li.setText(exp.getName());
			li.addContent(prepSubjectUL(slist));
		}

		StringWriter sw = new StringWriter(4096);
		FileUtils.saveXML(doc, sw);

		return sw.toString();
	}

	@Override
	public String getExperiments(String webUser, String pwd) throws Exception {
		String dbID = getDBID();
		UserInfo ui = ApplicationUtils.authenticate(webUser, pwd, dbID);
		System.out.println("ui=" + ui);

		IDBCache dbCache = ServiceFactory.getDBCache(dbID);

		List<Experiment> expList = dbCache.getExperiments(ui, false);
		Document doc = createSkeletonXHTML();
		Element body = doc.getRootElement().getChild("body");
		Element ul = new Element("ul");
		ul.setAttribute("class", "experiment");
		body.addContent(ul);
		for (Experiment exp : expList) {
			Element li = new Element("li");
			li.setText(exp.getName());
			ul.addContent(li);
		}
		StringWriter sw = new StringWriter(300);
		FileUtils.saveXML(doc, sw);

		return sw.toString();
	}

	@Override
	public String getSubjects4Experiment(String webUser, String pwd,
			String expName) throws Exception {
		String dbID = getDBID();
		UserInfo ui = ApplicationUtils.authenticate(webUser, pwd, dbID);
		System.out.println("ui=" + ui);
		ISubjectVisitManagement svm = ServiceFactory
				.getSubjectVisitManagement(dbID);
		List<Subjexperiment> slist = svm.getSubjectsForExperiment(ui, expName);

		Document doc = createSkeletonXHTML();
		Element body = doc.getRootElement().getChild("body");
		body.addContent(prepSubjectUL(slist));

		StringWriter sw = new StringWriter(1024);
		FileUtils.saveXML(doc, sw);

		return sw.toString();
	}

	@Override
	public String searchDD(String webUser, String pwd, String query)
			throws Exception {
		String dbID = getDBID();
		UserInfo ui = ApplicationUtils.authenticate(webUser, pwd, dbID);
		System.out.println("ui=" + ui);
		SimpleQueryLanguageParser parser = new SimpleQueryLanguageParser(query);
		parser.parse();
		List<Predicate> predicates = parser.getPredicates();
		IDerivedImageDataService dids = ServiceFactory
				.getDerivedImageDataService(dbID);
		Map<String, Predicate> predMap = new HashMap<String, Predicate>(7);
		for (Predicate pred : predicates) {
			predMap.put(pred.attribute, pred);
		}
		String subjectID = null;
		String expName = null;
		if (predMap.containsKey("subjectid")) {
			subjectID = predMap.get("subjectid").value;
		}
		if (predMap.containsKey("expname")) {
			expName = predMap.get("expname").value;
		}

		List<Deriveddata> ddList = null;
		String fileTypeClass = null;
		if (predMap.containsKey("filetype")) {
			String fileType = predMap.get("filetype").value.toLowerCase();
			fileTypeClass = fileType;
			if (!fileType.startsWith("."))
				fileType = "." + fileType;
			ddList = dids.findDerivedData(ui, subjectID, expName, null, null,
					fileType);
		} else {
			ddList = dids.findDerivedData(ui, subjectID, expName, null, null);
		}

		Document doc = createSkeletonXHTML();
		Element body = doc.getRootElement().getChild("body");
		Element ul = new Element("ul");
		if (fileTypeClass != null)
			ul.setAttribute("class", fileTypeClass);
		body.addContent(ul);
		for (Deriveddata dd : ddList) {
			Element li = new Element("li");
			Element aEl = new Element("a");
			ul.addContent(li);
			li.addContent(aEl);
			String uri = null;
			String dataURI = dd.getDatauri();
			if (dd.getDatauri().startsWith("/")) {
				uri = "/downloaduri" + dataURI;
			} else {
				uri = "/downloaduri/" + dataURI;
			}
			aEl.setAttribute("href", uri);
			// TODO no sanity check here
			String file = dataURI.substring(dataURI.lastIndexOf('/') + 1);
			aEl.setText(file);
		}

		if (!ddList.isEmpty()) {
			Set<String> ddURIRootSet = new HashSet<String>(3);
			Pattern p = Pattern.compile("\\.\\w+$");
			for (Deriveddata dd : ddList) {
				String uri = dd.getDatauri();
				String uriRoot = uri;
				if (uri.endsWith("/")) {
					uriRoot = uri.replaceAll("\\/+$", "");
				} else if (p.matcher(uriRoot).find()) {
					uriRoot = uri.replaceFirst("\\/[^\\/|\\.]+\\.\\w+$", "");
				}
				ddURIRootSet.add(uriRoot);
			}
			Element dl = new Element("dl");
			dl.setAttribute("class", "md");
			body.addContent(dl);
			prepDTDD(dl, "ddURIRoots", ddURIRootSet);
		}

		StringWriter sw = new StringWriter(1024);
		FileUtils.saveXML(doc, sw);

		return sw.toString();
	}

	protected void prepDTDD(Element parent, String className,
			Set<String> ddURIRootSet) {
		Element dt = new Element("dt");
		dt.setText(className);
		parent.addContent(dt);
		Element dd = new Element("dd");
		parent.addContent(dd);
		Element ul = new Element("ul");
		ul.setAttribute("class", className);
		dd.addContent(ul);
		for (String ddURIRoot : ddURIRootSet) {
			Element li = new Element("li");
			Element aEl = new Element("a");
			ul.addContent(li);
			li.addContent(aEl);
			aEl.setAttribute("href", ddURIRoot);
		}
	}

	protected Element prepSubjectUL(List<Subjexperiment> slist) {
		Element ul = new Element("ul");
		ul.setAttribute("class", "subject");
		for (Subjexperiment se : slist) {
			Element li = new Element("li");
			li.setText(se.getSubjectid());
			ul.addContent(li);
		}
		return ul;
	}

	protected String getDBID() throws Exception {
		IAppConfigService cs = AppConfigService.getInstance();
		String dbID = cs.getParamValue("fbirn.dbid");

		if (dbID == null) {
			throw new Exception(
					"Cannot detect data source! Possible misconfiguration!");
		}
		return dbID;
	}

	protected Document createSkeletonXHTML() {
		Document doc = new Document();
		DocType type = new DocType("html",
				"-//W3C//DTD XHTML 1.0 Transitional//EN",
				"http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd");
		doc.setDocType(type);
		Element root = new Element("html");
		doc.setRootElement(root);
		// root.setAttribute("xmlns", "http://www.w3.org/xhtml");
		// root.setAttribute("xml:lang", "en");
		root.addContent(new Element("head"));
		root.addContent(new Element("body"));
		return doc;
	}

	public static class Predicate {
		protected String attribute;
		protected String value;
		protected int op;
		protected int valueCode;
		protected int code;

		public Predicate(String attribute) {
			super();
			this.attribute = attribute;
		}

		public String toString() {
			StringBuilder sb = new StringBuilder();
			sb.append("Predicate::[");
			sb.append("attribute=").append(attribute);
			sb.append(",value=").append(value);
			sb.append(']');
			return sb.toString();
		}

	}// ;

	public static class SimpleQueryLanguageParser implements ITokens {
		StreamTokenizer stok;
		Map<String, Integer> reservedMap = new HashMap<String, Integer>(17);
		protected List<Predicate> predicates = new ArrayList<Predicate>(3);

		public SimpleQueryLanguageParser(String query) {
			StringReader sr = new StringReader(query);
			stok = new StreamTokenizer(sr);
			stok.resetSyntax();
			stok.lowerCaseMode(false);
			stok.quoteChar('\'');
			stok.eolIsSignificant(false);
			stok.ordinaryChar('=');
			stok.ordinaryChar('&');
			stok.wordChars('a', 'z');
			stok.wordChars('A', 'Z');
			stok.wordChars('0', '9');
			stok.wordChars('_', '_');
			stok.whitespaceChars(' ', ' ');
			stok.whitespaceChars('\t', '\t');
			reservedMap.put("subjectid", SUBJECTID);
			reservedMap.put("expname", EXPNAME);
			reservedMap.put("filetype", FILETYPE);
		}

		public void parse() throws IOException, ParseException {
			int tc;
			while ((tc = stok.nextToken()) != StreamTokenizer.TT_EOF) {
				if (tc == StreamTokenizer.TT_WORD) {
					String tok = stok.sval;
					if (reservedMap.containsKey(tok.toLowerCase())) {
						tok = tok.toLowerCase();
						Predicate p = new Predicate(tok);
						p.code = reservedMap.get(tok);
						parsePredicate(p);
						predicates.add(p);
					}
				}
			}
		}

		protected void parsePredicate(Predicate p) throws IOException,
				ParseException {
			int tc = stok.nextToken();
			checkEOF(p, tc);
			if (stok.ttype == '=') {
				p.op = EQUAL;
				tc = stok.nextToken();
				checkEOF(p, tc);
				if (tc == StreamTokenizer.TT_WORD) {
					p.value = stok.sval;
				} else if (stok.ttype == '\'') {
					p.value = stok.sval;
				} else {
					throw new ParseException("Bad predicate format for "
							+ p.attribute);
				}
			} else {
				throw new ParseException("Bad predicate format for "
						+ p.attribute);
			}
		}

		protected void checkEOF(Predicate p, int tc) throws ParseException {
			if (tc == StreamTokenizer.TT_EOF) {
				throw new ParseException(
						"Unexpected end of query for predicate " + p.attribute);
			}
		}

		public List<Predicate> getPredicates() {
			return predicates;
		}
	}

	public static class ParseException extends Exception {
		private static final long serialVersionUID = 1L;

		public ParseException(String message) {
			super(message);
		}

	}// ;

	public static void main(String[] args) throws Exception {
		SimpleQueryLanguageParser p = new SimpleQueryLanguageParser(
				"subjectID=000812345678 & expName='Exp 1' & fileType=MRML");
		p.parse();
		for (Predicate pr : p.getPredicates())
			System.out.println(pr);
	}

}
