package clinical.test.framework;

import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.Timestamp;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.jdom.Element;

import clinical.cache.CacheUtils;
import clinical.server.dao.AnimalspeciesDAO;
import clinical.server.dao.ExpcomponentDAO;
import clinical.server.dao.ExperimentDAO;
import clinical.server.dao.ExpsegmentDAO;
import clinical.server.dao.HumansubjectDAO;
import clinical.server.dao.PersonDAO;
import clinical.server.dao.ResearchgroupDAO;
import clinical.server.dao.SubjexperimentDAO;
import clinical.server.vo.Animalspecies;
import clinical.server.vo.Databaseuser;
import clinical.server.vo.Expcomponent;
import clinical.server.vo.Experiment;
import clinical.server.vo.Expsegment;
import clinical.server.vo.Humansubject;
import clinical.server.vo.Person;
import clinical.server.vo.Researchgroup;
import clinical.server.vo.Researchgrouptype;
import clinical.server.vo.Subjexperiment;
import clinical.utils.Assertion;
import clinical.utils.FileUtils;
import clinical.web.ConnectionSupportMixin;
import clinical.web.DAOFactory;
import clinical.web.DBUtils;
import clinical.web.ISequenceHelper;
import clinical.web.MinimalServiceFactory;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.query.TSQLProcessor;

/**
 * This class provides functionality for replacing experiment/subject/site
 * dependent information with test specific info in a given DBUnit database dump
 * in flat XML format.
 *
 * @author I. Burak Ozyurt
 * @version $Id: TestDataTemplateUtils.java,v 1.1.2.1 2008/10/02 00:42:58
 *          bozyurt Exp $
 */
public class TestDataTemplateUtils {
   String testPrefix = "HID_TEST_";
   String testExpID;
   String testSubjectID;
   String testVisitID = "2";
   String testSegmentID = "1";
   protected ConnectionSupportMixin mixin;
   ISequenceHelper seqHelper;
   String flatXmlFile;
   int startIdx;
   String dbUserID;
   Map<String, ColumnInfo> columnMap = new HashMap<String, ColumnInfo>();

   public TestDataTemplateUtils(String testPropsFile, String flatXmlFile)
         throws Exception {
      this.flatXmlFile = flatXmlFile;
      mixin = new ConnectionSupportMixin(testPropsFile, true);
      MinimalServiceFactory.setMimimalOpMode(true);
      ServiceFactory.setMimimalOpMode(true);
      mixin.startup();
      dbUserID = getDatabaseUser();
      seqHelper = ServiceFactory.getSequenceHelper(mixin.getDbID());
      ensureTestSubjectExists();
      startIdx = seqHelper.getNextUID(mixin.getConnection(), "", "").intValue();

   }

   public TestDataTemplateUtils(ConnectionSupportMixin mixin, String flatXmlFile)
         throws Exception {
      this.mixin = mixin;
      this.flatXmlFile = flatXmlFile;
      dbUserID = getDatabaseUser();
      seqHelper = ServiceFactory.getSequenceHelper(mixin.getDbID());
      ensureTestSubjectExists();
      startIdx = seqHelper.getNextUID(mixin.getConnection(), "", "").intValue();
   }

   public void shutdown() {
      try {
         mixin.shutdown();
      } catch (Exception e) {
         e.printStackTrace();
      }
   }

   public String getDatabaseUser() throws Exception {
      TSQLProcessor tsp = new TSQLProcessor(mixin.getSqlDialect());
      List<?> results = tsp
            .executeQuery(mixin.getConnection(),
                  "select u.* from Databaseuser as u where u.name ='ADMIN' and u.isgroup = false");
      Assertion.assertFalse(results.isEmpty());
      Databaseuser dbUser = (Databaseuser) results.get(0);
      return dbUser.getUniqueid().toString();
   }

   public BigDecimal getContactPerson() throws Exception {
      PersonDAO dao = DAOFactory.createPersonDAO(mixin.getDbID());
      List<Person> persons = dao.find(mixin.getConnection(), new Person());
      return persons.get(0).getUniqueid();
   }

   public BigDecimal getAnimalSpecies() throws Exception {
      AnimalspeciesDAO dao = DAOFactory.createAnimalspeciesDAO(mixin.getDbID());
      List<Animalspecies> asList = dao.find(mixin.getConnection(),
            new Animalspecies());
      return asList.get(0).getUniqueid();
   }

   public BigDecimal getRGControlType() throws Exception {
      IDBCache dbCache = ServiceFactory.getDBCache(mixin.getDbID());
      List<Researchgrouptype> rgTypes = dbCache.getResearchGroupTypes(mixin
            .getUi(), false);
      Researchgrouptype crgt = null;
      for (Researchgrouptype rgt : rgTypes) {
         if (rgt.getName().equalsIgnoreCase("control")) {
            crgt = rgt;
            break;
         }
      }
      Assertion.assertNotNull(crgt);
      return crgt.getUniqueid();
   }

   public Researchgroup getOrInsertResearchGroup(BigDecimal expID)
         throws Exception {
      IDBCache dbCache = ServiceFactory.getDBCache(mixin.getDbID());
      List<Researchgroup> rgList = dbCache.getResearchGroups(mixin.getUi(),
            false);
      Researchgroup crg = null;
      for (Researchgroup rg : rgList) {
         if (rg.getName().equals("Control")
               && rg.getNcExperimentUniqueid().equals(expID)) {
            crg = rg;
            break;
         }
      }
      if (crg != null)
         return crg;
      ResearchgroupDAO rgDAO = DAOFactory.createResearchgroupDAO(mixin
            .getDbID());
      Researchgroup rg = new Researchgroup();
      BigDecimal rgtID = getRGControlType();
      rg.setName("Control");
      rg.setNcExperimentUniqueid(expID);
      rg.setNcResearchgrouptypeUniqueid(rgtID);
      int tableID = DBUtils.getTableID(mixin.getDbID(), mixin.getConnection(),
            "NC_RESEARCHGROUP");
      rg.setTableid(new BigDecimal(tableID));
      rg.setModtime(new Date());
      rg.setModuser(new BigDecimal(dbUserID));
      rg.setOwner(new BigDecimal(dbUserID));
      BigDecimal uniqueId = seqHelper.getNextUID(mixin.getConnection(), "", "");
      rg.setUniqueid(uniqueId);

      rgDAO.insert(mixin.getConnection(), rg);
      return rg;
   }

   public void ensureTestSubjectExists() throws Exception {
      Connection con = null;
      try {
         BigDecimal uniqueId = null;

         String dbID = mixin.getDbID();
         String siteID = CacheUtils.getDBID2SiteIDMap().get(dbID);
         this.testSubjectID = siteID + "12345678";
         con = mixin.getConnection();
         ExperimentDAO expDAO = DAOFactory.createExperimentDAO(dbID);
         Experiment crExp = new Experiment();
         crExp.setName("HID_TEST_EXPERIMENT");
         List<Experiment> expList = expDAO.find(con, crExp);
         Experiment exp = null;
         if (expList.isEmpty()) {
            exp = new Experiment();
            exp.setContactperson(getContactPerson());
            exp.setName("HID_TEST_EXPERIMENT");
            int tableID = DBUtils.getTableID(dbID, con, "NC_EXPERIMENT");
            exp.setTableid(new BigDecimal(tableID));
            exp.setModtime(new Date());
            exp.setModuser(new BigDecimal(dbUserID));
            exp.setOwner(new BigDecimal(dbUserID));
            uniqueId = seqHelper.getNextUID(mixin.getConnection(), "", "");
            exp.setUniqueid(uniqueId);
            this.testExpID = uniqueId.toString();
            expDAO.insert(con, exp);
         } else {
            this.testExpID = expList.get(0).getUniqueid().toString();
         }

         BigDecimal expID = new BigDecimal(this.testExpID);

         HumansubjectDAO hsDAO = DAOFactory.createHumansubjectDAO(dbID);
         Humansubject hs = new Humansubject();
         hs.setSubjectid(testSubjectID);
         List<Humansubject> hsList = hsDAO.find(con, hs);
         Humansubject subject = null;
         if (hsList.isEmpty()) {
            subject = new Humansubject();
            subject.setSubjectid(this.testSubjectID);
            int tableID = DBUtils.getTableID(dbID, con, "NC_HUMANSUBJECT");
            subject.setTableid(new BigDecimal(tableID));
            subject.setModtime(new Date());
            subject.setModuser(new BigDecimal(dbUserID));
            subject.setOwner(new BigDecimal(dbUserID));
            uniqueId = seqHelper.getNextUID(mixin.getConnection(), "", "");
            subject.setUniqueid(uniqueId);
            subject.setExtensionname("humanSubject");
            subject.setNcAnimalspeciesUniqueid(getAnimalSpecies());
            hsDAO.insert(con, subject);
         }

         Researchgroup rg = getOrInsertResearchGroup(expID);

         SubjexperimentDAO seDAO = DAOFactory.createSubjexperimentDAO(dbID);
         Subjexperiment seCR = new Subjexperiment();
         seCR.setNcExperimentUniqueid(expID);
         seCR.setSubjectid(testSubjectID);
         List<Subjexperiment> seList = seDAO.find(con, seCR);
         if (seList.isEmpty()) {
            Subjexperiment se = new Subjexperiment();
            se.setNcExperimentUniqueid(expID);
            se.setNcResearchgroupUniqueid(rg.getUniqueid());
            se.setSubjectid(testSubjectID);
            int tableID = DBUtils.getTableID(dbID, con, "NC_SUBJEXPERIMENT");
            se.setTableid(new BigDecimal(tableID));
            se.setModtime(new Date());
            se.setModuser(new BigDecimal(dbUserID));
            se.setOwner(new BigDecimal(dbUserID));
            uniqueId = seqHelper.getNextUID(con, "", "");
            se.setUniqueid(uniqueId);
            seDAO.insert(con, se);
         }

         ExpcomponentDAO visitDAO = DAOFactory.createExpcomponentDAO(dbID);
         BigDecimal visitID = new BigDecimal(testVisitID);
         Expcomponent visitCR = new Expcomponent();
         visitCR.setSubjectid(testSubjectID);
         visitCR.setNcExperimentUniqueid(new BigDecimal(testExpID));
         visitCR.setComponentid(visitID);
         List<Expcomponent> visitList = visitDAO.find(con, visitCR);
         if (visitList.isEmpty()) {
            Expcomponent v = new Expcomponent();
            v.setComponentid(visitID);
            v.setName("Test Visit");
            v.setNcExperimentUniqueid(expID);
            v.setSubjectid(this.testSubjectID);
            v.setVisittype("fMRI scan");
            int tableID = DBUtils.getTableID(dbID, con, "NC_EXPCOMPONENT");
            v.setTableid(new BigDecimal(tableID));
            v.setModtime(new Date());
            v.setModuser(new BigDecimal(dbUserID));
            v.setOwner(new BigDecimal(dbUserID));
            uniqueId = seqHelper.getNextUID(mixin.getConnection(), "", "");
            v.setUniqueid(uniqueId);
            v.setTimeStamp(new Timestamp(System.currentTimeMillis()));
            visitDAO.insert(con, v);
         }

         ExpsegmentDAO segDAO = DAOFactory.createExpsegmentDAO(dbID);
         BigDecimal segID = new BigDecimal(testSegmentID);
         Expsegment segCR = new Expsegment();
         segCR.setNcExperimentUniqueid(new BigDecimal(testExpID));
         segCR.setSubjectid(testSubjectID);
         segCR.setComponentid(visitID);
         List<Expsegment> segList = segDAO.find(con, segCR);
         if (segList.isEmpty()) {
            Expsegment s = new Expsegment();
            s.setNcExperimentUniqueid(new BigDecimal(testExpID));
            s.setSubjectid(testSubjectID);
            s.setComponentid(visitID);
            s.setSegmentid(segID);
            s.setName("t1");
            s.setIsbad(Boolean.FALSE);
            s.setIstimeinterval(Boolean.FALSE);
            s.setStudyid(null);
            s.setProtocolid("Subjects screening protocol");
            s.setProtocolversion(new BigDecimal(1));
            s.setTimeStamp(new Timestamp(System.currentTimeMillis()));
            int tableID = DBUtils.getTableID(dbID, con, "NC_EXPSEGMENT");
            s.setTableid(new BigDecimal(tableID));
            s.setModtime(new Date());
            s.setModuser(new BigDecimal(dbUserID));
            s.setOwner(new BigDecimal(dbUserID));
            uniqueId = seqHelper.getNextUID(mixin.getConnection(), "", "");
            s.setUniqueid(uniqueId);

            segDAO.insert(con, s);
         }
         con.commit();
      } catch (Exception x) {
         x.printStackTrace();
         con.rollback();
      }
   }

   public void removeTestExperiment() throws Exception {
      Connection con = null;
      try {
         String dbID = mixin.getDbID();
         con = mixin.getConnection();
         BigDecimal expID = new BigDecimal(this.testExpID);

         ExpsegmentDAO segDAO = DAOFactory.createExpsegmentDAO(dbID);
         ExpcomponentDAO visitDAO = DAOFactory.createExpcomponentDAO(dbID);
         SubjexperimentDAO seDAO = DAOFactory.createSubjexperimentDAO(dbID);
         ResearchgroupDAO rgDAO = DAOFactory.createResearchgroupDAO(dbID);
         HumansubjectDAO hsDAO = DAOFactory.createHumansubjectDAO(dbID);
         ExperimentDAO expDAO = DAOFactory.createExperimentDAO(dbID);

         Expsegment segCR = new Expsegment();
         segCR.setNcExperimentUniqueid(expID);
         segCR.setSubjectid(testSubjectID);
         segDAO.delete(con, segCR);

         Expcomponent visitCR = new Expcomponent();
         visitCR.setNcExperimentUniqueid(expID);
         visitCR.setSubjectid(testSubjectID);
         visitDAO.delete(con, visitCR);

         Subjexperiment seCR = new Subjexperiment();
         seCR.setNcExperimentUniqueid(expID);
         seCR.setSubjectid(testSubjectID);
         seDAO.delete(con, seCR);

         Researchgroup rgCR = new Researchgroup();
         rgCR.setNcExperimentUniqueid(expID);
         rgDAO.delete(con, rgCR);

         Humansubject hsCR = new Humansubject();
         hsCR.setSubjectid(testSubjectID);
         hsDAO.delete(con, hsCR);

         Experiment expCR = new Experiment();
         expCR.setUniqueid(expID);
         expDAO.delete(con, expCR);

         con.commit();
      } catch (Exception x) {
         x.printStackTrace();
         con.rollback();
      }
   }

   public void rewrite(String outFile) throws Exception {
      ColumnInfo ci = getCI("nc_humansubject", "subjectid");
      ci.old2NewMap.put("001029291693", this.testSubjectID);

      ci = getCI("nc_experiment", "experimentid");
      ci.old2NewMap.put("385", testExpID);
      ci = getCI("nc_experiment", "nc_experiment_uniqueid");
      ci.old2NewMap.put("385", testExpID);
      ci = getCI("nc_expcomponent", "componentid");
      ci.old2NewMap.put("2", testVisitID);
      ci = getCI("nc_expsegment", "segmentid");
      ci.old2NewMap.put("15", testSegmentID);

      Element rootEl = FileUtils.loadXML(flatXmlFile);
      List<?> tableEls = rootEl.getChildren();
      for (Iterator<?> it = tableEls.iterator(); it.hasNext();) {
         Element row = (Element) it.next();
         String tableName = row.getName();
         updateRow(row, tableName);
      }

      FileUtils.saveXML(rootEl, outFile);
   }

   public void updateRow(Element row, String tableName) {
      Set<String> noPKTableSet = new HashSet<String>();
      noPKTableSet.add("nc_analysisflow");
      noPKTableSet.add("nc_executedtransformflow");
      noPKTableSet.add("nc_transforminputdata");
      String columnName = null;
      String uniqueId = String.valueOf(startIdx++);
      String oldUniqueId = row.getAttributeValue("uniqueid");
      if (!noPKTableSet.contains(tableName)
            || tableName.equalsIgnoreCase("nc_transforminputdata")) {
         row.setAttribute("uniqueid", uniqueId);
      }
      if (tableName.equals("nc_analysis")) {
         columnName = "name";
         ColumnInfo ci;
         adjustAttribute(row, tableName, columnName);

         ci = getCI(tableName, "analysisid");
         ci.old2NewMap.put(oldUniqueId, uniqueId);
         ci = getCI(tableName, "basetupleid");
         ci.old2NewMap.put(oldUniqueId, uniqueId);
      } else if (tableName.equals("nc_transformation")) {
         ColumnInfo ci = getCI(tableName, "transformationid");
         ci.old2NewMap.put(oldUniqueId, uniqueId);
         ci = getCI(tableName, "logicaltransformid");
         ci.old2NewMap.put(oldUniqueId, uniqueId);
      } else if (tableName.equals("nc_analysiscomponent")) {
         replaceAttribute(row, "analysisid");
         replaceAttribute(row, "transformationid");
         ColumnInfo ci = getCI(tableName, "analysiscomponentid");
         ci.old2NewMap.put(oldUniqueId, uniqueId);
      } else if (tableName.equals("nc_analysisflow")) {
         replaceAttribute(row, "analysisid");
      } else if (tableName.equals("nc_machine")) {
         ColumnInfo ci = getCI(tableName, "machineid");
         ci.old2NewMap.put(oldUniqueId, uniqueId);
      } else if (tableName.equals("nc_extendedtuple")) {
         ColumnInfo ci = getCI(tableName, "extendedtupleid");
         ci.old2NewMap.put(oldUniqueId, uniqueId);
         adjustAttribute(row, tableName, "name");
      } else if (tableName.equals("nc_tuplecolumns")) {
         replaceAttribute(row, "extendedtupleid");

      } else if (tableName.equals("nc_executedtransform")) {
         replaceAttribute(row, "logicaltransformid");
         replaceAttribute(row, "machineid");
         replaceAttribute(row, "analysisid");
         row.setAttribute("nc_databaseuser_uniqueid", dbUserID);
         ColumnInfo ci = getCI(tableName, "executedtransformationid");
         ci.old2NewMap.put(oldUniqueId, uniqueId);
         ci = getCI(tableName, "nc_executedtransform_uniqueid");
         ci.old2NewMap.put(oldUniqueId, uniqueId);
         ci = getCI(tableName, "startexecutedtransformid");
         ci.old2NewMap.put(oldUniqueId, uniqueId);
         ci = getCI(tableName, "priorexecutedtransformid");
         ci.old2NewMap.put(oldUniqueId, uniqueId);
      } else if (tableName.equals("nc_executedtransformflow")) {
         replaceAttribute(row, "executedtransformationid");
         replaceAttribute(row, "startexecutedtransformid");
         replaceAttribute(row, "priorexecutedtransformid");
         replaceAttribute(row, "analysisid");
      } else if (tableName.equals("nc_analysisresult")) {
         ColumnInfo ci = getCI(tableName, "basetupleid");
         ci.old2NewMap.put(oldUniqueId, uniqueId);
         replaceAttribute(row, "segmentid");
         replaceAttribute(row, "componentid");
         replaceAttribute(row, "experimentid");
         replaceAttribute(row, "subjectid");
         replaceAttribute(row, "executedtransformationid");
         replaceAttribute(row, "analysisid");
      } else if (tableName.equals("nc_storedtuple")) {
         replaceAttribute(row, "basetupleid");
         replaceAttribute(row, "extendedtupleid");
         ColumnInfo ci = getCI(tableName, "storedtupleid");
         ci.old2NewMap.put(oldUniqueId, uniqueId);
      } else if (tableName.equals("nc_tuplevarchar")
            || tableName.equals("nc_tuplefloat")
            || tableName.equals("nc_tupleinteger")
            || tableName.equals("nc_tupledata")) {
         replaceAttribute(row, "extendedtupleid");
         replaceAttribute(row, "storedtupleid");
      } else if (tableName.equals("nc_researchdata")) {
         // no op
      } else if (tableName.equals("nc_deriveddata")) {
         replaceAttribute(row, "segmentid");
         replaceAttribute(row, "componentid");
         replaceAttribute(row, "nc_experiment_uniqueid");
         replaceAttribute(row, "subjectid");
         ColumnInfo ci = getCI(tableName, "nc_researchdata_uniqueid");
         ci.old2NewMap.put(oldUniqueId, uniqueId);
      } else if (tableName.equals("nc_transformeddata")
            || tableName.equals("nc_transforminputdata")) {
         replaceAttribute(row, "nc_researchdata_uniqueid");
         replaceAttribute(row, "nc_executedtransform_uniqueid");
      }

      // update owner and moduser
      if (!noPKTableSet.contains(tableName)) {
         row.setAttribute("owner", dbUserID);
         row.setAttribute("moduser", dbUserID);
      }
   }

   protected void replaceAttribute(Element row, String columnName) {
      String oldValue = row.getAttributeValue(columnName);
      if (oldValue == null)
         return;
      ColumnInfo ci = getCI(null, columnName);
      String newValue = ci.old2NewMap.get(oldValue);
      Assertion.assertNotNull(newValue);
      row.setAttribute(columnName, newValue);
   }

   protected void adjustAttribute(Element row, String tableName,
         String columnName) {
      String oldValue = row.getAttributeValue(columnName);
      String newValue = testPrefix + oldValue;
      ColumnInfo ci = getCI(tableName, columnName);

      ci.old2NewMap.put(oldValue, newValue);
      row.setAttribute(columnName, newValue);
   }

   protected ColumnInfo getCI(String tableName, String columnName) {
      ColumnInfo ci = columnMap.get(columnName);
      if (ci == null) {
         ci = new ColumnInfo(columnName, tableName);
         columnMap.put(columnName, ci);
      }
      return ci;
   }

   public static class ColumnInfo {
      String name;
      String tableName;
      Map<String, String> old2NewMap = new HashMap<String, String>(3);

      public ColumnInfo(String name, String tableName) {
         super();
         this.name = name;
         this.tableName = tableName;
      }

   }

   public static void main(String[] args) throws Exception {
      TestDataTemplateUtils t = null;
      try {
         String flatXmlFile = "/home/bozyurt3/dev/java/B_SLICER_WS/BIRN/clinical/test/db_test_data/fs_small_test_template.xml";
         String outFile = "/home/bozyurt3/dev/java/B_SLICER_WS/BIRN/clinical/test/db_test_data/fs_small_test.xml";
         t = new TestDataTemplateUtils("test.properties", flatXmlFile);
         t.rewrite(outFile);
         // t.removeTestExperiment();

      } finally {
         if (t != null)
            t.shutdown();
      }
   }
}
