package clinical.web;

import java.sql.Connection;
import java.util.List;
import java.util.Map;

import clinical.server.vo.Expcomponent;
import clinical.server.vo.Experiment;
import clinical.server.vo.Humansubject;
import clinical.server.vo.Researchgroup;
import clinical.server.vo.Subjexperiment;
import clinical.web.common.UserInfo;
import clinical.web.common.query.SearchCriteria;
import clinical.web.exception.DBPoolServiceException;
import clinical.web.exception.SubjectVisitManagementException;
import clinical.web.vo.Study;
import clinical.web.vo.StudySegment;
import clinical.web.vo.StudySegmentInfo;
import clinical.web.vo.Subject;
import clinical.web.vo.Visit;
import clinical.web.vo.VisitSegment;

/**
 * Session facade interface for subject/visit data management.
 * 
 * @author I. Burak Ozyurt
 * @version $Id: ISubjectVisitManagement.java,v 1.11 2006/03/01 02:47:26 bozyurt
 *          Exp $
 */

public interface ISubjectVisitManagement {

	public Connection getConnection(UserInfo ui) throws DBPoolServiceException;

	/**
	 * Returns the experiments a subject has particated in.
	 * 
	 * @param ui
	 *           User Information used to use the correct named database user
	 *           connection from the connection pool
	 * @param subjectID
	 *           the BIRN ID of the subject
	 * @return A list of value objects of type
	 *         {@link clinical.server.vo.Experiment}
	 * @throws SubjectVisitManagementException
	 */
	public List<Experiment> getExperimentsForSubject(UserInfo ui,
			String subjectID) throws SubjectVisitManagementException;

	/**
	 * Returns the list of reasearch groups this particular subject is involved.
	 * 
	 * @param ui
	 *           User Information used to use the correct named database user
	 *           connection from the connection pool
	 * @param subjectID
	 *           the BIRN ID of the subject
	 * @return A list of value objects of type
	 *         {@link clinical.server.vo.Researchgroup}
	 * @throws SubjectVisitManagementException
	 * 
	 * @see clinical.server.vo.Researchgroup
	 */
	public List<Researchgroup> getResarchGroupsForSubject(UserInfo ui,
			String subjectID) throws SubjectVisitManagementException;

	/**
	 * Given a search criteria object, retrieves all matching human subjects.
	 * 
	 * @param ui
	 *           User Information used to use the correct named database user
	 *           connection from the connection pool
	 * @param sc
	 *           search criteria consisting of search predicates combined with
	 *           AND or OR
	 * @return A list of value object of type
	 *         {@link clinical.server.vo.Humansubject}
	 * @throws SubjectVisitManagementException
	 * 
	 */
	public List<Humansubject> getMatchingSubjects(UserInfo ui, SearchCriteria sc)
			throws SubjectVisitManagementException;

	/**
	 * 
	 * @param ui
	 *           User Information used to use the correct named database user
	 *           connection from the connection pool
	 * @param expID
	 *           int
	 * @throws SubjectVisitManagementException
	 * @return A list of value object of type
	 *         {@link clinical.server.vo.Subjexperiment}
	 */
	public List<Subjexperiment> getSubjectsForExperiment(UserInfo ui, int expID)
			throws SubjectVisitManagementException;

	public Humansubject getSubjectByID(UserInfo ui, String subjectID)
			throws SubjectVisitManagementException;

	/**
	 * Given a new subject's information, adds it to the database.
	 * 
	 * @param ui
	 *           User Information used to use the correct named database user
	 *           connection from the connection pool
	 * @param subject
	 *           a value object containing subject information to be inserted
	 *           into the database
	 * @throws SubjectVisitManagementException
	 * 
	 * @see Subject
	 */
	public void addSubject(UserInfo ui, Subject subject)
			throws SubjectVisitManagementException;

	public void addSubject(Connection con, UserInfo ui, Subject subject)
			throws Exception;

	/**
	 * Updates information of a given subject in the database.
	 * 
	 * @param ui
	 *           User Information used to use the correct named database user
	 *           connection from the connection pool
	 * @param subject
	 *           a value object containing subject information to be updated
	 * @throws SubjectVisitManagementException
	 */
	public void updateSubject(UserInfo ui, Subject subject)
			throws SubjectVisitManagementException;

	/**
	 * Removes a particular subject from the database. The passed
	 * {@link clinical.web.vo.Subject} object needs to have subjectID property
	 * set, even though primary keys cannot be updated, subjectID is used to find
	 * the subject to update.
	 * 
	 * @param ui
	 *           User Information used to use the correct named database user
	 *           connection from the connection pool
	 * @param subject
	 *           a value object containing subject information to find the
	 *           subject to be deleted
	 * @throws SubjectVisitManagementException
	 * @see clinical.web.vo.Subject
	 */
	public void deleteSubject(UserInfo ui, Subject subject)
			throws SubjectVisitManagementException;

	/**
	 * Given a subject ID, returns the visits associated with this subject.
	 * 
	 * @param ui
	 *           User Information used to use the correct named database user
	 *           connection from the connection pool
	 * @param subjectID
	 *           the BIRN ID of the subject
	 * @return A list of {@link clinical.web.vo.Visit} objects each also contains
	 *         the associated visit segments as
	 *         {@link clinical.web.vo.VisitSegment} value objects.
	 * @throws SubjectVisitManagementException
	 * @see clinical.web.vo.Visit
	 * @see clinical.web.vo.VisitSegment
	 */
	public List<Visit> getAllVisitsForSubject(UserInfo ui, String subjectID)
			throws SubjectVisitManagementException;

	/**
	 * Given a search criteria on the {@link clinical.server.vo.Expcomponent}
	 * object's properties, returns the matching visits.
	 * 
	 * @param ui
	 *           User Information used to use the correct named database user
	 *           connection from the connection pool
	 * @param sc
	 *           search criteria consisting of search predicates combined with
	 *           AND or OR
	 * @return A list of {@link clinical.web.vo.Visit} objects
	 * @throws SubjectVisitManagementException
	 * @see clinical.web.vo.Visit
	 * @see clinical.server.vo.Expcomponent
	 */
	public List<Expcomponent> getVisitsForSubject(UserInfo ui, SearchCriteria sc)
			throws SubjectVisitManagementException;

	/**
	 * Given a (composite) {@link clinical.web.vo.Visit} object, adds the visit
	 * and its segments (if any) to the subject as determined by the SubjectID
	 * property of the {@link clinical.web.vo.Visit} object.
	 * 
	 * @param ui
	 *           User Information used to use the correct named database user
	 *           connection from the connection pool
	 * @param visit
	 *           a presentation layer value object holding visit and segment
	 *           information
	 * @return visitID
	 * @throws SubjectVisitManagementException
	 * @see clinical.web.vo.Visit
	 */
	public int addVisitForSubject(UserInfo ui, Visit visit)
			throws SubjectVisitManagementException;

	public void addVisitForSubjectNonContiguous(Connection con, UserInfo ui,
			Visit visit) throws SubjectVisitManagementException;

	/**
	 * Given a (composite) {@link clinical.web.vo.Visit} object, adds updates the
	 * visit and its segments (if any) of the subject as determined by the
	 * SubjectID property of the {@link clinical.web.vo.Visit} object.
	 * 
	 * @param ui
	 *           User Information used to use the correct named database user
	 *           connection from the connection pool
	 * @param visit
	 *           a presentation layer value object holding visit and segment
	 *           information
	 * @throws SubjectVisitManagementException
	 */
	public void updateVisitForSubject(UserInfo ui, Visit visit)
			throws SubjectVisitManagementException;

	/**
	 * Given a (composite) {@link clinical.web.vo.Visit} object, removes the
	 * visit and its segments (if any) of the subject as determined by the
	 * SubjectID property of the {@link clinical.web.vo.Visit} object from the
	 * database.
	 * 
	 * @param ui
	 *           User Information used to use the correct named database user
	 *           connection from the connection pool
	 * @param visit
	 *           a presentation layer value object holding visit and segment
	 *           information
	 * @throws SubjectVisitManagementException
	 */
	public void deleteVisitForSubject(UserInfo ui, Visit visit)
			throws SubjectVisitManagementException;

	/**
	 * Adds a segment to an existing visit for a subject as identified by the
	 * corresponding properties in the {@link clinical.web.vo.VisitSegment}
	 * object.
	 * 
	 * @param ui
	 *           User Information used to use the correct named database user
	 *           connection from the connection pool
	 * @param visitSegment
	 *           a presentation layer value object holding segment information
	 *           for a visit
	 * @return newSegmentID
	 * @throws SubjectVisitManagementException
	 * @see clinical.web.vo.VisitSegment
	 */
	public int addVisitSegmentForSubject(UserInfo ui, VisitSegment visitSegment)
			throws SubjectVisitManagementException;

	public int handleAddVisitSegment(Connection con, UserInfo ui,
			VisitSegment visitSegment) throws Exception;

	/**
	 * Updates a segment for an existing visit for a subject as identified by the
	 * corresponding properties in the {@link clinical.web.vo.VisitSegment}
	 * object.
	 * 
	 * @param ui
	 *           User Information used to use the correct named database user
	 *           connection from the connection pool
	 * @param visitSegment
	 *           a presentation layer value object holding segment information
	 *           for a visit
	 * @throws SubjectVisitManagementException
	 * @see clinical.web.vo.VisitSegment
	 */
	public void updateVisitSegmentForSubject(UserInfo ui,
			VisitSegment visitSegment) throws SubjectVisitManagementException;

	/**
	 * Removes a segment for an existing visit for a subject as identified by the
	 * corresponding properties in the {@link clinical.web.vo.VisitSegment}
	 * object, from the database.
	 * 
	 * @param ui
	 *           User Information used to use the correct named database user
	 *           connection from the connection pool
	 * @param visitSegment
	 *           a presentation layer value object holding segment information
	 *           for a visit
	 * @throws SubjectVisitManagementException
	 * @see clinical.web.vo.VisitSegment
	 */
	public void deleteVisitSegmentForSubject(UserInfo ui,
			VisitSegment visitSegment) throws SubjectVisitManagementException;

	/**
	 * 
	 * @param ui
	 *           User Information used to use the correct named database user
	 *           connection from the connection pool
	 * @return A list of value objects of type
	 *         {@link clinical.server.vo.Experiment}
	 * @throws SubjectVisitManagementException
	 */
	public List<Experiment> getAllExperiments(UserInfo ui)
			throws SubjectVisitManagementException;

	public Study addStudyForVisit(UserInfo ui, Study study)
			throws SubjectVisitManagementException;

	public void updateStudyForVisit(UserInfo ui, Study study)
			throws SubjectVisitManagementException;

	public int addStudySegmentForSubject(UserInfo ui, StudySegment studySegment)
			throws SubjectVisitManagementException;

	public void updateStudySegmentForSubject(UserInfo ui,
			StudySegment studySegment) throws SubjectVisitManagementException;

	/**
	 * 
	 * @param ui
	 *           User Information used to use the correct named database user
	 *           connection from the connection pool
	 * @param subjectID
	 *           the BIRN ID of the subject
	 * @param visitID
	 *           int
	 * @param segmentID
	 *           int
	 * @param experimentID
	 *           int
	 * @param assessmentID
	 *           int
	 * @throws SubjectVisitManagementException
	 */
	public void deleteAssessmentFromSegment(UserInfo ui, String subjectID,
			int visitID, int segmentID, int experimentID, int assessmentID)
			throws SubjectVisitManagementException;

	/**
	 * 
	 * @param ui
	 *           User Information used to use the correct named database user
	 *           connection from the connection pool
	 * @param subjectIDList
	 *           a list of {@link clinical.server.vo.Subjexperiment} objects
	 * @throws SubjectVisitManagementException
	 * @return List
	 */
	public List<StudySegmentInfo> findStudyInfosForSubjects(UserInfo ui,
			List<String> subjectIDList) throws SubjectVisitManagementException;

	/**
	 * 
	 * @param ui
	 *           UserInfo
	 * @param sc
	 *           {@link clinical.web.query.SearchCriteria}
	 * @throws SubjectVisitManagementException
	 * @return List
	 */
	public List<Subjexperiment> getMatchingExperimentSubjects(UserInfo ui,
			SearchCriteria sc) throws SubjectVisitManagementException;

	public List<Subjexperiment> getSubjectsForExperiment(UserInfo ui,
			String expName) throws SubjectVisitManagementException;

	public Map<Experiment, List<Subjexperiment>> getExperimentSubjects(
			UserInfo ui) throws SubjectVisitManagementException;

}
