package clinical.web.services;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import net.nbirn.mediator.client.Mediator;
import net.nbirn.mediator.client.Query;
import net.nbirn.mediator.client.Results;
import net.nbirn.mediator.client.ServiceNames;
import net.nbirn.mediator.client.Tuple;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.server.dao.BrainsegmentdataDAO;
import clinical.server.vo.Brainsegmentdata;
import clinical.web.Constants;
import clinical.web.DAOFactory;
import clinical.web.DBUtils;
import clinical.web.IDerivedDataService;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBPoolService;
import clinical.web.common.UserInfo;
import clinical.web.common.query.DerivedDataQueryBuilder;
import clinical.web.common.query.MediatedDerivedDataQueryBuilder;
import clinical.web.common.query.Operator;
import clinical.web.helpers.SubCorticalVarInfo;
import clinical.web.vo.SubjectDerivedDataStats;
import clinical.web.vo.SubjectDerivedDataValueSummary;

/**
 * Implements <code>IDerivedDataService</code> interface.
 *
 * @author I. Burak Ozyurt
 * @version $Id: DerivedDataService.java 90 2009-08-17 23:37:45Z bozyurt $
 */

public class DerivedDataService implements IDerivedDataService {
	private IDBPoolService pool = null;

	private BrainsegmentdataDAO dao;

	private String dbID = null;

	private Log log = LogFactory.getLog("clinical");

	public DerivedDataService(String dbID) {
		pool = ServiceFactory.getPoolService(dbID);
		dao = DAOFactory.createBrainsegmentdataDAO(dbID);
		this.dbID = dbID;
	}

	/**
	 * default constructor does not initialize connection pool, mainly for
	 * mediated queries
	 */
	public DerivedDataService() {
	}

	public List<Brainsegmentdata> getAllSubCorticalVariables(UserInfo userInfo) throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(userInfo.getName());
			List<?> bsds = dao.findDistinctBrainRegions(con);
			List<Brainsegmentdata> bsdList = new ArrayList<Brainsegmentdata>( bsds.size());
			for(Object item : bsds) {
				Brainsegmentdata bsd = (Brainsegmentdata) item;
				bsdList.add(bsd);
			}
			return bsdList;
		} finally {
			pool.releaseConnection(userInfo.getName(), con);
		}
	}

	public Map<String, Float> getAllSubcorticalRegionVolumesForSubject(
			UserInfo userInfo, String subjectID, int experimentID)
			throws Exception {
		StringBuffer buf = new StringBuffer(256);
		buf.append("select brainregionname,laterality,measurement from nc_brainsegmentdata where subjectid='");
		buf.append(subjectID).append("' and nc_experimentid=").append(
				experimentID).append(" AND measurementtype = 'volume'");
		Statement st = null;
		ResultSet rs = null;
		Connection con = null;
		Map<String, Float> resultMap = new LinkedHashMap<String, Float>();
		try {
			con = pool.getConnection(userInfo.getName());
			st = con.createStatement();
			String sql = buf.toString();
			rs = st.executeQuery(sql);
			while (rs.next()) {
				String regionName = rs.getString(1);
				String laterality = rs.getString(2);
				String key = "";
				if (rs.wasNull() || laterality.trim().length() == 0
						|| laterality.equalsIgnoreCase("n/a")) {
					key = regionName;
				} else {
					key = laterality + "_" + regionName;
				}
				float measurement = rs.getFloat(3);
				resultMap.put(key, new Float(measurement));
			}
		} finally {
			DBUtils.close(st, rs);
			pool.releaseConnection(userInfo.getName(), con);
		}
		return resultMap;
	}

	public SubjectDerivedDataStats getAllSubcorticalVolumeStatsForSubject(
			UserInfo userInfo, String subjectID, int experimentID)
			throws Exception {
		StringBuffer buf = new StringBuffer(128);
		buf
				.append("select brainregionname, laterality, measurement from nc_brainsegmentdata"
						+ " where measurementtype='volume'");
		buf.append(" AND subjectid='").append(subjectID).append(
				"' AND nc_experimentid=").append(experimentID);
		Statement st = null;
		ResultSet rs = null;
		Connection con = null;

		SubjectDerivedDataStats allStats = null;
		SubjectDerivedDataStats sds = new SubjectDerivedDataStats(subjectID);
		try {
			con = pool.getConnection(userInfo.getName());
			allStats = getDerivedDataStats(con, experimentID);
			st = con.createStatement();
			String sql = buf.toString();
			rs = st.executeQuery(sql);
			while (rs.next()) {
				String regionName = rs.getString(1);
				String laterality = rs.getString(2);
				float measurement = rs.getFloat(3);
				String key = prepareKey(regionName, laterality);
				SubjectDerivedDataStats.DerivedVarStats varStats = allStats
						.getDerivedVarStats(key);
				if (varStats != null) {
					SubjectDerivedDataStats.DerivedVarStats dvs = new SubjectDerivedDataStats.DerivedVarStats(
							varStats);
					dvs.setSubjectValue(measurement);
					sds.addDerivedVarStats(dvs);
				}
			}
		} finally {
			if (st != null)
				try {
					st.close();
				} catch (Exception x) {
				}
			pool.releaseConnection(userInfo.getName(), con);
		}

		return sds;
	}

	public SubjectDerivedDataStats getAllSubcorticalVolumeStatsForSubject(
			UserInfo userInfo, String subjectID, int experimentID,
			String[] subjectsInPop) throws Exception {
		StringBuffer buf = new StringBuffer(128);
		buf
				.append("select brainregionname, laterality, measurement from nc_brainsegmentdata where measurementtype='volume'");
		buf.append(" AND subjectid='").append(subjectID).append(
				"' AND nc_experimentid=").append(experimentID);
		Statement st = null;
		ResultSet rs = null;
		Connection con = null;

		SubjectDerivedDataStats allStats = null;
		SubjectDerivedDataStats sds = new SubjectDerivedDataStats(subjectID);
		try {
			con = pool.getConnection(userInfo.getName());
			allStats = getPopulationDerivedDataStats(con, experimentID,
					subjectsInPop);
			st = con.createStatement();
			String sql = buf.toString();
			rs = st.executeQuery(sql);
			while (rs.next()) {
				String regionName = rs.getString(1);
				String laterality = rs.getString(2);
				float measurement = rs.getFloat(3);
				String key = prepareKey(regionName, laterality);
				SubjectDerivedDataStats.DerivedVarStats varStats = allStats
						.getDerivedVarStats(key);
				if (varStats != null) {
					SubjectDerivedDataStats.DerivedVarStats dvs = new SubjectDerivedDataStats.DerivedVarStats(
							varStats);
					dvs.setSubjectValue(measurement);
					sds.addDerivedVarStats(dvs);
				}
			}
		} finally {
			if (st != null)
				try {
					st.close();
				} catch (Exception x) {
				}
			pool.releaseConnection(userInfo.getName(), con);
		}
		return sds;
	}

	protected String prepareKey(String regionName, String laterality) {
		String key = "";
		if (laterality.trim().length() == 0
				|| laterality.trim().equalsIgnoreCase("n/a")) {
			key = regionName;
		} else {
			key = laterality + "_" + regionName;
		}
		return key;
	}

	protected SubjectDerivedDataStats getDerivedDataStats(Connection con,
			int experimentID) throws SQLException {
		StringBuffer buf = new StringBuffer(256);
		buf
				.append("select avg(measurement), min(measurement), max(measurement), ");
		buf.append("stddev(measurement), brainregionname, laterality ");
		buf
				.append("from nc_brainsegmentdata where measurementtype='volume' AND nc_experimentid=");
		buf.append(experimentID);
		buf.append(" group by brainregionname, laterality");
		Statement st = null;
		ResultSet rs = null;
		SubjectDerivedDataStats sds = new SubjectDerivedDataStats(null);
		try {
			st = con.createStatement();
			String sql = buf.toString();
			rs = st.executeQuery(sql);
			while (rs.next()) {
				float avg = rs.getFloat(1);
				float min = rs.getFloat(2);
				float max = rs.getFloat(3);
				float std = rs.getFloat(4);
				String regionName = rs.getString(5);
				String laterality = rs.getString(6);
				SubCorticalVarInfo scvi = new SubCorticalVarInfo(regionName,
						laterality, "volume", "");
				SubjectDerivedDataStats.DerivedVarStats dvs = new SubjectDerivedDataStats.DerivedVarStats(
						scvi, max, min, avg, std, -1);
				sds.addDerivedVarStats(dvs);
			}

		} finally {
			if (rs != null)
				try {
					rs.close();
				} catch (Exception x) {
				}
			if (st != null)
				try {
					st.close();
				} catch (Exception x) {
				}
		}

		return sds;
	}

	protected SubjectDerivedDataStats getPopulationDerivedDataStats(
			Connection con, int experimentID, String[] popSubjectIDs)
			throws SQLException {
		StringBuffer buf = new StringBuffer(256);
		buf
				.append("select avg(measurement), min(measurement), max(measurement), stddev(measurement), brainregionname, laterality ");
		buf
				.append(
						"from nc_brainsegmentdata where measurementtype='volume' AND nc_experimentid=")
				.append(experimentID);
		buf.append(" AND subjectid IN (");
		for (int i = 0; i < popSubjectIDs.length; i++) {
			buf.append("'").append(popSubjectIDs[i]).append("'");
			if ((i + 1) < popSubjectIDs.length)
				buf.append(',');
		}
		buf.append(") group by brainregionname, laterality");
		Statement st = null;
		ResultSet rs = null;
		SubjectDerivedDataStats sds = new SubjectDerivedDataStats(null);
		try {
			st = con.createStatement();
			String sql = buf.toString();
			log.info("DerivedDataService: sql=" + sql);
			rs = st.executeQuery(sql);
			while (rs.next()) {
				float avg = rs.getFloat(1);
				float min = rs.getFloat(2);
				float max = rs.getFloat(3);
				float std = rs.getFloat(4);
				String regionName = rs.getString(5);
				String laterality = rs.getString(6);
				SubCorticalVarInfo scvi = new SubCorticalVarInfo(regionName,
						laterality, "volume", "");
				SubjectDerivedDataStats.DerivedVarStats dvs = new SubjectDerivedDataStats.DerivedVarStats(
						scvi, max, min, avg, std, -1);
				sds.addDerivedVarStats(dvs);
			}

		} finally {
			if (rs != null)
				try {
					rs.close();
				} catch (Exception x) {
				}
			if (st != null)
				try {
					st.close();
				} catch (Exception x) {
				}
		}

		return sds;
	}

	public List<SubjectDerivedDataValueSummary> getSubCorticalValuesForSubjects(
			UserInfo userInfo, String dbID, List<SubCorticalVarInfo> derivedDataList,
			Operator rootOp) throws Exception {
		Connection con = null;

		if (rootOp == null || rootOp.isEmpty()) {
			if (log.isDebugEnabled())
				log.debug("*** The op tree is empty!");
			return new LinkedList<SubjectDerivedDataValueSummary>();
		}

		DerivedDataQueryBuilder qb = new DerivedDataQueryBuilder();
		qb.visit(rootOp);
		if ( log.isDebugEnabled())
			log.debug("Derived Data query=" + qb.getQuery());
		Map<String, SubCorticalVarInfo> ddiMap = new HashMap<String, SubCorticalVarInfo>();
		for (SubCorticalVarInfo sv : derivedDataList) {
			ddiMap.put(sv.getBrainRegionName() + sv.getLaterality(), sv);
			if ( log.isDebugEnabled())
				log.debug(">> sv=" + sv);
		}

		List<SubjectDerivedDataValueSummary> results = new LinkedList<SubjectDerivedDataValueSummary>();
		Statement st = null;
		ResultSet rs = null;
		IDBPoolService aPool = null;
		try {
			// use the connection pool as specified by the dbID
			aPool = ServiceFactory.getPoolService(dbID);

			con = aPool.getConnection(userInfo.getName());
			st = con.createStatement();
			String sql = qb.getQuery();
			rs = st.executeQuery(sql);
			while (rs.next()) {
				String subjectID = rs.getString(1);
				SubjectDerivedDataValueSummary sdvs = new SubjectDerivedDataValueSummary();
				sdvs.setSubjectID(subjectID);
				String brainRegionName = rs.getString(2);
				String laterality = rs.getString(3);
				sdvs.setValue(rs.getFloat(4));
				sdvs.setVisitID(rs.getInt(5));
				sdvs.setSegmentID(rs.getInt(6));
				sdvs.setExperimentID(rs.getInt(7));

				SubCorticalVarInfo sv = ddiMap
						.get(brainRegionName + laterality);
				log.debug("sv=" + sv);
				sdvs.setScVarInfo(sv);

				results.add(sdvs);
			}

		} finally {
			DBUtils.close(st, rs);
			if (aPool != null) {
				aPool.releaseConnection(userInfo.getName(), con);
			}
		}
		return results;
	}

	public List<SubjectDerivedDataValueSummary> getSubCorticalValuesForSubjects(
			UserInfo userInfo, List<String> subjectList,
			List<SubCorticalVarInfo> derivedDataList, Operator rootOp)
			throws Exception {
		Connection con = null;

		if (rootOp == null || rootOp.isEmpty()) {
			if (log.isDebugEnabled())
				log.debug("*** The op tree is empty!");
			return new LinkedList<SubjectDerivedDataValueSummary>();
		}

		DerivedDataQueryBuilder qb = new DerivedDataQueryBuilder();
		qb.visit(rootOp);

		log.debug("Derived Data query=" + qb.getQuery());
		Map<String, SubCorticalVarInfo> ddiMap = new HashMap<String, SubCorticalVarInfo>();
		for (SubCorticalVarInfo sv : derivedDataList) {
			ddiMap.put(sv.getBrainRegionName() + sv.getLaterality(), sv);
			if ( log.isDebugEnabled())
				log.debug(">> sv=" + sv);
		}
		Map<String, String> sidMap = new HashMap<String, String>();
		for (String subjectID : subjectList) {
			sidMap.put(subjectID, subjectID);
		}

		List<SubjectDerivedDataValueSummary> results = new LinkedList<SubjectDerivedDataValueSummary>();
		Statement st = null;
		ResultSet rs = null;
		try {
			con = pool.getConnection(userInfo.getName());
			st = con.createStatement();
			String sql = qb.getQuery();
			rs = st.executeQuery(sql);
			while (rs.next()) {
				String subjectID = rs.getString(1);
				if (sidMap.get(subjectID) == null)
					continue;
				SubjectDerivedDataValueSummary sdvs = new SubjectDerivedDataValueSummary();
				sdvs.setSubjectID(subjectID);
				String brainRegionName = rs.getString(2);
				String laterality = rs.getString(3);
				sdvs.setValue(rs.getFloat(4));
				sdvs.setVisitID(rs.getInt(5));
				sdvs.setSegmentID(rs.getInt(6));
				sdvs.setExperimentID(rs.getInt(7));

				SubCorticalVarInfo sv = ddiMap
						.get(brainRegionName + laterality);
				log.debug("sv=" + sv);
				sdvs.setScVarInfo(sv);

				results.add(sdvs);
			}

		} finally {
			DBUtils.close(st, rs);
			pool.releaseConnection(userInfo.getName(), con);
		}

		return results;
	}

	public List<SubjectDerivedDataValueSummary> getMediatedSubCorticalValues(
			List<SubCorticalVarInfo> derivedDataList, Operator rootOp) throws Exception {
		if (rootOp == null || rootOp.isEmpty()) {
			log.info("*** The op tree is empty!");
			return new LinkedList<SubjectDerivedDataValueSummary>();
		}

		MediatedDerivedDataQueryBuilder mqb = new MediatedDerivedDataQueryBuilder();
		mqb.visit(rootOp);

		log.info("Mediated Derived Data query=" + mqb.getQuery());
		Map<String, SubCorticalVarInfo> ddiMap = new HashMap<String, SubCorticalVarInfo>();
		for (SubCorticalVarInfo sv : derivedDataList) {
			ddiMap.put(sv.getBrainRegionName() + sv.getLaterality(), sv);
		}
		List<SubjectDerivedDataValueSummary> results = new LinkedList<SubjectDerivedDataValueSummary>();
		net.nbirn.mediator.client.Connection con = new net.nbirn.mediator.client.Connection(
				Constants.MEDIATOR_URL, ServiceNames.MEDIATOR_SERVICE);

		Mediator m = new Mediator("mhid", con);

		String queryStr = mqb.getQuery();
		Query q = new Query(queryStr, m);
		Results rs = new Results(q);
		rs.init();
		while (rs.hasMore()) {
			Tuple tuple = (Tuple) rs.next();
			String subjectID = tuple.getAttValue(0);

			SubjectDerivedDataValueSummary sdvs = new SubjectDerivedDataValueSummary();
			sdvs.setSubjectID(subjectID);
			String brainRegionName = tuple.getAttValue(1);
			String laterality = tuple.getAttValue(2);
			sdvs.setValue(Float.parseFloat(tuple.getAttValue(3)));
			sdvs.setVisitID(Integer.parseInt(tuple.getAttValue(4)));
			sdvs.setSegmentID(Integer.parseInt(tuple.getAttValue(5)));
			sdvs.setExperimentID(Integer.parseInt(tuple.getAttValue(6)));

			SubCorticalVarInfo sv = ddiMap.get(brainRegionName + laterality);
			if (log.isDebugEnabled()) {
				log.debug("sv=" + sv);
			}
			sdvs.setScVarInfo(sv);

			results.add(sdvs);
		}
		return results;
	}

	public List<SubjectDerivedDataValueSummary> getMediatedSubCorticalValuesForSubjects(
			List<String> subjectList, List<SubCorticalVarInfo> derivedDataList, Operator rootOp)
			throws Exception {
		if (rootOp == null || rootOp.isEmpty()) {
			log.info("*** The op tree is empty!");
			return new LinkedList<SubjectDerivedDataValueSummary>();
		}

		MediatedDerivedDataQueryBuilder mqb = new MediatedDerivedDataQueryBuilder();
		mqb.visit(rootOp);

		log.info("Mediated Derived Data query=" + mqb.getQuery());
		Map<String, SubCorticalVarInfo> ddiMap = new HashMap<String, SubCorticalVarInfo>();
		for (SubCorticalVarInfo sv : derivedDataList) {
			ddiMap.put(sv.getBrainRegionName() + sv.getLaterality(), sv);
		}

		Map<String, String> sidMap = new HashMap<String, String>();
		for (String subjectID : subjectList) {
			sidMap.put(subjectID, subjectID);
		}

		List<SubjectDerivedDataValueSummary> results = new LinkedList<SubjectDerivedDataValueSummary>();
		net.nbirn.mediator.client.Connection con = new net.nbirn.mediator.client.Connection(
				Constants.MEDIATOR_URL, ServiceNames.MEDIATOR_SERVICE);

		Mediator m = new Mediator("mhid", con);

		String queryStr = mqb.getQuery();
		Query q = new Query(queryStr, m);
		Results rs = new Results(q);
		rs.init();
		while (rs.hasMore()) {
			Tuple tuple = (Tuple) rs.next();
			String subjectID = tuple.getAttValue(0);
			if (sidMap.get(subjectID) == null)
				continue;
			SubjectDerivedDataValueSummary sdvs = new SubjectDerivedDataValueSummary();
			sdvs.setSubjectID(subjectID);
			String brainRegionName = tuple.getAttValue(1);
			String laterality = tuple.getAttValue(2);
			sdvs.setValue(Float.parseFloat(tuple.getAttValue(3)));
			sdvs.setVisitID(Integer.parseInt(tuple.getAttValue(4)));
			sdvs.setSegmentID(Integer.parseInt(tuple.getAttValue(5)));
			sdvs.setExperimentID(Integer.parseInt(tuple.getAttValue(6)));

			SubCorticalVarInfo sv = ddiMap.get(brainRegionName + laterality);
			if (log.isDebugEnabled()) {
				log.debug("sv=" + sv);
			}
			sdvs.setScVarInfo(sv);

			results.add(sdvs);
		}

		return results;
	}

	public String getDbID() {
		return dbID;
	}

}
