package clinical.web.services;

import java.math.BigDecimal;
import java.security.MessageDigest;
import java.security.SecureRandom;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Random;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.server.dao.TokenDAO;
import clinical.server.vo.Token;
import clinical.web.DAOFactory;
import clinical.web.ISequenceHelper;
import clinical.web.MinimalServiceFactory;
import clinical.web.common.UserInfo;

/**
 * @author I. Burak Ozyurt
 * @version $Id: FileManagementServiceImpl.java 90 2009-08-17 23:37:45Z bozyurt $
 */
public class FileManagementServiceImpl extends AbstractServiceImpl {
	private Random rng;
	public final static long MAX_AGE = 24 * 3600 * 1000; // 1 day
	protected static Log log = LogFactory.getLog(FileManagementServiceImpl.class);

	public FileManagementServiceImpl(String dbID) throws Exception {
		super(dbID);
		rng = SecureRandom.getInstance("SHA1PRNG");
		((SecureRandom) rng).setSeed(SecureRandom.getSeed(128));
	}

	public List<Token> createOneTimeTokens(UserInfo ui, int noTokensRequested)
			throws Exception {
		Connection con = null;
		TokenDAO dao = DAOFactory.createTokenDAO(theDBID);
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			ISequenceHelper sequenceHelper = MinimalServiceFactory
					.getSequenceHelper(theDBID);
			BigDecimal groupId = sequenceHelper.getNextUID(con, "NC_TOKEN",
					"groupid");
			MessageDigest md = MessageDigest.getInstance("MD5");

			List<Token> tokens = new ArrayList<Token>(noTokensRequested);
			Date now = new Date();
			for (int i = 0; i < noTokensRequested; i++) {
				String token = null;
				do {
					token = createToken(groupId, md);
					if (!hasCollision(con, token)) {
						break;
					}
				} while (true);

				Token tk = new Token();
				tk.setToken(token);
				tk.setGroupid(groupId);
				tk.setTimecreated(now);
				tk.setExpired(new Boolean(false));
				tk.setServed(new Boolean(false));

				dao.insert(con, tk);
				tokens.add(tk);

			}
			con.commit();
			return tokens;
		} catch (Exception x) {
			handleErrorAndRollBack(con, "createOneTimeTokens", x, true);
			return null;
		} finally {
			releaseConnection(con, ui);
		}
	}

	public boolean isTokenValid(UserInfo ui, String token) throws Exception {
		if (token == null || token.trim().length() == 0) {
			return false;
		}
		Connection con = null;
		TokenDAO dao = DAOFactory.createTokenDAO(theDBID);
		boolean ok = false;
		long now = System.currentTimeMillis();
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			Token cr = new Token();
			cr.setToken(token);
			List<Token> tkList = getTokens(con, cr);
			if (tkList.isEmpty()) {
				return false;
			}
			Token tk = tkList.get(0);
			long age = now - tk.getTimecreated().getTime();
			if (age > MAX_AGE) {
				if (!tk.getExpired().booleanValue()) {
					// tag all in group as expired
					Token update = new Token();
					update.setExpired(new Boolean(true));
					updateTokens(con, tk.getGroupid(), update);
				}
			} else {
				if (!tk.getServed().booleanValue()) {
					ok = true;
					Token update = new Token();
					update.setServed(new Boolean(true));
					dao.update(con, update, tk);
				} else {
					// check if all in group are served
					// if not allow the token to be reused till all in group
					// is served at once
					cr = new Token();
					cr.setGroupid(tk.getGroupid());
					tkList = getTokens(con, cr);
					boolean allServed = true;
					for (Token tk2 : tkList) {
						if (!tk2.getServed().booleanValue()) {
							allServed = false;
							break;
						}
					}
					if (!allServed) {
						ok = true;
					}
				}
			}
			con.commit();
		} catch (Exception x) {
			handleErrorAndRollBack(con, "createOneTimeTokens", x, true);
		} finally {
			releaseConnection(con, ui);
		}
		return ok;
	}

	protected boolean hasCollision(Connection con, String token)
			throws Exception {
		TokenDAO dao = DAOFactory.createTokenDAO(theDBID);
		Token cr = new Token();
		cr.setToken(token);
		List<Token> list = dao.find(con, cr);
		return !list.isEmpty();
	}

	protected List<Token> getTokens(Connection con, Token criteria)
			throws Exception {
		TokenDAO dao = DAOFactory.createTokenDAO(theDBID);
		List<Token> list = dao.find(con, criteria);
		return list;
	}

	protected void updateTokens(Connection con, BigDecimal groupID, Token update)
			throws Exception {
		TokenDAO dao = DAOFactory.createTokenDAO(theDBID);
		Token cr = new Token();
		cr.setGroupid(groupID);
		dao.update(con, update, cr);
	}

	

	// utility methods

	private String createToken(BigDecimal groupId, MessageDigest md) {
		StringBuffer buf = new StringBuffer();
		long number = rng.nextLong();
		buf.append(number).append('_').append(groupId);

		md.update(buf.toString().getBytes());
		byte[] bytes = md.digest();

		return toHex(bytes);
	}

	protected static String toHex(byte[] bytes) {
		StringBuffer buf = new StringBuffer();
		for (int i = 0; i < bytes.length; ++i) {
			int byteVal = (bytes[i] < 0) ? bytes[i] + 256 : bytes[i];
			String s = Integer.toHexString(byteVal);
			s = (s.length() == 1) ? "0" + s : s;
			buf.append(s);
		}
		return buf.toString();
	}

}
