package clinical.web.services;

import java.math.BigDecimal;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.cache.DBChangeNotifySupport;
import clinical.server.dao.AnalysisDAO;
import clinical.server.dao.AnalysisresultDAO;
import clinical.server.dao.ExtendedtupleDAO;
import clinical.server.dao.MeasurementsystemDAO;
import clinical.server.dao.MeasurementtypeDAO;
import clinical.server.dao.MeasurementunitDAO;
import clinical.server.dao.StoredtupleDAO;
import clinical.server.dao.TuplebooleanDAO;
import clinical.server.dao.TupleclassDAO;
import clinical.server.dao.TuplecolumnsDAO;
import clinical.server.dao.TuplefloatDAO;
import clinical.server.dao.TupleintegerDAO;
import clinical.server.dao.TuplesubclassDAO;
import clinical.server.dao.TupletimestampDAO;
import clinical.server.dao.TuplevarcharDAO;
import clinical.server.vo.Analysis;
import clinical.server.vo.Analysisresult;
import clinical.server.vo.Databaseuser;
import clinical.server.vo.Executedtransform;
import clinical.server.vo.Experiment;
import clinical.server.vo.Expsegment;
import clinical.server.vo.Extendedtuple;
import clinical.server.vo.Measurementsystem;
import clinical.server.vo.Measurementtype;
import clinical.server.vo.Measurementunit;
import clinical.server.vo.Ontologyconcept;
import clinical.server.vo.Ontologysource;
import clinical.server.vo.Storedtuple;
import clinical.server.vo.Tupleboolean;
import clinical.server.vo.Tupleclass;
import clinical.server.vo.Tuplecolumns;
import clinical.server.vo.Tupledata;
import clinical.server.vo.Tuplefloat;
import clinical.server.vo.Tupleinteger;
import clinical.server.vo.Tuplesubclass;
import clinical.server.vo.Tupletimestamp;
import clinical.server.vo.Tuplevarchar;
import clinical.utils.Assertion;
import clinical.utils.GenUtils;
import clinical.web.Constants;
import clinical.web.DAOFactory;
import clinical.web.IProvenanceService;
import clinical.web.IRemoteDBServices;
import clinical.web.ISequenceHelper;
import clinical.web.IXCEDEDerivedDataService;
import clinical.web.MinimalServiceFactory;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.UserInfo;
import clinical.web.common.query.TSQLProcessor;
import clinical.web.common.vo.CommonTupleDataInfo;
import clinical.web.exception.BaseException;
import clinical.web.helpers.ExtendedTupleHelper;
import clinical.xml.xcede2.AbstractEntityT;
import clinical.xml.xcede2.AnalysisT;
import clinical.xml.xcede2.AnatomicalEntityT;
import clinical.xml.xcede2.LevelDataRefsT;
import clinical.xml.xcede2.MeasurementGroupT;
import clinical.xml.xcede2.ObservationT;
import clinical.xml.xcede2.TerminologyStringT;

/**
 * @author I. Burak Ozyurt
 * @version $Id: XCEDEDerivedDataServiceImpl.java,v 1.5 2008/03/14 19:19:26
 *          bozyurt Exp $
 */

public class XCEDEDerivedDataServiceImpl extends AbstractServiceImpl implements
      IXCEDEDerivedDataService {
   protected static Log log = LogFactory
         .getLog(XCEDEDerivedDataServiceImpl.class);

   public XCEDEDerivedDataServiceImpl(String dbID) throws BaseException {
      super(dbID);
   }

   public void deleteAnalysesForSubject(UserInfo ui, AnalysisT provenanceAT)
         throws Exception {
      Connection con = null;
      try {
         con = pool.getConnection(ui.getName());
         con.setAutoCommit(false);
         IDBCache dbCache = ServiceFactory.getDBCache(theDBID);
         IRemoteDBServices rdbs = ServiceFactory.getRemoteDBServices();
         IProvenanceService prvs = ServiceFactory.getProvenanceService(theDBID);

         LevelDataRefsT ldr = provenanceAT.getInput().get(0);
         String subjectID = ldr.getSubjectID();
         String expName = ldr.getProjectID();
         String visitStr = ldr.getVisitID();
         String[] toks = visitStr.split("__");
         int visitID = GenUtils.toInt(toks[1], -1);

         String segmentName = ldr.getEpisodeID();
         Assertion.assertTrue(visitID != -1);

         List<Experiment> expList = dbCache.getExperiments(ui, false);
         Experiment theExp = findExperiment(expName, expList);
         if (theExp == null) {
            throw new RuntimeException("Experiment '" + expName
                  + "' does not exists in the database!");
         }
         Integer experimentID = theExp.getUniqueid().intValue();
         // TODO assumption acquisition site ID is retrieved from the BIRN ID
         String siteID = subjectID.substring(0, 4);
         Expsegment segment = rdbs.getSegmentInfo(ui, siteID, expName,
               subjectID, visitID, segmentName);
         Assertion.assertNotNull(segment);
         Integer segmentID = segment.getSegmentid().intValue();

         prvs.deleteProvenanceForAnalysisInstance(con, ui, provenanceAT);

         deleteAnalysesForSubject(con, ui, subjectID, experimentID, visitID,
               segmentID);

         con.commit();
      } catch (Exception x) {
         handleErrorAndRollBack(con, "deleteAnalysesForSubject", x, true);
      } finally {
         releaseConnection(con, ui);
      }
   }

   public void addAnalysesForSubject(UserInfo ui, String subjectID,
         String expName, Integer visitID, String segmentName,
         AnalysisT provenanceAT, List<AnalysisT> remAtList,
         String shortAnalysisName, String snapshotID) throws Exception {
      Connection con = null;
      try {
         con = pool.getConnection(ui.getName());
         con.setAutoCommit(false);
         IDBCache dbCache = ServiceFactory.getDBCache(theDBID);
         IRemoteDBServices rdbs = ServiceFactory.getRemoteDBServices();

         List<Experiment> expList = dbCache.getExperiments(ui, false);
         Experiment theExp = findExperiment(expName, expList);
         Assertion.assertNotNull(theExp);

         // TODO assumption acquisition site ID is retrieved from the BIRN ID
         String siteID = subjectID.substring(0, 4);
         Expsegment segment = rdbs.getSegmentInfo(ui, siteID, expName,
               subjectID, visitID, segmentName);
         Assertion.assertNotNull(segment);

         IProvenanceService prvs = ServiceFactory.getProvenanceService(theDBID);
         List<Executedtransform> etList = prvs
               .addProvenanceForAnalysisInstance(con, ui, provenanceAT,
                     shortAnalysisName, snapshotID);
         // for the list of AnalysisT records containing summary derived data

         // TODO assumption all the analysis result records are associated
         // with the first executed transformation in the chain
         Executedtransform theET = etList.get(0);

         for (AnalysisT at : remAtList) {
            addAnalysisForSubject(con, ui, subjectID, theExp.getUniqueid()
                  .intValue(), visitID, segment.getSegmentid().intValue(),
                  theET.getUniqueid().intValue(), at, provenanceAT.getType());
         }
         con.commit();
      } catch (Exception x) {
         handleErrorAndRollBack(con, "addAnalysesForSubject", x, true);
      } finally {
         releaseConnection(con, ui);
      }

   }

   protected Experiment findExperiment(String expName, List<Experiment> expList) {
      Experiment theExp = null;
      for (Experiment exp : expList) {
         if (exp.getName().equals(expName)) {
            theExp = exp;
            break;
         }
      }
      return theExp;
   }

   public void addAnalysisForSubject(Connection con, UserInfo ui,
         String subjectID, Integer experimentID, Integer visitID,
         Integer segmentID, Integer executedTransformID, AnalysisT analysis,
         String analysisTypeName) throws Exception {

      Databaseuser dbUser = getDatabaseUser(ui, Constants.USERCLASS_ADMIN);
      Assertion.assertNotNull(dbUser);
      List<MeasurementGroupT> mgList = analysis.getMeasurementGroup();
      // MeasurementGroupT amg = mgList.get(0);
      BigDecimal extTupleID = findMatchingExtendedTuple(con, analysis);
      Assertion.assertNotNull(extTupleID);

      Analysis an = findAnalysisByName(con, analysisTypeName);
      Assertion.assertNotNull(an);
      AnalysisGroup anGroup = new AnalysisGroup(analysis);

      Analysisresult ar = addAnalysisResultRec(ui, con, subjectID,
            experimentID, visitID, segmentID, executedTransformID, analysis
                  .getID(), an.getUniqueid().intValue(), anGroup);

      BigDecimal baseTableID = getTableID(ui, Constants.ANALYSIS_RESULT);
      BigDecimal baseTupleID = ar.getUniqueid();

      for (MeasurementGroupT mg : mgList) {
         Assertion.assertTrue(mg.getEntity().size() == 1);

         Storedtuple st = addStoredTuple(ui, con, baseTableID, baseTupleID,
               extTupleID);

         AbstractEntityT at = mg.getEntity().get(0);
         CommonTupleDataInfo ctdi = new CommonTupleDataInfo(st.getUniqueid(),
               dbUser.getUniqueid(), extTupleID);

         if (at instanceof AnatomicalEntityT) {
            AnatomicalEntityT aet = (AnatomicalEntityT) at;
            TerminologyStringT ts = aet.getLabel().get(0);
            String atLabel = ts.getValue();
            ctdi.setColumnType(Constants.VARCHAR_TYPE);
            ctdi.setColumnName("anatomicalEntity");
            addTupleVarchar(ui, con, ctdi, atLabel);
            ctdi.setColumnName("nomenclature");
            addTupleVarchar(ui, con, ctdi, ts.getNomenclature());
            if (ts.getTermID() != null) {
               ctdi.setColumnName("termID");
               addTupleVarchar(ui, con, ctdi, ts.getTermID());
            }
            if (aet.getLaterality() != null) {
               ctdi.setColumnName("laterality");
               addTupleVarchar(ui, con, ctdi, aet.getLaterality());
            }
            if (aet.getTissueType() != null) {
               ctdi.setColumnName("tissueType");
               addTupleVarchar(ui, con, ctdi, aet.getTissueType());
            }
         }

         for (ObservationT ob : mg.getObservation()) {
            String columnType = getObservationType(ob);
            ctdi.setColumnType(columnType);
            ctdi.setColumnName(ob.getName());
            if (columnType.equals(Constants.FLOAT_TYPE)) {
               addTupleFloat(ui, con, ctdi, ob.getValue());
            } else if (columnType.equals(Constants.VARCHAR_TYPE)) {
               addTupleVarchar(ui, con, ctdi, ob.getValue());
            } else if (columnType.equals(Constants.INTEGER_TYPE)) {
               addTupleInteger(ui, con, ctdi, ob.getValue());
            }
         }
      }
   }

   public static class AnalysisGroup {
      String groupName;
      String groupLongName;
      String groupOntologySource;
      String groupConceptId;

      public AnalysisGroup(AnalysisT analysis) {
         groupName = analysis.getAbbreviation();
         groupLongName = analysis.getPreferredLabel();
         groupOntologySource = analysis.getNomenclature();
         groupConceptId = analysis.getTermID();
      }
   }// ;

   public void addAnalysisForSubject(UserInfo ui, String subjectID,
         Integer experimentID, Integer visitID, Integer segmentID,
         Integer executedTransformID, AnalysisT analysis,
         String analysisTypeName) throws Exception {

      Connection con = null;
      try {
         con = pool.getConnection(ui.getName());
         con.setAutoCommit(false);

         addAnalysisForSubject(con, ui, subjectID, experimentID, visitID,
               segmentID, executedTransformID, analysis, analysisTypeName);
         con.commit();
      } catch (Exception x) {
         handleErrorAndRollBack(con, "addAnalysisForSubject", x, true);
      } finally {
         releaseConnection(con, ui);
      }
   }

   public Analysis findAnalysisByName(Connection con, String analysisName)
         throws Exception {
      AnalysisDAO dao = DAOFactory.createAnalysisDAO(theDBID);
      Analysis cr = new Analysis();
      cr.setName(analysisName);
      List<Analysis> results = dao.find(con, cr);
      if (results.isEmpty())
         return null;
      Assertion.assertTrue(results.size() == 1);
      return results.get(0);
   }

   public static String getObservationType(ObservationT ob) {
      if (ob.getType() != null) {
         return ob.getType().value();
      }
      String columnType = Constants.FLOAT_TYPE;
      if (ob.getValue() != null && !GenUtils.isDecimalNumber(ob.getValue())) {
         columnType = Constants.INTEGER_TYPE;
      }
      return columnType;
   }

   protected Analysisresult addAnalysisResultRec(UserInfo ui, Connection con,
         String subjectID, Integer experimentID, Integer visitID,
         Integer segmentID, Integer executedTransformID, String analysisName,
         Integer analysisUniqueID, AnalysisGroup ag) throws Exception {
      ISequenceHelper sequenceHelper = MinimalServiceFactory
            .getSequenceHelper(theDBID);
      Databaseuser dbUser = getDatabaseUser(ui, Constants.USERCLASS_ADMIN);
      Assertion.assertNotNull(dbUser);

      AnalysisresultDAO dao = DAOFactory.createAnalysisResultDAO(theDBID);
      Analysisresult ar = new Analysisresult();
      BigDecimal uniqueID = sequenceHelper.getNextUID(con,
            Constants.ANALYSIS_RESULT, "uniqueid");
      ar.setUniqueid(uniqueID);
      ar.setTableid(getTableID(ui, Constants.ANALYSIS_RESULT));
      if (segmentID != null)
         ar.setSegmentid(new BigDecimal(segmentID));
      if (visitID != null)
         ar.setComponentid(new BigDecimal(visitID));
      ar.setExperimentid(new BigDecimal(experimentID));
      ar.setSubjectid(subjectID);
      if (executedTransformID != null)
         ar.setExecutedtransformid(new BigDecimal(executedTransformID));
      ar.setOwner(dbUser.getUniqueid());
      ar.setModtime(new Date());
      ar.setModuser(dbUser.getUniqueid());
      ar.setOntologysource(Constants.DEFAULT_ONT_SRC);
      ar.setConceptid(Constants.DEFAULT_ONT_CONCEPT);
      ar.setIsbad(new Boolean(false));
      ar.setAnalysisname(analysisName);
      ar.setAnalysisid(new BigDecimal(analysisUniqueID));
      if (ag != null) {
         ar.setGroupname(ag.groupName);
         ar.setGrouplongname(ag.groupLongName);
         ar.setGroupontologysource(ag.groupOntologySource);
         ar.setGroupconceptid(ag.groupConceptId);
      }
      dao.insert(con, ar);
      return ar;
   }

   protected Storedtuple addStoredTuple(UserInfo ui, Connection con,
         BigDecimal baseTableID, BigDecimal baseTupleID,
         BigDecimal extendedTupleID) throws Exception {
      ISequenceHelper sequenceHelper = MinimalServiceFactory
            .getSequenceHelper(theDBID);
      Databaseuser dbUser = getDatabaseUser(ui, Constants.USERCLASS_ADMIN);
      Assertion.assertNotNull(dbUser);

      StoredtupleDAO dao = DAOFactory.createStoredtupleDAO(theDBID);
      Storedtuple st = new Storedtuple();
      BigDecimal uniqueID = sequenceHelper.getNextUID(con,
            Constants.STORED_TUPLE, "uniqueid");
      st.setUniqueid(uniqueID);
      st.setBasetableid(baseTableID);
      st.setBasetupleid(baseTupleID);
      st.setExtendedtupleid(extendedTupleID);
      st.setTableid(getTableID(ui, Constants.STORED_TUPLE));
      st.setOwner(dbUser.getUniqueid());
      st.setModtime(new Date());
      st.setModuser(dbUser.getUniqueid());

      dao.insert(con, st);
      return st;
   }

   protected void addTupleInteger(UserInfo ui, Connection con,
         CommonTupleDataInfo ctdi, String value) throws Exception {
      ISequenceHelper sequenceHelper = MinimalServiceFactory
            .getSequenceHelper(theDBID);

      TupleintegerDAO dao = DAOFactory.createTupleintegerDAO(theDBID);

      Tupleinteger ti = new Tupleinteger();
      ti.setColumnname(ctdi.getColumnName());
      ti.setColumntype(Constants.INTEGER_TYPE);
      ti.setComments(ctdi.getComments());
      ti.setDatavalue(new BigDecimal(value));
      ti.setTextvalue(value);
      ti.setExtendedtupleid(ctdi.getExtendedTupleID());
      ti.setStoredtupleid(ctdi.getStoredTupleID());
      ti.setOwner(ctdi.getDbUserID());
      ti.setModuser(ctdi.getDbUserID());
      ti.setModtime(new Date());
      BigDecimal uniqueID = sequenceHelper.getNextUID(con,
            Constants.TUPLE_INTEGER, "uniqueid");
      ti.setUniqueid(uniqueID);
      ti.setTableid(getTableID(ui, Constants.TUPLE_INTEGER));

      dao.insert(con, ti);
   }

   protected void addTupleVarchar(UserInfo ui, Connection con,
         CommonTupleDataInfo ctdi, String value) throws Exception {
      ISequenceHelper sequenceHelper = MinimalServiceFactory
            .getSequenceHelper(theDBID);

      TuplevarcharDAO dao = DAOFactory.createTuplevarcharDAO(theDBID);

      Tuplevarchar tv = new Tuplevarchar();
      tv.setColumnname(ctdi.getColumnName());
      tv.setColumntype(Constants.VARCHAR_TYPE);
      tv.setComments(ctdi.getComments());
      tv.setDatavalue(value);
      tv.setTextvalue(value);
      tv.setExtendedtupleid(ctdi.getExtendedTupleID());
      tv.setStoredtupleid(ctdi.getStoredTupleID());
      tv.setOwner(ctdi.getDbUserID());
      tv.setModuser(ctdi.getDbUserID());
      tv.setModtime(new Date());
      BigDecimal uniqueID = sequenceHelper.getNextUID(con,
            Constants.TUPLE_VARCHAR, "uniqueid");
      tv.setUniqueid(uniqueID);
      tv.setTableid(getTableID(ui, Constants.TUPLE_VARCHAR));

      dao.insert(con, tv);
   }

   protected void addTupleFloat(UserInfo ui, Connection con,
         CommonTupleDataInfo ctdi, String value) throws Exception {
      ISequenceHelper sequenceHelper = MinimalServiceFactory
            .getSequenceHelper(theDBID);

      TuplefloatDAO dao = DAOFactory.createTuplefloatDAO(theDBID);

      Tuplefloat ti = new Tuplefloat();
      ti.setColumnname(ctdi.getColumnName());
      ti.setColumntype(Constants.FLOAT_TYPE);
      ti.setComments(ctdi.getComments());
      ti.setDatavalue(new Float(value));
      ti.setTextvalue(value);
      ti.setExtendedtupleid(ctdi.getExtendedTupleID());
      ti.setStoredtupleid(ctdi.getStoredTupleID());
      ti.setOwner(ctdi.getDbUserID());
      ti.setModuser(ctdi.getDbUserID());
      ti.setModtime(new Date());
      BigDecimal uniqueID = sequenceHelper.getNextUID(con,
            Constants.TUPLE_FLOAT, "uniqueid");
      ti.setUniqueid(uniqueID);
      ti.setTableid(getTableID(ui, Constants.TUPLE_FLOAT));
      System.out.println("inserting " + ti);
      dao.insert(con, ti);
   }

   public void deleteExtendedTuple(UserInfo ui, String tupleName)
         throws BaseException {
      Assertion.assertNotNull(tupleName);
      ExtendedtupleDAO etDAO = DAOFactory.createExtendedtupleDAO(theDBID);
      TuplecolumnsDAO tcDAO = DAOFactory.createTuplecolumnsDAO(theDBID);
      Connection con = null;
      try {
         con = pool.getConnection(ui.getName());
         con.setAutoCommit(false);

         Extendedtuple etCR = new Extendedtuple();
         etCR.setName(tupleName);
         List<Extendedtuple> etList = etDAO.find(con, etCR);
         Assertion.assertTrue(etList.isEmpty() || etList.size() == 1);

         Extendedtuple et = etList.get(0);
         Tuplecolumns tcCR = new Tuplecolumns();
         tcCR.setExtendedtupleid(et.getUniqueid());
         tcDAO.delete(con, tcCR);
         etCR = new Extendedtuple();
         etCR.setUniqueid(et.getUniqueid());
         etDAO.delete(con, etCR);

         con.commit();
         String siteID = ServiceFactory.getSecurityService().findSiteIDByDbID(
               theDBID);
         DBChangeNotifySupport.getInstance().fireDBChangeNotifyEvent(siteID,
               "extendedtuple");
      } catch (Exception x) {
         handleErrorAndRollBack(con, "deleteExtendedTuple", x, true);
      } finally {
         releaseConnection(con, ui);
      }
   }

   public void addExtendedTuple(UserInfo ui, String tupleName,
         MeasurementGroupT mg, String tupleClass, String tupleSubClass)
         throws BaseException {
      ExtendedtupleDAO etDAO = DAOFactory.createExtendedtupleDAO(theDBID);
      ISequenceHelper sequenceHelper = MinimalServiceFactory
            .getSequenceHelper(theDBID);
      Connection con = null;

      try {
         con = pool.getConnection(ui.getName());
         con.setAutoCommit(false);
         List<Extendedtuple> existingETList = etDAO.find(con,
               new Extendedtuple());
         for (Extendedtuple et : existingETList) {
            if (et.getName().equals(tupleName)) {
               log.info("Extendedtuple already exists:" + et);
               con.commit();
               return;
            }
         }
         Databaseuser dbUser = getDatabaseUser(ui, Constants.USERCLASS_ADMIN);
         Assertion.assertNotNull(dbUser);

         Extendedtuple et = new Extendedtuple();
         Assertion.assertTrue(mg.getEntity().size() == 1);

         List<Tupleclass> tcList = getTupleclassList(con);
         boolean foundTC = false;
         for (Tupleclass tc : tcList) {
            if (tupleClass.equals(tc.getTupleclass())) {
               foundTC = true;
               break;
            }
         }
         if (!foundTC) {
            addTupleClass(ui, tupleClass);
            foundTC = true;
         }

         boolean foundTSC = false;
         if (foundTC) {
            List<Tuplesubclass> tscList = getTuplesubclassList(con);
            for (Tuplesubclass tsc : tscList) {
               if (tupleClass.equals(tsc.getTupleclass())
                     && tupleSubClass.equals(tsc.getTuplesubclass())) {
                  foundTSC = true;
                  break;
               }
            }
         }
         if (!foundTSC) {
            addTupleSubClass(ui, tupleClass, tupleSubClass, con);
         }

         et.setName(tupleName);
         et.setTupleclass(tupleClass);
         et.setTuplesubclass(tupleSubClass);

         et.setOwner(dbUser.getUniqueid());
         et.setModuser(dbUser.getUniqueid());
         et.setModtime(new Date());
         et.setTableid(getTableID(ui, Constants.EXTENDED_TUPLE));
         BigDecimal uniqueid = sequenceHelper.getNextUID(con,
               Constants.EXTENDED_TUPLE, "uniqueid");
         et.setUniqueid(uniqueid);

         etDAO.insert(con, et);

         // add entity attributes as extended tuple columns
         AbstractEntityT at = mg.getEntity().get(0);

         if (at instanceof AnatomicalEntityT) {
            AnatomicalEntityT aet = (AnatomicalEntityT) at;
            TerminologyStringT ts = aet.getLabel().get(0);
            ts.getValue();

            MeasurementInfo mi = new MeasurementInfo("unitless", "metric",
                  "no-unit");
            addTupleColumn(ui, con, "anatomicalEntity", Constants.VARCHAR_TYPE,
                  mi, et);
            addTupleColumn(ui, con, "nomenclature", Constants.VARCHAR_TYPE, mi,
                  et);
            addTupleColumn(ui, con, "termID", Constants.VARCHAR_TYPE, mi, et);

            addTupleColumn(ui, con, "laterality", Constants.VARCHAR_TYPE, mi,
                  et);
            addTupleColumn(ui, con, "tissueType", Constants.VARCHAR_TYPE, mi,
                  et);
         }

         for (Iterator<ObservationT> it = mg.getObservation().iterator(); it
               .hasNext();) {
            ObservationT ot = it.next();
            String columnName = ot.getName();
            String unit = ot.getUnits() != null ? ot.getUnits() : "unitless";
            MeasurementInfo mi = null;
            if (unit.equals("unitless")) {
               mi = new MeasurementInfo(unit, "metric", "no-unit");
            } else {
               mi = new MeasurementInfo(unit, "metric", "volume");
            }

            String columnType = Constants.FLOAT_TYPE;
            if (ot.getType() != null) {
               String otType = ot.getType().value();
               if (otType.equals(Constants.FLOAT_TYPE)) {
                  columnType = Constants.FLOAT_TYPE;
               } else if (otType.equals(Constants.INTEGER_TYPE)) {
                  columnType = Constants.INTEGER_TYPE;
               }
            } else {
               // try to guess
               if (ot.getValue() != null
                     && !GenUtils.isDecimalNumber(ot.getValue())) {
                  columnType = Constants.INTEGER_TYPE;
               }

            }

            addTupleColumn(ui, con, columnName, columnType, mi, et);
         }
         con.commit();
         String siteID = ServiceFactory.getSecurityService().findSiteIDByDbID(
               theDBID);
         DBChangeNotifySupport.getInstance().fireDBChangeNotifyEvent(siteID,
               new String[] { "extendedtuple", "tuplecolumn" });
      } catch (Exception x) {
         handleErrorAndRollBack(con, "addExtendedTuple", x, true);
      } finally {
         releaseConnection(con, ui);
      }
   }

   public Tupleclass addTupleClass(UserInfo ui, String tupleClass)
         throws Exception {
      Connection con = null;
      try {
         con = pool.getConnection(ui.getName());
         Tupleclass tc = addTupleClass(ui, tupleClass, con);
         return tc;
      } catch (Exception x) {
         handleErrorAndRollBack(con, "addTupleClass", x, true);
         return null;
      } finally {
         releaseConnection(con, ui);
      }
   }

   protected Tupleclass addTupleClass(UserInfo ui, String tupleClass,
         Connection con) throws BaseException, Exception {
      ISequenceHelper sequenceHelper = MinimalServiceFactory
            .getSequenceHelper(theDBID);
      Databaseuser dbUser = getDatabaseUser(ui, Constants.USERCLASS_ADMIN);
      Assertion.assertNotNull(dbUser);

      TupleclassDAO dao = DAOFactory.createTupleclassDAO(theDBID);
      Tupleclass tc = new Tupleclass();
      tc.setTupleclass(tupleClass);
      tc.setOwner(dbUser.getUniqueid());
      tc.setModuser(dbUser.getUniqueid());
      tc.setModtime(new Date());
      BigDecimal uniqueid = sequenceHelper.getNextUID(con,
            Constants.TUPLE_CLASS, "uniqueid");
      tc.setUniqueid(uniqueid);
      tc.setTableid(getTableID(ui, Constants.TUPLE_CLASS));

      dao.insert(con, tc);
      return tc;
   }

   public void addTupleSubClass(UserInfo ui, String tupleClass,
         String tupleSubclass) throws Exception {
      Connection con = null;
      try {
         con = pool.getConnection(ui.getName());
         addTupleSubClass(ui, tupleClass, tupleSubclass, con);
      } catch (Exception x) {
         handleErrorAndRollBack(con, "addTupleSubClass", x, true);
      } finally {
         releaseConnection(con, ui);
      }
   }

   protected void addTupleSubClass(UserInfo ui, String tupleClass,
         String tupleSubclass, Connection con) throws BaseException, Exception {
      ISequenceHelper sequenceHelper = MinimalServiceFactory
            .getSequenceHelper(theDBID);
      Databaseuser dbUser = getDatabaseUser(ui, Constants.USERCLASS_ADMIN);
      Assertion.assertNotNull(dbUser);

      TuplesubclassDAO dao = DAOFactory.createTuplesubclassDAO(theDBID);
      Tuplesubclass tc = new Tuplesubclass();
      tc.setTupleclass(tupleClass);
      tc.setTuplesubclass(tupleSubclass);
      tc.setOwner(dbUser.getUniqueid());
      tc.setModuser(dbUser.getUniqueid());
      tc.setModtime(new Date());
      BigDecimal uniqueid = sequenceHelper.getNextUID(con,
            Constants.TUPLE_SUBCLASS, "uniqueid");
      tc.setUniqueid(uniqueid);
      tc.setTableid(getTableID(ui, Constants.TUPLE_SUBCLASS));

      dao.insert(con, tc);
   }

   protected List<Tuplesubclass> getTuplesubclassList(Connection con)
         throws Exception {
      TuplesubclassDAO dao = DAOFactory.createTuplesubclassDAO(theDBID);
      return dao.find(con, new Tuplesubclass());
   }

   protected List<Tupleclass> getTupleclassList(Connection con)
         throws Exception {
      TupleclassDAO dao = DAOFactory.createTupleclassDAO(theDBID);
      return dao.find(con, new Tupleclass());
   }

   public void addMeasurementInfo(UserInfo ui, MeasurementInfo mi)
         throws BaseException {
      Connection con = null;
      try {
         con = pool.getConnection(ui.getName());
         con.setAutoCommit(false);
         IDBCache dbCache = ServiceFactory.getDBCache(theDBID);
         Map<String, Measurementunit> muMap = dbCache.getMeasurementUnitMap(ui,
               true);
         Measurementunit mu = muMap.get(mi.getUnit());
         if (mu != null
               && mu.getMeasurementsystem().equals(mi.getMeasurementSystem())) {
            // already exists
            return;
         }
         ISequenceHelper sequenceHelper = MinimalServiceFactory
               .getSequenceHelper(theDBID);
         Databaseuser dbUser = getDatabaseUser(ui, Constants.USERCLASS_ADMIN);
         Assertion.assertNotNull(dbUser);

         Map<String, Measurementsystem> msMap = dbCache
               .getMeasurementSystemMap(ui, true);
         if (!msMap.containsKey(mi.getMeasurementSystem())) {
            // add new measurement system
            MeasurementsystemDAO dao = DAOFactory
                  .createMeasurementsystemDAO(theDBID);

            BigDecimal uniqueId = sequenceHelper.getNextUID(con,
                  Constants.MEASUREMENT_SYSTEM, "uniqueid");
            Measurementsystem ms = new Measurementsystem();
            ms.setName(mi.getMeasurementSystem());
            ms.setDescription(mi.getMeasurementSystem());
            ms.setUniqueid(uniqueId);
            ms.setOwner(dbUser.getUniqueid());
            ms.setModuser(dbUser.getUniqueid());
            ms.setModtime(new Date());
            ms.setTableid(getTableID(ui, Constants.MEASUREMENT_SYSTEM));
            dao.insert(con, ms);
         }
         Map<String, Measurementtype> mtMap = dbCache.getMeasurementTypeMap(ui,
               true);
         if (!mtMap.containsKey(mi.getMeasurementType())) {
            MeasurementtypeDAO dao = DAOFactory
                  .createMeasurementtypeDAO(theDBID);
            Measurementtype mt = new Measurementtype();
            mt.setName(mi.getMeasurementType());
            mt.setDescription(mi.getMeasurementType());
            BigDecimal uniqueId = sequenceHelper.getNextUID(con,
                  Constants.MEASUREMENT_TYPE, "uniqueid");
            mt.setUniqueid(uniqueId);
            mt.setOwner(dbUser.getUniqueid());
            mt.setModuser(dbUser.getUniqueid());
            mt.setModtime(new Date());
            mt.setTableid(getTableID(ui, Constants.MEASUREMENT_TYPE));
            dao.insert(con, mt);
         }

         MeasurementunitDAO muDAO = DAOFactory
               .createMeasurementunitDAO(theDBID);
         mu = new Measurementunit();
         mu.setMeasurementsystem(mi.getMeasurementSystem());
         mu.setUnit(mi.getUnit());
         mu.setDescription(mi.getUnit());
         mu.setMeasurementtype(mi.getMeasurementType());
         BigDecimal uniqueId = sequenceHelper.getNextUID(con,
               Constants.MEASUREMENT_UNIT, "uniqueid");
         mu.setUniqueid(uniqueId);
         mu.setOwner(dbUser.getUniqueid());
         mu.setModuser(dbUser.getUniqueid());
         mu.setModtime(new Date());
         mu.setTableid(getTableID(ui, Constants.MEASUREMENT_UNIT));
         mu.setConversionsystem(mi.getConversionSystem());
         mu.setConversionunit(mi.getUnit());
         muDAO.insert(con, mu);

         con.commit();

         String siteID = ServiceFactory.getSecurityService().findSiteIDByDbID(
               theDBID);
         DBChangeNotifySupport.getInstance().fireDBChangeNotifyEvent(
               siteID,
               new String[] { "measurementsystem", "measurementtype",
                     "measurementunit" });
      } catch (Exception x) {
         handleErrorAndRollBack(con, "addMeasurementInfo", x, true);
      } finally {
         releaseConnection(con, ui);
      }
   }

   public List<Analysisresult> deleteAnalysesForSubject(Connection con,
         UserInfo ui, String subjectID, Integer experimentID, Integer visitID,
         Integer segmentID) throws Exception {
      StoredtupleDAO stDAO = DAOFactory.createStoredtupleDAO(theDBID);
      AnalysisresultDAO arDAO = DAOFactory.createAnalysisResultDAO(theDBID);

      TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
      StringBuffer qBuf = new StringBuffer(128);
      qBuf.append("select a.uniqueid, a.subjectid, a.segmentid,");
      qBuf.append("a.componentid, a.experimentid, a.executedtransformid");
      qBuf.append(", st.uniqueid from Analysisresult as a,");
      qBuf.append("Storedtuple as st where ");
      qBuf.append("a.uniqueid = st.basetupleid and ");
      qBuf.append("a.subjectid = '").append(subjectID);
      qBuf.append("'  and a.experimentid = ").append(experimentID);
      if (visitID != null)
         qBuf.append(" and a.componentid = ").append(visitID);
      if (segmentID != null)
         qBuf.append(" and a.segmentid = ").append(segmentID);
      qBuf.append(" order by a.uniqueid, st.extendedtupleid, ");
      qBuf.append("st.uniqueid");

      System.out
            .println("query:" + GenUtils.applyWordWrapping(qBuf.toString()));
      List<?> results = tsp.executeQuery(con, qBuf.toString());
      Map<BigDecimal, Analysisresult> arMap = new HashMap<BigDecimal, Analysisresult>(
            17);
      Storedtuple stCR = new Storedtuple();
      for (Iterator<?> it = results.iterator(); it.hasNext();) {
         Object[] row = (Object[]) it.next();
         Analysisresult ar = (Analysisresult) row[0];
         arMap.put(ar.getUniqueid(), ar);
         Storedtuple st = (Storedtuple) row[1];

         deleteTupleFloatRec(con, st.getUniqueid());
         deleteTupleVarcharRec(con, st.getUniqueid());
         deleteTupleIntegerRec(con, st.getUniqueid());
         deleteTupleBooleanRec(con, st.getUniqueid());
         deleteTupleTimestampRec(con, st.getUniqueid());

         stCR.setUniqueid(st.getUniqueid());
         stDAO.delete(con, stCR);
      }
      Analysisresult arCR = new Analysisresult();
      for (Analysisresult ar : arMap.values()) {
         arCR.setUniqueid(ar.getUniqueid());
         arDAO.delete(con, arCR);
      }
      return new ArrayList<Analysisresult>(arMap.values());
   }

   /**
    *
    * @param ui
    * @param subjectID
    * @param experimentID
    * @param visitID
    * @param segmentID
    * @return List<Analysisresult>
    * @throws Exception
    */
   public List<Analysisresult> deleteAnalysesForSubject(UserInfo ui,
         String subjectID, Integer experimentID, Integer visitID,
         Integer segmentID) throws Exception {
      Connection con = null;
      try {
         con = pool.getConnection(ui.getName());
         con.setAutoCommit(false);

         List<Analysisresult> arList = deleteAnalysesForSubject(con, ui,
               subjectID, experimentID, visitID, segmentID);

         con.commit();
         return arList;
      } catch (Exception x) {
         handleErrorAndRollBack(con, "deleteAnalysesForSubject", x, true);
         return null;
      } finally {
         releaseConnection(con, ui);
      }
   }

   protected void deleteTupleFloatRec(Connection con, BigDecimal storedTupleID)
         throws Exception {
      TuplefloatDAO dao = DAOFactory.createTuplefloatDAO(theDBID);
      Tuplefloat cr = new Tuplefloat();
      cr.setStoredtupleid(storedTupleID);
      dao.delete(con, cr);
   }

   protected void deleteTupleIntegerRec(Connection con, BigDecimal storedTupleID)
         throws Exception {
      TupleintegerDAO dao = DAOFactory.createTupleintegerDAO(theDBID);
      Tupleinteger cr = new Tupleinteger();
      cr.setStoredtupleid(storedTupleID);
      dao.delete(con, cr);
   }

   protected void deleteTupleVarcharRec(Connection con, BigDecimal storedTupleID)
         throws Exception {
      TuplevarcharDAO dao = DAOFactory.createTuplevarcharDAO(theDBID);
      Tuplevarchar cr = new Tuplevarchar();
      cr.setStoredtupleid(storedTupleID);
      dao.delete(con, cr);
   }

   protected void deleteTupleTimestampRec(Connection con,
         BigDecimal storedTupleID) throws Exception {
      TupletimestampDAO dao = DAOFactory.createTupletimestampDAO(theDBID);
      Tupletimestamp cr = new Tupletimestamp();
      cr.setStoredtupleid(storedTupleID);
      dao.delete(con, cr);
   }

   protected void deleteTupleBooleanRec(Connection con, BigDecimal storedTupleID)
         throws Exception {
      TuplebooleanDAO dao = DAOFactory.createTuplebooleanDAO(theDBID);
      Tupleboolean cr = new Tupleboolean();
      cr.setStoredtupleid(storedTupleID);
      dao.delete(con, cr);
   }

   public List<AnalysisT> getAnalysesForSubject(UserInfo ui, String subjectID,
         Integer experimentID) throws Exception {
      Connection con = null;
      try {
         ExtendedTupleHelper eth = new ExtendedTupleHelper(theDBID);

         con = pool.getConnection(ui.getName());

         TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
         StringBuffer qBuf = new StringBuffer(128);
         qBuf.append("select a.uniqueid, a.subjectid, a.segmentid,");
         qBuf.append("a.componentid, a.experimentid, a.executedtransformid");
         qBuf.append(", st.uniqueid, td.* from Analysisresult as a,");
         qBuf.append("Storedtuple as st, Tupledata as td where ");
         qBuf.append("a.uniqueid = st.basetupleid and ");
         qBuf.append("st.uniqueid = td.storedtupleid and a.subjectid = '");
         qBuf.append(subjectID).append("'  and a.experimentid = ").append(
               experimentID);
         qBuf.append(" order by a.uniqueid, st.extendedtupleid, ");
         qBuf.append("st.uniqueid, td.columnname");

         System.out.println("query:"
               + GenUtils.applyWordWrapping(qBuf.toString()));
         List<?> results = tsp.executeQuery(con, qBuf.toString());

         Map<BigDecimal, Map<String, Tuplecolumns>> extID2ColMapMap = new HashMap<BigDecimal, Map<String, Tuplecolumns>>(
               7);
         Map<BigDecimal, AnalysisT> analysisMap = new LinkedHashMap<BigDecimal, AnalysisT>(
               17);
         MeasurementGroupT mg = null;
         BigDecimal storedTupleID = null;
         AnatomicalEntityT at = null;
         for (Iterator<?> it = results.iterator(); it.hasNext();) {
            Object[] row = (Object[]) it.next();
            Analysisresult ar = (Analysisresult) row[0];
            Storedtuple st = (Storedtuple) row[1];
            Tupledata td = (Tupledata) row[2];
            Map<String, Tuplecolumns> tcMap = extID2ColMapMap.get(st
                  .getExtendedtupleid());
            if (tcMap == null) {
               tcMap = eth.getColumMapForTuple(con, st.getExtendedtupleid());
               extID2ColMapMap.put(st.getExtendedtupleid(), tcMap);
            }
            AnalysisT a = analysisMap.get(ar.getUniqueid());
            if (a == null) {
               a = new AnalysisT();
               analysisMap.put(ar.getUniqueid(), a);
               a.setSubjectID(subjectID);
               // TODO experiment information ??
               a.setProjectID(experimentID.toString());

               mg = new MeasurementGroupT();
               a.getMeasurementGroup().add(mg);
               storedTupleID = st.getUniqueid();
               at = new AnatomicalEntityT();
               mg.getEntity().add(at);
            }
            if (!storedTupleID.equals(st.getUniqueid())) {
               mg = new MeasurementGroupT();
               
               a.getMeasurementGroup().add(mg);
               storedTupleID = st.getUniqueid();
               at = new AnatomicalEntityT();
               mg.getEntity().add(at);
            }

            Tuplecolumns tc = tcMap.get(td.getColumnname());
            if (tc == null) {
               continue;
            }
            if (td.getColumnname().equals("anatomicalEntity")) {
               eth.addAnatomicalEntity(td.getTextvalue(), at);
               continue;
            } else if (td.getColumnname().equals("nomenclature")) {
               eth.addNomenclature(td.getTextvalue(), at);
               continue;
            } else if (td.getColumnname().equals("termID")) {
               eth.addTermID(td.getTextvalue(), at);
               continue;
            } else if (td.getColumnname().equals("laterality")) {
               eth.addLaterality(td.getTextvalue(), at);
               continue;
            } else if (td.getColumnname().equals("tissueType")) {
               eth.addTissueType(td.getTextvalue(), at);
               continue;
            }

            ObservationT ob = new ObservationT();
            ob.setName(td.getColumnname());
            ob.setType(ExtendedTupleHelper.getColumnType(td.getColumntype()));
            if (!tc.getMeasurementunit().equals("unitless")) {
               ob.setUnits(tc.getMeasurementunit());
            }
            ob.setValue(td.getTextvalue());

            mg.getObservation().add(ob);

            // System.out.println(td);
         }

         return new ArrayList<AnalysisT>(analysisMap.values());

      } catch (Exception x) {
         handleErrorAndRollBack(con, "getAnalysesForSubject", x, true);
         return null;
      } finally {
         releaseConnection(con, ui);
      }
   }

   protected void addTupleColumn(UserInfo ui, Connection con,
         String columnName, String columnType, MeasurementInfo mi,
         Extendedtuple et) throws Exception {
      ISequenceHelper sequenceHelper = MinimalServiceFactory
            .getSequenceHelper(theDBID);
      TuplecolumnsDAO tcDAO = DAOFactory.createTuplecolumnsDAO(theDBID);
      Databaseuser databaseUser = getDatabaseUser(ui, Constants.USERCLASS_ADMIN);
      IDBCache dbCache = ServiceFactory.getDBCache(theDBID);
      Map<String, Ontologysource> ontSrcMap = dbCache.getOntologySourceMap(ui,
            false);
      Map<String, Ontologyconcept> ontConceptMap = dbCache
            .getOntologyConceptMap(ui, false);
      Ontologysource os = ontSrcMap.get(Constants.DEFAULT_ONT_SRC);
      Ontologyconcept oc = ontConceptMap.get(Constants.DEFAULT_ONT_CONCEPT);
      Assertion.assertNotNull(os);
      Assertion.assertNotNull(oc);

      Tuplecolumns tc = new Tuplecolumns();
      tc.setExtendedtupleid(et.getUniqueid());
      tc.setColumnname(columnName);
      tc.setColumntype(columnType);
      tc.setNullable(false);
      tc.setOwner(databaseUser.getUniqueid());
      tc.setModuser(databaseUser.getUniqueid());
      tc.setModtime(new Date());
      tc.setTableid(getTableID(ui, Constants.TUPLE_COLUMNS));
      BigDecimal uniqueId = sequenceHelper.getNextUID(con,
            Constants.TUPLE_COLUMNS, "uniqueid");
      tc.setUniqueid(uniqueId);
      tc.setMeasurementsystem(mi.getMeasurementSystem());
      tc.setMeasurementunit(mi.getUnit());
      tc.setColumnontology(os.getOntologysource());
      tc.setColumnconcept(oc.getConceptid());

      tcDAO.insert(con, tc);
   }

   protected BigDecimal findMatchingExtendedTuple(Connection con,
         AnalysisT analysis) throws Exception {
      ExtendedtupleDAO dao = DAOFactory.createExtendedtupleDAO(theDBID);
      Extendedtuple cr = new Extendedtuple();
      List<Extendedtuple> etList = dao.find(con, cr);
      for (Extendedtuple et : etList) {
         if (et.getName().equals(analysis.getType())) {
            return et.getUniqueid();
         }
      }
      return null;
   }

   public static class MeasurementInfo {
      String measurementSystem;
      String measurementType;
      String unit;
      String conversionSystem;
      String conversionUnit;

      public MeasurementInfo(String unit, String measurementSystem,
            String measurementType) {
         super();
         this.unit = unit;
         this.measurementSystem = measurementSystem;
         this.measurementType = measurementType;
         this.conversionSystem = measurementSystem;
         this.conversionUnit = unit;
      }

      public String getMeasurementSystem() {
         return measurementSystem;
      }

      public String getMeasurementType() {
         return measurementType;
      }

      public String getUnit() {
         return unit;
      }

      public String getConversionSystem() {
         return conversionSystem;
      }

      public String getConversionUnit() {
         return conversionUnit;
      }
   }// ;

}
