package clinical.xml.importer.client;

import java.io.File;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.QName;
import javax.xml.ws.BindingProvider;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.methods.multipart.FilePart;
import org.apache.commons.httpclient.methods.multipart.MultipartRequestEntity;
import org.apache.commons.httpclient.methods.multipart.Part;
import org.apache.commons.httpclient.methods.multipart.StringPart;

import clinical.utils.XCEDEUtils;
import clinical.web.dd.XCEDE2AnalysisMerger;
import clinical.ws.BaseCLICLient;
import clinical.xml.xcede2.AnalysisT;
import clinical.xml.xcede2.LevelDataRefsT;
import clinical.xml.xcede2.XCEDE;

/**
 * @author I. Burak Ozyurt
 * @version $Id: FSAImportClient.java 91 2009-08-17 23:38:26Z bozyurt $
 */

public class FSAImportClient extends BaseCLICLient {
	private static final QName SERVICE_NAME = new QName(
			"http://importer.xml.clinical/", "XCEDE2DDService");

	protected DdService getDDService() {
		URL wsdlURL = getClass().getClassLoader().getResource(
				"DerivedDataService.wsdl");
		System.out.println("WSDL URL:" + wsdlURL);
		XCEDE2DDService ss = new XCEDE2DDService(wsdlURL, SERVICE_NAME);
		DdService servicePort = ss.getXCEDE2DDServiceImplPort();

		BindingProvider bp = (BindingProvider) servicePort;
		String endPointAddress = getHostPort() + "/" + contextName
				+ "/services/DerivedDataService";
		bp.getRequestContext().put(BindingProvider.ENDPOINT_ADDRESS_PROPERTY,
				endPointAddress);
		System.out.println("EndPoint Address:"
				+ bp.getRequestContext().get(
						BindingProvider.ENDPOINT_ADDRESS_PROPERTY));
		if (useHTTPS) {
			prepForHTTPS(servicePort);
		}
		return servicePort;
	}

	public void importFreesurferAnalyses(String webUser, String pwd,
			String mergedXcedeFile, String shortAnalysisName, String snapshotID,
			String tarFilePath, List<String> files, String fileType,
			boolean uploadTarFile) throws Exception {
		List<AnalysisT> atList = loadMergedFSXCEDE(mergedXcedeFile);
		AnalysisT provenanceAT = findProvenanceAnalysis(atList);
		LevelDataRefsT ldr = provenanceAT.getInput().get(0);
		final String subjectID = ldr.getSubjectID();
		final String projectID = ldr.getProjectID();
		final String visitIDStr = ldr.getVisitID();
		String episodeID = ldr.getEpisodeID();
		Map<String, String> paramMap = new HashMap<String, String>(17);
		paramMap.put("webUser", webUser);
		paramMap.put("pwd", pwd);
		paramMap.put("projectID", projectID);
		paramMap.put("subjectID", subjectID);
		paramMap.put("visitID", visitIDStr);
		paramMap.put("studyID", ldr.getStudyID());
		paramMap.put("episodeID", episodeID);
		paramMap.put("analysisShortName", shortAnalysisName);
		paramMap.put("snapshotVersion", snapshotID);

		if (uploadTarFile) {
			String targetURL = getHostPort() + "/" + contextName
					+ "/uploadfile.do";
			System.out.println("targetURL:" + targetURL);
			uploadFile(targetURL, tarFilePath, paramMap);
			if (fileType != null && files != null && !files.isEmpty()) {
				paramMap.put("dd", "true");
				for (String file : files) {
					uploadFile(targetURL, file, paramMap);
				}
			}
		}

		importFSAnalysisDD(webUser, pwd, mergedXcedeFile, shortAnalysisName,
				snapshotID);
	}

	protected void importFSAnalysisDD(String webUser, String pwd,
			String mergedXcedeFile, String shortAnalysisName, String snapshotID)
			throws Exception {
		DdService servicePort = getDDService();

		String mergedXcedeContent = loadXMLContent(mergedXcedeFile);
		servicePort.importFSAnalysesForSession(webUser, pwd, mergedXcedeContent,
				shortAnalysisName, snapshotID);
	}

	public List<AnalysisT> loadMergedFSXCEDE(String mergedXcedeFile)
			throws Exception {
		XCEDE xcede = XCEDEUtils.unmarshal(mergedXcedeFile);
		List<Object> aList = xcede.getAnnotationListOrRevisionListOrProject();
		List<AnalysisT> analysisTypes = new ArrayList<AnalysisT>(5);
		if (!aList.isEmpty() && aList.get(0) instanceof AnalysisT) {
			for (Iterator<?> it = aList.iterator(); it.hasNext();) {
				AnalysisT a = (AnalysisT) it.next();
				analysisTypes.add(a);
			}
		}
		return analysisTypes;
	}

	protected AnalysisT findProvenanceAnalysis(List<AnalysisT> analysisTypeList) {
		for (AnalysisT a : analysisTypeList) {
			if (!a.getInput().isEmpty() && !a.getProvenance().isEmpty()) {
				return a;
			}
		}
		return null;
	}

	public void uploadFile(String targetURL, String path,
			Map<String, String> paramMap) throws Exception {
		File targetFile = new File(path);
		if (!targetFile.exists() || !targetFile.isFile()) {
			throw new Exception("Not a valid path: " + path);
		}

		PostMethod post = new PostMethod(targetURL);
		try {
			Part[] parts = new Part[paramMap.size() + 1];
			int i = 0;
			for (Map.Entry<String, String> entry : paramMap.entrySet()) {
				StringPart stringPart = new StringPart(entry.getKey(), entry
						.getValue());
				parts[i++] = stringPart;
			}
			parts[i] = new FilePart(targetFile.getName(), targetFile);
			post.setRequestEntity(new MultipartRequestEntity(parts, post
					.getParams()));

			HttpClient client = new HttpClient();
			client.getHttpConnectionManager().getParams().setConnectionTimeout(
					5000);

			int status = client.executeMethod(post);
			if (status == HttpStatus.SC_OK) {
				System.out.println("Succesfull upload of " + targetFile);
			} else {
				throw new Exception("problem in upload of " + targetFile
						+ "\nHTTP Status: " + status);
			}
		} finally {
			post.releaseConnection();
		}
	}

	public String getHost() {
		return host;
	}

	public static void usage() {
		System.err.println("\n\n");
		System.err
				.println("Usage: FSAImportClient -u <webUser> -p <webUserPwd> -x <xcedeDir> -a <shortAnalysisName> \\\n"
						+ "-s <snapshotID> [-host <HID-web-app-host>] [-port <port-number>] \\ \n"
						+ "[-context <application-context>] [-useSSL] [-mrml] [-skip-tarball-upload]");
		System.err.println("For example:\n");
		System.err.println("fsa_import.sh -u b -p b -a FreeSurfer -s 0001 \\ ");
		System.err
				.println("-x /home/bozyurt/work/fsanalysis/fBIRNPhaseII__0010/Data/001029291693/scanVisit__0010__0002/MRI__0001/t1/Analysis/FreeSurfer__0010__0001 \\");
		System.err.println("-host localhost -port 8080");
		System.err.println();
		System.err.println("-x xcedeDir");
		System.err
				.println("\tXCEDE directory contains XCEDE XML files for the Freesurfer (or other) summary ");
		System.err
				.println("\tderived data including provenance information (provenance.xml). Also the actual ");
		System.err
				.println("\tderived data is included as a tarball (.tgz or tar.gz) in this directory.");
		System.err.println("-host <HID-web-app-host>");
		System.err.println("\t(Default: localhost) e.g. loci.ucsd.edu");
		System.err.println("-port <port-number>");
		System.err.println("\t(Default: 8080)");
		System.err.println("-context <application-context>");
		System.err.println("\t(Default: clinical)");
		System.err
				.println("-mrml also upload Slicer MRML files included in the data dir to be uploaded");

		System.err
				.println("-skip-tarball-upload do not upload the derived data tar ball to its location in the distributed storage");
		System.exit(1);
	}

	public static File findTarBall(String dir) throws Exception {
		File[] files = new File(dir).listFiles();
		File tarFile = null;
		for (File f : files) {
			if (f.getName().endsWith(".tgz") || f.getName().endsWith(".tar.gz")) {
				if (tarFile != null) {
					throw new Exception(
							"More than one tar file is detected in directory: " + dir);
				} else {
					tarFile = f;
				}
			}
		}
		if (tarFile == null) {
			throw new Exception("A tar file is expected in directory: " + dir);
		}
		return tarFile;
	}

	public static List<String> findFilesOfType(String dir, String fileExtension)
			throws Exception {
		File[] files = new File(dir).listFiles();
		List<String> fileList = new ArrayList<String>(10);
		for (File f : files) {
			if (f.isFile() && f.getName().endsWith(fileExtension))
				fileList.add(f.getAbsolutePath());
		}
		return fileList;
	}

	public static String getArgValue(String[] args, int idx) {
		if (args.length <= (idx + 1)) {
			usage();
		}
		if (args[idx + 1].startsWith("-")) {
			usage();
		}
		return args[idx + 1];
	}

	public static void main(String[] args) {
		// String rootDir =
		// "/home/bozyurt/work/fsanalysis/fBIRNPhaseII__0010/Data/001029291693/scanVisit__0010__0002/MRI__0001/t1/Analysis/FreeSurfer__0010__0001";
		// String mergedXcedeFile = rootDir + "/merged.xml";
		// String tarFilePath = rootDir + "/001029291693__0002.tgz";

		if (args.length < 10) {
			usage();
		}
		try {
			boolean uploadTarBall = true;
			String webUser = null;
			String pwd = null;
			String xcedeDir = null;
			String shortAnalysisName = "FreeSurfer";
			String snapshotID = "0001";
			String host = null;
			boolean useSSL = false;
			boolean uploadMRML = false;
			String contextName = null;
			int port = -1;
			int i = 0;
			while (i < args.length) {
				if (args[i].equals("-u")) {
					webUser = getArgValue(args, i);
				} else if (args[i].equals("-p")) {
					pwd = getArgValue(args, i);
				} else if (args[i].equals("-x")) {
					xcedeDir = getArgValue(args, i);
				} else if (args[i].equals("-a")) {
					shortAnalysisName = getArgValue(args, i);
				} else if (args[i].equals("-s")) {
					snapshotID = getArgValue(args, i);
				} else if (args[i].equals("-host")) {
					host = getArgValue(args, i);
				} else if (args[i].equals("-port")) {
					String portStr = getArgValue(args, i);
					try {
						port = Integer.parseInt(portStr);
					} catch (NumberFormatException nfe) {
						port = -1;
					}
				} else if (args[i].equals("-useSSL")) {
					useSSL = true;
					i++;
					continue;
				} else if (args[i].equals("-context")) {
					contextName = getArgValue(args, i);
				} else if (args[i].equals("-mrml")) {
					uploadMRML = true;
					i++;
					continue;
				} else if (args[i].equals("-skip-tarball-upload")) {
					uploadTarBall = false;
					i++;
					continue;
				}
				i += 2;
			}
			FSAImportClient client = new FSAImportClient();
			if (host != null)
				client.setHost(host);
			if (port != -1)
				client.setPort(port);
			if (useSSL)
				client.setUseHTTPS(useSSL);
			if (contextName != null)
				client.setContextName(contextName);
			if (xcedeDir == null || webUser == null || pwd == null) {
				usage();
			}

			String tarFilePath = findTarBall(xcedeDir).getAbsolutePath();

			XCEDE2AnalysisMerger merger = new XCEDE2AnalysisMerger();
			List<File> xcedeFiles = merger.findXCEDEFiles(new File(xcedeDir));
			File mergedFile = new File(xcedeDir, "merged.xml");
			XCEDE xcede = merger.merge(xcedeFiles);

			merger.saveMerged(xcede, mergedFile);

			if (uploadMRML) {
				List<String> mrmlFiles = findFilesOfType(xcedeDir, ".mrml");
				client.importFreesurferAnalyses(webUser, pwd, mergedFile
						.getAbsolutePath(), shortAnalysisName, snapshotID,
						tarFilePath, mrmlFiles, ".mrml", uploadTarBall);
			} else {
				client.importFreesurferAnalyses(webUser, pwd, mergedFile
						.getAbsolutePath(), shortAnalysisName, snapshotID,
						tarFilePath, null, null, uploadTarBall);
			}
			System.exit(0);
		} catch (Throwable t) {
			t.printStackTrace();
			System.exit(1);
		}
	}

}
