package clinical.web.actions;

import java.beans.BeanInfo;
import java.beans.PropertyDescriptor;
import java.math.BigDecimal;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpSession;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.server.vo.Assessment;
import clinical.server.vo.Experiment;
import clinical.server.vo.Humansubject;
import clinical.server.vo.Protocol;
import clinical.server.vo.Researchgroup;
import clinical.server.vo.Visittype;
import clinical.utils.DateTimeUtils;
import clinical.web.Constants;
import clinical.web.IAssessmentService;
import clinical.web.ISubjectAssessmentManagement;
import clinical.web.ISubjectVisitManagement;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.UserInfo;
import clinical.web.common.query.QueryUtils;
import clinical.web.common.query.SearchCriteria;
import clinical.web.common.query.SearchPredicate;
import clinical.web.common.query.SearchPredicateList;
import clinical.web.exception.BaseException;
import clinical.web.exception.SubjectVisitManagementException;
import clinical.web.forms.AssessmentSelector;
import clinical.web.forms.SubjectVisitManagementForm;
import clinical.web.helpers.MissingEntry;
import clinical.web.helpers.PropertyQueryPartInfo;
import clinical.web.helpers.ScoreValuesDisplayHelper;
import clinical.web.helpers.ScoreValuesDisplayLayout;
import clinical.web.helpers.SegmentFormInfo;
import clinical.web.helpers.SegmentTimestampSelector;
import clinical.web.helpers.StudyFormInfo;
import clinical.web.helpers.StudySelector;
import clinical.web.helpers.SubjectExperimentsSelector;
import clinical.web.helpers.VisitFormInfo;
import clinical.web.vo.AssessmentInfo;
import clinical.web.vo.AssessmentScoreValues;
import clinical.web.vo.ScoreValue;
import clinical.web.vo.Study;
import clinical.web.vo.StudySegment;
import clinical.web.vo.Subject;
import clinical.web.vo.Visit;
import clinical.web.vo.VisitSegment;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: SubjectManagementHelper.java,v 1.24 2006/06/01 01:19:51 bozyurt
 *          Exp $
 */

public class SubjectManagementHelper {
	public static final int RETRIEVE_NON_VALIDATED = 100;
	public static final int RETRIEVE_VALIDATED = 101;
	public static final int RETRIEVE_BOTH = 102;

	protected static Log log = LogFactory.getLog(SubjectManagementHelper.class);

	protected SubjectManagementHelper() {
	}

	public static String[] getVisitTypes(String dbID, UserInfo ui)
			throws BaseException {
		IDBCache dbCache = ServiceFactory.getDBCache(dbID);
		List<Visittype> visitTypes = dbCache.getVisitTypes(ui, false);
		String[] visitTypesArr = new String[visitTypes.size()];
		int idx = 0;
		for (Visittype vt : visitTypes) {
			visitTypesArr[idx++] = vt.getVisittype();
		}
		return visitTypesArr;
	}

	public static Experiment findExperiment(List<Experiment> experiments,
			int expID) {
		for (Experiment exp : experiments) {
			if (exp.getUniqueid().intValue() == expID) {
				return exp;
			}
		}
		return null;
	}

	public static Experiment findExperimentByName(List<Experiment> experiments,
			String experimentName) {
		for (Experiment exp : experiments) {
			if (exp.getName().equals(experimentName)) {
				return exp;
			}
		}
		return null;
	}

	public static List<Protocol> getProtocols(String dbID, UserInfo ui)
			throws BaseException {
		IDBCache dbCache = ServiceFactory.getDBCache(dbID);
		List<Protocol> protocols = dbCache.getProtocols(ui, false);
		return protocols;
	}

	public static Object[] getProtocolNameVersions(List<Protocol> protocols) {
		String[] protocolNames = new String[protocols.size()];
		int[] protocolVersions = new int[protocols.size()];
		int idx = 0;
		for (Iterator<Protocol> iter = protocols.iterator(); iter.hasNext();) {
			Protocol pr = iter.next();
			protocolNames[idx] = pr.getName();
			protocolVersions[idx] = pr.getProtocolversion().intValue();
			++idx;
		}
		Object[] oarr = new Object[2];
		oarr[0] = protocolNames;
		oarr[1] = protocolVersions;
		return oarr;
	}

	public static String getProtocolValue(List<Protocol> protocols,
			String segmentProtocolID, int segmentProtocolVersion) {
		for (Iterator<Protocol> iter = protocols.iterator(); iter.hasNext();) {
			Protocol p = iter.next();
			if (p.getProtocolid().equals(segmentProtocolID)
					&& p.getProtocolversion().intValue() == segmentProtocolVersion) {
				return p.getName() + " " + segmentProtocolVersion;
			}
		}
		return null;
	}

	public static void prepareStaticInfo(String dbID,
			SubjectVisitManagementForm svmForm, UserInfo ui)
			throws BaseException {
		IDBCache dbCache = null;
		dbCache = ServiceFactory.getDBCache(dbID);

		if (svmForm.getVisitTypes() == null
				|| svmForm.getVisitTypes().isEmpty()) {
			List<Visittype> visitTypes = dbCache.getVisitTypes(ui, false);
			for (Visittype vt : visitTypes) {
				svmForm.addVisitType(vt.getVisittype());
			}
		}
		if (svmForm.getExperiments() == null
				|| svmForm.getExperiments().isEmpty()) {
			List<Experiment> experiments = dbCache.getExperiments(ui, false);
			for (Experiment exp : experiments) {
				svmForm.addExperiment(exp);
			}
		}

		if (svmForm.getProtocols() == null || svmForm.getProtocols().isEmpty()) {
			List<Protocol> protocols = dbCache.getProtocols(ui, false);
			for (Protocol pr : protocols) {
				svmForm.addProtocol(pr);
			}
		}
	}

	public static SearchCriteria prepareSearchCriteria(
			SubjectVisitManagementForm svmForm, BeanInfo bi, Map<String,String> voColMap) throws Exception {
		List<PropertyQueryPartInfo> queryParts = svmForm.getQueryParts();
		SearchPredicateList spList = new SearchPredicateList();
		int idx = 0;
		SearchPredicate prev = null;
		for (PropertyQueryPartInfo pqp : queryParts) {

			SearchPredicate.Range range = null;
			SearchPredicate sp = null;

			int relOp = pqp.getOperator();
			int spType = 0;

			PropertyDescriptor pd = getPropertyDescriptor(bi, pqp
					.getPropertyName());

			spType = QueryUtils.getSearchPredicateTypeForProperty(pd
					.getPropertyType());

			Object value = QueryUtils.convertToType(pqp.getRhs(), spType);

			log.info(pqp.getPropertyName() + "  rhs= " + pqp.getRhs());

			Object attribute = voColMap.get(pqp.getPropertyName());
			if ( attribute == null) {
				attribute = pqp.getPropertyName();
			}

			if (pqp.getLowBound() != null && pqp.getLowBound().length() > 0
					&& pqp.getUppBound() != null
					&& pqp.getUppBound().length() > 0) {
				Number lowBound = (Number) QueryUtils.convertToType(pqp
						.getLowBound(), spType);
				Number uppBound = (Number) QueryUtils.convertToType(pqp
						.getUppBound(), spType);

				range = new SearchPredicate.Range(lowBound, uppBound);
			}

			if (range == null) {
				sp = new SearchPredicate(attribute, value, relOp, spType);
			} else {
				sp = new SearchPredicate(attribute, range, relOp, spType);
			}

			if (prev == null) {
				spList.addSearchPredicate(sp, SearchPredicateList.NONE);
			} else {
				int logOp = getSelectedCombinator(idx, svmForm);
				spList.addSearchPredicate(sp, prev, logOp);
			}

			prev = sp;
			++idx;
		}
		return new SearchCriteria(spList);
	}

	public static PropertyDescriptor getPropertyDescriptor(BeanInfo bi,
			String propertyName) {
		PropertyDescriptor[] pds = bi.getPropertyDescriptors();
		for (int i = 0; i < pds.length; i++) {
			if (pds[i].getName().equals(propertyName)) {
				return pds[i];
			}
		}
		return null;
	}

	protected static int getSelectedCombinator(int idx,
			SubjectVisitManagementForm form) {
		String combStr = form.getCombinator(idx - 1);
		return (combStr.equals("AND")) ? SearchPredicateList.AND
				: SearchPredicateList.OR;
	}

	public static String[] getExperimentNames(String dbID, UserInfo ui,
			SubjectVisitManagementForm svmForm) throws BaseException {
		return getExperimentNames(dbID, ui, svmForm, null);
	}

	public static String[] getExperimentNames(String dbID, UserInfo ui,
			SubjectVisitManagementForm svmForm, String subjectID)
			throws BaseException {
		ISubjectVisitManagement isvm = ServiceFactory
				.getSubjectVisitManagement(dbID);
		List<?> experiments = null;
		if (subjectID == null) {
			Subject subject = svmForm.getSubject();

			// get only the experiments this subject belongs
			experiments = isvm.getExperimentsForSubject(ui, subject
					.getSubjectID());
		} else {
			// get only the experiments this subject belongs
			experiments = isvm.getExperimentsForSubject(ui, subjectID);
		}
		String[] expNames = new String[experiments.size()];
		int idx = 0;
		svmForm.getExperiments().clear();
		for (Iterator<?> iter = experiments.iterator(); iter.hasNext();) {
			Experiment exp = (Experiment) iter.next();
			expNames[idx++] = exp.getName();
			// cache experiments in the svmForm
			svmForm.addExperiment(exp);
		}
		return expNames;
	}

	public static void prepareSegmentTimestampList(
			SubjectVisitManagementForm svmForm) {
		prepareSegmentTimestampList(svmForm, -1);
	}

	public static void prepareSegmentTimestampList(
			SubjectVisitManagementForm svmForm, int selectedSegmentID) {
		// prepare the segment time stamp
		List<SegmentFormInfo> sfis = null;
		StudyFormInfo studyFI = svmForm.getCurrentStudy();
		final boolean usingStudy = studyFI != null;
		log.info("prepareSegmentTimestampList  --- usingStudy=" + usingStudy);
		if (studyFI != null) {
			sfis = new ArrayList<SegmentFormInfo>(studyFI.getSegmentFormInfos());
		} else {
			sfis = new ArrayList<SegmentFormInfo>(svmForm.getCurrentVisit()
					.getSegmentFormInfos());
		}
		Collections.sort(sfis, new Comparator<SegmentFormInfo>() {
			public int compare(SegmentFormInfo si1, SegmentFormInfo si2) {
				if (usingStudy) {
					return si2.getStudySegment().getTimeStamp().compareTo(
							si1.getStudySegment().getTimeStamp());
				} else {
					return si2.getVisitSegment().getTimeStamp().compareTo(
							si1.getVisitSegment().getTimeStamp());
				}
			}
		});

		boolean first = true;

		SegmentTimestampSelector selector = new SegmentTimestampSelector();
		svmForm.setSegTimestampSelector(selector);

		// svmForm.getSegmentTimestamps().clear();
		for (SegmentFormInfo sfi : sfis) {
			Timestamp tsv = null;
			int segmentID = -1;
			String name = null;
			if (usingStudy) {
				tsv = sfi.getStudySegment().getTimeStamp();
				segmentID = sfi.getStudySegment().getSegmentID();
				name = sfi.getStudySegment().getName();
			} else {
				tsv = sfi.getVisitSegment().getTimeStamp();
				segmentID = sfi.getVisitSegment().getSegmentID();
				name = sfi.getVisitSegment().getName();
			}

			String formattedTimeStamp = DateTimeUtils.formatTimestamp(tsv);
			String label = formattedTimeStamp;
			if (name != null) {
				label += " - " + name;
			}
			selector.add(label, segmentID);

			if (first) {
				if (selectedSegmentID == -1) {
					svmForm.setSelectedSegmentID(String.valueOf(segmentID));
					// make sure that the timestamp selected matches the segment
					svmForm.setSegmentTimestampValue(formattedTimeStamp);
					log.info("setSegmentTimestampValue = " + formattedTimeStamp
							+ " selectedSegmentID=" + selectedSegmentID);
				}
				first = false;
			}
			if (selectedSegmentID != -1) {
				if (selectedSegmentID == segmentID) {
					svmForm.setSelectedSegmentID(String.valueOf(segmentID));
					// make sure that the timestamp selected matches the segment
					svmForm.setSegmentTimestampValue(formattedTimeStamp);
					log.info("setSegmentTimestampValue = " + formattedTimeStamp
							+ "selectedSegemntID=" + selectedSegmentID);
				}
			}
		}

		log
				.info(" svmForm.selectedSegmentID="
						+ svmForm.getSelectedSegmentID());
		log.info("*** end of prepareSegmentTimestampList");

	}

	public static void prepareCurrentSubject(String dbID, UserInfo ui,
			String selSubjectID, SubjectVisitManagementForm svmForm,
			int selExperimentID) throws BaseException {
		prepareCurrentSubject(dbID, ui, selSubjectID, svmForm, selExperimentID,
				-1);
	}

	/**
	 * Gets the selected subject's info from the database. Retrieves the visits
	 * associated with the subject, sorts them by date. Depending on the visit
	 * type the visits are added to the {@link clinical.web.vo.Subject} object.
	 * <p>
	 * The visit date LabelValue beans are populated in the <code>svmForm</code>.
	 * The selected visit is set to the latest visit. Then the experiments this
	 * subject is participating is retrieved from the database and cached in the
	 * <code>svmForm</code> ActionForm. The protocols are retrieved from the
	 * database or from the cache and {@link clinical.web.helpers.VisitFormInfo}
	 * object is prepared for the selected (latest) visit.
	 * <p>
	 * The {@link clinical.web.helpers.SegmentFormInfo} objects are created for
	 * {@link clinical.web.vo.VisitSegment} objects of the latest visit ({@link clinical.web.vo.Visit}).
	 * After that it sets the <code>currentVisit</code> and
	 * <code>selectedVisitID</code> on the <code>svmForm</code>.
	 *
	 * @param dbID
	 * @param ui
	 *            User Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param selSubjectID
	 *            the user selected subject's unique ID
	 * @param svmForm
	 *            the ActionForm for subject and visit management
	 * @exception BaseException
	 *                if an error occurs
	 *
	 * @see clinical.web.vo.Subject
	 * @see clinical.web.helpers.VisitFormInfo
	 * @see clinical.web.helpers.SegmentFormInfo
	 * @see clinical.web.vo.Visit
	 * @see clinical.web.vo.VisitSegment
	 */

	public static void prepareCurrentSubject(String dbID, UserInfo ui,
			String selSubjectID, SubjectVisitManagementForm svmForm,
			int selExperimentID, int selVisitID) throws BaseException {
		ISubjectVisitManagement isvm = ServiceFactory
				.getSubjectVisitManagement(dbID);

		Subject subject = prepareSubjectFromHumanSubject(ui, selSubjectID, isvm);
		svmForm.setSelectedSubjectID(selSubjectID);
		// get the visits for the selected subject
		List<?> visits = isvm
				.getAllVisitsForSubject(ui, subject.getSubjectID());

		for (Iterator<?> iter = visits.iterator(); iter.hasNext();) {
			Visit vi = (Visit) iter.next();
			// sort the visits by date also
			subject.addVisit(vi, true);
		}
		List<Visit> expVisits = null;
		int selectedExpID = selExperimentID;
		if (selectedExpID == -1) {
			selectedExpID = subject.getFirstExperimentID();
		}
		Experiment selectedExp = null;
		if (selectedExpID != -1) {
			expVisits = subject.getVisitsForExperiment(String
					.valueOf(selectedExpID));
			selectedExp = SubjectManagementHelper.findExperiment(svmForm
					.getExperiments(), selectedExpID);
		}

		String[] expNames = prepareExperiments(ui, svmForm, isvm, subject);

		svmForm.setSubject(subject);

		if (expVisits != null && !expVisits.isEmpty()) {
			// repopulate visitdates
			svmForm.getVisitDates().clear();
			for (Visit vi : expVisits) {
				svmForm.addVisitDate(DateTimeUtils.formatTimestampToDate(vi
						.getTs()), String.valueOf(vi.getComponentID()));
			}
			Visit latestVisit = null;
			if (selVisitID == -1) {
				latestVisit = expVisits.get(0);
			} else {
				for (Visit vi : expVisits) {
					if (vi.getComponentID() == selVisitID) {
						latestVisit = vi;
						break;
					}
				}
			}

			// the first visit shown is the latest
			svmForm.setVisitDateValue(latestVisit.getComponentIDAsString());

			String[] visitTypes = SubjectManagementHelper.getVisitTypes(dbID,
					ui);

			VisitFormInfo vfi = new VisitFormInfo(latestVisit, expNames,
					visitTypes, svmForm);
			vfi.setExperiment(selectedExp.getName());
			// set the protocols
			List<Protocol> protocols = SubjectManagementHelper.getProtocols(
					dbID, ui);
			svmForm.setProtocols(protocols);

			Object[] protocolNameVersions = SubjectManagementHelper
					.getProtocolNameVersions(protocols);

			if (!latestVisit.getStudies().isEmpty()) {
				StudySelector studySelector = new StudySelector(latestVisit
						.getStudies());
				svmForm.setStudySelector(studySelector);
				svmForm.setSelectedStudyID(String.valueOf(latestVisit
						.getStudies().size()));
				studySelector.setSelectedStudyID(latestVisit.getStudies()
						.size());

				prepareStudy(svmForm, latestVisit, vfi, protocols,
						protocolNameVersions, studySelector);
			} else {
				// there are no studies
				svmForm.setStudySelector(null);
				// set segments
				log.info("no studies found setSegmentsForVisit ... ");
				setSegmentsForVisit(svmForm, latestVisit, vfi, protocols,
						protocolNameVersions);
			}

			svmForm.setCurrentVisit(vfi);
			svmForm.setSelectedVisitID(String.valueOf(vfi.getVisit()
					.getComponentID()));
		} // ! expvisits,isEmpty()
	}

	public static Subject prepareSubjectFromHumanSubject(UserInfo ui,
			String selSubjectID, ISubjectVisitManagement isvm)
			throws SubjectVisitManagementException {
		SearchPredicateList spList = new SearchPredicateList();
		spList.addSearchPredicate(new SearchPredicate("Subjectid",
				selSubjectID, SearchPredicate.EQUAL, SearchPredicate.STRING),
				SearchPredicateList.NONE);
		List<?> subjects = isvm.getMatchingSubjects(ui, new SearchCriteria(
				spList));
		// should return a single subject
		assert (subjects.size() == 1);
		Humansubject hs = (Humansubject) subjects.get(0);

		Subject subject = new Subject(hs);
		return subject;
	}

	public static void prepareCurrentSubjectAndVisit(String dbID, UserInfo ui,
			String selSubjectID, int selectedVisitID, int selectedSegmentID,
			SubjectVisitManagementForm svmForm) throws BaseException {
		/** @todo very similar to SubjectManagementAction.manageSubjects refactor */
		ISubjectVisitManagement isvm = ServiceFactory
				.getSubjectVisitManagement(dbID);

		Subject subject = prepareSubjectFromHumanSubject(ui, selSubjectID, isvm);

		// get the visits for the selected subject
		List<?> visits = isvm
				.getAllVisitsForSubject(ui, subject.getSubjectID());
		// sort the visits by date
		Collections.sort(visits, new Comparator<Object>() {
			public int compare(Object o1, Object o2) {
				Visit v1 = (Visit) o1;
				Visit v2 = (Visit) o2;
				return v2.getTs().compareTo(v1.getTs());
			}
		});

		svmForm.getVisitDates().clear();
		for (Object element : visits) {
			Visit visit = (Visit) element;
			/** @todo get rid of hardcoded constants */
			if (visit.getVisitType()
					.equalsIgnoreCase(Constants.VISIT_TYPE_SCAN)) {
				subject.addScanVisit(visit);
			} else {
				subject.addClinicalVisit(visit);
			}
			// set the visit date after formatting
			svmForm.addVisitDate(DateTimeUtils.formatTimestampToDate(visit
					.getTs()), String.valueOf(visit.getComponentID()));
		}

		String[] expNames = prepareExperiments(ui, svmForm, isvm, subject);

		svmForm.setSubject(subject);

		if (!visits.isEmpty()) {
			Visit selectedVisit = null;
			for (Object element : visits) {
				Visit visit = (Visit) element;
				if (visit.getComponentID() == selectedVisitID) {
					selectedVisit = visit;
					break;
				}
			}
			svmForm.setVisitDateValue(selectedVisit.getComponentIDAsString());

			String[] visitTypes = SubjectManagementHelper.getVisitTypes(dbID,
					ui);

			VisitFormInfo vfi = new VisitFormInfo(selectedVisit, expNames,
					visitTypes, svmForm);

			// set the protocols
			List<Protocol> protocols = SubjectManagementHelper.getProtocols(
					dbID, ui);
			svmForm.setProtocols(protocols);

			Object[] protocolNameVersions = SubjectManagementHelper
					.getProtocolNameVersions(protocols);

			if (!selectedVisit.getStudies().isEmpty()) {
				StudySelector studySelector = svmForm.getStudySelector();
				if (studySelector == null) {
					studySelector = new StudySelector(selectedVisit
							.getStudies());
					svmForm.setStudySelector(studySelector);
				}
				prepareStudy(svmForm, selectedVisit, vfi, protocols,
						protocolNameVersions, studySelector);
			} else {
				// there are no studies
				svmForm.setStudySelector(null);
				// set segments
				log.info("no studies found setSegmentsForVisit ... ");
				setSegmentsForVisit(svmForm, selectedVisit, vfi, protocols,
						protocolNameVersions);
			}

			svmForm.setCurrentVisit(vfi);
			svmForm.setSelectedVisitID(String.valueOf(vfi.getVisit()
					.getComponentID()));

			if (log.isDebugEnabled()) {
				for (SegmentFormInfo sfi : vfi.getSegmentFormInfos()) {
					log.debug("** " + sfi.toString());
				}
			}

		} // ! visits,isEmpty()
	}

	public static String[] prepareExperiments(UserInfo ui,
			SubjectVisitManagementForm svmForm, ISubjectVisitManagement isvm,
			Subject subject) throws SubjectVisitManagementException {
		// get only the experiments this subject belongs
		List<?> experiments = isvm.getExperimentsForSubject(ui, subject
				.getSubjectID());

		String[] expNames = new String[experiments.size()];
		int idx = 0;
		svmForm.getExperiments().clear();
		svmForm.getExperimentLVBeans().clear();
		Experiment selectedExperiment = null;

		for (Object element : experiments) {
			Experiment exp = (Experiment) element;
			expNames[idx++] = exp.getName();
			// cache experiments in the svmForm
			svmForm.addExperiment(exp);
			svmForm.addExperimentLVBean(exp);
			if (selectedExperiment == null) {
				selectedExperiment = exp;
			}
		}
		return expNames;
	}

	public static void prepareStudy(SubjectVisitManagementForm svmForm,
			Visit aVisit, VisitFormInfo vfi, List<Protocol> protocols,
			Object[] protocolNameVersions, StudySelector studySelector) {
		// add study value objects used in form
		vfi.clearStudyFormInfos();
		for (Study s : aVisit.getStudies()) {
			StudyFormInfo sgi = new StudyFormInfo(s);
			setSegmentsForStudy(svmForm, s, sgi, protocols,
					protocolNameVersions);
			vfi.addStudyFormInfo(sgi);
		}

		int selStudyID = studySelector.getSelectedStudyID();
		if (selStudyID == StudySelector.DEFAULT_STUDY) {
			// 6/19/2007 IBO
			if (getNumberOfVisitSegments(aVisit) == 0
					&& !aVisit.getStudies().isEmpty()) {
				// select the first study instead of default
				selStudyID = ((Study) aVisit.getStudies().get(0)).getStudyID();
				studySelector.setSelectedStudyID(selStudyID);
			}
		}
		if (selStudyID == StudySelector.DEFAULT_STUDY) {
			log.info("default study  setSegmentsForVisit ... ");
			setSegmentsForVisit(svmForm, aVisit, vfi, protocols,
					protocolNameVersions);
		} else {
			// an actual study is selected
			StudyFormInfo selectedSFI = findStudyFormInfoByID(vfi, selStudyID);
			vfi.setCurrentStudyFormInfo(selectedSFI);
			svmForm.setSelectedStudyID(String.valueOf(selStudyID));
		}
	}

	private static StudyFormInfo findStudyFormInfoByID(VisitFormInfo vfi,
			int selStudyID) {
		for (StudyFormInfo sfi : vfi.getStudyFormInfos()) {
			if (sfi.getStudy().getStudyID() == selStudyID) {
				return sfi;
			}
		}
		return null;
	}

	public static int getNumberOfVisitSegments(Visit aVisit) {
		return aVisit.getVisitSegments().size();
	}

	public static void setSegmentsForVisit(SubjectVisitManagementForm svmForm,
			Visit aVisit, VisitFormInfo vfi, List<Protocol> protocols,
			Object[] protocolNameVersions) {
		vfi.clearSegmentFormInfos();
		for (VisitSegment vs : aVisit.getVisitSegments()) {

			SegmentFormInfo sfi = new SegmentFormInfo(vs,
					(String[]) protocolNameVersions[0],
					(int[]) protocolNameVersions[1], svmForm);
			String protocolValue = SubjectManagementHelper.getProtocolValue(
					protocols, vs.getProtocolID(), vs.getProtocolVersion());
			sfi.setProtocolValue(protocolValue);
			vfi.addSegmentFormInfo(sfi);
		}
	}

	public static void setSegmentsForStudy(SubjectVisitManagementForm svmForm,
			Study theStudy, StudyFormInfo sgi, List<Protocol> protocols,
			Object[] protocolNameVersions) {
		log.info("in setSegmentsForStudy");
		sgi.clearSegmentFormInfos();
		List<StudySegment> studySegments = theStudy.getStudySegments();
		log.info("studySegments=" + studySegments);
		for (StudySegment ss : studySegments) {
			SegmentFormInfo sfi = new SegmentFormInfo(ss,
					(String[]) protocolNameVersions[0],
					(int[]) protocolNameVersions[1], svmForm);
			String protocolValue = SubjectManagementHelper.getProtocolValue(
					protocols, ss.getProtocolID(), ss.getProtocolVersion());
			sfi.setProtocolValue(protocolValue);
			sgi.addSegmentFormInfo(sfi);
		}
	}

	@SuppressWarnings("unchecked")
	public static Map<String, String> getAssessmentNamesMap(
			HttpSession session, String dbID, UserInfo ui) throws Exception {
		Map<String, String> assessmentNameMap = (Map<String, String>) session
				.getAttribute(Constants.ASSESSMENT_NAME_MAP);
		if (assessmentNameMap == null) {
			IAssessmentService asService = ServiceFactory
					.getAssessmentService(dbID);
			List<?> assessments = asService.getAllAssessments(ui);
			assessmentNameMap = new LinkedHashMap<String, String>(17);
			for (Object element : assessments) {
				Assessment as = (Assessment) element;
				assessmentNameMap.put(as.getName(), as.getName());
			}
		}
		return assessmentNameMap;
	}

	public static void dumpScoreValuesMap(
			Map<String, AssessmentScoreValues> scoreValuesMap, Log logger) {
		logger.info("****************** ScoreValuesMap dump **************");
		for (Object element : scoreValuesMap.values()) {
			AssessmentScoreValues asv = (AssessmentScoreValues) element;
			logger.info("assessment - " + asv.getAssessmentName() + " visitID="
					+ asv.getVisitID() + " segmentID=" + asv.getSegmentID());
			StringBuffer sb = new StringBuffer(200);
			for (ScoreValue sv : asv.getScoreValues()) {
				sb.append("name=").append(sv.getName()).append(" value=")
						.append(sv.getValue()).append("\n");
			}
			logger.info(sb.toString());
		}
		logger
				.info("****************** end ScoreValuesMap dump **************");
	}

	public static void dumpAsiList(List<AssessmentInfo> asiList, Log logger) {
		logger.info("****************** asiList dump **************");
		for (AssessmentInfo asi : asiList) {
			log.info(asi.toString());
		}
		logger.info("****************** end asiList dump **************");
	}

	public static List<AssessmentInfo> getAsInfosForVisit(
			List<AssessmentInfo> asiList, int visitID) {
		List<AssessmentInfo> curAsInfos = new LinkedList<AssessmentInfo>();
		for (AssessmentInfo asi : asiList) {
			if (asi.getVisitID() == visitID) {
				curAsInfos.add(asi);
			}
		}
		return curAsInfos;
	}

	public static List<AssessmentInfo> getAsInfosForSegment(
			List<AssessmentInfo> asiList, int visitID, int segmentID) {
		List<AssessmentInfo> curAsInfos = new LinkedList<AssessmentInfo>();
		for (Iterator<AssessmentInfo> iter = asiList.iterator(); iter.hasNext();) {
			AssessmentInfo asi = iter.next();
			if (asi.getVisitID() == visitID && asi.getSegmentID() == segmentID) {
				curAsInfos.add(asi);
			}
		}
		return curAsInfos;

	}

	public static List<String> getAsNamesToAdd(List<String> asNames,
			List<AssessmentInfo> visitAsiList) {
		List<String> asNames2Add = new LinkedList<String>();
		Map<String, String> map = new HashMap<String, String>(17);
		for (Iterator<AssessmentInfo> iter = visitAsiList.iterator(); iter
				.hasNext();) {
			AssessmentInfo asi = iter.next();
			map.put(asi.getName(), asi.getName());
		}
		for (Iterator<String> iter = asNames.iterator(); iter.hasNext();) {
			String asName = iter.next();
			if (map.get(asName) == null) {
				asNames2Add.add(asName);
			}
		}
		return asNames2Add;
	}

	public static void prepareForSegmentManPage(HttpSession session,
			UserInfo ui, String dbID, SubjectVisitManagementForm svmForm,
			String subjectID, int experimentID) throws Exception {
		prepareForSegmentManPage(session, ui, dbID, svmForm, subjectID,
				experimentID, -1, -1, RETRIEVE_NON_VALIDATED, false);
	}

	public static void prepareForSegmentManPage(HttpSession session,
			UserInfo ui, String dbID, SubjectVisitManagementForm svmForm,
			String subjectID, int experimentID, int entryID2Use)
			throws Exception {
		prepareForSegmentManPage(session, ui, dbID, svmForm, subjectID,
				experimentID, -1, entryID2Use, RETRIEVE_NON_VALIDATED, false);
	}

	public static int getSelectedEntryID(SubjectVisitManagementForm svmForm) {
		String entryID = (String) svmForm.getSelectedEntryID();
		if ((entryID == null) || (entryID.length() == 0)) {
			entryID = "-1";
		}
		return Integer.parseInt(entryID);
	}

	public static void dumpAsScoreValuesList(
			List<AssessmentScoreValues> valuesList) {
		System.out
				.println("******************* dumpAsScoreValuesList ****************");
		System.out
				.println("**********************************************************");

		for (Object element : valuesList) {
			AssessmentScoreValues asv = (AssessmentScoreValues) element;
			System.out.println(asv.toString());
		}
		System.out
				.println("************** end ofdumpAsScoreValuesList ***************");
		System.out
				.println("**********************************************************");
	}

	/**
	 *
	 * @param session
	 *            HttpSession
	 * @param ui
	 *            UserInfo
	 * @param dbID
	 *            String
	 * @param svmForm
	 *            SubjectVisitManagementForm
	 * @param subjectID
	 *            String
	 * @param experimentID
	 *            int
	 * @param selectedSegmentID
	 *            int
	 * @param entryID2Use
	 *            the entryID for the assessment instance(s) you want see
	 *            summaries for (entryIDs are 1 based, use -1 if you don't want
	 *            to specify the entry ID)
	 * @param retrieveMode
	 * @throws Exception
	 */
	public static void prepareForSegmentManPage(HttpSession session,
			UserInfo ui, String dbID, SubjectVisitManagementForm svmForm,
			String subjectID, int experimentID, int selectedSegmentID,
			int entryID2Use, int retrieveMode, boolean doDebug)
			throws Exception {
		// unselect any previous selection
		svmForm.setSelectedSegmentID(null);

		ISubjectAssessmentManagement isam = null;
		isam = ServiceFactory.getSubjectAssessmentManagement(dbID);

		List<AssessmentInfo> asiList = isam.getAssessmentsForSubject(ui,
				subjectID, experimentID);

		if (asiList != null && !asiList.isEmpty()) {
			int entryID = getSelectedEntryID(svmForm);
			if (entryID2Use > 0) {
				// overrides the current selected entry id in the subject visit
				// management form
				entryID = entryID2Use;
			}

			// retrieves assessment values for the subject for the provided
			// list of assessments
			List<AssessmentScoreValues> valuesList = null;
			List<AssessmentScoreValues> validatedValuesList = null;
			if (retrieveMode == RETRIEVE_VALIDATED
					|| retrieveMode == RETRIEVE_BOTH) {
				validatedValuesList = isam.getAssessmentValuesForSubject(ui,
						subjectID, experimentID, asiList, -1, true);
			}
			if (retrieveMode != RETRIEVE_VALIDATED) {
				valuesList = isam.getAssessmentValuesForSubject(ui, subjectID,
						experimentID, asiList, entryID, false);
			}
			if (retrieveMode == RETRIEVE_VALIDATED) {
				valuesList = validatedValuesList;
			} else if (retrieveMode == RETRIEVE_BOTH) {
				if (validatedValuesList != null
						&& !validatedValuesList.isEmpty()) {
					// combine both validated and not validated assessment
					// instances
					// for this segment
					List<AssessmentScoreValues> tempList = new ArrayList<AssessmentScoreValues>(
							validatedValuesList);
					for (Iterator<AssessmentScoreValues> iter = valuesList
							.iterator(); iter.hasNext();) {
						AssessmentScoreValues asv = iter.next();
						if (!alreadyInList(asv, validatedValuesList)) {
							tempList.add(asv);
						}
					}
					valuesList = tempList;
				}
			}

			if (doDebug) {
				// dumpAsScoreValuesList(valuesList);
			}
			// dumpAsScoreValuesList(valuesList);

			// order score values by score sequence
			/*
			 * log.info("ORDER AssessmentScoreValues"); for (Iterator iter =
			 * valuesList.iterator(); iter.hasNext(); ) { AssessmentScoreValues
			 * asv = (AssessmentScoreValues)iter.next(); asv.orderScoreValues(); //
			 * log.info(asv.toString()); } log.info("end ORDER
			 * AssessmentScoreValues");
			 */

			svmForm.setAssessmentInfos(asiList);

			svmForm.setAsScoreValuesMap(valuesList);

			if (log.isDebugEnabled()) {
				SubjectManagementHelper.dumpScoreValuesMap(svmForm
						.getScoreValuesMap(), log);
			}

		} else {
			svmForm.setAssessmentInfos(new LinkedList<AssessmentInfo>());
		}

		// prepare the segment time stamp , set the selected segment
		SubjectManagementHelper.prepareSegmentTimestampList(svmForm,
				selectedSegmentID);

		svmForm.setCurrentAsInfos();
		svmForm.resetAssessments();

		log.info(">> visit " + svmForm.getCurrentVisit());

		// prepare editable assessment selection dropdown info
		prepareEditableAsSelDropdown(session, ui, dbID, svmForm, asiList);

		// prepare entry counts keyed by the assessment name for the segment
		Map<String, Integer> entryCountMap = SubjectManagementHelper
				.prepareEntryCountMap(ui, dbID, asiList, svmForm);
		svmForm.setEntryMap(entryCountMap);
		SubjectManagementHelper
				.prepareReconStatsMap(ui, dbID, asiList, svmForm);
	}

	public static List<MissingEntry> getMissingEntryItems(UserInfo ui,
			String dbID, String subjectID, int experimentID, int visitID,
			int segmentID, int assessmentID) throws Exception {
		ISubjectAssessmentManagement isam = ServiceFactory
				.getSubjectAssessmentManagement(dbID);
		List<MissingEntry> misssingEntries = isam.getMissingEntryItems(ui,
				subjectID, experimentID, visitID, segmentID, assessmentID);
		return misssingEntries;
	}

	/**
	 *
	 * @param ui
	 *            UserInfo
	 * @param dbID
	 *            String
	 * @param subjectID
	 *            String
	 * @param experimentID
	 *            int
	 * @param visitID
	 *            int
	 * @param selectedSegmentID
	 *            int
	 * @param asi
	 *            AssessmentInfo
	 * @throws Exception
	 * @return a hash table keyed by entryID holding a
	 *         <code>AssessmentScoreValues</code> object per key
	 */
	public static Map<Integer, AssessmentScoreValues> getAssessmentEntryData(
			UserInfo ui, String dbID, String subjectID, int experimentID,
			int visitID, int selectedSegmentID, AssessmentInfo asi)
			throws Exception {
		ISubjectAssessmentManagement isam = ServiceFactory
				.getSubjectAssessmentManagement(dbID);
		List<AssessmentInfo> asiList = new ArrayList<AssessmentInfo>();
		// using an incomplete AssessmentInfo object
		asiList.add(asi);
		// getting a list of AssessmentScoreValues objects
		List<AssessmentScoreValues> valuesList = isam
				.getAssessmentValuesForSubject(ui, subjectID, experimentID,
						visitID, selectedSegmentID, asiList, 1, false);
		assert (valuesList.size() == 1);
		Map<Integer, AssessmentScoreValues> entryMap = new HashMap<Integer, AssessmentScoreValues>(
				3);

		AssessmentScoreValues asv = valuesList.get(0);
		entryMap.put(new Integer(1), asv);

		valuesList = isam.getAssessmentValuesForSubject(ui, subjectID,
				experimentID, visitID, selectedSegmentID, asiList, 2, false);
		assert (valuesList.size() == 1);
		asv = valuesList.get(0);
		entryMap.put(new Integer(2), asv);

		return entryMap;
	}

	public static ScoreValuesDisplayLayout prepareScoreValuesDisplayLayout(
			AssessmentScoreValues asv, AssessmentInfo asi) throws Throwable {
		Map<String, AssessmentScoreValues> asScoreValuesMap = new HashMap<String, AssessmentScoreValues>(
				3);
		asScoreValuesMap.put(SubjectManagementHelper.createScoreValueMapKey(asv
				.getAssessmentName(), asv.getSegmentID(), asv.getVisitID()),
				asv);

		ScoreValuesDisplayLayout svdl = ScoreValuesDisplayHelper
				.getScoreValuesDisplayLayout(asi, asv.getSegmentID(), asv
						.getVisitID(), asScoreValuesMap);

		return svdl;
	}

	/**
	 * assuming each segment will have only a single assessment of the same type
	 *
	 * @param asName
	 * @param segmentID
	 * @return
	 */
	public static String createScoreValueMapKey(String asName, int segmentID,
			int visitID) {
		StringBuilder sb = new StringBuilder();
		sb.append(asName).append(':').append(segmentID).append(':').append(
				visitID);
		return sb.toString();
	}

	protected static boolean alreadyInList(AssessmentScoreValues asv,
			List<AssessmentScoreValues> validatedValuesList) {
		for (Iterator<AssessmentScoreValues> iter = validatedValuesList
				.iterator(); iter.hasNext();) {
			AssessmentScoreValues vasv = iter.next();
			if (vasv.getAssessmentID() == asv.getAssessmentID()
					&& vasv.getVisitID() == asv.getVisitID()
					&& vasv.getSegmentID() == asv.getSegmentID()) {
				return true;
			}
		}
		return false;
	}

	public static void prepareEditableAsSelDropdown(HttpSession session,
			UserInfo ui, String dbID, SubjectVisitManagementForm svmForm,
			List<AssessmentInfo> asiList) throws Exception {
		SegmentManagementHelper smHelper = new SegmentManagementHelper();
		Map<String, String> assessmentNamesMap = SubjectManagementHelper
				.getAssessmentNamesMap(session, dbID, ui);
		List<String> asNames = smHelper
				.getEditableAssessments(assessmentNamesMap);

		if (!asNames.isEmpty()) {
			svmForm.setAllEditableAsNames(asNames);

			int selectedSegmentID = Integer.parseInt(svmForm
					.getSelectedSegmentID());
			log.info("prepareEditableAsSelDropdown: selectedSegmentID ="
					+ selectedSegmentID);

			// filter the available editable assessment list, removing the ones
			// already present in current visit

			VisitFormInfo curVFI = svmForm.getCurrentVisit();
			List<AssessmentInfo> segmentAsiList = SubjectManagementHelper
					.getAsInfosForSegment(asiList, curVFI.getVisit()
							.getComponentID(), selectedSegmentID);

			List<String> filteredAsNames = SubjectManagementHelper
					.getAsNamesToAdd(asNames, segmentAsiList);

			if (!filteredAsNames.isEmpty()) {

				Collections.sort(filteredAsNames, new Comparator<String>() {
					public int compare(String s1, String s2) {
						return s1.compareToIgnoreCase(s2);
					}
				});

				AssessmentSelector selector = new AssessmentSelector(
						filteredAsNames);
				svmForm.setEditableAssessmentSelector(selector);
			} else {
				svmForm.setEditableAssessmentSelector(null);
			}
		}
	}

	public static void prepareReconStatsMap(UserInfo ui, String dbID,
			List<AssessmentInfo> asiList, SubjectVisitManagementForm svmForm)
			throws BaseException {
		ISubjectAssessmentManagement isam = ServiceFactory
				.getSubjectAssessmentManagement(dbID);
		int segmentID = Integer.parseInt(svmForm.getSelectedSegmentID());
		int visitID = svmForm.getCurrentVisit().getVisit().getComponentID();
		int expID = svmForm.getCurrentVisit().getVisit().getExperimentID();

		List<AssessmentInfo> segmentAsiList = SubjectManagementHelper
				.getAsInfosForSegment(asiList, visitID, segmentID);
		for (Iterator<AssessmentInfo> iter = segmentAsiList.iterator(); iter
				.hasNext();) {
			AssessmentInfo asi = iter.next();
			boolean reconciled = isam.isAssessmentReconciled(ui, svmForm
					.getSelectedSubjectID(), expID, visitID, segmentID, asi
					.getAssessmentID());
			if (reconciled) {
				svmForm.setReconStatus(String.valueOf(asi.getAssessmentID()),
						SubjectVisitManagementForm.RECONCILED);
				continue;
			}
			boolean canReconcile = isam.isAssessmentReadyForReconcile(ui,
					svmForm.getSelectedSubjectID(), expID, visitID, segmentID,
					asi.getAssessmentID());
			String status = (String) svmForm.getReconStatus(String.valueOf(asi
					.getAssessmentID()));
			if (!status.equals(SubjectVisitManagementForm.RECONCILED)) {
				if (canReconcile) {
					svmForm.setReconStatus(String
							.valueOf(asi.getAssessmentID()),
							SubjectVisitManagementForm.READY_FOR_RECON);
				} else {
					svmForm.setReconStatus(String
							.valueOf(asi.getAssessmentID()),
							SubjectVisitManagementForm.NOT_READY_FOR_RECON);
				}
			}
		}
	}

	public static Map<String, Integer> prepareEntryCountMap(UserInfo ui,
			String dbID, List<AssessmentInfo> asiList,
			SubjectVisitManagementForm svmForm) throws BaseException {
		ISubjectAssessmentManagement isam = ServiceFactory
				.getSubjectAssessmentManagement(dbID);
		int segmentID = Integer.parseInt(svmForm.getSelectedSegmentID());
		int visitID = svmForm.getCurrentVisit().getVisit().getComponentID();
		int expID = svmForm.getCurrentVisit().getVisit().getExperimentID();

		List<AssessmentInfo> segmentAsiList = SubjectManagementHelper
				.getAsInfosForSegment(asiList, visitID, segmentID);
		Map<String, AssessmentInfo> existingAsMap = new HashMap<String, AssessmentInfo>(
				17);
		for (Iterator<AssessmentInfo> iter = segmentAsiList.iterator(); iter
				.hasNext();) {
			AssessmentInfo asi = iter.next();
			existingAsMap.put(asi.getName(), asi);
		}
		segmentAsiList = null;

		Map<String, Integer> entryCountMap = new HashMap<String, Integer>(23);
		for (Iterator<AssessmentInfo> iter = asiList.iterator(); iter.hasNext();) {
			AssessmentInfo asi = iter.next();
			int entryCount = 0;
			if (existingAsMap.get(asi.getName()) != null) {
				entryCount = isam.getNumOfEntries(ui, svmForm
						.getSelectedSubjectID(), expID, visitID, segmentID, asi
						.getAssessmentID());
			}
			entryCountMap.put(String.valueOf(asi.getAssessmentID()),
					new Integer(entryCount));
			log.info(asi.getName() + " (asID=" + asi.getAssessmentID()
					+ ") --> entryCount =" + entryCount);
		}
		return entryCountMap;
	}

	public static int findExperimentID(String expName,
			SubjectVisitManagementForm svmForm) {
		for (Experiment exp : svmForm.getExperiments()) {
			if (exp.getName().equals(expName)) {
				return exp.getUniqueid().intValue();
			}
		}
		return -1;
	}

	public static void setExperimentIDForVisitSegment(
			SubjectVisitManagementForm svmForm, VisitSegment vs) {
		for (Experiment exp : svmForm.getExperiments()) {
			if (exp.getName().equals(svmForm.getCurrentVisit().getExperiment())) {
				vs.setExperimentID(exp.getUniqueid().intValue());
				break;
			}
		}
	}

	public static void setExperimentIDForStudySegment(
			SubjectVisitManagementForm svmForm, StudySegment ss) {
		for (Object element : svmForm.getExperiments()) {
			Experiment exp = (Experiment) element;
			if (exp.getName().equals(svmForm.getCurrentVisit().getExperiment())) {
				ss.setExperimentID(exp.getUniqueid().intValue());
				break;
			}
		}
	}

	public static Experiment prepareExperiments(
			SubjectVisitManagementForm svmForm, List<Experiment> experiments,
			String[] expNames, int selectedExpID) {
		int idx = 0;
		svmForm.getExperiments().clear();
		SubjectExperimentsSelector expSelector = new SubjectExperimentsSelector(
				experiments);
		expSelector.setSelectedExpID(selectedExpID);
		svmForm.setExpSelector(expSelector);

		Experiment selectedExperiment = null;
		for (Experiment exp : experiments) {
			expNames[idx++] = exp.getName();
			// cache experiments in the svmForm
			svmForm.addExperiment(exp);
			log.debug(">>> adding " + exp.toString());
			if (exp.getUniqueid().intValue() == selectedExpID) {
				selectedExperiment = exp;
			}
		}

		return selectedExperiment;
	}

	public static void prepare4SubjectEnrollment(String dbID, UserInfo ui,
			SubjectVisitManagementForm svmForm) throws BaseException {
		IDBCache dbCache = ServiceFactory.getDBCache(dbID);

		Map<BigDecimal, Experiment> expMap = new HashMap<BigDecimal, Experiment>(
				7);
		List<Experiment> experiments = dbCache.getExperiments(ui, true);
		List<Researchgroup> researchGroups = dbCache
				.getResearchGroups(ui, true);

		for (Iterator<Experiment> it = experiments.iterator(); it.hasNext();) {
			Experiment exp = it.next();
			expMap.put(exp.getUniqueid(), exp);
		}

		Map<Experiment, List<Researchgroup>> exp2ResearchGroupListMap = new HashMap<Experiment, List<Researchgroup>>(
				7);

		for (Researchgroup rg : researchGroups) {
			Experiment exp = expMap.get(rg.getNcExperimentUniqueid());
			List<Researchgroup> rgList = exp2ResearchGroupListMap.get(exp);
			if (rgList == null) {
				rgList = new ArrayList<Researchgroup>(5);
				exp2ResearchGroupListMap.put(exp, rgList);
			}
			rgList.add(rg);
		}
		SubjectExperimentsSelector expSelector = new SubjectExperimentsSelector(
				experiments);
		svmForm.setEnrollExperimentSelector(expSelector);

		StringBuffer jsonBuf = new StringBuffer(500);
		jsonBuf.append('{');
		for (Iterator< Map.Entry<Experiment, List<Researchgroup>> > it = exp2ResearchGroupListMap
				.entrySet().iterator(); it.hasNext();) {
			Map.Entry<Experiment, List<Researchgroup>> entry = it.next();
			Experiment exp = entry.getKey();
			List<Researchgroup> rgList = entry.getValue();
			Collections.sort(rgList, new Comparator<Researchgroup>() {
				public int compare(Researchgroup rg1, Researchgroup rg2) {
					return rg1.getName().compareTo(rg2.getName());
				}
			});
			jsonBuf.append(exp.getUniqueid()).append(" : {expName : \"");
			jsonBuf.append(exp.getName()).append("\", rgList : [");
			for (Iterator<Researchgroup> it2 = rgList.iterator(); it2.hasNext();) {
				Researchgroup rg = it2.next();
				jsonBuf.append("{ id: ").append(rg.getUniqueid());
				jsonBuf.append(",name :\"").append(rg.getName()).append("\"}");
				if (it2.hasNext())
					jsonBuf.append(',');
			}
			jsonBuf.append("] }");
			if (it.hasNext())
				jsonBuf.append(',');
			jsonBuf.append("\n");
		}

		jsonBuf.append('}');
		svmForm.setExp2RGListJSON(jsonBuf.toString());
	}

}
