package clinical.web.common.query;

import java.beans.BeanInfo;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;


/**
 * @author I. Burak Ozyurt
 * @version $Id: TSQLQueryBuilder.java 91 2009-08-17 23:38:26Z bozyurt $
 */

public class TSQLQueryBuilder {
	/**
	 * The class of the value object for which a query will be build and
	 * optionally executed.
	 */
	protected Class<?> voClass;

	/**
	 * holds the search predicate list used to build the where clause of the SQL
	 * query
	 */
	protected SearchPredicateList spList;

	/**
	 * provides explicit information about the methods, properties, events, etc,
	 * of the value object bean.
	 */
	protected BeanInfo bi;
	protected SimpleDateFormat df = new SimpleDateFormat("dd-MMM-yy");
	private static Log log = LogFactory.getLog(TSQLQueryBuilder.class);

	public TSQLQueryBuilder(Class<?> valueObjectClass,
			SearchPredicateList spList) throws Exception {
		this.voClass = valueObjectClass;
		this.spList = spList;
		bi = Introspector.getBeanInfo(voClass, java.lang.Object.class);
	}
	
	public TSQLQueryBuilder(Class<?> valueObjectClass,
			SearchCriteria sc) throws Exception {
		this.voClass = valueObjectClass;
		this.spList = sc.getSpList();
		bi = Introspector.getBeanInfo(voClass, java.lang.Object.class);
	}

	
	public static String getBaseClassName(String fullClassName) {
		int idx = fullClassName.lastIndexOf('.');
		if (idx != -1) {
			return fullClassName.substring(idx + 1);
		}
		return fullClassName;
	}
	
	public String buildQuery() throws Exception {
		StringBuffer query = new StringBuffer(256);
		Map<String, PropertyDescriptor> propMap = new HashMap<String, PropertyDescriptor>(
				19);
		PropertyDescriptor[] pds = bi.getPropertyDescriptors();
		for (int i = 0; i < pds.length; i++) {
			if (log.isDebugEnabled()) {
				log.debug(pds[i].getName());
			}
			propMap.put(pds[i].getName().toLowerCase(), pds[i]);
		}

		String beanName = getBaseClassName(voClass.getName());
		String alias =   String.valueOf( Character.toLowerCase( beanName.charAt(0)) ); 

		query.append("select ").append(alias).append(".* ");
		
		query.append(" from ").append(beanName).append(" as ").append(alias).append(' ');

		boolean first = true;
		for (Iterator<SearchPredicateList.SearchPredicateInfo> iter = spList
				.iterator(); iter.hasNext();) {
			SearchPredicateList.SearchPredicateInfo spi = iter.next();
			String attr = (String) spi.getSearchPredicate().getAttribute();
			if (propMap.get(attr.toLowerCase()) != null) {
				if (first)
					query.append("where ");
				else {
					if (spi.getLogicOp() == SearchPredicateList.AND)
						query.append(" and ");
					else if (spi.getLogicOp() == SearchPredicateList.OR)
						query.append(" or ");
				}
				String colName = attr;
				if (log.isDebugEnabled()) {
					log.debug(">>colName=" + colName + "attr=" + attr);
				}
				query.append(buildPredicate(spi.getSearchPredicate(), alias));

				first = false;
			}
		}
		return query.toString();
	}

	protected String buildPredicate(SearchPredicate sp, String alias) {
		StringBuffer buf = new StringBuffer();
		buf.append("( ");
		buf.append(alias).append('.').append(sp.getAttribute());
		int op = sp.getOperator();

		if (sp.getValue() == null
				|| sp.getValue().toString().trim().equals("*")) {
			buf.append(" is not null ) ");
			return buf.toString();
		}
		buf.append(QueryUtils.getRelationalOperator(op));

		if (op == SearchPredicate.BETWEEN) {
			SearchPredicate.Range range = (SearchPredicate.Range) sp.getValue();
			buf.append(range.getLowBound()).append(" and ").append(
					range.getUppBound());
		} else if (op == SearchPredicate.STARTS_WITH) {
			buf.append("'").append(sp.getValue()).append("%' ");
		} else if (op == SearchPredicate.ENDS_WITH) {
			buf.append("'%").append(sp.getValue()).append("' ");
		} else {
			if (sp.getType() == SearchPredicate.STRING) {
				buf.append("'").append(sp.getValue()).append("' ");
			} else if (sp.getType() == SearchPredicate.DATE) {
				preparePredicateDateValue(sp, buf);
			} else {
				buf.append(sp.getValue()).append(' ');
			}
		}
		buf.append(") ");

		return buf.toString();
	}

	protected void preparePredicateDateValue(SearchPredicate sp,
			StringBuffer buf) {
		Date date = (Date) sp.getValue();
		buf.append(df.format(date)).append(' ');
	}
}
