package clinical.web.services;

import java.io.File;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.utils.GenUtils;
import clinical.web.IAppConfigService;
import clinical.web.ImageHandler;
import clinical.web.ServiceFactory;
import clinical.web.exception.BaseException;

/**
 * Cleanup thread for AFNI BRIK file cache maintenance.
 * 
 * @author I. Burak Ozyurt
 * @version $Id: FileCacheCleanupService.java,v 1.5 2007/04/16 17:05:04 bozyurt
 *          Exp $
 */

public class FileCacheCleanupService implements Runnable {
	protected String cacheRoot;
	protected long cacheCheckInterval;
	protected long cacheSizeThreshold = -1L;
	protected long maxAge;
	protected IAppConfigService configService;
	protected ImageHandler imageHandler;
	protected boolean silentOperation = true;
	protected static FileCacheCleanupService instance;
	private static Log log = LogFactory.getLog("cache-cleanup");

	protected FileCacheCleanupService(String cacheRoot,
			IAppConfigService configService) throws BaseException {
		this.cacheRoot = cacheRoot;
		this.configService = configService;

		this.maxAge = GenUtils.toLong(configService
				.getParamValue("file.cache.max_age"), (24 * 3600));
		String value = configService.getParamValue("file.cache.max_size");
		if (value != null && Long.parseLong(value) > 0) {
			cacheSizeThreshold = Long.parseLong(value);
		}
		cacheCheckInterval = GenUtils.toLong(configService
				.getParamValue("file.cache.cache_check_interval"), 300); // default
																			// 5
																			// minutes
		cacheCheckInterval *= 1000; // to millisecs
		imageHandler = ServiceFactory.getImageHandlerService();
	}

	public synchronized static FileCacheCleanupService getInstance(
			String cacheRoot, IAppConfigService configService)
			throws BaseException {
		if (instance == null)
			instance = new FileCacheCleanupService(cacheRoot, configService);
		return instance;
	}

	public long getTotCacheSize() {
		File f = new File(cacheRoot);
		File[] files = f.listFiles();
		long totSize = 0;
		for (int i = 0; i < files.length; i++) {
			if (files[i].isDirectory()) {
				totSize += getDirSize(files[i]);
			}
		}
		return totSize;
	}

	static long getDirSize(File dir) {
		File[] files = dir.listFiles();
		long totSize = 0;
		for (int i = 0; i < files.length; i++) {
			if (files[i].isFile())
				totSize += files[i].length();
		}
		return totSize;
	}

	public void run() {
		while (true) {
			synchronized (this) {
				try {
					wait(cacheCheckInterval);
				} catch (InterruptedException iex) {
				}
			}
			try {
				if (log.isDebugEnabled()) {
					log.debug("doing routine file cache sweeping...");
				}
				imageHandler.doCacheCleanup(cacheRoot, configService, maxAge,
						cacheSizeThreshold, silentOperation);

			} catch (Exception x) {
				log.error(x, x);
			}
		}
	}

	public static class DirSizeInfo {
		File dir;
		int size;

		public DirSizeInfo(File dir, int size) {
			this.dir = dir;
			this.size = size;
		}
	}

	public boolean isSilentOperation() {
		return silentOperation;
	}

	public void setSilentOperation(boolean silentOperation) {
		this.silentOperation = silentOperation;
	}

}
