package clinical.web.services;

import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.server.dao.JobsDAO;
import clinical.server.vo.Databaseuser;
import clinical.server.vo.Jobs;
import clinical.server.vo.Tableid;
import clinical.web.Constants;
import clinical.web.DAOFactory;
import clinical.web.IJobManagementService;
import clinical.web.ISequenceHelper;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBPoolService;
import clinical.web.common.ISecurityService;
import clinical.web.common.UserInfo;
import clinical.web.common.security.User;
import clinical.web.exception.BaseException;
import clinical.web.exception.DBPoolServiceException;
import clinical.web.exception.JobManServiceException;
import clinical.web.scheduler.JobInfo;
import clinical.web.scheduler.JobRecord;

/**
 * @author I. Burak Ozyurt
 * @version $Id: JobManagementServiceImpl.java,v 1.5 2008/01/12 01:00:33 bozyurt
 *          Exp $
 */
public class JobManagementServiceImpl implements IJobManagementService {
	/** Connection pool service interface */
	protected IDBPoolService dbPoolService;
	protected String dbType;
	protected ISecurityService securityService;

	/**
	 * interface for retrieving and caching nearly static data from the
	 * database.
	 */
	protected DBCache dbCache;
	/** unique number generator for primary keys */
	protected ISequenceHelper seqHelper;
	protected String dbID;
	private Log log = LogFactory.getLog(JobManagementServiceImpl.class);

	public JobManagementServiceImpl(String dbID) throws BaseException {
		dbPoolService = ServiceFactory.getPoolService(dbID);
		dbCache = DBCache.getInstance(dbID);
		seqHelper = ServiceFactory.getSequenceHelper(dbID);
		this.dbID = dbID;
		securityService = ServiceFactory.getSecurityService();
		this.dbType = securityService.getDBType(dbID);
	}

	public List<Jobs> getAllJobs(UserInfo ui) throws JobManServiceException {
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			JobsDAO dao = DAOFactory.createJobsDAO(dbID);
			Jobs criteriaBean = new Jobs();
			return dao.find(con, criteriaBean);
		} catch (Exception x) {
			log.error("Error in getAllJobs", x);
			throw new JobManServiceException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void removeJob(UserInfo ui, String jobID)
			throws JobManServiceException {
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			JobsDAO dao = DAOFactory.createJobsDAO(dbID);
			Jobs criteriaBean = new Jobs();
			criteriaBean.setJobid(jobID);
			dao.delete(con, criteriaBean);
			con.commit();
		} catch (Exception x) {
			try {
				con.rollback();
			} catch (SQLException e) {
				log.error("rollback", e);
			}
			log.error("Error in removeJob", x);
			throw new JobManServiceException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void expireJob(UserInfo ui, String jobID)
			throws JobManServiceException {
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			JobsDAO dao = DAOFactory.createJobsDAO(dbID);
			Jobs criteriaBean = new Jobs();
			criteriaBean.setJobid(jobID);
			List<Jobs> jl = dao.find(con, criteriaBean);
			if (!jl.isEmpty()) {
				Jobs job = jl.get(0);
				Jobs updateBean = new Jobs();
				updateBean.setJobstatus(job.getJobstatus() + "_expired");
				dao.update(con, updateBean, criteriaBean);
				con.commit();
			}
		} catch (Exception x) {
			try {
				con.rollback();
			} catch (SQLException e) {
				log.error("rollback", e);
			}
			log.error("Error in removeJob", x);
			throw new JobManServiceException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public JobRecord findJob(UserInfo ui, String jobUser, String jobID)
			throws JobManServiceException {
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			JobsDAO dao = DAOFactory.createJobsDAO(dbID);
			Jobs criteriaBean = new Jobs();
			criteriaBean.setJobuser(jobUser);
			criteriaBean.setJobid(jobID);
			List<Jobs> jobList = dao.find(con, criteriaBean);
			if (jobList.isEmpty()) {
				return null;
			}
			if (jobList.size() > 1) {
				throw new Exception("The criteria jobUser:" + jobUser
						+ " and jobID:" + jobID
						+ " returned more than one record!");
			}

			Jobs job = jobList.get(0);
			if (JobInfo.REMOVED.equals(job.getJobstatus())
					|| JobInfo.CANCELED.equals(job.getJobstatus())) {
				return null;
			}
			JobRecord jr = new JobRecord(job.getJobuser(), job.getJobtype(),
					job.getJobid(), job.getJobstartdate(), job.getJobstatus(),
					job.getDescription(), 
					JobManagementHelper.fromDBSavedResultsFileString(job
									.getSavedresultfile()),
								false);
			int downloadCount = job.getDownloadcount() != null ? job
					.getDownloadcount().intValue() : 0;
			jr.setDownloadCount(downloadCount);

			return jr;
		} catch (Exception x) {
			log.error("Error in findJob", x);
			throw new JobManServiceException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public List<JobRecord> getJobsForUser(UserInfo ui, String jobUser)
			throws JobManServiceException {
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			JobsDAO dao = DAOFactory.createJobsDAO(dbID);
			Jobs criteriaBean = new Jobs();
			criteriaBean.setJobuser(jobUser);

			List<Jobs> jobList = dao.find(con, criteriaBean);
			List<JobRecord> jrList = new ArrayList<JobRecord>(jobList.size());
			for (Iterator<Jobs> iterator = jobList.iterator(); iterator
					.hasNext();) {
				Jobs job = iterator.next();
				if (JobInfo.REMOVED.equals(job.getJobstatus())
						|| JobInfo.CANCELED.equals(job.getJobstatus())) {
					continue;
				}
				// TODO newlyAdded?

				JobRecord jr = new JobRecord(job.getJobuser(),
						job.getJobtype(), job.getJobid(),
						job.getJobstartdate(), job.getJobstatus(), job
								.getDescription(), 								
						JobManagementHelper.fromDBSavedResultsFileString(job.getSavedresultfile()),
						false);
				if (job.getJobsize() != null
						&& job.getJobsize().longValue() > 0) {
					jr.setJobSize(job.getJobsize().longValue());
				}
				if (job.getDownloadcount() != null
						&& job.getDownloadcount().intValue() > 0) {
					jr.setDownloadCount(job.getDownloadcount().intValue());
				}
				if (job.getErrormsg() != null) {
					jr.setErrorMsg(job.getErrormsg());
				}
				jrList.add(jr);
			}
			return jrList;
		} catch (Exception x) {
			log.error("Error in getJobsForUser", x);
			throw new JobManServiceException(x);
		} finally {
			releaseConnection(con, ui);
		}

	}

	public void updateJob(UserInfo ui, JobRecord jobRec)
			throws JobManServiceException {
		log.info("updateJob " + jobRec);
		Connection con = null;
		try {
			con = dbPoolService.getConnection(ui.getName());
			con.setAutoCommit(false);
			Jobs criteriaBean = new Jobs();
			criteriaBean.setJobid(jobRec.getJobID());
			criteriaBean.setJobuser(jobRec.getUser());

			Jobs updateBean = new Jobs();
			updateBean.setJobstatus(jobRec.getStatus());
			if (jobRec.getDateFinished() != null) {
				updateBean.setJobenddate(jobRec.getDateFinished());
			}
			if (jobRec.getJobSize() > 0) {
				updateBean.setJobsize(new BigDecimal(jobRec.getJobSize()));
			}
			if (jobRec.getDownloadCount() > 0) {
				updateBean.setDownloadcount(new BigDecimal(jobRec
						.getDownloadCount()));
			}
			if (jobRec.getLastDownloadTime() != null) {
				updateBean.setLastdownloadtime(jobRec.getLastDownloadTime());
			}
			if (jobRec.getSavedResultFiles() != null) {
				String s = JobManagementHelper.buildSavedResultStringForDB( jobRec.getSavedResultFiles());
				updateBean.setSavedresultfile(s);
			}
			if (jobRec.getErrorMsg() != null) {
				updateBean.setErrormsg(jobRec.getErrorMsg());
			}

			updateBean.setModtime(new Date());
			JobsDAO dao = DAOFactory.createJobsDAO(dbID);
			dao.update(con, updateBean, criteriaBean);
			con.commit();
		} catch (Exception x) {
			handleErrorAndRollBack(con, "Error in updateJob", x, true);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void addJob(UserInfo ui, JobRecord jobRec)
			throws JobManServiceException {
		log.info("addJob " + jobRec);
		Connection con = null;
		try {
			if (securityService == null) {
				securityService = ServiceFactory.getSecurityService();
			}
			con = dbPoolService.getConnection(ui.getName());
			con.setAutoCommit(false);

			Jobs job = new Jobs();
			job.setJobuser(jobRec.getUser());
			job.setJobtype(jobRec.getType());
			job.setJobid(jobRec.getJobID());
			job.setJobstartdate(jobRec.getDate());
			job.setJobstatus(jobRec.getStatus());
			// TODO job size
			job.setDescription(jobRec.getDescription());

			String srfStr = JobManagementHelper
					.buildSavedResultStringForDB(jobRec.getSavedResultFiles());

			job.setSavedresultfile(srfStr);
			if (jobRec.getErrorMsg() != null) {
				job.setErrormsg(jobRec.getErrorMsg());
			}

			Map<String, User> userMap = securityService.getAllUsers(this.dbID);
			User u = userMap.get(ui.getName());

			Databaseuser databaseUser = dbCache.getDatabaseUser(ui, u
					.getDbUser().getName(), Constants.USERCLASS_ADMIN);
			job.setModuser(databaseUser.getUniqueid());
			job.setOwner(databaseUser.getUniqueid());
			Tableid tid = dbCache.getTableID(ui, Constants.JOBS_DB_TABLE);
			// TODO make sure nc_jobs table id exists
			job.setTableid(tid.getUniqueid());
			job.setUniqueid(seqHelper.getNextUID(ui, Constants.JOBS_DB_TABLE,
					"uniqueid"));
			job.setModtime(new Date());

			JobsDAO dao = DAOFactory.createJobsDAO(dbID);

			dao.insert(con, job);

			con.commit();
		} catch (Exception x) {
			handleErrorAndRollBack(con, "Error in addJob", x, true);
		} finally {
			releaseConnection(con, ui);
		}

	}

	protected void handleErrorAndRollBack(Connection con, String msg,
			Exception x, boolean doRollback) throws JobManServiceException {
		if (con != null) {
			try {
				con.rollback();
			} catch (SQLException se) {
				log.error("", se);
				throw new JobManServiceException(se.getMessage());
			}
		}
		log.error(msg, x);
		throw new JobManServiceException(x);
	}

	protected void releaseConnection(Connection con, UserInfo ui) {
		if (con == null) {
			return;
		}
		try {
			dbPoolService.releaseConnection(ui.getName(), con);
		} catch (DBPoolServiceException x) {
			log.error("Cannot release connection for user " + ui.getName(), x);
		}
	}
}
