package clinical.web.services;

import java.io.File;
import java.io.InputStream;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.input.SAXBuilder;

import clinical.utils.DBTableInfo;
import clinical.web.DBUtils;
import clinical.web.MinimalServiceFactory;
import clinical.web.common.IAuthenticationService;
import clinical.web.common.IAuthorizationService;
import clinical.web.common.IDBPoolService;
import clinical.web.common.ISecurityAdminService;
import clinical.web.common.ISecurityService;
import clinical.web.common.UserInfo;
import clinical.web.common.security.DBConfig;
import clinical.web.common.security.Privilege;
import clinical.web.common.security.User;
import clinical.web.exception.AuthenticationException;
import clinical.web.exception.BaseException;
import clinical.web.exception.UnknownUserException;

/**
 * A simple implementation of basic authentication and authorization services.
 * The authentication is based on an XML file mapping application users with the
 * named database users.
 *
 * @author I. Burak Ozyurt
 * @version $Id: SimpleSecurityService.java,v 1.31 2007/11/02 22:00:49 bozyurt
 *          Exp $
 */

public class SimpleSecurityService implements IAuthenticationService,
		IAuthorizationService, ISecurityService, ISecurityAdminService {
	private Document doc;
	private Map<String, Privilege> privilegesMap = new HashMap<String, Privilege>(
			23);
	private Map<String, DBConfig> dbConfigMap = Collections
			.synchronizedMap(new HashMap<String, DBConfig>(7));
	private Map<String, Map<String, String>> masterTableMap = new HashMap<String, Map<String, String>>();
	private DBConfig currentDBConfig;
	private Log log = LogFactory.getLog("security");
	private static SimpleSecurityService instance = null;

	private SimpleSecurityService(InputStream is) throws BaseException {
		try {
			SAXBuilder builder = null;
			builder = new SAXBuilder(false); // no validation
			doc = builder.build(is);
			extractData();
		} catch (Exception x) {
			throw new BaseException(x);
		}
	}

	private SimpleSecurityService(String userInfoFile) throws BaseException {
		SAXBuilder builder = null;
		try {
			builder = new SAXBuilder(false); // no validation
			doc = builder.build(new File(userInfoFile));
			extractData();
		} catch (Exception x) {
			throw new BaseException(x);
		}
	}

	public List<Privilege> getPrivileges() {
		List<Privilege> privList = new ArrayList<Privilege>(privilegesMap
				.values());
		Collections.sort(privList, new Comparator<Privilege>() {
			public int compare(Privilege p1, Privilege p2) {
				return p1.getName().compareTo(p2.getName());
			}
		});
		return privList;
	}

	public Map<String, Privilege> getPrivilegesMap() {
		return privilegesMap;
	}

	public String getDefaultDBID() {
		return currentDBConfig.getId();
	}

	public String[] getAllDBIDs() {
		String[] arr = new String[dbConfigMap.size()];
		int i = 0;
		for (Iterator<String> iter = dbConfigMap.keySet().iterator(); iter
				.hasNext();) {
			String dbID = iter.next();
			arr[i++] = dbID;
		}
		return arr;
	}

	public DBConfig getDBConfig(String dbID) {
		return dbConfigMap.get(dbID);
	}

	public String getDBType(String dbID) {
		DBConfig dbConfig = dbConfigMap.get(dbID);
		return dbConfig.getDbType();
	}

	/**
	 * Assumption: only one database per site
	 *
	 * @param siteID
	 * @return
	 */
	public DBConfig findBySiteID(String siteID) {
		siteID = siteID.toLowerCase();
		for (Iterator<DBConfig> iter = dbConfigMap.values().iterator(); iter
				.hasNext();) {
			DBConfig dbConfig = iter.next();
			String dbID = dbConfig.getId();
			if (dbID.toLowerCase().startsWith(siteID)) {
				return dbConfig;
			}
		}
		return null;
	}

	public String findSiteIDByDbID(String dbID) {
		for (DBConfig dbConfig : dbConfigMap.values()) {
			if (dbConfig.getId().equals(dbConfig)) {
				return dbConfig.getSiteID();
			}
		}
		return null;
	}

	public Map<String, String> getSiteURLs() {
		Map<String, String> map = new HashMap<String, String>(17);
		for (DBConfig dbConfig : dbConfigMap.values()) {
			map.put(dbConfig.getId(), dbConfig.getSiteURL());
		}
		return map;
	}

	public String findDBForSiteID(String siteID) {
		for (DBConfig dbConf : this.dbConfigMap.values()) {
			if (dbConf.getSiteID().equals(siteID)) {
				return dbConf.getId();
			}
		}
		return null;
	}


	private SimpleSecurityService(String userInfoFile, String schemaFile)
			throws BaseException {
		SAXBuilder builder = null;
		try {
			File f = null;
			if (schemaFile != null && (f = new File(schemaFile)).exists()) {
				builder = new SAXBuilder("org.apache.xerces.parsers.SAXParser",
						true); // validation
				builder.setFeature(
						"http://apache.org/xml/features/validation/schema",
						true);

				builder
						.setProperty(
								"http://apache.org/xml/properties/schema/external-noNamespaceSchemaLocation",
								f.toURI().toURL().toString());
				log.info("Using XML Schema " + schemaFile
						+ " for web-user/database user map validation.");
			} else {
				builder = new SAXBuilder(false); // no validation
			}
			doc = builder.build(new File(userInfoFile));
			extractData();
			log.info("finished parsing the users.xml file.");

		} catch (Exception x) {
			log.error("SimpleSecurityService", x);
			throw new BaseException(x);
		}
	}

	/**
	 * Checks if the given table name can be accessed by using the given
	 * connection.
	 *
	 * @param con
	 *            database connection
	 * @param tableName
	 *            database table name
	 * @return true if the given table name can be accessed by using the given
	 *         connection.
	 */
	public static boolean canAccessTable(Connection con, String tableName) {
		Statement st = null;
		try {
			st = con.createStatement();
			ResultSet rs = st.executeQuery("select 1 from " + tableName);
			rs.close();
			return true;
		} catch (SQLException x) {
			return false;
		} finally {
			DBUtils.close(st);
		}
	}

	/**
	 * Creates or returns the singleton simple security service.
	 *
	 * @param userInfoFile
	 *            the XML file containing the application user , named database
	 *            user mapping
	 * @return the singleton SimpleSecurityService
	 * @throws java.lang.BaseException
	 */
	public static synchronized SimpleSecurityService getInstance(
			String userInfoFile) throws BaseException {
		if (instance == null) {
			instance = new SimpleSecurityService(userInfoFile);
		}
		return instance;
	}

	/**
	 * Creates or returns the singleton simple security service.
	 *
	 * @param is
	 *            input stream for the XML file containing the application user ,
	 *            named database user mapping
	 * @return the singleton SimpleSecurityService
	 * @throws java.lang.BaseException
	 */
	public static synchronized SimpleSecurityService getInstance(InputStream is)
			throws BaseException {
		if (instance == null) {
			instance = new SimpleSecurityService(is);
		}
		return instance;
	}

	/**
	 * Creates or returns the singleton simple security service.
	 *
	 * @param userInfoFile
	 *            the XML file containing the application user , named database
	 *            user mapping
	 * @param xmlSchema
	 *            the XML Schema file full path used for validatio
	 * @return the singleton SimpleSecurityService
	 * @throws java.lang.BaseException
	 */
	public static synchronized SimpleSecurityService getInstance(
			String userInfoFile, String xmlSchema) throws BaseException {
		if (instance == null) {
			instance = new SimpleSecurityService(userInfoFile, xmlSchema);
		}
		return instance;
	}

	/**
	 *
	 * @return the singleton SimpleSecurityService
	 * @throws BaseException
	 *             if not created before
	 */
	public static synchronized SimpleSecurityService getInstance()
			throws BaseException {
		if (instance == null)
			throw new BaseException("Service is not initialized!");
		return instance;
	}

	/**
	 * Gets the master table/view list from the database meta data and caches.
	 *
	 * @param dbID
	 *            the ID of the database to get the table/view list
	 * @throws java.lang.Exception
	 */
	public void prepareTableCache(String dbID) throws Exception {
		// get the master table/view list and cache

		Map<String, String> dbMasterTableMap = new HashMap<String, String>(67);

		UserInfo ui = new UserInfo("admin", null, null);

		DBConfig adbc = dbConfigMap.get(dbID);
		Map<String, DBTableInfo> tableInfoMap = DBUtils.prepareTableInfoCache(
				ui, dbID, false, adbc.getDbType());

		for (DBTableInfo dbti : tableInfoMap.values()) {
			if (log.isDebugEnabled()) {
				log.debug("prepareTableCache: Adding " + dbti.getTableName());
			}
			dbMasterTableMap.put(dbti.getTableName().toUpperCase(), dbti
					.getTableName());
		}

		masterTableMap.put(dbID, dbMasterTableMap);
	}

	public DBConfig getCurrentDBConfig() {
		return currentDBConfig;
	}

	public void extractData() throws Exception {

		Element root = doc.getRootElement();
		Element databasesElem = root.getChild("databases");
		List<?> dbChildren = databasesElem.getChildren("database");
		boolean first = true;
		for (Iterator<?> iter = dbChildren.iterator(); iter.hasNext();) {
			Element elem = (Element) iter.next();
			String siteId = null;
			String siteName = null;
			if ( elem.getAttribute("siteName") != null)
				siteName = elem.getAttributeValue("siteName");
			if ( elem.getAttribute("siteId") != null)
				siteId = elem.getAttributeValue("siteId");

			DBConfig dbConfig = new DBConfig(elem.getAttributeValue("id"),
					siteName, siteId);
			Element dbURLElem = (Element) elem.getChild("db-url");
			dbConfig.setDbURL(dbURLElem.getText().trim());
			dbConfigMap.put(dbConfig.getId(), dbConfig);

			Element dbTypeElem = (Element) elem.getChild("db-type");
			if (dbTypeElem != null) {
				dbConfig.setDbType(dbTypeElem.getText().trim());
			} else {
				// default database type is oracle
				dbConfig.setDbType(DBConfig.ORACLE);
			}

			if (first) {
				currentDBConfig = dbConfig;
				first = false;
			}
			if (elem.getAttribute("default") != null) {
				if (elem.getAttributeValue("default").equalsIgnoreCase("true")) {
					currentDBConfig = dbConfig;
					dbConfig.setDefaultDB(true);
				}
			}

			if (elem.getAttribute("force-schema-owner-check") != null) {
				if (elem.getAttributeValue("force-schema-owner-check")
						.equalsIgnoreCase("true")) {
					dbConfig.setForceSchemaOwnerCheck(true);
				}
			}
		}
		currentDBConfig.setDefaultDB(true);

		Element dbUsersElem = root.getChild("dbusers");
		List<?> children = dbUsersElem.getChildren("dbuser");
		for (Iterator<?> it = children.iterator(); it.hasNext();) {
			Element elem = (Element) it.next();
			String dbID = elem.getAttributeValue("dbid");
			DBConfig dbConfig = dbConfigMap.get(dbID);
			if (dbConfig == null)
				throw new BaseException(dbID + " is not a valid database id!");
			User dbUser = new User(elem.getAttributeValue("name"), elem
					.getAttributeValue("pwd"));
			dbConfig.addDbUser(dbUser);

			// dbUserMap.put(dbUser.getName(), dbUser);
		}
		Element privsElem = root.getChild("privileges");
		children = privsElem.getChildren("privilege");
		for (Iterator<?> it = children.iterator(); it.hasNext();) {
			Element elem = (Element) it.next();
			String name = elem.getAttributeValue("name");
			Privilege privilege = new Privilege(name);
			Element descElem = elem.getChild("description");
			if (descElem != null) {
				privilege.setDescription(descElem.getText());
			}
			privilegesMap.put(name, privilege);

		}

		Element usersElem = root.getChild("users");
		children = usersElem.getChildren("user");
		for (Iterator<?> it = children.iterator(); it.hasNext();) {
			Element elem = (Element) it.next();
			String dbUserName = elem.getAttributeValue("dbuser");
			String dbID = elem.getAttributeValue("dbid");
			DBConfig dbConfig = dbConfigMap.get(dbID);
			if (dbConfig == null)
				throw new BaseException(dbID + " is not a valid database id!");

			User user = new User(elem.getAttributeValue("name"), elem
					.getAttributeValue("pwd"));
			User dbUser = dbConfig.getDBUser(dbUserName);
			// User dbUser = (User) dbUserMap.get(dbUserName);
			if (dbUser == null)
				throw new BaseException("Database User <" + dbUserName
						+ "> does not exists!");
			user.setDbUser(dbUser);
			// get privileges also
			setUserPrivileges(elem, user);
			dbConfig.addUser(user);
		}
	}

	protected void setUserPrivileges(Element userElm, User user) {
		Element privsElem = userElm.getChild("privileges");
		if (privsElem == null)
			return;
		List<?> children = privsElem.getChildren("privilege");
		for (Iterator<?> it = children.iterator(); it.hasNext();) {
			Element privElem = (Element) it.next();
			if (privElem.getAttributeValue("name").equals("all")) {

				for (Iterator<Privilege> it2 = privilegesMap.values()
						.iterator(); it2.hasNext();) {
					user.addPrivilege(it2.next());
				}
				break;
			}
			Privilege priv = privilegesMap.get(privElem
					.getAttributeValue("name"));
			if (priv != null)
				user.addPrivilege(priv);
		}
	}

	public UserInfo getDefaultUser(String dbID, UserInfo ui)
			throws AuthenticationException {
		// a simple security check with the assumption that if the passed ui is
		// a valid user info
		// possible it is safe the return the default user information for
		// database connection
		DBConfig dbConfig = dbConfigMap.get(dbID);
		log.info("dbID=" + dbID + " dbConfig=" + dbConfig);
		User u = dbConfig.getUser("admin");
		if (u.getAvailableTables() == null || u.getAvailableTables().isEmpty()) {
			// occurs if the default user from a secondary database is requested
			prepareAvailableTables(u, dbID);
		}

		return new UserInfo("admin", null, u.getAvailableTables());
	}

	public User getRestrictedUser(String dbID) {
		DBConfig dbConfig = dbConfigMap.get(dbID);
		log.info("dbID=" + dbID + " dbConfig=" + dbConfig);

		User guestUser = dbConfig.getUser("guest");
		if (guestUser == null) {
			guestUser = new User("guest", "");
			guestUser.setDbUser(getDefaultDBUser(dbConfig));
			dbConfig.addUser(guestUser);

			if (guestUser.getAvailableTables() == null
					|| guestUser.getAvailableTables().isEmpty()) {
				// occurs if the default user from a secondary database is
				// requested
				prepareAvailableTables(guestUser, dbID);
			}
		}
		return guestUser;
	}

	private User getDefaultDBUser(DBConfig dbConfig) {
		// just return the first dbUser encountered as the default user
		for (User dbUser : dbConfig.getDBUserMap().values()) {
			return dbUser;
		}
		return null;
	}

	/**
	 * simple password based authentication (mainly relies on SSL)
	 *
	 * @param user
	 * @param pwd
	 * @throws AuthenticationException
	 *
	 */
	public UserInfo authenticate(String user, String pwd, String dbID)
			throws AuthenticationException {
		DBConfig dbConfig = dbConfigMap.get(dbID);
		if (dbConfig == null) {
			throw new AuthenticationException("Unknown dbID:" + dbID);
		}
		User u = dbConfig.getUser(user);
		if (u == null) {
			// throw new UnknownUserException("errors.notrecognized",user);
			throw new UnknownUserException("User '" + user
					+ "' is not recognized!");
		}
		if (!u.getPwd().equals(pwd)) {
			// throw new AuthenticationException("errors.invalid","Password
			// supplied");
			throw new AuthenticationException();
		}

		// find all available tables for this user and cache them
		// for the DAO adapters so that they can fetch from the correct view or
		// table
		// and return appropriate messages for the hidden columns
		prepareAvailableTables(u, dbID);

		return new UserInfo(user, null, u.getAvailableTables());
	}

	public UserInfo authenticateAnonymous(String email, String dbID)
			throws AuthenticationException {
		final String PUBLIC_USER = "public";
		DBConfig dbConfig = dbConfigMap.get(dbID);
		if (dbConfig == null) {
			throw new AuthenticationException("Unknown dbID:" + dbID);
		}
		// check for email format (very primitive)
		if (email.indexOf("@") == -1) {
			throw new AuthenticationException("Not a valid email address");
		}

		User u = dbConfig.getUser(PUBLIC_USER);
		if (u == null)
			throw new UnknownUserException("errors.notrecognized", PUBLIC_USER);

		// find all available tables for this user and cache them
		// for the DAO adapters so that they can fetch from the correct view or
		// table and return appropriate messages for the hidden columns
		prepareAvailableTables(u, dbID);

		return new UserInfo(PUBLIC_USER, email, null, u.getAvailableTables());
	}

	protected void prepareAvailableTables(User user, String dbID) {
		Connection con = null;
		IDBPoolService pool = null;
		if (!user.getAvailableTables().isEmpty())
			return;
		try {
			pool = MinimalServiceFactory.getPoolService(dbID);
			con = pool.getConnection(user.getName());
			Map<String, String> dbMasterTableMap = masterTableMap.get(dbID);
			for (String tableName : dbMasterTableMap.keySet()) {
				if (canAccessTable(con, tableName)) {
					user.addAvailableTable(tableName);
				} else {
					log.error(user.getName() + " cannot access table: "
							+ tableName);
				}
			}
		} catch (Exception x) {
			x.printStackTrace();
			log.error(x, x);
		} finally {
			try {
				log.info("****** Releasing connection " + user.getName()
						+ " con=" + con + " dbID=" + dbID);
				pool.releaseConnection(user.getName(), con);
			} catch (Exception ex) {
				log.error(ex, ex);
			}
		}
	}

	public boolean isAuthorized(UserInfo userInfo, String dbID, Object action) {
		if (!getCurrentDBConfig().getId().equals(dbID)) {
			return false;
		}
		User u = getCurrentDBConfig().getUser(userInfo.getName());
		if (u == null)
			return false;
		String privName = (String) action;
		return u.hasPrivilege(privName);
	}

	public Map<String, User> getAllUsers(String dbID) {
		return dbConfigMap.get(dbID).getUserMap();
	}

	public Map<String, User> getAllNamedUsers(String dbID) {
		return dbConfigMap.get(dbID).getDBUserMap();
	}

	/**
	 *
	 * @param dbConfig
	 * @throws java.lang.SecurityException
	 */
	public synchronized void updateUserPoolForDatabase(DBConfig dbConfig)
			throws SecurityException {
		DBConfig dbc = dbConfigMap.get(dbConfig.getId());
		if (dbc == null) {
			dbConfigMap.put(dbConfig.getId(), dbConfig);
		} else {
			synchronized (dbConfigMap) {
				updateUsers(dbc, dbConfig);
			}

			// find removed users

			// find added users
			// find updated user info
		}
	}

	private void updateUsers(DBConfig origDBConfig, DBConfig dbConfig) {
		for (User dbUser : dbConfig.getDBUserMap().values()) {
			User origDBUser = origDBConfig.getDBUser(dbUser.getName());
			/** @todo update available tables */
			if (origDBUser == null) {
				// add the new user
				origDBConfig.addDbUser(dbUser);
			} else {
				if (!origDBUser.isSame(dbUser)) {
					// just replaces the old user info with the updated one
					origDBConfig.addDbUser(dbUser);
				}
			}
		}
	}

	/**
	 * returns the actual <code>DBConfig</code> hash table keyed by the
	 * database ID
	 *
	 * @return the actual <code>DBConfig</code> hash table
	 */
	public Map<String, DBConfig> getDBConfigMap() {
		return dbConfigMap;
	}

	public static void main(String[] args) {
		SimpleSecurityService sss = null;
		try {
			sss = SimpleSecurityService
					.getInstance(
							"/data1/opt/tomcat-4.1.24/webapps/clinical/WEB-INF/users.xml",
							"/home/bozyurt/dev/java/clinical/conf/users.xsd");
			Map<String, User> users = sss.getAllUsers("ucsd_mbirn");
			for (String userName : users.keySet()) {
				System.out.println(userName);
			}

		} catch (Exception x) {
			x.printStackTrace();
		}
	}
}
