package clinical.utils;

import java.sql.*;
import java.text.*;
import java.util.*;
import java.util.Date;

/**
 *
 *  A collection of time stamp, date arithmetic, formatting utility methods.
 *
 *  @author I. Burak Ozyurt
 *  @version $Id: DateTimeUtils.java 88 2009-08-17 23:02:38Z bozyurt $
 */
public class DateTimeUtils {
  static SimpleDateFormat tf = new SimpleDateFormat("HH:mm");
  static SimpleDateFormat df = new SimpleDateFormat("MM/dd/yyyy");
  static SimpleDateFormat df2 = new SimpleDateFormat("yyyy-MM-dd");
  
  static SimpleDateFormat tsf =  new SimpleDateFormat("MM/dd/yyyy HH:mm");
  static SimpleDateFormat tsf2 =  new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

  static {
    tf.setLenient(false);
    df.setLenient(false);
    df2.setLenient(false);
    tsf.setLenient(false);
    tsf2.setLenient(false);
  }

  protected DateTimeUtils() {}

  /**
   * Given a date return a string formatted as mm/dd/yyyy.
   * @param date the <code>java.util.Date</code> to be formatted
   * @return the date formatted as mm/dd/yyyy
   */
  public static String formatDate(Date date) {
     return df.format(date);
  }

  /**
   * Given a long containing a time value return a string formatted as mm/dd/yyyy hh:mm.
   * @param time a long containing a time value
   * @return formatted as mm/dd/yyyy hh:mm
   */
  public static String formatDate(long time) {
    return tsf.format( new Date(time) );
  }
  /**
   * Given a time stamp return a string formatted as mm/dd/yyyy.
   *
   * @param ts time stamp
   * @return the date formatted as mm/dd/yyyy
   */
   public static String formatTimestampToDate(Timestamp ts) {
     return df.format( new Date(ts.getTime()) );
   }

   /**
   * Given a time stamp return a string formatted as mm/dd/yyyy hh:mm.
   *
   * @param ts time stamp
   * @return the date formatted as mm/dd/yyyy hh:mm
   */

   public static String formatTimestamp(Timestamp ts) {
     return tsf.format( new Date(ts.getTime()) );
   }

 /**
   * Given a string holding a date in the <code>mm/dd/yyyy</code> format, returns
   * the corresponding  <code>java.util.Date</code> object.
   *
   * @param dateString the date in the <code>mm/dd/yyyy</code> format
   * @return a <code>java.util.Date</code> object initialized from the specified dateString
   */
  public static Date toDate(String dateString) {
     try {
         return df.parse(dateString);
     } catch(ParseException pe) {
        return null;
     }
  }


  /**
   * Given a time stamp and a time string in the format of hh:mm (for example 13:05)
   * returns a timestamp resulting from the addition of the old time stamp with the time.
   *
   * @param oldTs old time stamp
   * @param time a string with military time format hours:minutes (for example 13:05)
   * @return the new time stamp resulting from the addition of the time with the old time stamp.
   * If the time is not valid it returns null
   *
   * @see Timestamp
   */
  public static Timestamp prepareTimeStamp(Timestamp oldTs, String time) {
    GregorianCalendar cal = new GregorianCalendar();
    cal.setTime( new Date(oldTs.getTime()) );

    int month = cal.get(Calendar.MONTH);
    int day = cal.get( Calendar.DAY_OF_MONTH);
    int year = cal.get( Calendar.YEAR );
    int hour = -1;
    int min = -1;
    try {
      Date d = tf.parse(time);
      cal.setTime(d);
      hour = cal.get( Calendar.HOUR_OF_DAY);
      min = cal.get( Calendar.MINUTE);
    } catch(ParseException pe) {
        return null;
    }
    if ( min != -1 && hour != -1) {
        cal.set(year, month, day, hour, min);
    } else {
      cal.set(year, month, day);
    }

     return new Timestamp( cal.getTime().getTime() );
  }

  /**
   * Given a string holding a date in the <code>mm/dd/yyyy</code> format, returns
   * the corresponding time stamp
   *
   * @param ts the date in the <code>mm/dd/yyyy</code> format
   * @return the time stamp
   *
   * @see Timestamp
   */
  public static Timestamp toTimeStamp(String ts) {
     try {
         return new Timestamp( df.parse(ts).getTime() );
     } catch(ParseException pe) {
        return null;
     }
  }
  
  public static Timestamp toTimeStampTablet(String ts) {
	     try {
	         return new Timestamp( df2.parse(ts).getTime() );
	     } catch(ParseException pe) {
	        return null;
	     }
	  }

  /**
   * Given a time stamp returns the date in <code>mm/dd/yyyy</code> format.
   *
   * @param ts a time stamp
   * @return the date in <code>mm/dd/yyyy</code> format.
   * @see Timestamp
   */
  public static String fromTimestamp(Timestamp ts) {
    return df.format( new Date(ts.getTime()) );
  }

  /**
   * Given a time stamp returns the time portion of it in the <code>hh:mm</code> format
   * @param ts a time stamp
   * @return time in the <code>hh:mm</code> format
   * @see Timestamp
   */
  public static String extractTimePart(Timestamp ts) {
    return tf.format( new Date( ts.getTime()) );
  }

  /**
   *
   * @param ts the time stamp in the <code>mm/dd/yyyy hh:mm</code> format
   * @return
   */
  public static Date convertTimestamp(String ts) {
     try {
         return new Date( tsf.parse(ts).getTime() );
     } catch(ParseException pe) {
        return null;
     }
  }
  
  /**
  *
  * @param ts the time stamp in the <code>mm/dd/yyyy hh:mm:ss</code> format
  * @return
  */
 public static Date convertTimestampTablet(String ts) {
    try {
        return new Date( tsf2.parse(ts).getTime() );
    } catch(ParseException pe) {
       return null;
    }
 }



  public static void main(String[] args) {
    DateTimeUtils.prepareTimeStamp(new Timestamp( System.currentTimeMillis()) , "3:45");
    System.out.println("" + DateTimeUtils.extractTimePart(new Timestamp( System.currentTimeMillis()) ) );
  }

}
