package clinical.web;

import java.io.IOException;
import java.io.InputStream;
import java.sql.Connection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import net.sf.ehcache.Cache;
import net.sf.ehcache.CacheManager;
import net.sf.ehcache.Element;

import clinical.utils.FileUtils;
import clinical.utils.GenUtils;
import clinical.web.common.UserInfo;
import clinical.web.common.query.OracleSQLDialect;
import clinical.web.common.query.PostgresSQLDialect;
import clinical.web.common.security.DBConfig;
import clinical.web.exception.DBPoolServiceException;
import clinical.web.services.AppConfigService;
import clinical.web.services.DBPoolService;
import clinical.web.services.SimpleSecurityService;

/**
 * @author I. Burak Ozyurt
 * @version $Id: ConnectionSupportMixin.java,v 1.4 2008/01/12 01:00:32 bozyurt
 *          Exp $
 */
public class ConnectionSupportMixin {
	protected Properties props;
	protected DBPoolService dbPoolService;
	protected UserInfo ui;
	protected String dbID;
	protected String dbType;
	protected boolean skipQueryProcessorCache = false;
	protected String[] dbIDs;
	protected ISQLDialect sqlDialect;
	protected String usersFile;
	protected InputStream is;
	protected boolean doPrepareTableCache = true;

	public ConnectionSupportMixin(String propsFile,
			boolean skipQueryProcessorCache) throws IOException {
		this.skipQueryProcessorCache = skipQueryProcessorCache;
		props = GenUtils.loadProperties(propsFile);
	}

	public ConnectionSupportMixin(String usersXmlFile) throws IOException {
		this.is = ConnectionSupportMixin.class.getClassLoader()
				.getResourceAsStream(usersXmlFile);
	}

	public void startup() throws Exception {
		System.out.println("---- startup --------");
		String adminUser = null;
		String adminPwd = null;
		System.out.println("Creating cache manager");
		CacheManager.create();
		SimpleSecurityService secService = null;
		if (props != null) {
			String driverClass = props.getProperty("fbirn.driver_class");
			Class.forName(driverClass);

			this.usersFile = props.getProperty("fbirn.users_file");
			adminUser = props.getProperty("fbirn.admin_user");
			adminPwd = props.getProperty("fbirn.admin_pwd");
			this.dbID = props.getProperty("fbirn.dbid");

			secService = SimpleSecurityService.getInstance(usersFile);
		} else if (is != null) {
			secService = SimpleSecurityService.getInstance(is);
			this.dbID = secService.getDefaultDBID();
			DBConfig dbConfig = secService.getDBConfig(this.dbID);
			adminUser = "admin";
			adminPwd = dbConfig.findUser("admin").getPwd();
		}

		this.dbIDs = secService.getAllDBIDs();
		String defaultDBID = secService.getDefaultDBID();
		this.dbType = secService.getDBConfig(defaultDBID).getDbType();
		if (dbType.equals(DBConfig.ORACLE)) {
			sqlDialect = new OracleSQLDialect();
		} else if (dbType.equals(DBConfig.POSTGRES)) {
			sqlDialect = new PostgresSQLDialect();
		} else {
			throw new RuntimeException("Not a supported database dialect:"
					+ dbType);
		}
		System.out.println("getting pool service...");
		dbPoolService = DBPoolService.getInstance(secService, secService
				.getDBConfigMap());
		System.out.println("got pool service...");

		DAOFactory.bootstrap();
		//System.out.println("initialized DAOFactory");

		Map<String, String> dbID2SiteIDMap = Collections
		.synchronizedMap(new HashMap<String, String>(17));

		DBPoolService poolService = null;
		// startup AppConfigService
		for (Object element : secService.getDBConfigMap().values()) {
			DBConfig dbConfig = (DBConfig) element;

			poolService = DBPoolService.getInstance(dbConfig.getId());
			poolService.startup();

			if (dbConfig.getId().equals(defaultDBID)) {
				AppConfigService.getInstance(poolService, defaultDBID);
			}

			dbID2SiteIDMap.put(dbConfig.getId(), dbConfig.getSiteID());
		}
		// cache dbid 2 siteid map
		Cache cache = CacheManager.getInstance().getCache("remote");
		Element el = new Element(Constants.DBID2SITEID_MAP, dbID2SiteIDMap);
		el.setEternal(true);
		cache.put(el);

		//System.out.println("after AppConfigService startup");

		for (Object element : secService.getDBConfigMap().values()) {
			DBConfig dbConfig = (DBConfig) element;
			poolService = DBPoolService.getInstance(dbConfig.getId());
			if (isDoPrepareTableCache()) {
				// System.out.println("prepareTableCache");
				secService.prepareTableCache(dbConfig.getId());
			}
		}
		// System.out.println("authenticating");
		ui = secService.authenticate(adminUser, adminPwd, dbID);

		System.out.println("------------ end startup -----------------");
	}

	public void shutdown() throws Exception {
		if (dbPoolService != null)
			try {
				dbPoolService.shutdown();
				dbPoolService = null;
			} catch (Exception x) {
			}
		FileUtils.close(is);
	}

	public DBPoolService getDbPoolService() {
		return dbPoolService;
	}

	public String getDbID() {
		return dbID;
	}

	public String[] getDbIDs() {
		return dbIDs;
	}

	public UserInfo getUi() {
		return ui;
	}

	public String getProperty(String propName) {
		return props.getProperty(propName);
	}

	public Connection getConnection() throws DBPoolServiceException {
		return dbPoolService.getConnection(ui.getName());
	}

	public void releaseConnection(Connection con) throws DBPoolServiceException {
		dbPoolService.releaseConnection(ui.getName(), con);
	}

	public ISQLDialect getSqlDialect() {
		return sqlDialect;
	}

	public String getUsersFile() {
		return usersFile;
	}

	public boolean isDoPrepareTableCache() {
		return doPrepareTableCache;
	}

	public void setDoPrepareTableCache(boolean doPrepareTableCache) {
		this.doPrepareTableCache = doPrepareTableCache;
	}

	public String getDbType() {
		return dbType;
	}
}
