package clinical.web.common.query;

import java.io.Serializable;
import java.util.Date;

/**
 * A search predicate has the following form;
 * 
 * <pre>
 *  &lt;variable&gt;  &lt;relational operator&gt; &lt;value&gt;
 * </pre>
 * 
 * where the extended set of relational operators are;
 * <ul>
 * <li>=</li>
 * <li>&gt; (for numbers only) </li>
 * <li>&gt;= (for numbers only)</li>
 * <li>&lt; (for numbers only)</li>
 * <li>&lt;= (for numbers only)</li>
 * <li>starts with - (for strings only) returns true if the variable starts
 * with the string denoted by the <code>value</code> </li>
 * <li>ends with - (for strings only) returns true if the variable ends with
 * the string denoted by the <code>value</code> </li>
 * <li>any - (for strings only) exactly matches any of the comma delimited
 * values supplied</li>
 * </ul>
 * 
 * @version $Id: SearchPredicate.java 91 2009-08-17 23:38:26Z bozyurt $
 * @author I. Burak Ozyurt
 */

public class SearchPredicate implements Serializable {
	private static final long serialVersionUID = -2209022818025333399L;
	private Object attribute;
	private Object value;
	private int operator;
	private int type;

	public final static int EQUAL = 1;
	public final static int GREATER = 2;
	public final static int LESS = 3;
	public final static int NOT_EQUAL = 4;
	public final static int GREATER_EQUAL = 5;
	public final static int LESS_EQUAL = 6;
	public final static int BETWEEN = 7;

	public final static int STARTS_WITH = 100;
	public final static int ENDS_WITH = 101;
	public final static int ANY = 102;

	public final static int INTEGER = 1000;
	public final static int STRING = 1001;
	public final static int FLOAT = 1002;
	public final static int BOOLEAN = 1003;

	public final static int DATE = 1004;

	public SearchPredicate() {
		operator = EQUAL;
		type = INTEGER;
	}

	public SearchPredicate(Object attribute, Object value, int operator,
			int type) {
		this.attribute = attribute;
		this.value = value;
		this.operator = operator;
		this.type = type;
	}

	// ---------------------- setters --------------
	public void setAttribute(Object newAttribute) {
		this.attribute = newAttribute;
	}

	public void setValue(Object newValue) {
		this.value = newValue;
	}

	public void setOperator(int newOperator) {
		this.operator = newOperator;
	}

	public void setType(int newType) {
		this.type = newType;
	}

	// ---------------------- getters --------------
	public Object getAttribute() {
		return this.attribute;
	}

	public Object getValue() {
		return this.value;
	}

	public int getOperator() {
		return this.operator;
	}

	public int getType() {
		return this.type;
	}

	public int hashCode() {
		int hc = attribute.hashCode() ^ type ^ value.hashCode() ^ operator;
		return hc;
	}

	public boolean equals(Object other) {
		if (other == null || !(other instanceof SearchPredicate))
			return false;
		SearchPredicate osp = (SearchPredicate) other;
		return attribute.equals(osp.attribute) && value.equals(osp.value)
				&& type == osp.type && operator == operator;
	}

	public static class DateRange {
		private Date lowBound;
		private Date uppBound;

		public DateRange(Date lowBound, Date uppBound) {
			this.lowBound = lowBound;
			this.uppBound = uppBound;
		}

		public void setLowBound(Date newLowBound) {
			this.lowBound = newLowBound;
		}

		public void setUppBound(Date newUppBound) {
			this.uppBound = newUppBound;
		}

		public Date getLowBound() {
			return this.lowBound;
		}

		public Date getUppBound() {
			return this.uppBound;
		}

		public int hashCode() {
			return lowBound.hashCode() ^ uppBound.hashCode();
		}

		public boolean equals(Object other) {
			if (other == null || !(other instanceof Range))
				return false;
			Range otr = (Range) other;
			return lowBound.equals(otr.lowBound)
					&& uppBound.equals(otr.uppBound);
		}
	}

	/**
	 * 
	 * Encapsulates the boundary values of a numeric range.
	 * 
	 * @author I. Burak Ozyurt
	 * 
	 */
	public static class Range {
		private Number lowBound;
		private Number uppBound;

		public Range(Number lowBound, Number uppBound) {
			this.lowBound = lowBound;
			this.uppBound = uppBound;
		}

		// ---------------------- setters --------------
		public void setLowBound(Number newLowBound) {
			this.lowBound = newLowBound;
		}

		public void setUppBound(Number newUppBound) {
			this.uppBound = newUppBound;
		}

		// ---------------------- getters --------------
		public Number getLowBound() {
			return this.lowBound;
		}

		public Number getUppBound() {
			return this.uppBound;
		}

		public int hashCode() {
			return lowBound.hashCode() ^ uppBound.hashCode();
		}

		public boolean equals(Object other) {
			if (other == null || !(other instanceof Range))
				return false;
			Range otr = (Range) other;
			return lowBound.equals(otr.lowBound)
					&& uppBound.equals(otr.uppBound);
		}
	}
	
	
	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();
		sb.append("SearchPredicate::[");
		sb.append("attribute=").append(attribute);
		sb.append(",operator=").append(operator);
		sb.append(",value=").append(value);
		sb.append(",type=").append(type);
		sb.append(']');
		return sb.toString();
	}
	
}
