package clinical.web.services;

import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.Statement;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import net.nbirn.mediator.client.Mediator;
import net.nbirn.mediator.client.Query;
import net.nbirn.mediator.client.Results;
import net.nbirn.mediator.client.ServiceNames;
import net.nbirn.mediator.client.Tuple;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import sun.security.util.BigInt;

import clinical.cache.CacheUtils;
import clinical.server.dao.AssessmentDAO;
import clinical.server.dao.AssessmentscoreDAO;
import clinical.server.dao.ExpcomponentDAO;
import clinical.server.dao.ExpsegmentDAO;
import clinical.server.dao.ExpstudyDAO;
import clinical.server.dao.ResearchgroupDAO;
import clinical.server.dao.StoredqueryDAO;
import clinical.server.dao.SubjexperimentDAO;
import clinical.server.vo.Assessment;
import clinical.server.vo.Assessmentscore;
import clinical.server.vo.Databaseuser;
import clinical.server.vo.Expcomponent;
import clinical.server.vo.Expsegment;
import clinical.server.vo.Expstudy;
import clinical.server.vo.Humansubject;
import clinical.server.vo.Researchgroup;
import clinical.server.vo.Site;
import clinical.server.vo.Storedassessment;
import clinical.server.vo.Storedquery;
import clinical.server.vo.Subjexperiment;
import clinical.server.vo.Tableid;
import clinical.utils.Assertion;
import clinical.utils.DateTimeUtils;
import clinical.utils.GenUtils;
import clinical.web.Constants;
import clinical.web.DAOFactory;
import clinical.web.DBUtils;
import clinical.web.IAssessmentService;
import clinical.web.IRemoteDBServices;
import clinical.web.ISQLDialect;
import clinical.web.ServiceFactory;
import clinical.web.common.AssessmentMapping;
import clinical.web.common.IDBCache;
import clinical.web.common.IDBPoolService;
import clinical.web.common.ISecurityService;
import clinical.web.common.UserInfo;
import clinical.web.common.query.AssessmentQueryBuilder;
import clinical.web.common.query.MediatedAssessmentQueryBuilder;
import clinical.web.common.query.MultiSiteAssessmentQueryBuilder;
import clinical.web.common.query.Operator;
import clinical.web.common.query.QueryUtils;
import clinical.web.common.query.TSQLProcessor;
import clinical.web.common.security.User;
import clinical.web.common.vo.AsScoreInfo;
import clinical.web.common.vo.AssessmentSelectionInfo;
import clinical.web.download.SubjectImageDataBundle;
import clinical.web.exception.BaseException;
import clinical.web.exception.DBPoolServiceException;
import clinical.web.helpers.SubCorticalVarInfo;
import clinical.web.services.AssessmentServiceHelper.SiteAsiInfo;
import clinical.web.vo.AssessmentInfo;
import clinical.web.vo.QuerySummary;
import clinical.web.vo.RangeInfo;
import clinical.web.vo.SubjectAsScoreValueSummary;
import clinical.web.vo.VisitDateQuery;

import com.jamonapi.Monitor;
import com.jamonapi.MonitorFactory;
import com.sun.corba.se.impl.orbutil.closure.Constant;

/**
 * Implements <code>IAssessmentService</code> interface.
 *
 * @version $Id: DebugAssessmentService.java,v 1.24 2006/06/19 22:42:27 bozyurt
 *          Exp $
 * @author I. Burak Ozyurt
 */

public class DebugAssessmentService implements IAssessmentService {	
	private IDBPoolService pool = null;
	private AssessmentDAO asDAO = null;
	private String theDBID;
	private ISQLDialect sqlDialect;
	private Log log = LogFactory.getLog("clinical");

	public DebugAssessmentService(String dbID) throws BaseException {
		pool = ServiceFactory.getPoolService(dbID);
		asDAO = DAOFactory.createAssessmentDAO(dbID);
		theDBID = dbID;
		sqlDialect = ServiceFactory.getSQLDialect(theDBID);
	}

	public DebugAssessmentService() {
	}

	/** returns all assessments available from persistent store */
	public List<Assessment> getAllAssessments(UserInfo userInfo)
			throws Exception {
		java.sql.Connection con = null;
		try {
			con = pool.getConnection(userInfo.getName());
			List<Assessment> assessments = asDAO.find(con, new Assessment());
			return assessments;
		} finally {
			pool.releaseConnection(userInfo.getName(), con);
		}
	}

	/**
	 * returns a list of Assessmentscore objects for Assessment
	 */
	public List<Assessmentscore> getScoresForAssessment(UserInfo userInfo,
			Assessment assessment) throws Exception {
		java.sql.Connection con = null;
		try {
			con = pool.getConnection(userInfo.getName());
			AssessmentscoreDAO dao = DAOFactory
					.createAssessmentscoreDAO(theDBID);
			Assessmentscore criteria = new Assessmentscore();
			criteria.setAssessmentid(assessment.getAssessmentid());

			List<?> scores = dao.find(con, criteria);
			List<Assessmentscore> scoreList = new ArrayList<Assessmentscore>(
					scores.size());
			for (Object item : scores) {
				Assessmentscore score = (Assessmentscore) item;
				scoreList.add(score);
			}
			return scoreList;
		} finally {
			pool.releaseConnection(userInfo.getName(), con);
		}
	}

	public List<AssessmentSelectionInfo> getAssessmentsWithScores(UserInfo ui,
			List<AssessmentSelectionInfo> asiList) throws Exception {
		if (asiList == null || asiList.isEmpty())
			return asiList;
		java.sql.Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			TSQLProcessor tsql = new TSQLProcessor(ServiceFactory
					.getSQLDialect(theDBID));
			StringBuilder sb = new StringBuilder(512);
			sb.append("select s.scorename, s.scoretype, s.parentscore,");
			sb.append("s.scorelevel, s.assessmentid ");
			sb.append("from Assessmentscore as s where s.assessmentid in (");
			for (Iterator<AssessmentSelectionInfo> it = asiList.iterator(); it
					.hasNext();) {
				AssessmentSelectionInfo asi = it.next();
				sb.append(asi.getAssessmentID());
				if (it.hasNext())
					sb.append(',');
			}
			sb.append(")");
			log.info("getAssessmentsWithScores:: query \n"
					+ GenUtils.applyWordWrapping(sb.toString()));
			List<?> results = tsql.executeQuery(con, sb.toString());
			Map<BigDecimal, AssessmentSelectionInfo> asiMap = new HashMap<BigDecimal, AssessmentSelectionInfo>();
			for (AssessmentSelectionInfo asi : asiList) {
				asiMap.put(asi.getAssessmentID(), asi);
			}
			for (Object item : results) {
				Assessmentscore sc = (Assessmentscore) item;
				AssessmentSelectionInfo asi = asiMap.get(sc.getAssessmentid());
				AsScoreInfo si = new AsScoreInfo(sc.getAssessmentid());
				si.setName(sc.getScorename());
				si.setType(sc.getScoretype());
				si.setParentScore(sc.getParentscore());
				si.setLevel(sc.getScorelevel() == null ? -1 : sc
						.getScorelevel().intValue());
				asi.addScore(si);
			}
			return asiList;
		} finally {
			pool.releaseConnection(ui.getName(), con);
		}
	}

	public AssessmentInfo getAssessmentWithScores(UserInfo userInfo,
			int assessmentID) throws Exception {
		java.sql.Connection con = null;
		AssessmentInfo asi = null;
		try {
			con = pool.getConnection(userInfo.getName());
			AssessmentDAO asDAO = DAOFactory.createAssessmentDAO(theDBID);
			AssessmentscoreDAO dao = DAOFactory
					.createAssessmentscoreDAO(theDBID);

			Assessment ac = new Assessment();
			ac.setAssessmentid(new BigDecimal(assessmentID));
			List<?> asList = asDAO.find(con, ac);
			if (asList.isEmpty())
				return null;
			Assessment a = (Assessment) asList.get(0);
			asi = new AssessmentInfo(a.getAssessmentid().intValue(), a
					.getName(), a.getModtime(), -1, -1);

			Assessmentscore criteria = new Assessmentscore();
			criteria.setAssessmentid(a.getAssessmentid());
			List<?> scores = dao.find(con, criteria);
			for (Iterator<?> iter = scores.iterator(); iter.hasNext();) {
				Assessmentscore sc = (Assessmentscore) iter.next();
				int parentAsID = (sc.getParentasid() != null) ? sc
						.getParentasid().intValue() : 0;
				AssessmentInfo.ScoreInfo si = new AssessmentInfo.ScoreInfo(sc
						.getScorename(), sc.getScoretype(), sc.getScorelevel()
						.intValue(), parentAsID, sc.getParentscore(), sc
						.getScoresequence().intValue());
				asi.addScore(si);
			}

		} finally {
			pool.releaseConnection(userInfo.getName(), con);
		}
		return asi;
	}

	public List<SubjectAsScoreValueSummary> queryForScores(UserInfo userInfo,
			Operator op, List<AssessmentSelectionInfo> asiList,
			String queryScope, String primarySiteID) throws Exception {
		return queryForScores(userInfo, op, asiList, null, queryScope,
				primarySiteID, null);
	}
	
	public List<SubjectAsScoreValueSummary> queryForScores(UserInfo userInfo,
			Operator op, List<AssessmentSelectionInfo> asiList,
			List<Integer> experimentIds, String queryScope, String primarySiteID)
			throws Exception{
		return queryForScores(userInfo, op, asiList, experimentIds, queryScope, primarySiteID, null);
	}
			
	public List<SubjectAsScoreValueSummary> queryForScores(UserInfo userInfo,
			Operator op, List<AssessmentSelectionInfo> asiList,
			List<Integer> experimentIds, String queryScope, String primarySiteID,
			String queryType)
			throws Exception {
		return queryForScores(userInfo, op, asiList, experimentIds, queryScope, primarySiteID, queryType, Constants.IsValidated);
	}

	public List<SubjectAsScoreValueSummary> queryForScores(UserInfo userInfo,
			Operator op, List<AssessmentSelectionInfo> asiList,
			List<Integer> experimentIds, String queryScope, String primarySiteID,
			String queryType, String isValidated)
			throws Exception {
		AssessmentQueryBuilder aqb = new AssessmentQueryBuilder(sqlDialect,
				asiList, experimentIds, isValidated);
		// prepare the query
		aqb.visit(op);

		List<SubjectAsScoreValueSummary> summaryList = new LinkedList<SubjectAsScoreValueSummary>();
		java.sql.Connection con = null;
		Statement st = null;
		ResultSet rs = null;
		Map<String, SubjectAsScoreValueSummary> summaryMap = new HashMap<String, SubjectAsScoreValueSummary>();
		Monitor primary = null;
		try {
			primary = MonitorFactory.startPrimary("queryForScores1");
			Monitor conMon = MonitorFactory
					.start("queryForScores1.getConnection");
			con = pool.getConnection(userInfo.getName());
			conMon.stop();
			st = con.createStatement();
			// log.info("query=" + aqb.getQuery());
			Monitor queryMon = MonitorFactory.start("queryForScores1.doQuery");
			log.info("queryForScores = " + aqb.getQuery());
			String query = aqb.getQuery();
			if(queryType!=null){
				if(queryType.equals(Constants.ASSESSMENT_OR_IMAGING)){
				String imagingQuery = "SELECT c.subjectid, null, 1, null, -1, null, null, " + 
				   "c.NC_EXPERIMENT_UNIQUEID, c.modtime, null " + 
				   "FROM nc_rawdata c, nc_dataobject d, nc_dataobjecttype e " +
				   "WHERE c.uniqueid = d.dataid " +
				   "AND d.objecttypeid=e.uniqueid ";
				query = query + " UNION " + imagingQuery;
				}
			}
			rs = st.executeQuery(query);
			while (rs.next()) {
				AssessmentServiceHelper.prepareSubjectAsScoreSummaryRec(
						summaryList, rs, summaryMap);
			}
			queryMon.stop();
			summaryMap = null;

			// attach site information
			String dbID = CacheUtils.getDBID4SiteID(primarySiteID);
			IDBCache dbCache = ServiceFactory.getDBCache(dbID);
			List<Site> sites = dbCache.getSites(userInfo, false);
			Map<String, SubjectInfo> siMap = getSubjectInfos(con, summaryList,
					sites, primarySiteID);
			
			//Jinran added, additional three comlumns 'visit name' and 'visit date' and 'studyGroup' in Batch Query wide format
			//Add several more columns "SegmentName", "SegmentDate"
			ExpcomponentDAO dao = DAOFactory.createExpcomponentDAO(dbID);
			Expcomponent expComp = new Expcomponent();
			ExpsegmentDAO expSegDao = DAOFactory.createExpsegmentDAO(dbID);
			Expsegment expSegment = new Expsegment();
			ExpstudyDAO expStudyDao = DAOFactory.createExpstudyDAO(dbID);
			Expstudy expStudy = new Expstudy();
			
			for(SubjectAsScoreValueSummary sasvs : summaryList){
				int componentId = sasvs.getVisitID();
				int expId = sasvs.getExperimentID();
				BigDecimal studyId = null;
				expComp.setComponentid(BigDecimal.valueOf(componentId));
				expComp.setNcExperimentUniqueid(BigDecimal.valueOf(expId));
				expComp.setSubjectid(sasvs.getSubjectID());
				
				List<Expcomponent> expCompList = dao.find(con, expComp);
				if(expCompList!=null && !expCompList.isEmpty()){
					sasvs.setVisitName(expCompList.get(0).getName());
					sasvs.setVisitDate(expCompList.get(0).getTimeStamp().toString());						
				}
				
				expSegment.setComponentid(BigDecimal.valueOf(componentId));
				expSegment.setNcExperimentUniqueid(BigDecimal.valueOf(expId));
				expSegment.setSubjectid(sasvs.getSubjectID());
				expSegment.setSegmentid(BigDecimal.valueOf(sasvs.getSegmentID()));
				List<Expsegment> expSegList = expSegDao.find(con, expSegment);
				if(expSegList!=null && !expSegList.isEmpty()){
					sasvs.setSegmentName(expSegList.get(0).getName());
					sasvs.setSegmentDate(expSegList.get(0).getModtime().toString());
					studyId = expSegList.get(0).getStudyid();
				}
				
				expStudy.setExperimentid(BigDecimal.valueOf(expId));
				expStudy.setSubjectid(sasvs.getSubjectID());
				expStudy.setComponentid(BigDecimal.valueOf(componentId));
				expStudy.setStudyid(studyId);
				List<Expstudy> expStudyList = expStudyDao.find(con, expStudy);
				if(expStudyList!=null && !expStudyList.isEmpty()){
					sasvs.setStudyName(expStudyList.get(0).getName());
					sasvs.setStudyDate(expStudyList.get(0).getModtime().toString());
					sasvs.setStudyId(studyId.toString());
				}
				
				SubjexperimentDAO seDao = DAOFactory.createSubjexperimentDAO(dbID);
				Subjexperiment subjExp = new Subjexperiment();
				ResearchgroupDAO rgDao = DAOFactory.createResearchgroupDAO(dbID);
				Researchgroup rgVO = new Researchgroup();
				subjExp.setNcExperimentUniqueid(BigDecimal.valueOf(expId));
				subjExp.setSubjectid(sasvs.getSubjectID());
				List<Subjexperiment> subjExpList = seDao.find(con, subjExp);
				if(subjExpList!=null && !subjExpList.isEmpty()){
					rgVO.setUniqueid(subjExpList.get(0).getNcResearchgroupUniqueid());
					List<Researchgroup> rgList = rgDao.find(con, rgVO);
					if(rgList!=null && !rgList.isEmpty()){
						sasvs.setStudyGroup(rgList.get(0).getName());						
					}	
				}				
			}				

			ISecurityService secService = ServiceFactory.getSecurityService();
			String siteID = secService.findSiteIDByDbID(theDBID);
			IRemoteDBServices rds = ServiceFactory.getRemoteDBServices();
			Set<VisitDateQuery> vdqSet = new HashSet<VisitDateQuery>();
			for (SubjectAsScoreValueSummary summary : summaryList) {
				VisitDateQuery vdq = new VisitDateQuery(summary.getSubjectID(),
						summary.getVisitID(), summary.getExperimentID());
				vdqSet.add(vdq);
			}
			Map<String, Date> visitDateMap = rds.getVisitDates(siteID,
					new ArrayList<VisitDateQuery>(vdqSet), true);

			for (SubjectAsScoreValueSummary summary : summaryList) {
				SubjectInfo si = siMap.get(summary.getSubjectID());
				summary.setSiteID(si.siteID);
				summary.setSiteName(si.siteName);
				summary.setRemote(si.remote);

				String key = summary.getSubjectID() + "_"
						+ summary.getExperimentID() + "_"
						+ summary.getVisitID();
				Date visitDate = visitDateMap.get(key);
				if (visitDate != null) {
					summary.setTimeStamp(DateTimeUtils.formatDate(visitDate));
				}

			}

		} finally {
			DBUtils.close(st, rs);
			if (con != null)
				pool.releaseConnection(userInfo.getName(), con);
			if (primary != null) {
				primary.stop();
			}
		}

		// FIXME for testing
		/*
		 * List<Object> list = new ArrayList<Object>(2);
		 * list.add(summaryList); list.add(op); GenUtils.serialize(list,
		 * System.getProperty("user.home") + "/query.ser");
		 */

		// post-process the search results filtering segments which do not match
		// exactly
		if(queryType==null || (queryType!=null && !queryType.equals(Constants.ASSESSMENT_OR_IMAGING))){
			if (queryScope.equals(Constants.SEGMENT_SCOPE)) {
				log.info("Using segment scope...");
				summaryList = AssessmentServiceHelper.filterSearchResults(
						summaryList, op);
			} else if (queryScope.equals(Constants.VISIT_SCOPE)) {
				log.info("Using visit scope...");
				summaryList = AssessmentServiceHelper
						.filterSearchResultsGroupedByVisit(summaryList, op);
			}			
		}
		
		return summaryList;
	}

	
	public List<SubjectAsScoreValueSummary> queryForScores(UserInfo userInfo,
			String dbID, String firstDBID,
			Map<Integer, List<AssessmentMapping>> siteAsMap, Operator op,
			List<AssessmentSelectionInfo> asiList, List<Integer> experimentIDs)
			throws Exception {
		/**
		 * TODO assumption all quasi-mediated databases are of the same type,
		 * ie. all Oracle or all postgres needs to be relaxed
		 */

		MultiSiteAssessmentQueryBuilder aqb = null;

		List<SubjectAsScoreValueSummary> summaryList = new LinkedList<SubjectAsScoreValueSummary>();
		java.sql.Connection con = null;
		Statement st = null;
		ResultSet rs = null;
		IDBPoolService aPool = null;
		Map<String, SubjectAsScoreValueSummary> summaryMap = new HashMap<String, SubjectAsScoreValueSummary>();

		Monitor primary = null;
		try {

			aPool = ServiceFactory.getPoolService(dbID);
			primary = MonitorFactory.startPrimary("MSQueryForScores");
			Monitor conMon = MonitorFactory
					.start("MSQueryForScores.getConnection");

			con = aPool.getConnection(userInfo.getName());
			conMon.stop();

			Map<Integer, String> expID2NameMap = null;
			if (experimentIDs != null) {
				expID2NameMap = AssessmentServiceHelper.prepareExpID2NameMap(
						firstDBID, userInfo.getName(), experimentIDs);
			}
			log.info("******** using sqlDialect "
					+ this.sqlDialect.getDialectName());
			aqb = new MultiSiteAssessmentQueryBuilder(this.sqlDialect, asiList,
					siteAsMap, firstDBID, dbID, expID2NameMap);
			// prepare the query
			aqb.visit(op);

			st = con.createStatement();
			log.info("firstDBID=" + firstDBID + " ,query for site(" + dbID
					+ ")=" + aqb.getQuery());
			Monitor queryMon = MonitorFactory.start("MSQueryForScores.doQuery");
			rs = st.executeQuery(aqb.getQuery());
			while (rs.next()) {
				AssessmentServiceHelper.prepareSubjectAsScoreSummaryRec(
						summaryList, rs, summaryMap);
			}
			queryMon.stop();
		} finally {
			DBUtils.close(st, rs);
			Monitor closeCon = MonitorFactory
					.start("MSQueryForScores.closeConnection");
			if (con != null) {
				aPool.releaseConnection(userInfo.getName(), con);
			}
			closeCon.stop();
			if (primary != null) {
				primary.stop();
			}
		}

		return summaryList;
	}


	protected Map<String, SubjectInfo> getSubjectInfos(Connection con,
			List<SubjectAsScoreValueSummary> summaryList, List<Site> sites,
			String primarySiteID) throws Exception {
		Set<String> uniqSubjectIDSet = new HashSet<String>();
		for (SubjectAsScoreValueSummary summary : summaryList) {
			uniqSubjectIDSet.add(summary.getSubjectID());
		}
		Map<BigDecimal, Site> siteMap = new HashMap<BigDecimal, Site>(17);
		Map<String, Site> siteID2SiteMap = new HashMap<String, Site>(17);

		Site primarySite = null;
		for (Site site : sites) {
			siteMap.put(site.getUniqueid(), site);
			siteID2SiteMap.put(site.getSiteid(), site);
			if (site.getSiteid().equals(primarySiteID))
				primarySite = site;
		}
		Assertion.assertNotNull(primarySite);
		ISecurityService secService = ServiceFactory.getSecurityService();


		Map<String, SubjectInfo> subjectMap = new HashMap<String, SubjectInfo>();
		if (uniqSubjectIDSet == null || uniqSubjectIDSet.isEmpty()) {
			return subjectMap;
		}

		TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
		StringBuilder sb = new StringBuilder(512);
		sb.append("select s.subjectid, s.siteid, s.isremote ");
		sb.append("from Humansubject as s where s.subjectid in (");
		for (Iterator<String> it = uniqSubjectIDSet.iterator(); it.hasNext();) {
			String subjectID = it.next();
			sb.append("'").append(subjectID).append("'");
			if (it.hasNext())
				sb.append(',');
		}
		sb.append(')');
		List<?> results = tsp.executeQuery(con, sb.toString());
		for (Iterator<?> it = results.iterator(); it.hasNext();) {
			Humansubject hs = (Humansubject) it.next();
			Site site = siteMap.get(hs.getSiteid());
			if ( site == null) {
				String theSiteID = secService.findSiteIDByDbID(this.theDBID);
				site = siteID2SiteMap.get(theSiteID);
			}
			Assertion.assertNotNull(site);
			boolean isRemote = hs.getIsremote() == null ? false : hs
					.getIsremote();
			SubjectInfo si = new SubjectInfo(hs.getSubjectid(), site
					.getSiteid(), site.getSitename(), isRemote);
			subjectMap.put(si.subjectID, si);
		}
		return subjectMap;
	}

	public static class SubjectInfo {
		String subjectID;
		String siteID;
		String siteName;
		boolean remote;

		public SubjectInfo(String subjectID, String siteID, String siteName,
				boolean remote) {
			super();
			this.subjectID = subjectID;
			this.siteID = siteID;
			this.siteName = siteName;
			this.remote = remote;
		}

	}

	public BatchQueryResult doBatchQuery(UserInfo userInfo,
			List<QuerySummary> qsList,
			List<AssessmentSelectionInfo> asiList) throws Exception {
		List<SubjectAsScoreValueSummary> combinedList = new ArrayList<SubjectAsScoreValueSummary>();

		Map<String, SiteAsiInfo> saiMap = AssessmentServiceHelper
				.findSitesWithData(userInfo, qsList, asiList);
		Map<Integer, List<String>> expSubjectIDListMap = AssessmentServiceHelper
				.prepareExpSubjectIDListMapFromQuerySummaries(qsList);
		retrieveAndCombine(userInfo, expSubjectIDListMap, combinedList, saiMap);

		BatchQueryResult bqr = new BatchQueryResult(combinedList, saiMap);
		return bqr;
	}

	public BatchQueryResult doBatchQuery(UserInfo userInfo,
			List<QuerySummary> qsList,
			List<AssessmentSelectionInfo> asiList,
			String sValidated) throws Exception {
		List<SubjectAsScoreValueSummary> combinedList = new ArrayList<SubjectAsScoreValueSummary>();

		Map<String, SiteAsiInfo> saiMap = AssessmentServiceHelper
				.findSitesWithData(userInfo, qsList, asiList);
		Map<Integer, List<String>> expSubjectIDListMap = AssessmentServiceHelper
				.prepareExpSubjectIDListMapFromQuerySummaries(qsList);
		retrieveAndCombine(userInfo, expSubjectIDListMap, combinedList, saiMap, sValidated);

		BatchQueryResult bqr = new BatchQueryResult(combinedList, saiMap);
		return bqr;
	}
	
	protected void retrieveAndCombine(UserInfo userInfo,
			Map<Integer, List<String>> expSubjectIDListMap,
			List<SubjectAsScoreValueSummary> combinedList,
			Map<String, SiteAsiInfo> saiMap) throws DBPoolServiceException {
		java.sql.Connection con = null;

		for (Iterator<SiteAsiInfo> iter = saiMap.values().iterator(); iter
				.hasNext();) {
			SiteAsiInfo sai = iter.next();
			IDBPoolService poolService = null;
			try {
				poolService = ServiceFactory.getPoolService(sai.getDbID());
				con = poolService.getConnection(Constants.ADMIN_USER);

				List<Storedassessment> saList = AssessmentServiceHelper
						.getStoredAsRecords(con, expSubjectIDListMap, sai
								.getAsiList(), sai.getSqlDialect());
				List<SubjectAsScoreValueSummary> fullList = AssessmentServiceHelper
						.getFullAssessments(con, saList, sai.getSqlDialect());	
				
				//Jinran added, additional two comlumns 'visit name' and 'visit date' in Batch Query wide format
				String dbID = sai.getDbID();
				ExpcomponentDAO dao = DAOFactory.createExpcomponentDAO(dbID);
				Expcomponent expComp = new Expcomponent();
				SubjexperimentDAO seDao = DAOFactory.createSubjexperimentDAO(dbID);
				Subjexperiment subjExp = new Subjexperiment();
				ResearchgroupDAO rgDao = DAOFactory.createResearchgroupDAO(dbID);
				Researchgroup rgVO = new Researchgroup();
				
				for(SubjectAsScoreValueSummary sasvs : fullList){
					int componentId = sasvs.getVisitID();
					int expId = sasvs.getExperimentID();
					expComp.setComponentid(BigDecimal.valueOf(componentId));
					expComp.setNcExperimentUniqueid(BigDecimal.valueOf(expId));
					expComp.setSubjectid(sasvs.getSubjectID());					
					List<Expcomponent> expCompList = dao.find(con, expComp);
					if(expCompList!=null && !expCompList.isEmpty()){
						sasvs.setVisitName(expCompList.get(0).getName());
						sasvs.setVisitDate(expCompList.get(0).getTimeStamp().toString());						
					}
					
					subjExp.setNcExperimentUniqueid(BigDecimal.valueOf(expId));
					subjExp.setSubjectid(sasvs.getSubjectID());
					List<Subjexperiment> subjExpList = seDao.find(con, subjExp);
					if(subjExpList!=null && !subjExpList.isEmpty()){
						rgVO.setUniqueid(subjExpList.get(0).getNcResearchgroupUniqueid());
						List<Researchgroup> rgList = rgDao.find(con, rgVO);
						if(rgList!=null && !rgList.isEmpty()){
							sasvs.setStudyGroup(rgList.get(0).getName());
						}	
					}					
				}				
				
				for(SubjectAsScoreValueSummary sasvs : fullList) {
					sasvs.setSiteID(sai.getSiteID());
				}
				combinedList.addAll(fullList);
			} catch (Exception x) {
				// just log the error and continue with the next site
				log.error(x);
			} finally {
				if (poolService != null && con != null) {
					poolService.releaseConnection(Constants.ADMIN_USER, con);
				}
			}
		}
	}

	protected void retrieveAndCombine(UserInfo userInfo,
			Map<Integer, List<String>> assSubjectIDListMap,
			List<SubjectAsScoreValueSummary> combinedList,
			Map<String, SiteAsiInfo> saiMap, 
			String assessmentId, 
			Map<Integer, List<String>> assExpIDListMap) throws DBPoolServiceException {
		java.sql.Connection con = null;

		for (Iterator<SiteAsiInfo> iter = saiMap.values().iterator(); iter
				.hasNext();) {
			SiteAsiInfo sai = iter.next();
			IDBPoolService poolService = null;
			try {
				poolService = ServiceFactory.getPoolService(sai.getDbID());
				con = poolService.getConnection(Constants.ADMIN_USER);

				List<Storedassessment> saList = AssessmentServiceHelper
						.getStoredAsRecordsNew(con, assSubjectIDListMap, sai
								.getAsiList(), sai.getSqlDialect(), assExpIDListMap, assessmentId);
				List<SubjectAsScoreValueSummary> fullList = AssessmentServiceHelper
						.getAssessmentsOnAsId(con, saList, sai.getSqlDialect(), assessmentId);
				for(SubjectAsScoreValueSummary sasvs : fullList) {
					sasvs.setSiteID(sai.getSiteID());
				}
				combinedList.addAll(fullList);
			} catch (Exception x) {
				// just log the error and continue with the next site
				log.error(x);
			} finally {
				if (poolService != null && con != null) {
					poolService.releaseConnection(Constants.ADMIN_USER, con);
				}
			}
		}
	}

	protected void retrieveAndCombine(UserInfo userInfo,
			Map<Integer, List<String>> expSubjectIDListMap,
			List<SubjectAsScoreValueSummary> combinedList,
			Map<String, SiteAsiInfo> saiMap,
			String sValidated) throws DBPoolServiceException {
		java.sql.Connection con = null;

		for (Iterator<SiteAsiInfo> iter = saiMap.values().iterator(); iter
				.hasNext();) {
			SiteAsiInfo sai = iter.next();
			IDBPoolService poolService = null;
			try {
				poolService = ServiceFactory.getPoolService(sai.getDbID());
				con = poolService.getConnection(Constants.ADMIN_USER);

				List<Storedassessment> saList = AssessmentServiceHelper
						.getStoredAsRecords(con, expSubjectIDListMap, sai
								.getAsiList(), sai.getSqlDialect(), sValidated);
				List<SubjectAsScoreValueSummary> fullList = AssessmentServiceHelper
						.getFullAssessments(con, saList, sai.getSqlDialect(), sValidated);	
				
				//Jinran added, additional two comlumns 'visit name' and 'visit date' in Batch Query wide format
				//add one additional column "study group"
				// add another two columns "segment name" and "segment date"
				String dbID = sai.getDbID();
				ExpcomponentDAO dao = DAOFactory.createExpcomponentDAO(dbID);
				Expcomponent expComp = new Expcomponent();
				SubjexperimentDAO seDao = DAOFactory.createSubjexperimentDAO(dbID);
				Subjexperiment subjExp = new Subjexperiment();
				ResearchgroupDAO rgDao = DAOFactory.createResearchgroupDAO(dbID);
				Researchgroup rgVO = new Researchgroup();
				ExpsegmentDAO esDao = DAOFactory.createExpsegmentDAO(dbID);
				Expsegment expSeg = new Expsegment();
				ExpstudyDAO expStudyDao = DAOFactory.createExpstudyDAO(dbID);
				Expstudy expStudy = new Expstudy();
				
				for(SubjectAsScoreValueSummary sasvs : fullList){
					if(sasvs==null){
						int a =0;	a++;
					}					
					
					int componentId = sasvs.getVisitID();
					int expId = sasvs.getExperimentID();
					BigDecimal studyId = null;
					
					expComp.setComponentid(BigDecimal.valueOf(componentId));
					expComp.setNcExperimentUniqueid(BigDecimal.valueOf(expId));
					expComp.setSubjectid(sasvs.getSubjectID());					
					List<Expcomponent> expCompList = dao.find(con, expComp);
					if(expCompList!=null && !expCompList.isEmpty()){
						if(expCompList.get(0)==null) {
							int a=0; a++;
						}
						sasvs.setVisitName(expCompList.get(0).getName());
						sasvs.setVisitDate(expCompList.get(0).getTimeStamp().toString());						
					}
					
					subjExp.setNcExperimentUniqueid(BigDecimal.valueOf(expId));
					subjExp.setSubjectid(sasvs.getSubjectID());
					List<Subjexperiment> subjExpList = seDao.find(con, subjExp);
					if(subjExpList!=null && !subjExpList.isEmpty()){
						if(subjExpList.get(0)==null) {
							int a=0; a++;
						}
						rgVO.setUniqueid(subjExpList.get(0).getNcResearchgroupUniqueid());
						List<Researchgroup> rgList = rgDao.find(con, rgVO);
						if(rgList!=null && !rgList.isEmpty()){
							sasvs.setStudyGroup(rgList.get(0).getName());							
						}	
					}			
					
					expSeg.setNcExperimentUniqueid(BigDecimal.valueOf(expId));
					expSeg.setSubjectid(sasvs.getSubjectID());
					expSeg.setSegmentid(BigDecimal.valueOf(sasvs.getSegmentID()));
					expSeg.setComponentid(BigDecimal.valueOf(componentId));
					List<Expsegment> expSegList = esDao.find(con, expSeg);
					if(expSegList!=null && !expSegList.isEmpty()){
						if(expSegList.get(0)==null){
							int a=0; a++;
						}
						sasvs.setSegmentName(expSegList.get(0).getName());
						sasvs.setSegmentDate(expSegList.get(0).getModtime().toString());
						studyId = expSegList.get(0).getStudyid();
					}
					
					expStudy.setExperimentid(BigDecimal.valueOf(expId));
					expStudy.setSubjectid(sasvs.getSubjectID());
					expStudy.setComponentid(BigDecimal.valueOf(componentId));
					expStudy.setStudyid(studyId);
					List<Expstudy> expStudyList = expStudyDao.find(con, expStudy);
					if(expStudyList!=null && !expStudyList.isEmpty()){
						if(expStudyList.get(0)==null){
							int a = 0; a++;
						}
						sasvs.setStudyName(expStudyList.get(0).getName());
						sasvs.setStudyDate(expStudyList.get(0).getModtime().toString());
						sasvs.setStudyId(String.valueOf(studyId));
					}
				}				
				
				for(SubjectAsScoreValueSummary sasvs : fullList) {
					sasvs.setSiteID(sai.getSiteID());
				}
				combinedList.addAll(fullList);
			} catch (Exception x) {
				// just log the error and continue with the next site
				log.error(x);
			} finally {
				if (poolService != null && con != null) {
					poolService.releaseConnection(Constants.ADMIN_USER, con);
				}
			}
		}
	}

	public BatchQueryResult getFullAssessments(UserInfo userInfo,
			List<SubjectImageDataBundle> sidbList,
			List<AssessmentSelectionInfo> asiList) throws Exception {
		List<SubjectAsScoreValueSummary> combinedList = new ArrayList<SubjectAsScoreValueSummary>();

		Map<String, SiteAsiInfo> saiMap = AssessmentServiceHelper
				.findSites4AssessmentDownload(userInfo, sidbList, asiList);
		Map<Integer, List<String>> expSubjectIDListMap = AssessmentServiceHelper
				.prepareExpSubjectIDListMapFromSIDB(sidbList);

		retrieveAndCombine(userInfo, expSubjectIDListMap, combinedList, saiMap);
		BatchQueryResult bqr = new BatchQueryResult(combinedList, saiMap);
		return bqr;
	}


	public List<SubjectAsScoreValueSummary> queryMediatorForScores(Operator op,
			List<AssessmentSelectionInfo> asiList,
			Map<Integer, List<AssessmentMapping>> siteAsMap) throws Exception {
		MediatedAssessmentQueryBuilder mqb = new MediatedAssessmentQueryBuilder(
				asiList, siteAsMap);
		mqb.visit(op);

		String query = mqb.getQuery();

		List<SubjectAsScoreValueSummary> summaryList = new LinkedList<SubjectAsScoreValueSummary>();

		net.nbirn.mediator.client.Connection con = new net.nbirn.mediator.client.Connection(
				Constants.MEDIATOR_URL, ServiceNames.MEDIATOR_SERVICE);

		Mediator m = new Mediator("mhid", con);
		log.info("assessment query=" + query);
		Query q = new Query(query, m);
		Results rs = new Results(q);
		rs.init();
		while (rs.hasMore()) {
			Tuple tuple = (Tuple) rs.next();
			SubjectAsScoreValueSummary ssv = new SubjectAsScoreValueSummary();
			ssv.setSubjectID(tuple.getAttValue(0));
			ssv.setValue(tuple.getAttValue(1));
			ssv.setScoreName(tuple.getAttValue(2));
			ssv.setAssessmentID(tuple.getAttValue(3));
			ssv.setVisitID(Integer.parseInt(tuple.getAttValue(4)));
			ssv.setSegmentID(Integer.parseInt(tuple.getAttValue(5)));
			ssv.setExperimentID(Integer.parseInt(tuple.getAttValue(6)));
			summaryList.add(ssv);
			if (log.isDebugEnabled())
				log.debug(ssv.toString());
		}

		return summaryList;
	}

	/**
	 * This method retrieves the assessments from mediated sources and creates
	 * an association table. It assumes the query initiating db is UCSD
	 *
	 * @param initiatingSrcName
	 * @param assessments
	 * @return
	 * @throws java.lang.Exception
	 */
	public Map<String, Integer> getAssessmentsFromMediator(
			String initiatingSrcName, String otherSrcName,
			List<Assessment> assessments) throws Exception {
		net.nbirn.mediator.client.Connection con = new net.nbirn.mediator.client.Connection(
				Constants.MEDIATOR_URL, ServiceNames.MEDIATOR_SERVICE);

		Map<String, Assessment> asMap = new HashMap<String, Assessment>();
		for (Assessment as : assessments) {
			asMap.put(as.getName(), as);
		}

		Map<String, Integer> medAsMap = new HashMap<String, Integer>();
		Mediator m = new Mediator("mhid", con);
		String query = "query.qa2(ASSESSMENTID, NAME) :- (view.nc_assessment_view(ASSESSMENTID,"
				+ "TABLEID,OWNER,MODTIME,MODUSER,NAME,DESCRIPTION))";
		Query q = new Query(query, m);
		Results rs = new Results(q);
		rs.init();
		while (rs.hasMore()) {
			Tuple tuple = (Tuple) rs.next();
			int asID = Integer.parseInt(tuple.getAttValue(0));
			String name = tuple.getAttValue(1);
			Assessment as = asMap.get(name);
			if (as != null) {
				if (as.getAssessmentid().intValue() == asID) {
					medAsMap.put(initiatingSrcName + "_" + name, new Integer(
							asID));
				} else {
					medAsMap.put(otherSrcName + "_" + name, new Integer(asID));
				}
			}
		}

		return medAsMap;
	}

	public Map<String, String> getAssessmentScoresFromMediator(
			String initiatingSrcName, String otherSrcName,
			List<Assessment> assessments) throws Exception {
		net.nbirn.mediator.client.Connection con = new net.nbirn.mediator.client.Connection(
				Constants.MEDIATOR_URL, ServiceNames.MEDIATOR_SERVICE);

		Map<String, Assessment> asMap = new HashMap<String, Assessment>();
		for (Assessment as : assessments) {
			asMap.put(as.getName(), as);
		}

		Map<String, String> medScoreMap = new HashMap<String, String>();
		Mediator m = new Mediator("mhid", con);
		StringBuffer queryBuf = new StringBuffer(256);
		queryBuf.append("query.qas1(ASSESSMENTID, SCORENAME) :-  "
				+ "(view.nc_assessmentscore_view(ASSESSMENTID,SCORENAME,");
		queryBuf
				.append("SCORESEQUENCE,SCORETYPE, TABLEID, UNIQUEID, OWNER, MODTIME, MODUSER, DEFAULTVALUE,");
		queryBuf
				.append("NULLABLE, SCORELEVEL, PARENTASID, PARENTSCORE, ASSESSMENTONTOLOGY,"
						+ "ASSESSMENTCONCEPT, SECURITYCLASSIFICATION))");

		Query q = new Query(queryBuf.toString(), m);
		Results rs = new Results(q);
		rs.init();
		while (rs.hasMore()) {
			Tuple tuple = (Tuple) rs.next();
			int asID = Integer.parseInt(tuple.getAttValue(0));
			String name = tuple.getAttValue(1);
			Assessment as = asMap.get(name);
			if (as != null) {
				if (as.getAssessmentid().intValue() == asID) {
					medScoreMap.put(initiatingSrcName + "_" + asID, name);
				} else {
					medScoreMap.put(otherSrcName + "_" + asID, name);
				}
			}
		}
		return medScoreMap;
	}

	public List<RangeInfo> getRangeInfos(UserInfo userInfo,
			List<AsScoreInfo> scoreInfos,
			List<SubCorticalVarInfo> subCorticalVars) throws Exception {
		List<RangeInfo> rangeInfos = new LinkedList<RangeInfo>();
		IDBCache dbCache;
		StringBuffer sb = new StringBuffer();
		java.sql.Connection con = null;
		ResultSet rs = null;
		Statement st = null;

		try {
			con = pool.getConnection(userInfo.getName());

			if (scoreInfos != null) {
				dbCache = ServiceFactory.getDBCache(this.theDBID);
				Tableid tid = dbCache.getTableID(userInfo,
						Constants.RANGEVALIDATOR_TABLE);

				sb.append("select lowLimit, upperLImit, incLowLimit, ");
				sb.append("incUpperLimit, uniqueID ");
				sb.append("from nc_rangeValidator where uniqueID IN (");

				Map<String, Integer> rangeValidatorMap = AssessmentServiceHelper
						.getValidatorIDs(theDBID, con, tid.getTableid());
				if (rangeValidatorMap.isEmpty()) {
					// there are no range validator records, so return empty
					// range info list
					return rangeInfos;
				}

				int count = 0;
				Map<Integer, AsScoreInfo> validatorIDScoreMap = new HashMap<Integer, AsScoreInfo>(
						11);
				for (Object element : scoreInfos) {
					AsScoreInfo si = (AsScoreInfo) element;

					String key = AssessmentServiceHelper.createScoreKey(si
							.getName(), si.getAssessmentID().intValue());
					Integer validatorID = (Integer) rangeValidatorMap.get(key);
					if (validatorID == null)
						continue;
					if (count > 0)
						sb.append(',');
					sb.append(validatorID);
					++count;
					validatorIDScoreMap.put(validatorID, si);
				}
				if (count == 0) {
					// no appropriate range validators are found for any of the
					// scores in the scoreInfoMap
					return rangeInfos;
				} else {
					sb.append(')');
				}
				try {
					st = con.createStatement();
					log.info("query:" + sb.toString());

					rs = st.executeQuery(sb.toString());
					while (rs.next()) {
						AsScoreInfo si = validatorIDScoreMap.get(new Integer(rs
								.getBigDecimal(5).toString()));
						if (si == null)
							continue;
						RangeInfo ri = new RangeInfo(rs.getBigDecimal(1)
								.toString(), rs.getBigDecimal(2).toString(), rs
								.getString(3).equals("Y"), rs.getString(4)
								.equals("Y"), si);

						log.info("range info = " + ri.toString());

						rangeInfos.add(ri);
					}
				} finally {
					DBUtils.close(st, rs);
				}
			}

			if (subCorticalVars == null) {
				return rangeInfos;
			}
			for (SubCorticalVarInfo scvi : subCorticalVars) {
				RangeInfo ri = AssessmentServiceHelper
						.findRangeForSubcorticalVariable(con, scvi);
				if (ri != null) {
					log.info("subcortical range info = " + ri.toString());
					rangeInfos.add(ri);
				}
			}
		} finally {
			if (con != null)
				pool.releaseConnection(userInfo.getName(), con);
		}
		return rangeInfos;
	}

	public List<Storedquery> getAvailableStoredQueries(UserInfo userInfo,
			String userName) throws Exception {
		List<Storedquery> storedQueries = new LinkedList<Storedquery>();
		java.sql.Connection con = null;
		ResultSet rs = null;
		Statement st = null;
		try {
			con = pool.getConnection(userInfo.getName());
			StringBuffer sb = new StringBuffer(128);
			sb
					.append("SELECT UNIQUEID, DESCRIPTION FROM NC_STOREDQUERY WHERE USERNAME = '");
			sb.append(userName).append("'");
			log.info("query: " + sb.toString());
			st = con.createStatement();
			rs = st.executeQuery(sb.toString());
			while (rs.next()) {
				Storedquery sq = new Storedquery();
				sq.setUniqueid(rs.getBigDecimal(1));
				sq.setDescription(rs.getString(2));
				sq.setUsername(userInfo.getName());
				storedQueries.add(sq);
			}

		} finally {
			DBUtils.close(st, rs);
			if (con != null)
				pool.releaseConnection(userInfo.getName(), con);

		}
		return storedQueries;
	}

	public Storedquery getFullStoredQuery(UserInfo userInfo,
			BigDecimal queryID, String userName) throws Exception {
		java.sql.Connection con = null;
		try {
			con = pool.getConnection(userInfo.getName());
			Storedquery criteria = new Storedquery();
			criteria.setUniqueid(queryID);
			criteria.setUsername(userName);
			StoredqueryDAO dao = DAOFactory.createStoredqueryDAO(theDBID);
			List<?> list = dao.find(con, criteria);
			if (!list.isEmpty()) {
				return (Storedquery) list.get(0);
			}

		} finally {
			if (con != null)
				pool.releaseConnection(userInfo.getName(), con);
		}

		return null;
	}

	public void saveUserQuery(UserInfo userInfo, String queryState,
			String description) throws Exception {
		java.sql.Connection con = null;
		try {
			con = pool.getConnection(userInfo.getName());
			Storedquery sq = new Storedquery();
			sq.setUsername(userInfo.getName());
			sq.setDescription(description);
			sq.setQuerystate(queryState);

			ISecurityService securityService = ServiceFactory
					.getSecurityService();
			IDBCache dbCache = ServiceFactory.getDBCache(theDBID);

			// set the owner and moduser
			Map<String, User> userMap = securityService.getAllUsers(theDBID);
			User u = userMap.get(userInfo.getName());

			Databaseuser databaseUser = dbCache.getDatabaseUser(userInfo, u
					.getDbUser().getName(), Constants.USERCLASS_ADMIN);

			sq.setModuser(databaseUser.getUniqueid());
			sq.setOwner(databaseUser.getUniqueid());

			sq.setModtime(new java.util.Date());

			// set the tableid
			Tableid tid = dbCache.getTableID(userInfo,
					Constants.STOREDQUERY_TABLE);
			sq.setTableid(tid.getTableid());

			BigDecimal uniqueid = ServiceFactory.getSequenceHelper(theDBID)
					.getNextUID(userInfo, Constants.STOREDQUERY_TABLE,
							"uniqueid");
			sq.setUniqueid(uniqueid);

			StoredqueryDAO dao = DAOFactory.createStoredqueryDAO(theDBID);
			dao.insert(con, sq);

		} finally {
			if (con != null)
				pool.releaseConnection(userInfo.getName(), con);
		}
	}
	
	//Jinran added
	public void deleteStoredAssessmentQuery(UserInfo userInfo, BigDecimal storedQueryId) throws Exception {
		java.sql.Connection con = null;
		try {
			con = pool.getConnection(userInfo.getName());
			Storedquery sq = new Storedquery();
			
			//set uniqueid
			sq.setUniqueid(storedQueryId);

			StoredqueryDAO dao = DAOFactory.createStoredqueryDAO(theDBID);
			dao.delete(con, sq);			

		} finally {
			if (con != null)
				pool.releaseConnection(userInfo.getName(), con);
		}
	}
	
	//Jinran added
	public void updateStoredAssessmentQuery(UserInfo userInfo, String queryState,
			String description, BigDecimal storedQueryId) throws Exception {
		java.sql.Connection con = null;
		try {
			con = pool.getConnection(userInfo.getName());
			Storedquery sq = new Storedquery();
			
			sq.setUsername(userInfo.getName());
			sq.setDescription(description);
			sq.setQuerystate(queryState);

			ISecurityService securityService = ServiceFactory
					.getSecurityService();
			IDBCache dbCache = ServiceFactory.getDBCache(theDBID);

			// set the owner and moduser
			Map<String, User> userMap = securityService.getAllUsers(theDBID);
			User u = userMap.get(userInfo.getName());

			Databaseuser databaseUser = dbCache.getDatabaseUser(userInfo, u
					.getDbUser().getName(), Constants.USERCLASS_ADMIN);

			sq.setModuser(databaseUser.getUniqueid());
			sq.setOwner(databaseUser.getUniqueid());

			sq.setModtime(new java.util.Date());

			// set the tableid
			Tableid tid = dbCache.getTableID(userInfo,
					Constants.STOREDQUERY_TABLE);
			sq.setTableid(tid.getTableid());

			Storedquery cr = new Storedquery();
			//set uniqueid
			cr.setUniqueid(storedQueryId);

			StoredqueryDAO dao = DAOFactory.createStoredqueryDAO(theDBID);
			dao.update(con, sq, cr);			

		} finally {
			if (con != null)
				pool.releaseConnection(userInfo.getName(), con);
		}
	}	

	public void shutdown() {
	}

	@Override
	public List<SubjectAsScoreValueSummary> queryForImagesOnlyOrGeneOnly(UserInfo userInfo, Integer selectedExperimentIDs, String queryScope, 
			String primarySiteID, String protocolIDs, String objectTypes,	String protocolVersions)
			throws Exception {

		String imagingQuery;		
		
//		imagingQuery = "SELECT b.subjectid, a.textvalue, a.scoreorder, a.scorename, a.assessmentid, " + 
//					   "b.componentid, b.segmentid, b.NC_EXPERIMENT_UNIQUEID, b.time_stamp, a.scoretype " + 
//					   "FROM  nc_assessmentvarchar a,  nc_storedassessment b " + 
//					   "WHERE a.STOREDASSESSMENTID = b.UNIQUEID " + 
//					   "AND a.isvalidated =  true ";
		
//		imagingQuery = "SELECT b.subjectid, null, a.scoreorder, null, a.assessmentid, " + 
//					   "b.componentid, b.segmentid, b.NC_EXPERIMENT_UNIQUEID, b.time_stamp, null " + 
//					   "FROM  nc_assessmentvarchar a,  nc_storedassessment b " +
//					   "WHERE a.STOREDASSESSMENTID = b.UNIQUEID " + 
//					   "AND a.isvalidated =  true " + 
//					   "AND b.subjectid in (SELECT DISTINCT c.subjectid " + 
//					   "					FROM nc_rawdata c, nc_dataobject d, nc_dataobjecttype e " +
//					   "					WHERE c.uniqueid = d.dataid " +
//					   "					AND d.objecttypeid=e.uniqueid ";

		imagingQuery = "SELECT c.subjectid, null, 1, null, '', null, null, " + 
					   "c.NC_EXPERIMENT_UNIQUEID, c.modtime, null " + 
					   "FROM nc_rawdata c, nc_dataobject d, nc_dataobjecttype e " +
					   "WHERE c.uniqueid = d.dataid " +
					   "AND d.objecttypeid=e.uniqueid ";
		
		if(!protocolIDs.equals(Constants.ALL_PROTOCOLS)){
			imagingQuery += "AND c.protocolid in (" + protocolIDs + ") ";			   
		}
		
		if(!protocolVersions.equals(Constants.ALL_VERSIONS)){
			imagingQuery += "AND c.protocolversion in (" + protocolVersions + ") ";
		}

		if(!objectTypes.equals(Constants.ALL_TYPES)){
			imagingQuery += "AND e.objecttype in (" + objectTypes + ") ";
		}
		
		//imagingQuery += ")";
		
		if(selectedExperimentIDs!=-1 && selectedExperimentIDs!=-2){ //not all experiment selected
			imagingQuery = imagingQuery + "AND c.NC_EXPERIMENT_UNIQUEID = " + selectedExperimentIDs;
		}

		List<SubjectAsScoreValueSummary> summaryList = new LinkedList<SubjectAsScoreValueSummary>();
		java.sql.Connection con = null;
		Statement st = null;
		ResultSet rs = null;
		Map<String, SubjectAsScoreValueSummary> summaryMap = new HashMap<String, SubjectAsScoreValueSummary>();
		Monitor primary = null;
		try {
			primary = MonitorFactory.startPrimary("queryForScores1");
			Monitor conMon = MonitorFactory
					.start("queryForScores1.getConnection");
			con = pool.getConnection(userInfo.getName());
			conMon.stop();
			st = con.createStatement();
			// log.info("query=" + aqb.getQuery());
			Monitor queryMon = MonitorFactory.start("queryForScores1.doQuery");
			log.info("queryForScores = " + imagingQuery);
			rs = st.executeQuery(imagingQuery);
			while (rs.next()) {
				AssessmentServiceHelper.prepareSubjectAsScoreSummaryRec(
						summaryList, rs, summaryMap);
			}
			queryMon.stop();
			summaryMap = null;

			// attach site information
			String dbID = CacheUtils.getDBID4SiteID(primarySiteID);
			IDBCache dbCache = ServiceFactory.getDBCache(dbID);
			List<Site> sites = dbCache.getSites(userInfo, false);
			Map<String, SubjectInfo> siMap = getSubjectInfos(con, summaryList,
					sites, primarySiteID);

			ISecurityService secService = ServiceFactory.getSecurityService();
			String siteID = secService.findSiteIDByDbID(theDBID);
			IRemoteDBServices rds = ServiceFactory.getRemoteDBServices();
			Set<VisitDateQuery> vdqSet = new HashSet<VisitDateQuery>();
			for (SubjectAsScoreValueSummary summary : summaryList) {
				VisitDateQuery vdq = new VisitDateQuery(summary.getSubjectID(),
						summary.getVisitID(), summary.getExperimentID());
				vdqSet.add(vdq);
			}
			Map<String, Date> visitDateMap = rds.getVisitDates(siteID,
					new ArrayList<VisitDateQuery>(vdqSet), true);

			for (SubjectAsScoreValueSummary summary : summaryList) {
				SubjectInfo si = siMap.get(summary.getSubjectID());
				summary.setSiteID(si.siteID);
				summary.setSiteName(si.siteName);
				summary.setRemote(si.remote);

				String key = summary.getSubjectID() + "_"
						+ summary.getExperimentID() + "_"
						+ summary.getVisitID();
				Date visitDate = visitDateMap.get(key);
				if (visitDate != null) {
					summary.setTimeStamp(DateTimeUtils.formatDate(visitDate));
				}

			}

		} finally {
			DBUtils.close(st, rs);
			if (con != null)
				pool.releaseConnection(userInfo.getName(), con);
			if (primary != null) {
				primary.stop();
			}
		}

		// FIXME for testing
		/*
		 * List<Object> list = new ArrayList<Object>(2);
		 * list.add(summaryList); list.add(op); GenUtils.serialize(list,
		 * System.getProperty("user.home") + "/query.ser");
		 */

		// post-process the search results filtering segments which do not match
		// exactly
		if (queryScope.equals(Constants.SEGMENT_SCOPE)) {
			log.info("Using segment scope...");
			summaryList = AssessmentServiceHelper.filterSearchResults(
					summaryList, null);
		} else if (queryScope.equals(Constants.VISIT_SCOPE)) {
			log.info("Using visit scope...");
			summaryList = AssessmentServiceHelper
					.filterSearchResultsGroupedByVisit(summaryList, null);
		}

		return summaryList;

		
		
	}

	@Override
	public BatchQueryResult getAssessmentsOnAssessId(UserInfo userInfo,
			List<SubjectImageDataBundle> sidbList,
			List<AssessmentSelectionInfo> asiList, 
			String assessmentId,
			String assResults)
			throws Exception {
		List<SubjectAsScoreValueSummary> combinedList = new ArrayList<SubjectAsScoreValueSummary>();

		Map<String, SiteAsiInfo> saiMap = AssessmentServiceHelper
				.findSites4AssessmentDownload(userInfo, sidbList, asiList);
		Map<Integer, List<String>> assSubjectIDListMap = AssessmentServiceHelper
				.prepareAssSubjectIDListMapFromSIDB(assResults, asiList);
		Map<Integer, List<String>> assExpIDListMap = AssessmentServiceHelper
				.prepareAssExpIDListMapFromSIDB(assResults, asiList, sidbList);
		
		retrieveAndCombine(userInfo, assSubjectIDListMap, combinedList, saiMap, assessmentId, assExpIDListMap);
		BatchQueryResult bqr = new BatchQueryResult(combinedList, saiMap);
	
		return bqr;
	}
	
	public List<SubjectAsScoreValueSummary> batchQueryForAll(UserInfo userInfo, Integer selectedExperimentIDs,
			String primarySiteID)
			throws Exception {

		String batchQueryAll;		
		
//		batchQueryAll = "SELECT c.subjectid, null, 1, null, '', null, null, " + 
//					    "c.NC_EXPERIMENT_UNIQUEID, c.modtime, null " + 
//					    "FROM nc_rawdata c, nc_dataobject d, nc_dataobjecttype e " +
//					    "WHERE c.uniqueid = d.dataid " +
//					    "AND d.objecttypeid=e.uniqueid ";

		batchQueryAll = "SELECT DISTINCT subjectid, nc_experiment_uniqueid FROM nc_storedassessment s ";
		if(selectedExperimentIDs!=-1){ //not all experiment selected
			batchQueryAll= batchQueryAll+ " where NC_EXPERIMENT_UNIQUEID = " + selectedExperimentIDs;
		}
		batchQueryAll += " ORDER BY nc_experiment_uniqueid, subjectid";
		List<SubjectAsScoreValueSummary> summaryList = new LinkedList<SubjectAsScoreValueSummary>();
		java.sql.Connection con = null;
		Statement st = null;
		ResultSet rs = null;
		
		try {
			con = pool.getConnection(userInfo.getName());
			st = con.createStatement();
			log.info("queryForScores = " + batchQueryAll);
			rs = st.executeQuery(batchQueryAll);
			while (rs.next()) {
				String subjectID = rs.getString(1);
				//Timestamp timeStamp = rs.getTimestamp(9);				
				int expID = rs.getInt(2);
//				String timeStampStr = "";
				
//				if (timeStamp != null) {
//					timeStampStr = QueryUtils.formatDate(new java.sql.Date(timeStamp
//							.getTime()));
//				}
				SubjectAsScoreValueSummary ssv = new SubjectAsScoreValueSummary();
				ssv.setSubjectID(subjectID);			
				ssv.setExperimentID(expID);
//				ssv.setTimeStamp(timeStampStr);
				summaryList.add(ssv);				
			}
			
			// attach site information
			String dbID = CacheUtils.getDBID4SiteID(primarySiteID);
			IDBCache dbCache = ServiceFactory.getDBCache(dbID);
			List<Site> sites = dbCache.getSites(userInfo, false);
			Map<String, SubjectInfo> siMap = getSubjectInfos(con, summaryList,
					sites, primarySiteID);

			ISecurityService secService = ServiceFactory.getSecurityService();
			String siteID = secService.findSiteIDByDbID(theDBID);
			IRemoteDBServices rds = ServiceFactory.getRemoteDBServices();
			Set<VisitDateQuery> vdqSet = new HashSet<VisitDateQuery>();
			for (SubjectAsScoreValueSummary summary : summaryList) {
				VisitDateQuery vdq = new VisitDateQuery(summary.getSubjectID(),
						summary.getVisitID(), summary.getExperimentID());
				vdqSet.add(vdq);
			}
			Map<String, Date> visitDateMap = rds.getVisitDates(siteID,
					new ArrayList<VisitDateQuery>(vdqSet), true);

			for (SubjectAsScoreValueSummary summary : summaryList) {
				SubjectInfo si = siMap.get(summary.getSubjectID());
				summary.setSiteID(si.siteID);
				summary.setSiteName(si.siteName);
				summary.setRemote(si.remote);

				String key = summary.getSubjectID() + "_"
						+ summary.getExperimentID() + "_"
						+ summary.getVisitID();
				Date visitDate = visitDateMap.get(key);
				if (visitDate != null) {
					summary.setTimeStamp(DateTimeUtils.formatDate(visitDate));
				}

			}
		} 
		finally {
			DBUtils.close(st, rs);
			if (con != null)
				pool.releaseConnection(userInfo.getName(), con);
		}

		return summaryList;		
		
	}
	
}
