package clinical.xml.export;

import java.io.StringWriter;
import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.Marshaller;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.server.vo.Assessment;
import clinical.server.vo.Experiment;
import clinical.server.vo.Humansubject;
import clinical.server.vo.Researchgroup;
import clinical.server.vo.Subjexperiment;
import clinical.utils.Assertion;
import clinical.utils.DateTimeUtils;
import clinical.utils.GenUtils;
import clinical.utils.XCEDEUtils;
import clinical.web.IDBUserManService;
import clinical.web.IExperimentManagement;
import clinical.web.ISQLDialect;
import clinical.web.ISubjectAssessmentManagement;
import clinical.web.ISubjectVisitManagement;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.IDBPoolService;
import clinical.web.common.UserInfo;
import clinical.web.common.query.TSQLProcessor;
import clinical.web.exception.SubjectAssessmentManagementException;
import clinical.web.vo.AssessmentInfo;
import clinical.web.vo.AssessmentScoreValues;
import clinical.web.vo.ScoreValue;
import clinical.web.vo.Subject;
import clinical.web.vo.Visit;
import clinical.web.vo.VisitSegment;
import clinical.web.vo.AssessmentInfo.ScoreInfo;
import clinical.xml.xcede2.AssessmentItemT;
import clinical.xml.xcede2.AssessmentT;
import clinical.xml.xcede2.ValueT;
import clinical.xml.xcede2.XCEDE;
import clinical.xml.xcede2.AssessmentT.DataInstance;

/**
 * @author I. Burak Ozyurt
 * @version $Id: XCEDE2SeriesExporter.java 174 2010-02-11 19:18:15Z bozyurt $
 */
public class XCEDE2SeriesExporter {
	private static Log log = LogFactory.getLog("XCEDE2SeriesExporter");
	protected String dbID;
	protected ISQLDialect sqlDialect;
	public final static String _Assessment_Date = "Assessment_Date";
	public final static String _Assessment_Time = "Assessment_Time";
	public final static String _Clinical_Rater = "Clinical_Rater";
	public final static String _InformantID = "InformantID";
	public final static String _InformantRelation = "InformantRelation";

	public XCEDE2SeriesExporter(String dbID, ISQLDialect sqlDialect) {
		this.dbID = dbID;
		this.sqlDialect = sqlDialect;
	}

	public String[] getAllSubjects(UserInfo ui) throws Exception {

		IDBPoolService dbPoolService = null;
		Connection con = null;
		try {
			dbPoolService = ServiceFactory.getPoolService(dbID);
			con = dbPoolService.getConnection(ui.getName());
			TSQLProcessor tp = new TSQLProcessor(sqlDialect);
			StringBuffer qBuf = new StringBuffer();
			qBuf.append("select h from Humansubject as h order by h.subjectid");
			log.info("TQL:" + qBuf.toString());
			List<?> results = tp.executeQuery(con, qBuf.toString());
			String[] subjectIDs = new String[results.size()];
			int i = 0;
			for (Iterator<?> it = results.iterator(); it.hasNext();) {
				Humansubject hs = (Humansubject) it.next();
				subjectIDs[i++] = hs.getSubjectid();
			}
			return subjectIDs;
		} finally {
			if (dbPoolService != null) {
				dbPoolService.releaseConnection(ui.getName(), con);
			}
		}
	}

	public String[] getAllSubjectsForExperiment(UserInfo ui, int expID)
			throws Exception {
		IDBPoolService dbPoolService = null;
		Connection con = null;
		try {
			dbPoolService = ServiceFactory.getPoolService(dbID);
			con = dbPoolService.getConnection(ui.getName());
			TSQLProcessor tp = new TSQLProcessor(sqlDialect);
			StringBuffer qBuf = new StringBuffer();
			qBuf
					.append("select s from Subjexperiment as s where s.ncExperimentUniqueid = ");
			qBuf.append(expID).append(" order by s.subjectid");
			log.info("TQL:" + qBuf.toString());
			List<?> results = tp.executeQuery(con, qBuf.toString());
			String[] subjectIDs = new String[results.size()];
			int i = 0;
			for (Iterator<?> it = results.iterator(); it.hasNext();) {
				Subjexperiment se = (Subjexperiment) it.next();
				subjectIDs[i++] = se.getSubjectid();
			}
			return subjectIDs;
		} finally {
			if (dbPoolService != null) {
				dbPoolService.releaseConnection(ui.getName(), con);
			}
		}

	}

	public int[] getAllExperimentIDs(UserInfo ui) throws Exception {
		ISubjectVisitManagement isvm = ServiceFactory
				.getSubjectVisitManagement(dbID);

		List<Experiment> exps = isvm.getAllExperiments(ui);
		int[] expIDs = new int[exps.size()];
		int i = 0;
		for (Iterator<Experiment> iter = exps.iterator(); iter.hasNext();) {
			Experiment exp = iter.next();
			expIDs[i++] = exp.getUniqueid().intValue();
		}
		return expIDs;
	}

	public String[] getExperimentNames(UserInfo ui, int[] expIDs)
			throws Exception {
		ISubjectVisitManagement isvm = ServiceFactory
				.getSubjectVisitManagement(dbID);

		List<Experiment> exps = isvm.getAllExperiments(ui);
		String[] expNames = null;

		Map<Integer, Experiment> expIDMap = new HashMap<Integer, Experiment>();
		for (Iterator<Experiment> iter = exps.iterator(); iter.hasNext();) {
			Experiment exp = iter.next();
			expIDMap.put(new Integer(exp.getUniqueid().toString()), exp);
		}
		exps = new LinkedList<Experiment>();
		for (int i = 0; i < expIDs.length; i++) {
			Experiment exp = expIDMap.get(new Integer(expIDs[i]));
			if (exp != null) {
				exps.add(exp);
			}
		}
		expNames = new String[exps.size()];
		int i = 0;
		for (Iterator<Experiment> iter = exps.iterator(); iter.hasNext();) {
			Experiment exp = iter.next();
			expNames[i++] = exp.getName();
		}
		return expNames;
	}

	public void importAssessmentDataFromXCEDE(UserInfo ui, String theDBID,
			String xcede, String protocolID, int protocolVersion) throws Exception {

		XCEDE xe = XCEDEUtils.unmarshalFromString(xcede);

		if (xe.getAnnotationListOrRevisionListOrProject().isEmpty()) {
			return;
		}
		System.out.println(xe.getAnnotationListOrRevisionListOrProject());
		AssessmentT as = (AssessmentT) xe
				.getAnnotationListOrRevisionListOrProject().get(0);

		DataInstance di = as.getDataInstance().get(0);
		String visitDate = null;
		String visitTime = null;
		String informantID = null;
		String informantRelation = null;
		String clinicalRater = null;
		String asName = as.getName();

		IDBPoolService pool = ServiceFactory.getPoolService(dbID);
		if (pool == null) {
			throw new Exception("Cannot get database connection service for dbID:"
					+ dbID);
		}
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			for (AssessmentItemT ai : di.getAssessmentItem()) {
				String id = ai.getID();
				if (id.endsWith(_Assessment_Date)) {
					visitDate = ai.getValue().getValue();
				} else if (id.endsWith(_Assessment_Time)) {
					visitTime = ai.getValue().getValue();
				} else if (id.endsWith(_InformantID)) {
					informantID = ai.getValue().getValue();
				} else if (id.endsWith(_InformantRelation)) {
					informantRelation = ai.getValue().getValue();
				} else if (id.endsWith(_Clinical_Rater)) {
					clinicalRater = ai.getValue().getValue();
				}
			}
			System.out.println("visitDate=" + visitDate + " visitTime="
					+ visitTime);

			if (!informantRelation.equalsIgnoreCase("self")) {
				informantID = null;
			}

			IExperimentManagement em = ServiceFactory
					.getExperimentManagement(theDBID);
			Experiment exp = em.getExperiment(ui, as.getProjectID());
			if (exp == null) {
				throw new Exception("Unknown experiment/project: "
						+ as.getProjectID());
			}
			List<Subjexperiment> subjExpList = em.getEnrolledSubjects(ui, exp
					.getUniqueid().intValue());
			Subjexperiment theSE = null;
			for (Subjexperiment se : subjExpList) {
				if (se.getSubjectid().equals(as.getSubjectID())) {
					theSE = se;
					break;
				}
			}
			String[] toks = as.getVisitID().split("__");
			Assertion.assertTrue(toks.length == 3);
			int visitID = GenUtils.toInt(toks[2], 1);
			int segmentID = GenUtils.toInt(as.getEpisodeID(), 1);

			if (theSE == null) {
				theSE = enrollSubject(con, ui, theDBID, as.getSubjectID(), as
						.getSubjectGroupID(), exp, em);

				addVisitForSubject(con, ui, theDBID, as.getVisitID(), visitDate,
						visitTime, as.getEpisodeID(), theSE, protocolID,
						protocolVersion);
			} else {
				ISubjectVisitManagement isvm = ServiceFactory
						.getSubjectVisitManagement(theDBID);
				List<Visit> visitList = isvm.getAllVisitsForSubject(ui, as
						.getSubjectID());
				if (visitList.isEmpty()) {
					addVisitForSubject(con, ui, theDBID, as.getVisitID(), visitDate,
							visitTime, as.getEpisodeID(), theSE, protocolID,
							protocolVersion);
				} else {
					// check if visit exists
					Visit theVisit = null;
					for (Visit v : visitList) {
						if (v.getComponentID() == visitID) {
							theVisit = v;
							break;
						}
					}
					if (theVisit == null) {
						addVisitForSubject(con, ui, theDBID, as.getVisitID(),
								visitDate, visitTime, as.getEpisodeID(), theSE,
								protocolID, protocolVersion);
					} else {
						// check if segment exists
						if (!theVisit.hasSegment(segmentID)) {
							addVisitSegmentForSubject(con, ui, theDBID, visitDate,
									visitTime, as.getEpisodeID(), theVisit, theSE,
									protocolID, protocolVersion);
						}
					}
				}
			}

			ISubjectAssessmentManagement isam = ServiceFactory
					.getSubjectAssessmentManagement(theDBID);

			Assessment assessment = isam.getAssessment(ui, asName);

			if (assessment == null) {
				assessment = addAssessment(ui, theDBID, as, isam);
			}

			AssessmentInfo asInfo = prepAssessmentInfo(as, assessment);
			AssessmentScoreValues asv = new AssessmentScoreValues(asInfo);
			asv.setSubjectID(as.getSubjectID());
			asv.setExperimentID(exp.getUniqueid().intValue());
			asv.setVisitID(visitID);
			asv.setSegmentID(segmentID);
			asv.setInformantRelation(informantRelation);
			asv.setInformantID(informantID);
			asv.setClinicalRater(clinicalRater);
			String tsStr = visitDate;
			if (visitTime != null)
				tsStr += " " + visitTime;

			asv.setTimeStamp(new Timestamp(DateTimeUtils.convertTimestampTablet(
					tsStr).getTime()));

			List<AssessmentInfo> aiList = isam.getAssessmentsForSubject(con, ui,
					theSE.getSubjectid(), exp.getUniqueid().intValue());

			boolean foundAsEntry = false;
			for (AssessmentInfo ai : aiList) {
				if (ai.getAssessmentID() == asInfo.getAssessmentID()) {
					if (visitID == ai.getVisitID() && segmentID == ai.getSegmentID()) {
						foundAsEntry = true;
						break;
					}
				}
			}

			if (foundAsEntry) {
				throw new Exception("An entry of the assessment '"
						+ asInfo.getName() + "' for visit with ID " + visitID
						+ " and segment(episode) ID " + segmentID
						+ " already exists!");
			}

			IDBUserManService dbusm = ServiceFactory.getDBUserManService(theDBID);
			dbusm.ensureClinicalRater(con, ui, clinicalRater);
			Map<String, Map<String, String>> dbVarMetaDataMap = new HashMap<String, Map<String, String>>(
					3);
			List<ScoreValue> svList = new ArrayList<ScoreValue>();
			for (AssessmentItemT ai : di.getAssessmentItem()) {
				if (isReservedItem(ai)) {
					continue;
				}

				String value = null;
				if (ai.getValue() != null) {
					value = ai.getValue().getValue();
				}
				ScoreValue sv = new ScoreValue(ai.getID(), value, 1);
				if (sv.getName().startsWith("d")) {
					Map<String, String> metadataMap = new HashMap<String, String>(3);
					metadataMap.put("format", "MM/dd/yyyy");
					dbVarMetaDataMap.put(sv.getName(), metadataMap);
				}

				asv.addScoreValue(sv);
				svList.add(sv);
			}
			BigDecimal entryID = new BigDecimal(1);
			isam.insertAssessmentValues(con, ui, theDBID, asv, entryID, true,
					dbVarMetaDataMap);

			con.commit();
		} catch (Throwable t) {
			log.error("error in importAssessmentDataFromXCEDE", t);
			try {
				con.rollback();
			} catch (Exception e) {
				log.error("rollback error", e);
			}
			throw new Exception(t.getMessage());
		} finally {
			if (pool != null) {
				pool.releaseConnection(ui.getName(), con);
			}
		}

	}

	protected Subjexperiment enrollSubject(Connection con, UserInfo ui,
			String theDBID, String subjectID, String subjectGroupID,
			Experiment exp, IExperimentManagement em) throws Exception {
		ISubjectVisitManagement isvm = ServiceFactory
				.getSubjectVisitManagement(theDBID);
		IDBCache dbCache = ServiceFactory.getDBCache(theDBID);

		Humansubject hs = isvm.getSubjectByID(ui, subjectID);
		if (hs == null) {
			Subject subject = new Subject(subjectID);
			isvm.addSubject(ui, subject);
		}
		List<Researchgroup> rgList = dbCache.getResearchGroups(ui, false);
		Researchgroup theRG = null;
		for (Researchgroup rg : rgList) {
			if (rg.getName().equals(subjectGroupID)
					&& rg.getNcExperimentUniqueid().equals(exp.getUniqueid())) {
				theRG = rg;
				break;
			}
		}
		if (theRG == null) {
			throw new Exception("Unknown subject group ID:" + subjectGroupID);
		}
		Subjexperiment se = em.enrollSubject(con, ui, exp.getUniqueid()
				.intValue(), subjectID, theRG.getUniqueid().intValue());
		return se;
	}

	protected void addVisitSegmentForSubject(Connection con, UserInfo ui,
			String theDBID, String visitDate, String visitTime, String episodeID,
			Visit theVisit, Subjexperiment se, String protocolID,
			int protocolVersion) throws Exception {
		ISubjectVisitManagement isvm = ServiceFactory
				.getSubjectVisitManagement(theDBID);
		VisitSegment segment = new VisitSegment();
		segment.setVisitID(theVisit.getComponentID());
		segment.setExperimentID(se.getNcExperimentUniqueid().intValue());
		segment.setProtocolID(protocolID);
		segment.setProtocolVersion(protocolVersion);
		segment.setSubjectID(se.getSubjectid());
		int segmentID = GenUtils.toInt(episodeID, 1);
		segment.setSegmentID(segmentID);
		String tsStr = visitDate;
		if (visitTime != null)
			tsStr += " " + visitTime;
      System.out.println("tsStr=" + tsStr);
		segment.setTimeStamp(new Timestamp(DateTimeUtils.convertTimestampTablet(tsStr)
				.getTime()));
		segment.setName("from tablet");

		// isvm.addVisitSegmentForSubject(ui, segment);
		isvm.handleAddVisitSegment(con, ui, segment);
	}

	protected Visit addVisitForSubject(Connection con, UserInfo ui,
			String theDBID, String visitStr, String visitDate, String visitTime,
			String episodeID, Subjexperiment se, String protocolID,
			int protocolVersion) throws Exception {
		ISubjectVisitManagement isvm = ServiceFactory
				.getSubjectVisitManagement(theDBID);
		String[] toks = visitStr.split("__");
		Assertion.assertTrue(toks.length == 3);
		int visitID = GenUtils.toInt(toks[2], 1);
		Visit v = new Visit();
		v.setSubjectID(se.getSubjectid());
		v.setExperimentID(se.getNcExperimentUniqueid().intValue());
		v.setComponentID(visitID);
		v.setName(toks[0]);
		v.setVisitType("clinical visit");
		v.setTimeStamp(DateTimeUtils.toTimeStampTablet(visitDate));

		VisitSegment segment = new VisitSegment();
		segment.setExperimentID(se.getNcExperimentUniqueid().intValue());
		segment.setProtocolID(protocolID);
		segment.setProtocolVersion(protocolVersion);
		segment.setSubjectID(se.getSubjectid());
		int segmentID = GenUtils.toInt(episodeID, 1);
		segment.setSegmentID(segmentID);
		String tsStr = visitDate;
		if (visitTime != null)
			tsStr += " " + visitTime;
		System.out.println("tsStr=" + tsStr);
		segment.setTimeStamp(new Timestamp(DateTimeUtils.convertTimestampTablet(
				tsStr).getTime()));
		segment.setName("from tablet");
		v.addVisitSegment(segment);
		isvm.addVisitForSubjectNonContiguous(con, ui, v);
		return v;
	}

	protected AssessmentInfo prepAssessmentInfo(AssessmentT as,
			Assessment assessment) {
		AssessmentInfo asInfo = new AssessmentInfo(assessment.getUniqueid()
				.intValue(), assessment.getName(), null, -1, -1);
		DataInstance di = as.getDataInstance().get(0);
		int i = 1;
		for (AssessmentItemT ai : di.getAssessmentItem()) {
			if (isReservedItem(ai)) {
				continue;
			}
			String value = null;
			if (ai.getValue() != null) {
				value = ai.getValue().getValue();
			}
			String scoreType = guessScoreType(ai.getID(), value);
			ScoreInfo si = new AssessmentInfo.ScoreInfo(ai.getID(), scoreType, 1,
					-1, null, i);
			asInfo.addScore(si);
			i++;
		}
		return asInfo;

	}

	protected Assessment addAssessment(UserInfo ui, String theDBID,
			AssessmentT as, ISubjectAssessmentManagement isam)
			throws SubjectAssessmentManagementException {
		AssessmentInfo asInfo = new AssessmentInfo(-1, as.getName(), null, -1, -1);

		DataInstance di = as.getDataInstance().get(0);

		// set description if any
		if (di.getAssessmentInfo() != null) {
			String description = di.getAssessmentInfo().getDescription();
			if (description != null && description.trim().length() > 0) {
				asInfo.setDescription(description);
			}
		}
		int i = 1;
		for (AssessmentItemT ai : di.getAssessmentItem()) {
			if (isReservedItem(ai)) {
				continue;
			}
			String value = null;
			if (ai.getValue() != null) {
				value = ai.getValue().getValue();
			}
			String scoreType = guessScoreType(ai.getID(), value);
			ScoreInfo si = new AssessmentInfo.ScoreInfo(ai.getID(), scoreType, 1,
					-1, null, i);
			asInfo.addScore(si);
			i++;
		}

		return isam.addAssessment(ui, asInfo);
	}

	public static String guessScoreType(String scoreName, String scoreValue) {
		if (scoreName.startsWith("s")) {
			return "varchar";
		}
		if (scoreName.startsWith("d")) {
			return "timestamp";
		}
		if (scoreName.startsWith("n")) {
			if (scoreValue == null || scoreValue.length() == 0) {
				return "integer";
			}
		}
		if (scoreValue == null) {
			// best bet
			return "varchar";
		}
		Number num = GenUtils.toNumber(scoreValue);
		if (num == null) {
			return "varchar";
		}
		if (num instanceof Integer) {
			return "integer";
		} else if (num instanceof Float) {
			return "float";
		}
		return "varchar";
	}

	public static boolean isReservedItem(AssessmentItemT ai) {
		String name = ai.getID();
		return (name.endsWith(_Assessment_Date)
				|| name.endsWith(_Assessment_Time) || name.endsWith(_InformantID)
				|| name.endsWith(_InformantRelation) || name
				.endsWith(_Clinical_Rater));
	}

	public String exportAssessmentDataForSubject(UserInfo ui, String subjectID,
			int expID) throws Exception {
		StringWriter out = new StringWriter(4096);

		XCEDE xe = new XCEDE();

		List<AssessmentT> astList = prepareAssessmentData(ui, subjectID, expID);
		for (Iterator<AssessmentT> it = astList.iterator(); it.hasNext();) {
			AssessmentT xa = it.next();
			xe.getAnnotationListOrRevisionListOrProject().add(xa);
		}
		JAXBContext jc = JAXBContext.newInstance(XCEDE.class);
		Marshaller marshaller = jc.createMarshaller();
		marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT,
				new Boolean(true));

		marshaller.marshal(xe, System.out);

		return out.toString();
	}

	public List<AssessmentT> prepareAssessmentData(UserInfo ui,
			String subjectID, int experimentID) throws Exception {

		ISubjectAssessmentManagement isam = ServiceFactory
				.getSubjectAssessmentManagement(dbID);

		List<AssessmentInfo> asiList = isam.getAssessmentsForSubject(ui,
				subjectID, experimentID);
		List<AssessmentScoreValues> asvList = isam.getAssessmentValuesForSubject(
				ui, subjectID, experimentID, asiList, -1, true);

		Map<String, AssessmentT> astMap = new LinkedHashMap<String, AssessmentT>();
		for (AssessmentScoreValues asv : asvList) {
			AssessmentT ast = astMap.get(asv.getAssessmentName());
			if (ast == null) {
				ast = new AssessmentT();
				ast.setName(asv.getAssessmentName());
				astMap.put(ast.getName(), ast);
			}
			ast.setSubjectID(subjectID);
			ast.setProjectID(String.valueOf(experimentID));
			ast.setVisitID(String.valueOf(asv.getVisitID()));
			ast.setEpisodeID(String.valueOf(asv.getSegmentID()));
			AssessmentT.DataInstance di = new AssessmentT.DataInstance();
			di.setValidated(new Boolean(true));
			ast.getDataInstance().add(di);
			for (ScoreValue sv : asv.getScoreValues()) {
				AssessmentItemT ai = new AssessmentItemT();
				ai.setID(sv.getName());
				ValueT value = new ValueT();
				value.setValue(sv.getValue());
				ai.setValue(value);
				di.getAssessmentItem().add(ai);
			}
		}

		return new ArrayList<AssessmentT>(astMap.values());
	}
	/*
	 * public static void main(String[] args) throws Exception {
	 * XCEDE2SeriesExporter se = new XCEDE2SeriesExporter(); UserInfo ui = null;
	 * try { // ui = se.startup();
	 * 
	 * int[] expIDs = se.getAllExperimentIDs(ui); for (int i = 0; i <
	 * expIDs.length; i++) { System.out.print(expIDs[i] + " "); }
	 * System.out.println(); String[] expNames = se.getExperimentNames(ui,
	 * expIDs); for (int i = 0; i < expNames.length; i++) {
	 * System.out.println(expNames[i]); } String[] sids = se.getAllSubjects(ui);
	 * for (int i = 0; i < sids.length; i++) { System.out.println(sids[i]); }
	 * 
	 * String s = se.exportAssessmentDataForSubject(ui, "001100000103", 1013);
	 * System.out.println(s); } finally { // se.shutdown(); } }
	 */

}
