package clinical.utils;

import java.io.BufferedReader;
import java.io.EOFException;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

/**
 * A Parser for comma separated value (CSV) files used in spreadsheet and
 * statistical packages like M$ Excel and SPSS.
 * 
 * @author I. Burak Ozyurt
 * @version $Id: CSVParser.java 88 2009-08-17 23:02:38Z bozyurt $
 */
public class CSVParser {
	protected List<List<String>> rows = new ArrayList<List<String>>(1000);

	public CSVParser() {
	}

	/**
	 * Given a CSV file, parses each line to columns and stores the result in
	 * the property rows to be retrieved.
	 * 
	 * @param filename
	 *            the CSV file to be parsed
	 * @throws IOException
	 * @see #getRows
	 */
	public void extractData(String filename) throws IOException {
		BufferedReader in = null;
		try {
			in = new BufferedReader(new FileReader(filename));

			try {
				while (true) {
					List<String> row = extractColumns(in);
					rows.add(row);
				}
			} catch (EOFException ex) {
				// ex.printStackTrace();
			}

		} finally {
			if (in != null)
				try {
					in.close();
				} catch (Exception x) {/* ignore */
				}
		}
	}

	/**
	 * 
	 * @return a list of <code>List</code> objects holding the column values
	 *         extracted as strings
	 */
	public List<List<String>> getRows() {
		return rows;
	}

	/**
	 * Extracts columns from a line of input from the data source and returns a
	 * list of
	 * <code>String<code> objects corresponding to the columns extracted for the parsed line.
	 *
	 * @param in a Reader to get the input for the CSV format data source
	 * @return a list of <code>String<code> objects corresponding to the columns extracted for the parsed line
	 * @throws IOException
	 */
	public List<String> extractColumns(BufferedReader in) throws IOException {
		List<String> cols = new ArrayList<String>(50);
		char[] sepArr = { ',' };
		char[] recSep = { '\n' };
		while (!checkForSeparator(in, recSep)) {
			StringBuffer buf = extractField(in, sepArr, false);
			// System.out.println(buf.toString());
			cols.add(buf.toString());
		}
		return cols;
	}

	private boolean checkForSeparator(BufferedReader in, char[] sepArr)
			throws IOException {
		int c = -1;
		in.mark(sepArr.length + 1);
		c = in.read();
		// System.out.println((char) c);
		if (c == sepArr[0]) {
			// in.mark( sepArr.length );
			boolean sepFound = true;
			for (int i = 1; i < sepArr.length; ++i) {
				c = in.read();
				if (c == -1) {
					throw new EOFException();
				} else if (c != sepArr[i]) {
					in.reset();
					return false;
				}
			}
			if (sepFound)
				return true;
		} else {
			in.reset();
			// System.out.println("checkForSeparator c="+ (char) c);
		}
		return false;
	}

	private StringBuffer extractField(BufferedReader in, char[] sepArr,
			boolean ignoreSpace) throws IOException {
		StringBuffer buf = new StringBuffer();

		boolean useCSV = false;
		boolean finished = false;
		boolean first = true;
		int lastChar = -1;
		while (!finished) {
			in.mark(sepArr.length);
			int c = getChar(in);
			if (c == -1)
				throw new EOFException();
			if (c == '\n' || c == '\r') {
				// encountered an eol
				in.reset();
				break;
			}

			if (first && ignoreSpace) {
				if (c == ' ' || c == '\t')
					continue;
				else if (c == '"' && sepArr[0] == ',') {
					useCSV = true;
				}
			} else {
				if (first && c == '"' && sepArr[0] == ',') {
					useCSV = true;
					first = false;
					continue;
				}
				first = false;
			}

			if (c == sepArr[0]) {
				// test
				if (useCSV && lastChar != '"') {
					buf.append((char) c);
					lastChar = c;
					continue;
				}
				in.mark(sepArr.length);
				boolean sepFound = true;
				for (int i = 1; i < sepArr.length; ++i) {
					c = in.read();
					if (c == -1) {
						finished = true;
						throw new EOFException();
					} else if (c != sepArr[i]) {
						in.reset();
						sepFound = false;
						break;
					}
				}
				if (sepFound)
					break; // number delimiter found so stop parsing
			} else if (useCSV && c == '"') {
				lastChar = c;
				in.mark(1);
				c = in.read();
				if (c == -1) {
					throw new EOFException();
				} else if (c == '"') {
					buf.append((char) c);
					// for cases like "", <some text>"
					lastChar = -1;
				} else {
					in.reset();
				}
			} else {
				buf.append((char) c);
				lastChar = c;
				first = false;
			}
		}
		return buf;

	}

	protected int getChar(BufferedReader in) throws IOException {
		int c = -1;
		c = in.read();
		return c;
	}

}