package clinical.utils;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.input.SAXBuilder;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

import clinical.web.download.Packager;
import clinical.web.download.PathWrapper;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: FileUtils.java 785 2013-03-19 18:20:46Z jinranc $
 */
public class FileUtils {
	protected FileUtils() {}
	
	//Jinran added
	public static void copyFolder(File fSource, File fDest){
		try {
	          if (fSource.isDirectory()) {
	          // A simple validation, if the destination is not exist then create it
	               if (!fDest.exists()) {
	                    fDest.mkdirs();
	               }
	 
	               // Create list of files and directories on the current source
	               // Note: with the recursion 'fSource' changed accordingly
	               String[] fList = fSource.list();
	 
	               for (int index = 0; index < fList.length; index++) {
	                    File dest = new File(fDest, fList[index]);
	                    File source = new File(fSource, fList[index]);
	 
	                    // Recursion call take place here
	                    copyFolder(source, dest);
	               }
	          }
	     }
	     catch (Exception ex) {
	          // Please handle all the relevant exceptions here
	     }
	}

	public static void copyFile(String sourceFile, String destFile)
			throws FileNotFoundException, IOException {

		BufferedInputStream bin = null;
		BufferedOutputStream bout = null;
		try {
			bin = new BufferedInputStream(new FileInputStream(sourceFile));
			bout = new BufferedOutputStream(new FileOutputStream(destFile));
			byte[] buffer = new byte[4096];
			int readBytes = 0;
			while ((readBytes = bin.read(buffer)) != -1) {
				bout.write(buffer, 0, readBytes);
			}

		} finally {
			if (bin != null)
				try {
					bin.close();
				} catch (Exception x) {}
			if (bout != null)
				try {
					bout.close();
				} catch (Exception x) {}
		}
	}

	public static void copyContent(String sourceFile, OutputStream destOut)
			throws FileNotFoundException, IOException {
		BufferedInputStream bin = null;
		try {
			bin = new BufferedInputStream(new FileInputStream(sourceFile));
			byte[] buffer = new byte[4096];
			int readBytes = 0;
			while ((readBytes = bin.read(buffer)) != -1) {
				destOut.write(buffer, 0, readBytes);
			}

		} finally {
			if (bin != null)
				try {
					bin.close();
				} catch (Exception x) {}
		}
	}

	public static String loadTextFile(String filename) throws IOException {
		BufferedReader in = null;
		StringBuffer buf = new StringBuffer(4096);
		try {
			in = new BufferedReader(new FileReader(filename));
			String line = null;

			while ((line = in.readLine()) != null) {
				buf.append(line);
				buf.append('\n');
			}
			return buf.toString();
		} finally {
			if (in != null) {
				try {
					in.close();
				} catch (Exception x) {}
			}
		}
	}

	public static void save2File(String content, String filename)
			throws IOException {
		BufferedWriter out = null;
		try {
			out = new BufferedWriter(new FileWriter(filename));
			out.write(content);
		} finally {
			if (out != null) {
				try {
					out.close();
				} catch (Exception x) {}
			}
		}
	}

	public static Element loadXML(String xmlFile) throws Exception {
		SAXBuilder builder = new SAXBuilder(false);
		Document doc = builder.build(xmlFile);
		return doc.getRootElement();
	}

	public static Element loadXML(Reader reader) throws Exception {
		SAXBuilder builder = new SAXBuilder(false);
		Document doc = builder.build(reader);
		return doc.getRootElement();
	}

	public static void saveXML(Element rootElem, String xmlFile)
			throws Exception {
		XMLOutputter xout = new XMLOutputter(Format.getPrettyFormat());
		BufferedWriter out = null;
		try {
			out = new BufferedWriter(new FileWriter(xmlFile));
			xout.output(rootElem, out);
		} finally {
			if (out != null) {
				try {
					out.close();
				} catch (Exception x) {}
			}
		}
	}

	public static void saveXML(Element rootElem, OutputStream outStream)
			throws IOException {
		XMLOutputter xout = new XMLOutputter(Format.getPrettyFormat());
		try {
			xout.output(rootElem, outStream);
		} finally {
			close(outStream);
		}
	}

	public static void saveXML(Element rootElem, Writer out) throws IOException {
		XMLOutputter xout = new XMLOutputter(Format.getPrettyFormat());
		try {
			xout.output(rootElem, out);
		} finally {
			close(out);
		}
	}

	public static void saveXML(Document doc, Writer out) throws IOException {
		XMLOutputter xout = new XMLOutputter(Format.getPrettyFormat());
		try {
			xout.output(doc, out);
		} finally {
			close(out);
		}
	}

	public static void close(InputStream in) {
		if (in != null) {
			try {
				in.close();
			} catch (Exception x) {}
		}
	}

	public static void close(OutputStream out) {
		if (out != null) {
			try {
				out.close();
			} catch (Exception x) {}
		}
	}

	public static void close(Writer out) {
		if (out != null) {
			try {
				out.close();
			} catch (Exception x) {}
		}
	}

	public static void close(Reader in) {
		if (in != null) {
			try {
				in.close();
			} catch (Exception x) {}
		}
	}

	public static void findLeafPaths(File parent, List<File> dirList) {
		if (parent.isFile()) {
			parent.delete();
		} else if (parent.exists()) {
			File[] files = parent.listFiles();
			if (files == null || files.length == 0) {
				dirList.add(parent);
			} else {
				for (int i = 0; i < files.length; i++) {
					findLeafPaths(files[i], dirList);
				}
			}
		}
	}

	public static void deleteRecursively(File dir) {
		if (dir.isFile()) {
			dir.delete();
		} else if (dir.exists()) {
			File[] files = dir.listFiles();
			if (files != null && files.length > 0) {
				for (int i = 0; i < files.length; i++) {
					if (files[i].isDirectory()) {
						deleteRecursively(files[i]);
					} else {
						files[i].delete();
					}
				}
			}
			dir.delete();
		}
	}

	public static void deleteSubdirs(String rootDir, String path) {
		File f = new File(path);
		List<File> dirList = new ArrayList<File>();
		if (f.isDirectory()) {
			findLeafPaths(f, dirList);
		}

		for (Iterator<File> it = dirList.iterator(); it.hasNext();) {
			File dir = it.next();
			do {
				if (dir.isDirectory()) {
					// don't care if deletion is not successful make an attempt
					dir.delete();
				}
				dir = dir.getParentFile();
			} while (!dir.getPath().equals(rootDir));
		}
	}

	public static String getRelativePath(String rootDir, String path) {
		int idx = path.indexOf(rootDir);
		assert (idx == 0);
		String relPath = path.substring(rootDir.length());
		relPath = relPath.replaceFirst("^\\/+", "");
		return relPath;
	}

	public static String getTopDir(String rootDir, File path) {
		String relPath = getRelativePath(rootDir, path.getAbsolutePath());
		int idx = relPath.indexOf("/");
		if (idx == -1) {
			return null;
		}
		return relPath.substring(0, idx);
	}

	public static boolean buildLocalPath(String rootDir, String relativePath) {
		File parent = new File(rootDir);
		Assertion.assertTrue(parent.isDirectory());
		File f = new File(parent, relativePath);
		f.getParentFile().mkdirs();
		return f.getParentFile().isDirectory();
	}

	/**
	 * build a full path by combining a root directory path and a relative path
	 * making sure the there is only one <code>/</code> between those.
	 * 
	 * @param rootDir
	 * @param relativePath
	 * @return
	 */
	public static String createFullPath(String rootDir, String relativePath) {
		StringBuffer buf = new StringBuffer();
		buf.append(rootDir);
		if (rootDir.endsWith("/")) {
			if (relativePath.startsWith("/")) {
				buf.append(relativePath.substring(1));
			} else {
				buf.append(relativePath);
			}
		} else {
			if (relativePath.startsWith("/")) {
				buf.append(relativePath);
			} else {
				buf.append('/').append(relativePath);
			}
		}
		return buf.toString();
	}

	/**
	 * 
	 * @param rootPath
	 * @return
	 * @throws Exception
	 */
	public static List<File> getAllFilesUnderDir(String rootPath) {
		List<File> paths = new ArrayList<File>();
		File dir = new File(rootPath);
		prepBranch(dir, paths);
		return paths;
	}

	protected static void prepBranch(File parentDir, List<File> paths) {
		if (parentDir == null)
			return;
		File[] files = parentDir.listFiles();
		if (files == null)
			return;
		for (int i = 0; i < files.length; i++) {
			if (files[i].isFile()) {
				paths.add(files[i]);
			} else {
				prepBranch(files[i], paths);
			}
		}
	}

	/**
	 * Given a (composite) object, and a filename, serialize the state of the
	 * object to the given filename.
	 * 
	 * @param o
	 *           the object to serialize
	 * @param filename
	 *           the filename to which the serialized object is written
	 * @throws IOException
	 */
	public static void serialize(Object o, String filename) throws IOException {
		assert (o != null);
		ObjectOutputStream out = null;
		try {
			out = new ObjectOutputStream(new BufferedOutputStream(
					new FileOutputStream(filename), 4096));
			out.writeObject(o);
		} finally {
			if (out != null) {
				try {
					out.close();
				} catch (Exception x) {}
			}
		}
	}

	/**
	 * Resurrects a serialized object.
	 * 
	 * @param filename
	 *           the filename from which the serialized object is read
	 * @return the deserialized object.
	 * @throws IOException
	 * @throws java.lang.ClassNotFoundException
	 */
	public static Object deserialize(String filename) throws IOException,
			ClassNotFoundException {
		assert (filename != null && new File(filename).isFile());
		ObjectInputStream in = null;
		try {
			in = new ObjectInputStream(new BufferedInputStream(
					new FileInputStream(filename), 4096));
			Object o = in.readObject();
			return o;
		} finally {
			if (in != null) {
				try {
					in.close();
				} catch (Exception x) {}
			}
		}
	}

	public static long getTotalSizeUnderDir(String relPath, String localRootDir) {
		File f = new File(localRootDir, relPath);

		List<File> allFilesUnderDir = FileUtils.getAllFilesUnderDir(f
				.getAbsolutePath());
		long totSize = 0;
		for (File file : allFilesUnderDir) {
			if (file.isFile())
				totSize += file.length();
		}
		return totSize;
	}

	public static boolean compressFilesToTarFile(List<File> files, File tarFile) throws Exception {
		 try{			
//			 Tar tar = new Tar();
//			 
//			 tar.setDestFile(tarFile);
//			 
//			 for(File file : files){
//				 tar.setIncludesfile(file);				 
//			 }
//			 tar.execute();
			 
			 Packager pkg = new Packager(files, tarFile,files.get(0).getParent(), tarFile.getName(), 0 );
			 //Packager pkg = new Packager(files.get(0).getParent(), files.get(0).getName(), 0 );
			 pkg.setDoCleanup(false);
			 List<PathWrapper> listFiles = new ArrayList<PathWrapper>();
			 for(File file : files){
				 listFiles.add(new PathWrapper(file.getName(), file.getParent() + "/"));	 
			 }
			 
			 pkg.includeFiles(files.get(0).getParent()+"/", listFiles);
			 pkg.pack();
			 
			 return true;
		 }
		 catch(Exception e){
			 return false;
		 }
		 finally{
			 
		 }
	}
		 
	public static byte[] getBytesFromFile(File file) throws Exception {
		 InputStream is = new FileInputStream(file);
		 // Get the size of the file
		 long length = file.length();
		 if (length > Integer.MAX_VALUE) {
		 // File is too large
		 }
		 // Create the byte array to hold the data
		 byte[] bytes = new byte[(int)length];
		 // Read in the bytes
		 int offset = 0;
		 int numRead = 0;
		 while (offset < bytes.length
		 &&  (numRead=is.read(bytes, offset, bytes.length-offset)) >= 0) {
		 offset += numRead;
		 }
		 // Ensure all the bytes have been read in
		 if (offset < bytes.length){
		 throw new Exception("Could not completely read the file "+file.getName());
		 }
		 // Close the input stream and return bytes
		 is.close();
		 return bytes;
	}
}
