package clinical.web;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.server.dao.DatabaseuserDAO;
import clinical.server.dao.TableidDAO;
import clinical.server.vo.Databaseuser;
import clinical.server.vo.Dataclassification;
import clinical.server.vo.Tableid;
import clinical.utils.DBTableInfo;
import clinical.web.common.IDBCache;
import clinical.web.common.IDBPoolService;
import clinical.web.common.ISecurityService;
import clinical.web.common.UserInfo;
import clinical.web.common.security.DBConfig;
import clinical.web.common.security.User;
import clinical.web.exception.BaseException;
import clinical.web.services.SimpleSecurityService;

/**
 * 
 * A collection of convenience database query methods.
 * 
 * @author I. Burak Ozyurt
 * @version $Id: DBUtils.java 91 2009-08-17 23:38:26Z bozyurt $
 */
public class DBUtils {
	protected static Log log = LogFactory.getLog(DBUtils.class);
	protected static Map<String, Map<String, DBTableInfo>> dbTableInfoMap = new HashMap<String, Map<String, DBTableInfo>>(
			5);

	protected DBUtils() {
	}

	/**
	 * 
	 * @param dbID
	 * @param con
	 *            JDBC connection
	 * @param tableName
	 *            the name of the database table for which the table id is
	 *            looked
	 * @return the table id of the specified table or -1 if the specifed table
	 *         is not foun in the database.
	 * @throws java.lang.Exception
	 */
	public static int getTableID(String dbID, Connection con, String tableName)
			throws Exception {
		Tableid criteria = new Tableid();
		criteria.setTablename(tableName);
		TableidDAO dao = DAOFactory.createTableidDAO(dbID);
		List<Tableid> results = dao.find(con, criteria);
		if (results.isEmpty())
			return -1;
		else
			return ((Tableid) results.get(0)).getTableid().intValue();
	}

	/**
	 * 
	 * @param dbID
	 *            database ID
	 * @param con
	 *            JDBC connection
	 * @param userName
	 *            the name of the user for whom the corresponding
	 *            <code>Databaseuser</code> record will be returned.
	 * 
	 * @return the corresponding <code>Databaseuser</code> record or null if
	 *         there is no record for the specified user
	 * @throws java.lang.Exception
	 */
	public static Databaseuser getDatabaseUser(String dbID, Connection con,
			String userName) throws Exception {
		Databaseuser criteria = new Databaseuser();
		criteria.setName(userName.trim().toUpperCase());
		DatabaseuserDAO dao = DAOFactory.createDatabaseuserDAO(dbID);
		List<Databaseuser> results = dao.find(con, criteria);
		if (results.isEmpty())
			return null;
		return results.get(0);
	}

	/**
	 * 
	 * @param ui
	 *            UserInfo object holding web user information. (Cannot be null)
	 * @param dbID
	 *            unique ID used to identify the connection pools ( or
	 *            indirectly the databases)
	 * @param force
	 *            if true drop the cache (if any) and load table metadata from
	 *            the database
	 * @param dbType
	 *            database type
	 * 
	 * @return an association table of <code>DBTableInfo</code> objects
	 * @see DBTableInfo
	 * @throws java.lang.Exception
	 */
	public static synchronized Map<String, DBTableInfo> prepareTableInfoCache(
			UserInfo ui, String dbID, boolean force, String dbType)
			throws Exception {
		if (!force) {
			String key = dbID + "_" + ui.getName();
			Map<String, DBTableInfo> map = dbTableInfoMap.get(key);
			if (map != null)
				return map;
		}

		int tableCount = 0;
		Map<String, DBTableInfo> dbTableInfoCache = new LinkedHashMap<String, DBTableInfo>(
				71);

		IDBPoolService pool = MinimalServiceFactory.getPoolService(dbID);
		ISecurityService secService = null;
		try {
			secService = MinimalServiceFactory.getSecurityService();
		} catch (BaseException be) {
			log.error(be);
			log.error("Using bootstrap security service instead.");
			// try Simple Security service (mainly for ConnectionSupportMixin
			// class
			// as used by many test cases and installation subsystems)
			secService = SimpleSecurityService.getInstance();
		}
		Map<String, User> userMap = secService.getAllUsers(dbID);
		log.info("retrieving and caching metadata for database " + dbID
				+ " ...");
		Connection con = null;
		IAppConfigService configService = MinimalServiceFactory
				.getAppConfigService();
		try {
			con = pool.getConnection(ui.getName());
			User user = (User) userMap.get(ui.getName());

			DatabaseMetaData dmd = con.getMetaData();
			String schemaName = user.getDbUser().getName().toUpperCase();
			// in case the database user is not a schema owner, read the
			// schema name from properties file
			String mediatorSchemaName = configService
					.getParamValue(Constants.MEDIATOR_SCHEMA_NAME_PROPERTY);
			if (mediatorSchemaName != null
					&& mediatorSchemaName.trim().length() > 0) {
				mediatorSchemaName = mediatorSchemaName.trim();
			}

			ResultSet rs = null;

			if (dbType.equals(DBConfig.ORACLE)) {
				boolean gotMetaData = false;
				rs = dmd.getTables(null, schemaName, "NC_%", new String[] {
						"TABLE", "VIEW" });
				while (rs.next()) {
					gotMetaData = true;
					String tableName = rs.getString(3);
					++tableCount;
					if (log.isDebugEnabled()) {
						log.debug("adding table " + tableName);
					}
					DBTableInfo dbti = new DBTableInfo(tableName, null);
					dbTableInfoCache.put(tableName, dbti);
				}
				if (!gotMetaData && mediatorSchemaName.length() > 0) {
					// try explicitly provided mediator schema name
					// if the database user is not the owner of the db schema
					schemaName = mediatorSchemaName;
					while (rs.next()) {
						gotMetaData = true;
						String tableName = rs.getString(3);
						DBTableInfo dbti = new DBTableInfo(tableName, null);
						dbTableInfoCache.put(tableName, dbti);
						++tableCount;
					}
				}

			} else if (dbType.equals(DBConfig.POSTGRES)) {
				schemaName = null;
				// in some database servers the schema object names are case
				// sensitive like Postgres
				// this is a work-around for this kind of databases
				rs = dmd.getTables(null, schemaName, "nc_%", new String[] {
						"TABLE", "VIEW" });
				while (rs.next()) {
					String tableName = rs.getString(3);
					if (log.isDebugEnabled()) {
						log.debug("adding table " + tableName);
					}
					DBTableInfo dbti = new DBTableInfo(tableName, null);
					dbTableInfoCache.put(tableName, dbti);
					++tableCount;
				}
			} else {
				throw new Exception("Unsupported database type:" + dbType);
			}

			for (Iterator<DBTableInfo> iter = dbTableInfoCache.values()
					.iterator(); iter.hasNext();) {
				DBTableInfo dbti = iter.next();
				rs = dmd.getColumns(null, schemaName, dbti.getTableName(), "%");
				Map<String, String> colNames = new HashMap<String, String>(19);
				while (rs.next()) {
					String colName = rs.getString(4);
					colNames.put(colName.toLowerCase(), colName.toLowerCase());
				}
				dbti.setColumnNames(colNames);
			}

			log.info("cached metadata for " + tableCount + " tables");

		} finally {
			pool.releaseConnection(ui.getName(), con);
		}
		// cache it
		dbTableInfoMap.put(dbID + "_" + ui.getName(), dbTableInfoCache);

		return dbTableInfoCache;
	}

	/**
	 * returns the missing value reason code vs reason hash table from cache
	 * 
	 * @param ui
	 *            UserInfo object holding web user information. (Cannot be null)
	 * @param dbID
	 *            String unique ID used to identify the connection pools ( or
	 *            indirectly the databases)
	 * @throws Exception
	 * @return Map
	 */
	public static Map<Integer, String> getMissingValueReasonMap(UserInfo ui,
			String dbID) throws Exception {
		IDBCache dbCache = MinimalServiceFactory.getDBCache(dbID);
		List<Dataclassification> dcList = dbCache.getDataClassications(ui,
				false);
		Map<Integer, String> dcMap = new HashMap<Integer, String>(17);
		for (Dataclassification dc : dcList) {
			dcMap.put(new Integer(dc.getUniqueid().intValue()), dc.getName());
		}
		return dcMap;
	}

	public static void close(Statement st, ResultSet rs) {
		if (rs != null)
			try {
				rs.close();
			} catch (SQLException x) {
			}
		if (st != null)
			try {
				st.close();
			} catch (SQLException x) {
			}
	}

	public static void close(Statement st) {
		if (st != null)
			try {
				st.close();
			} catch (SQLException x) {
			}
	}

	public static void close(Connection con) {
		if (con != null) {
			try {
				con.close();
			} catch (SQLException x) {
			}
		}
	}

	public static <T> String toInList(List<T> list) {
		StringBuffer buf = new StringBuffer();
		buf.append('(');
		for (Iterator<T> it = list.iterator(); it.hasNext();) {
			T t = it.next();
			if (t instanceof String) {
				buf.append("'").append(t).append("'");
			} else {
				buf.append(t);
			}
			if (it.hasNext())
				buf.append(',');
		}
		buf.append(')');
		return buf.toString();
	}

}
