package clinical.web;

import java.sql.Connection;
import java.util.List;
import java.util.Map;

import clinical.server.vo.Experiment;
import clinical.server.vo.Humansubject;
import clinical.server.vo.Person;
import clinical.server.vo.Researchgroup;
import clinical.server.vo.Subjexperiment;
import clinical.web.common.UserInfo;
import clinical.web.exception.ExperimentManagementException;
import clinical.web.vo.ExperimentInfo;
import clinical.web.vo.StudyGroupInfo;

/**
 * Session facade interface for the experiment/study group management related
 * data services.
 * 
 * @author I. Burak Ozyurt
 * @version $Id: IExperimentManagement.java,v 1.7 2006/05/02 01:06:36 bozyurt
 *          Exp $
 */
public interface IExperimentManagement {
   public final static int SUCCESS = 1;
   public final static int HAS_VISITS = -1;
   public final static int HAS_ERROR = -2;

   public Experiment getExperiment(UserInfo ui, String expName)
         throws ExperimentManagementException;

   /**
    * Returns a particular experiment as an
    * {@link clinical.server.vo.Experiment} object from the database given the
    * experiment ID.
    * 
    * @param ui User Information used to use the correct named database user
    *           connection from the connection pool
    * @param experimentID the ID of the
    *           <code>clinical.server.vo.Experiment</code> object
    * @return the {@link clinical.server.vo.Experiment} object
    * @throws ExperimentManagementException
    */
   public Experiment getExperiment(UserInfo ui, int experimentID)
         throws ExperimentManagementException;

   /**
    * 
    * @param ui User Information used to use the correct named database user
    *           connection from the connection pool
    * @param personID the ID of the {@link clinical.server.vo.Person} object
    * @return the {@link clinical.server.vo.Person} object
    * @throws ExperimentManagementException
    */
   public Person getContactPerson(UserInfo ui, int personID)
         throws ExperimentManagementException;

   /**
    * 
    * @param ui User Information used to use the correct named database user
    *           connection from the connection pool
    * @param experimentID experiment ID of the
    *           {@link clinical.server.vo.Experiment} object
    * @return a list of {@link clinical.server.vo.Subjexperiment} objects
    * @throws ExperimentManagementException
    */
   public List<Subjexperiment> getEnrolledSubjects(UserInfo ui, int experimentID)
         throws ExperimentManagementException;

   /**
    * 
    * @param ui User Information used to use the correct named database user
    *           connection from the connection pool
    * @param experimentID experiment ID of the
    *           {@link clinical.server.vo.Experiment} object
    * @return a list of {@link clinical.server.vo.Humansubject} objects not
    *         associated with the given aexperiment ID
    * @throws ExperimentManagementException
    */
   public List<Humansubject> getRemainingSubjects(UserInfo ui, int experimentID)
         throws ExperimentManagementException;

   /**
    * 
    * @param ui User Information used to use the correct named database user
    *           connection from the connection pool
    * @param experimentID the ID of the {@link clinical.server.vo.Experiment}
    *           object
    * @return
    * @throws ExperimentManagementException
    */
   public Map<Researchgroup, Integer> getResearchGroupSubjectCount(UserInfo ui,
         int experimentID) throws ExperimentManagementException;

   /**
    * 
    * @param ui User Information used to use the correct named database user
    *           connection from the connection pool
    * @return a list of {@link clinical.server.vo.Person} objects
    * @throws ExperimentManagementException
    */
   public List<Person> getAllContactPersons(UserInfo ui)
         throws ExperimentManagementException;

   /**
    * adds a new experiment to the database.
    * 
    * @param ui User Information used to use the correct named database user
    *           connection from the connection pool
    * @param expInfo holds information collected from the user for the
    *           experiment to be created
    * @return the ID of the {@link clinical.server.vo.Experiment} object created
    *         or -1 in case there is an error
    * @throws ExperimentManagementException
    */
   public int addExperiment(UserInfo ui, ExperimentInfo expInfo)
         throws ExperimentManagementException;

   /**
    * 
    * @param ui User Information used to use the correct named database user
    *           connection from the connection pool
    * @return a list of {@link clinical.server.vo.Experiment} object
    * @throws ExperimentManagementException
    */
   public List<Experiment> getAllExperiments(UserInfo ui)
         throws ExperimentManagementException;

   /**
    * 
    * @param ui User Information used to use the correct named database user
    *           connection from the connection pool
    * @param experimentID the experiment ID
    * @param subjectID the ID of the subject to be enrolled with this experiment
    * @param researchGroupID the study group the subject is enrolled
    * @return
    * @throws ExperimentManagementException
    */
   public Subjexperiment enrollSubject(UserInfo ui, int experimentID,
         String subjectID, int researchGroupID)
         throws ExperimentManagementException;

   public Subjexperiment enrollSubject(Connection con, UserInfo ui,
			int experimentID, String subjectID, int researchGroupID)
			throws ExperimentManagementException;
   
   /**
    * 
    * @param ui User Information used to use the correct named database user
    *           connection from the connection pool
    * @param experimentID the experiment ID
    * @param subjectID the ID of the subject to be unenrolled with this
    *           experiment
    * @throws ExperimentManagementException
    * @return int SUCCESS if unenroll was successful, HAS_VISITS if there are
    *         associated visits
    */
   public int unenrollSubject(UserInfo ui, int experimentID, String subjectID)
         throws ExperimentManagementException;

   /**
    * 
    * @param ui User Information used to use the correct named database user
    *           connection from the connection pool
    * @param experimentID the experiment ID
    * @param oldStudyGroupID the current study (research) group ID the subject
    *           is associated
    * @param newStudyGroupID the study (research) group ID the subject will be
    *           associated
    * @param subjectID the ID of the enrolled subject
    * @throws ExperimentManagementException
    */
   public void changeSubjectStudyGroup(UserInfo ui, int experimentID,
         int oldStudyGroupID, int newStudyGroupID, String subjectID)
         throws ExperimentManagementException;

   /**
    * 
    * Persists the updates to one or more fields of the current experiment.
    * 
    * @param ui User Information used to use the correct named database user
    *           connection from the connection pool
    * @param expInfo holds information collected from the user for the
    *           experiment to be updated
    * @throws ExperimentManagementException
    * @see clinical.server.vo.Experiment
    */
   public void updateExperiment(UserInfo ui, ExperimentInfo expInfo)
         throws ExperimentManagementException;

   /**
    * Deletes an experiment and any of the related subject experiment records
    * from the database. This does not delete associated visits/segments and
    * assessments.
    * 
    * @param ui User Information used to use the correct named database user
    *           connection from the connection pool
    * @param experimentID the ID of the experiment to be deleted
    * @throws ExperimentManagementException
    */
   public void deleteExperiment(UserInfo ui, int experimentID)
         throws ExperimentManagementException;

   /**
    * 
    * @param ui User Information used to use the correct named database user
    *           connection from the connection pool
    * @param sgi
    * @return the ID of the {@link clinical.server.vo.Experiment} object created
    *         or -1 in case there is an error
    * @throws ExperimentManagementException
    */
   public int addStudyGroup(UserInfo ui, StudyGroupInfo sgi)
         throws ExperimentManagementException;
}
