package clinical.web;

import java.math.BigDecimal;
import java.sql.Connection;
import java.util.List;

import clinical.server.vo.Rawdata;
import clinical.web.exception.ImageHandlerException;

/**
 * Session facade interface for MRI image handling and image cache management.
 * 
 * @author I. Burak Ozyurt
 * @version $Id: ImageHandler.java 91 2009-08-17 23:38:26Z bozyurt $
 */
public interface ImageHandler {

	/**
	 * Retrieves a particular slice from an image series in SRB and converts it
	 * to JPEG and saves it to outPath.
	 * 
	 * @param subjectID
	 *            the BIRN ID of the subject
	 * @param srbSeries
	 *            the collection holding the MRI image series in DICOM format
	 * @param dcmFile
	 *            the name of a particular DICOM slice file in the series
	 * @param outPath
	 *            the directory path to which the generated JPEG image will be
	 *            saved
	 * @return the full path to the cached jpeg file
	 * @throws ImageHandlerException
	 *             if any error occurs during image retrieval from SRB or during
	 *             DICOM to JPEG conversion
	 */
	public String convertToJpeg(String subjectID, String srbSeries,
			String dcmFile, String outPath) throws ImageHandlerException;

	/**
	 * 
	 * @param dbID
	 *            database ID
	 * @param con
	 *            JDBC connection
	 * @param subjectID
	 *            BIRN ID
	 * @param visitID
	 *            visit ID (expcomponent id)
	 * @param experimentID
	 *            id of the experiment the subject is participating
	 * @param segmentID
	 *            id of the segment for the visit
	 * @return a list of <code>RawData</code> value objects
	 * @throws java.lang.Exception
	 * @see clinical.server.vo.RawData
	 */
	public List<Rawdata> getRawDataForSubject(String dbID, Connection con,
			String subjectID, BigDecimal visitID, BigDecimal experimentID,
			BigDecimal segmentID) throws ImageHandlerException;

	/**
	 * 
	 * @param cacheRoot
	 * @param srbSeries
	 *            the collection holding the MRI image series in DICOM format
	 * @param configService
	 * @return
	 * @throws ImageHandlerException
	 */
	public String retrieveMRISeriesForSubject(String cacheRoot,
			String srbSeries, IAppConfigService configService) throws ImageHandlerException;

	/**
	 * Request a read lock for the particular image series to be used during the
	 * streaming to the client.
	 * 
	 * @param cacheRoot
	 *            the root directory where the image series file cache resides
	 * @param srbSeries
	 *            the collection holding the MRI image series in DICOM format
	 * @param configService
	 *            for external code/script execution like environment
	 *            variables
	 * @return true if lock was granted
	 * @throws java.lang.Exception
	 */
	public boolean lockImageSeries(String cacheRoot, String srbSeries,
			IAppConfigService configService) throws ImageHandlerException;

	/**
	 * Releases the read lock for the particular image series used during the
	 * streaming to the client, so that the cached image series can be reclaimed
	 * by the Cache Cleanup Service.
	 * 
	 * @param cacheRoot
	 *            the root directory where the image series file cache resides
	 * @param srbSeries
	 *            the collection holding the MRI image series in DICOM format
	 * @param configService
	 *            for external code/script execution like environment
	 *            variables
	 * @return true if lock release was successful
	 * @throws java.lang.Exception
	 */
	public boolean unlockImageSeries(String cacheRoot, String srbSeries,
			IAppConfigService configService) throws ImageHandlerException;

	/**
	 * Executes an external process to evict the least recently used files from
	 * the cache. Called periodically from a daemon process.
	 * 
	 * @param cacheRoot
	 *            cacheRoot the root directory where the image series file cache
	 *            resides
	 * @param configService
	 *            properties for configuration
	 * @param maxAgeSecs
	 *            maximum idle time for a file in seconds
	 * @param maxSize
	 *            maximum total size beyond which cache can not grow (a negative
	 *            number means disregard maxSize
	 * @param silent
	 *            if true logs no errors and throws no exceptions
	 * @return
	 * @throws java.lang.Exception
	 */
	public boolean doCacheCleanup(String cacheRoot, IAppConfigService configService,
			long maxAgeSecs, long maxSize, boolean silent)
			throws ImageHandlerException;

	/**
	 * @param dbID
	 * @param con
	 *            JDBC connection
	 * @param expID
	 *            int
	 * @throws ImageHandlerException
	 * @return String
	 */
	public String getExperimentBaseURI(String dbID, Connection con, int expID)
			throws ImageHandlerException;
}
