package clinical.web.helpers.security;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.math.BigDecimal;
import java.sql.Connection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.tools.ant.Project;
import org.apache.tools.ant.input.InputRequest;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.input.SAXBuilder;

import clinical.server.dao.ConfDatabaseDAO;
import clinical.server.dao.ConfDbuserDAO;
import clinical.server.dao.ConfPrivilegeDAO;
import clinical.server.dao.ConfWebuserDAO;
import clinical.server.dao.ConfWebuserPrivDAO;
import clinical.server.dao.SiteDAO;
import clinical.server.vo.ConfDatabase;
import clinical.server.vo.ConfDbuser;
import clinical.server.vo.ConfPrivilege;
import clinical.server.vo.ConfWebuser;
import clinical.server.vo.ConfWebuserPriv;
import clinical.server.vo.Databaseuser;
import clinical.server.vo.Site;
import clinical.utils.Assertion;
import clinical.utils.GenUtils;
import clinical.web.ConnectionSupportMixin;
import clinical.web.DAOFactory;
import clinical.web.ISQLDialect;
import clinical.web.ISequenceHelper;
import clinical.web.MinimalServiceFactory;
import clinical.web.common.query.TSQLProcessor;
import clinical.web.common.security.DBConfig;
import clinical.web.common.security.Privilege;
import clinical.web.common.security.User;
import clinical.web.exception.BaseException;

public class UsersXml2DBConverter {
	protected BufferedReader console;
	protected Project project;
	protected ConnectionSupportMixin csm;
	protected ISQLDialect sqlDialect;
	protected Connection con;
	protected DBConfig currentDBConfig;
	private Map<String, DBConfig> dbConfigMap = new LinkedHashMap<String, DBConfig>(
			7);
	private Map<String, Privilege> privilegesMap = new LinkedHashMap<String, Privilege>(
			23);

	public UsersXml2DBConverter() throws Exception {
		this(null);
	}

	public UsersXml2DBConverter(Project project) throws Exception {
		this.project = project;
		MinimalServiceFactory.setMimimalOpMode(true);
		csm = new ConnectionSupportMixin("users.xml");
		csm.startup();
		this.con = csm.getConnection();
		this.sqlDialect = csm.getSqlDialect();

		if (project == null) {
			console = new BufferedReader(new InputStreamReader(System.in));
		}
	}

	public void shutdown() throws Exception {
		if (csm != null)
			csm.shutdown();
	}

	public void populateDB(String usersXmlFile) throws Exception {
		loadDataFromUsersXml(usersXmlFile);
		TSQLProcessor tsp = new TSQLProcessor(sqlDialect);
		List<?> results = tsp
				.executeQuery(con,
						"select u.* from Databaseuser as u where u.name ='ADMIN' and u.isgroup = false");
		Assertion.assertFalse(results.isEmpty());
		Databaseuser dbUser = (Databaseuser) results.get(0);
		String dbID = csm.getDbID();
		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(dbID);
		ConfPrivilegeDAO cpDAO = DAOFactory.createConfPrivilegeDAO(dbID);
		ConfDatabaseDAO cdDAO = DAOFactory.createConfDatabaseDAO(dbID);
		SiteDAO siteDAO = DAOFactory.createSiteDAO(dbID);

		con.setAutoCommit(false);
		try {
			List<ConfPrivilege> existingPrivList = cpDAO.find(con,
					new ConfPrivilege());
			Set<String> existingPrivNames = new HashSet<String>(7);
			for (ConfPrivilege priv : existingPrivList) {
				existingPrivNames.add(priv.getName());
			}
			for (Privilege priv : privilegesMap.values()) {
				if (existingPrivNames.contains(priv.getName())) {
					continue;
				}
				ConfPrivilege cp = new ConfPrivilege();
				cp.setName(priv.getName());
				cp.setDescription(priv.getDescription());
				cp.setModTime(new Date());
				cp.setOwner(dbUser.getUniqueid());
				cp.setModUser(dbUser.getUniqueid());
				System.out.println(cp);
				cpDAO.insert(con, cp);
			}

			// databases
			int idx = 0;
			List<ConfDatabase> existingCDList = cdDAO
					.find(con, new ConfDatabase());
			Map<String, ConfDatabase> existingDbIdMap = new HashMap<String, ConfDatabase>(
					7);
			for (ConfDatabase cd : existingCDList) {
				existingDbIdMap.put(cd.getDbId(), cd);
			}
			for (DBConfig dbConfig : dbConfigMap.values()) {
				BigDecimal dbUniqueId = new BigDecimal(idx);
				ConfDatabase cd = null;
				boolean newDB = true;
				if (existingDbIdMap.containsKey(dbConfig.getId())) {
					cd = existingDbIdMap.get(dbConfig.getId());
					dbUniqueId = cd.getUniqueId();
					newDB = false;
				} else {
					Site siteCR = new Site();
					siteCR.setSiteid(dbConfig.getSiteID());
					List<Site> sites = siteDAO.find(con, siteCR);
					Assertion.assertTrue(sites.size() == 1);
					Site theSite = sites.get(0);
					dbUniqueId = sequenceHelper.getNextUID(con, "nc_conf_database",
							"uniqueId");
					cd = new ConfDatabase();
					// sequence
					cd.setUniqueId(dbUniqueId);
					cd.setDbId(dbConfig.getId());
					cd.setDbType(dbConfig.getDbType());
					cd.setIsDefault(new Boolean(dbConfig.isDefaultDB()));
					cd.setDbUrl(dbConfig.getDbURL());
					cd.setModTime(new Date());
					cd.setOwner(dbUser.getUniqueid());
					cd.setModUser(dbUser.getUniqueid());
					cd.setForceSchemaOwnerCheck(new Boolean(false));
					cd.setSiteUniqueId(theSite.getUniqueid());
				}

				if (newDB) {
					System.out.println("adding " + cd);
					cdDAO.insert(con, cd);
				}
				Map<String, User> dbUserMap = dbConfig.getDBUserMap();
				persistDBUsers(sequenceHelper, dbUniqueId, dbUserMap, dbUser);
				Map<String, User> webUserMap = dbConfig.getUserMap();
				persistWebUsers(dbUniqueId, sequenceHelper, webUserMap, dbUser);

				++idx;
			}
			con.commit();
			System.out
					.println("finished populating HID with database and user admin data.");
		} catch (Exception x) {
			x.printStackTrace();
			con.rollback();
		}
	}

	protected void persistDBUsers(ISequenceHelper seqHelper,
			BigDecimal dbUniqueId, Map<String, User> dbUserMap, Databaseuser owner)
			throws Exception {
		ConfDbuserDAO dao = DAOFactory.createConfDBuserDAO(csm.getDbID());
		List<ConfDbuser> existingDBUserList = dao.find(con, new ConfDbuser());
		Set<String> existingDBUserSet = new HashSet<String>(7);
		for (ConfDbuser dbUser : existingDBUserList) {
			existingDBUserSet.add(dbUser.getName());
		}

		for (User dbUser : dbUserMap.values()) {
			if (existingDBUserSet.contains(dbUser.getName())) {
				continue;
			}
			BigDecimal uniqueId = seqHelper.getNextUID(con, "nc_conf_dbuser",
					"uniqueId");
			ConfDbuser cu = new ConfDbuser();
			cu.setUniqueId(uniqueId);
			cu.setName(dbUser.getName());
			cu.setPwd(dbUser.getPwd());
			cu.setDbUniqueId(dbUniqueId);
			cu.setModTime(new Date());
			cu.setOwner(owner.getUniqueid());
			cu.setModUser(owner.getUniqueid());

			System.out.println(cu);
			dao.insert(con, cu);
		}
	}

	protected void persistWebUsers(BigDecimal dbUniqueId,
			ISequenceHelper seqHelper, Map<String, User> webUserMap,
			Databaseuser owner) throws Exception {
		ConfWebuserDAO dao = DAOFactory.createConfWebuserDAO(csm.getDbID());
		ConfWebuserPrivDAO privDAO = DAOFactory.createConfWebuserPrivDAO(csm
				.getDbID());

		ConfDbuserDAO duDAO = DAOFactory.createConfDBuserDAO(csm.getDbID());
		ConfDbuser duCR = new ConfDbuser();
		duCR.setDbUniqueId(dbUniqueId);
		List<ConfDbuser> duList = duDAO.find(con, duCR);
		Map<String, ConfDbuser> duMap = new HashMap<String, ConfDbuser>(7);
		for (ConfDbuser du : duList) {
			duMap.put(du.getName(), du);
		}
		duList = null;

		List<ConfWebuser> existingWebUserList = dao.find(con, new ConfWebuser());
		Set<String> existingWebUserSet = new HashSet<String>(7);
		for (ConfWebuser wu : existingWebUserList) {
			existingWebUserSet.add(wu.getName());
		}
		int idx = 1;
		for (User webUser : webUserMap.values()) {
			if (existingWebUserSet.contains(webUser.getName())) {
				pwd2MD5(webUser, dao);
				continue;
			}
			ConfWebuser wu = new ConfWebuser();

			BigDecimal wuUniqueId = new BigDecimal(idx);
			wuUniqueId = seqHelper.getNextUID(con, "nc_conf_webuser", "uniqueId");

			wu.setUniqueId(wuUniqueId);
			wu.setDbUniqueId(dbUniqueId);
			wu.setName(webUser.getName());
			ConfDbuser du = duMap.get(webUser.getDbUser().getName());
			Assertion.assertNotNull(du);

			wu.setDbuserId(du.getUniqueId());
			String digest = webUser.getPwd();
			if (!isAMD5Digest(webUser.getPwd())) {
				digest = GenUtils.getMD5Digest(webUser.getPwd());
			}
			wu.setPwd(digest);
			wu.setModTime(new Date());
			wu.setOwner(owner.getUniqueid());
			wu.setModUser(owner.getUniqueid());
			System.out.println(wu);
			dao.insert(con, wu);

			List<Privilege> privList = webUser.listPrivileges();
			int privIdx = 1;
			for (Privilege priv : privList) {
				ConfWebuserPriv cwp = new ConfWebuserPriv();
				BigDecimal privUniqueId = new BigDecimal(privIdx);
				privUniqueId = seqHelper.getNextUID(con, "nc_conf_webuser_priv",
						"uniqueId");
				cwp.setUniqueId(privUniqueId);
				cwp.setWebuserId(wuUniqueId);
				cwp.setPrivName(priv.getName());
				cwp.setModTime(new Date());
				cwp.setOwner(owner.getUniqueid());
				cwp.setModUser(owner.getUniqueid());
				System.out.println(cwp);

				privDAO.insert(con, cwp);
				++privIdx;
			}
			++idx;
		}
	}

	protected void pwd2MD5(User webUser, ConfWebuserDAO dao) throws Exception {
		ConfWebuser cr = new ConfWebuser();
		cr.setName(webUser.getName());
		List<ConfWebuser> list = dao.find(con, cr);
		if (list.isEmpty())
			return;
		ConfWebuser wu = list.get(0);
		if (!isAMD5Digest(wu.getPwd())) {
			String digest = GenUtils.getMD5Digest(wu.getPwd());
			cr.setUniqueId(wu.getUniqueId());
			ConfWebuser bean = new ConfWebuser();
			bean.setPwd(digest);
			dao.update(con, bean, cr);
		}
	}

	public static boolean isAMD5Digest(String s) {
		if (s.length() != 32)
			return false;
		char[] carr = s.toCharArray();
		for (int i = 0; i < carr.length; i++) {
			char c = carr[i];
			if (!Character.isDigit(c) && c != 'a' && c != 'b' && c != 'c'
					&& c != 'd' && c != 'e' && c != 'f') {
				return false;
			}
		}
		return true;
	}

	public String getInput(String prompt) throws IOException {
		while (true) {
			if (console != null) {
				System.out.print(prompt);
				String ans = console.readLine().trim();
				if (ans.length() > 0) {
					return ans;
				}
			} else {

				InputRequest request = new InputRequest(prompt);
				project.getInputHandler().handleInput(request);
				String ans = request.getInput().trim();
				if (ans.length() > 0) {
					return ans;
				}
			}
		}
	}

	public void loadDataFromUsersXml(String usersXmlFile) throws Exception {

		SAXBuilder builder = null;
		builder = new SAXBuilder(false);
		Document doc = builder.build(new File(usersXmlFile));
		Element root = doc.getRootElement();
		Element databasesElem = root.getChild("databases");
		List<?> dbChildren = databasesElem.getChildren("database");
		boolean first = true;
		for (Iterator<?> iter = dbChildren.iterator(); iter.hasNext();) {
			Element elem = (Element) iter.next();
			String dbID = elem.getAttributeValue("id");
			String siteId = null;
			String siteName = null;
			if (elem.getAttribute("siteId") != null) {
				siteId = elem.getAttributeValue("siteId");
			}
			if (elem.getAttribute("siteName") != null) {
				siteName = elem.getAttributeValue("siteName");
			}
			if (siteName == null || siteId == null) {
				System.out.println("Site Information:");
				System.out
						.println("This information must match the corresponding row for '"
								+ dbID + "' in nc_site table of your HID.");
				siteId = getInput("Zero padded four Digit SiteID (e.g. 0008):");
				siteName = getInput("Site Name (e.g. UCSD):");
			}
			DBConfig dbConfig = new DBConfig(elem.getAttributeValue("id"),
					siteName, siteId);
			Element dbURLElem = (Element) elem.getChild("db-url");
			dbConfig.setDbURL(dbURLElem.getText().trim());
			dbConfigMap.put(dbConfig.getId(), dbConfig);

			Element dbTypeElem = (Element) elem.getChild("db-type");
			if (dbTypeElem != null) {
				dbConfig.setDbType(dbTypeElem.getText().trim());
			} else {
				// default database type is oracle
				dbConfig.setDbType(DBConfig.ORACLE);
			}

			if (first) {
				currentDBConfig = dbConfig;
				first = false;
			}
			if (elem.getAttribute("default") != null) {
				if (elem.getAttributeValue("default").equalsIgnoreCase("true")) {
					currentDBConfig = dbConfig;
					dbConfig.setDefaultDB(true);
				}
			}

			if (elem.getAttribute("force-schema-owner-check") != null) {
				if (elem.getAttributeValue("force-schema-owner-check")
						.equalsIgnoreCase("true")) {
					dbConfig.setForceSchemaOwnerCheck(true);
				}
			}
		}
		currentDBConfig.setDefaultDB(true);

		Element dbUsersElem = root.getChild("dbusers");
		List<?> children = dbUsersElem.getChildren("dbuser");
		for (Iterator<?> it = children.iterator(); it.hasNext();) {
			Element elem = (Element) it.next();
			String dbID = elem.getAttributeValue("dbid");
			DBConfig dbConfig = dbConfigMap.get(dbID);
			if (dbConfig == null)
				throw new BaseException(dbID + " is not a valid database id!");
			User dbUser = new User(elem.getAttributeValue("name"), elem
					.getAttributeValue("pwd"));
			dbConfig.addDbUser(dbUser);

			// dbUserMap.put(dbUser.getName(), dbUser);
		}
		Element privsElem = root.getChild("privileges");
		children = privsElem.getChildren("privilege");
		for (Iterator<?> it = children.iterator(); it.hasNext();) {
			Element elem = (Element) it.next();
			String name = elem.getAttributeValue("name");
			Privilege privilege = new Privilege(name);
			Element descElem = elem.getChild("description");
			if (descElem != null) {
				privilege.setDescription(descElem.getText());
			}
			privilegesMap.put(name, privilege);

		}

		Element usersElem = root.getChild("users");
		children = usersElem.getChildren("user");
		for (Iterator<?> it = children.iterator(); it.hasNext();) {
			Element elem = (Element) it.next();
			String dbUserName = elem.getAttributeValue("dbuser");
			String dbID = elem.getAttributeValue("dbid");
			DBConfig dbConfig = dbConfigMap.get(dbID);
			if (dbConfig == null)
				throw new BaseException(dbID + " is not a valid database id!");

			User user = new User(elem.getAttributeValue("name"), elem
					.getAttributeValue("pwd"));
			User dbUser = dbConfig.getDBUser(dbUserName);
			// User dbUser = (User) dbUserMap.get(dbUserName);
			if (dbUser == null)
				throw new BaseException("Database User <" + dbUserName
						+ "> does not exists!");
			user.setDbUser(dbUser);
			// get privileges also
			setUserPrivileges(elem, user);
			dbConfig.addUser(user);
		}

	}

	protected void setUserPrivileges(Element userElm, User user) {
		Element privsElem = userElm.getChild("privileges");
		if (privsElem == null)
			return;
		List<?> children = privsElem.getChildren("privilege");
		for (Iterator<?> it = children.iterator(); it.hasNext();) {
			Element privElem = (Element) it.next();
			if (privElem.getAttributeValue("name").equals("all")) {

				for (Iterator<Privilege> it2 = privilegesMap.values().iterator(); it2
						.hasNext();) {
					user.addPrivilege(it2.next());
				}
				break;
			}
			Privilege priv = privilegesMap.get(privElem.getAttributeValue("name"));
			if (priv != null)
				user.addPrivilege(priv);
		}
	}

	public static void main(String[] args) throws Exception {
		UsersXml2DBConverter converter = null;
		String usersXmlFile = "/home/bozyurt/dev/java/BIRN/clinical/conf/users.xml";
		try {
			converter = new UsersXml2DBConverter();

			converter.populateDB(usersXmlFile);
		} finally {
			if (converter != null)
				converter.shutdown();
		}
	}
}
