package clinical.web.scheduler;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.utils.FileUtils;
import clinical.web.Constants;
import clinical.web.IJobManagementService;
import clinical.web.ServiceFactory;
import clinical.web.common.ICachePolicy;
import clinical.web.common.UserInfo;
import clinical.web.exception.BaseException;

/**
 * This class is responsible for scheduling of asynchronous jobs submitted via
 * the web interface. The asynchronous jobs are mainly used for long running
 * OLAP like queries and other long running tasks.
 *
 * @author I. Burak Ozyurt
 * @version $Id: JobScheduler.java 498 2011-12-16 01:06:29Z jinranc $
 */
public class JobScheduler implements Runnable {
	protected List<JobInfo> jobQueue = Collections
			.synchronizedList(new ArrayList<JobInfo>());
	protected Map<String, Thread> threadMap = Collections
			.synchronizedMap(new HashMap<String, Thread>(7));
	protected Map<String, IJob> jobIDMap = Collections
			.synchronizedMap(new HashMap<String, IJob>());
	protected Map<String, JobRecord> jobRecordMap = Collections
			.synchronizedMap(new HashMap<String, JobRecord>());
	protected ICachePolicy policy;

	/**
	 * the root directory where the finished processes will write their
	 * results/logs.
	 */
	protected String outputRootDir;
	protected int maxConcurrentJobs = 10;
	protected int runningJobCount = 0;
	private Log log = LogFactory.getLog(JobScheduler.class);

	protected static JobScheduler instance = null;

	protected JobScheduler(String outputRootDir, ICachePolicy policy) throws JobSchedulerException {
		super();
		this.outputRootDir = outputRootDir;
		this.policy = policy;
	}

	public static synchronized JobScheduler getInstance(String outputRootDir, ICachePolicy policy)
			throws JobSchedulerException {
		if (instance == null) {
			instance = new JobScheduler(outputRootDir, policy);
		}
		return instance;
	}

	public static synchronized JobScheduler getInstance()
			throws JobSchedulerException {
		if (instance == null) {
			throw new JobSchedulerException("No proper initialization!");
		}
		return instance;
	}

	public void shutdown() {

	}

	public synchronized void addJob(IJob job) throws BaseException {
		if (!jobIDMap.containsKey(job.getID())) {
			jobIDMap.put(job.getID(), job);
			jobQueue.add(new JobInfo(job));
			IJobManagementService jms = ServiceFactory
					.getJobManagementService(job.getDbID());
			JobRecord jr = new JobRecord(job.getUser(), job.getType(), job
					.getID(), new Date(), JobInfo.NOT_STARTED, job
					.getDescription(), job.getResultsFiles(), true);
			jms.addJob(job.getUserInfo(), jr);

			this.notify();
		}
	}

	protected String prepKey(JobRecord jr) {
		StringBuffer buf = new StringBuffer();
		buf.append(jr.jobID).append('_').append(jr.user);
		return buf.toString();
	}

	protected String prepKey(String jobUser, String jobID) {
		StringBuffer buf = new StringBuffer();
		buf.append(jobID).append('_').append(jobUser);
		return buf.toString();
	}

	public JobRecord getJobRecord(UserInfo ui, String dbID, String user,
			String jobID) throws BaseException {
		IJobManagementService jms = ServiceFactory
				.getJobManagementService(dbID);
		JobRecord jr = jms.findJob(ui, user, jobID);
		return jr;
	}

	public File getResultsFileForJob(String user, String jobID) {
		return new File(getOutputRootDir(), jobID + ".csv");
	}

	public File getGeneExportFileForJob(String user, String jobID, String fileType){
		return new File(getOutputRootDir(), jobID + "." + fileType);
		
	}

	public List<JobRecord> getJobsForUser(UserInfo ui, String dbID, String user)
			throws BaseException {
		IJobManagementService jms = ServiceFactory
				.getJobManagementService(dbID);

		List<JobRecord> jrList = jms.getJobsForUser(ui, user);
		for (Iterator<JobRecord> it = jrList.iterator(); it.hasNext();) {
			JobRecord jr = it.next();
			if (jr.getStatus().endsWith("_expired")) {
				it.remove();
			}
		}
		Collections.sort(jrList, new Comparator<JobRecord>() {
			public int compare(JobRecord jr1, JobRecord jr2) {
				return jr2.date.compareTo(jr1.date);
			}
		});

		return jrList;
	}

	public void run() {
		long checkInterval = 5000L;
		while (true) {
			synchronized (this) {
				try {
					this.wait(checkInterval);
				} catch (InterruptedException e) {
				}
			}
			if (jobQueue.isEmpty()) {
				continue;
			}

			synchronized (this) {
				// first cleanup any finished jobs
				for (Iterator<JobInfo> iter = jobQueue.iterator(); iter
						.hasNext();) {
					JobInfo jobInfo = iter.next();
					String jobID = jobInfo.getJob().getID();

					if (jobInfo.getStatus().equals(JobInfo.FINISHED)
							|| jobInfo.getStatus().equals(
									JobInfo.FINISHED_WITH_ERR)) {
						try {
							jobIDMap.remove(jobID);
							updateJobDB(jobInfo, jobID);
						} catch (BaseException e) {
							log.error("", e);
						} finally {
							iter.remove();
							log.info("removed job " + jobInfo.getJob().getID()
									+ " for user:" + jobInfo.getJob().getUser()
									+ " from job queue.");
							threadMap.remove(jobID);
						}
					}
				}

				if (threadMap.size() >= maxConcurrentJobs) {
					continue;
				}

				boolean hasEnoughSpace = hasEnoughSpace();
				for (Iterator<JobInfo> iter = jobQueue.iterator(); iter
						.hasNext();) {
					JobInfo jobInfo = iter.next();
					String jobID = jobInfo.getJob().getID();
					if (jobInfo.getStatus().equals(JobInfo.NOT_STARTED)) {
						if (threadMap.size() < maxConcurrentJobs
								&& hasEnoughSpace) {
							JobRunner runner = new JobRunner(this, jobInfo);
							log.info("*** starting job"
									+ jobInfo.getJob().getID());

							jobInfo.setStatus(JobInfo.RUNNING);
							try {
								updateJobDB(jobInfo, jobID);
								Thread thread = new Thread(runner);
								thread.setDaemon(true);
								thread.setPriority(Thread.NORM_PRIORITY - 1);
								threadMap.put(jobID, thread);
								thread.start();
								jobInfo.setStatus(JobInfo.RUNNING);
								updateJobDB(jobInfo, jobID);
							} catch (BaseException e) {
								e.printStackTrace();
								jobInfo.setStatus(JobInfo.FINISHED_WITH_ERR);
								jobInfo.setErrorMsg(e.getMessage());
								try {
									updateJobDB(jobInfo, jobID);
								} catch (Exception x) {
								}
							}
						}
					} else if (jobInfo.getStatus().equals(JobInfo.FINISHED)
							|| jobInfo.getStatus().equals(
									JobInfo.FINISHED_WITH_ERR)) {
						try {
							jobIDMap.remove(jobID);
							updateJobDB(jobInfo, jobID);
						} catch (BaseException e) {
							e.printStackTrace();
						} finally {
							iter.remove();
							threadMap.remove(jobID);
						}
					}
				}
			} // synchronized
		} // while
	}

	public void updateJob(UserInfo ui, String dbID, JobRecord jr)
			throws BaseException {
		IJobManagementService jms = ServiceFactory
				.getJobManagementService(dbID);
		jms.updateJob(ui, jr);
	}

	public void updateJobDB(JobInfo jobInfo, String jobID) throws BaseException {
		try {
			String user = jobInfo.getJob().getUser();
			String jobType = jobInfo.getJob().getType();
			String key = jobID + "_" + user;

			UserInfo ui = jobInfo.getJob().getUserInfo();
			IJobManagementService jms = ServiceFactory
					.getJobManagementService(jobInfo.getJob().getDbID());
			JobRecord jr = jms.findJob(ui, user, jobID);

			if (jr == null) {
				jr = new JobRecord(user, jobType, jobID, new Date(), jobInfo
						.getStatus(), jobInfo.getJob().getDescription(),
						jobInfo.getJob().getResultsFiles(), true);
				if (jobInfo.getErrorMsg() != null) {
					jr.setErrorMsg(jobInfo.getErrorMsg());
				}
				jms.addJob(ui, jr);
			} else {
				jr.setStatus(jobInfo.getStatus());
				jr.setDate(new Date());
				if (jobInfo.getErrorMsg() != null) {
					jr.setErrorMsg(jobInfo.getErrorMsg());
				}
				if (JobInfo.FINISHED.equals(jr.getStatus())
						&& jr.getSavedResultFiles() != null) {
					long jobSize = 0;
					// FIXME total job size
					for (int i = 0; i < jr.getSavedResultFiles().length; i++) {
						File f = new File(jr.getSavedResultFiles()[i]);
						if (f.isFile()) {
							jobSize += f.length();
						}
					}

					jr.setJobSize(jobSize);
				}
				jr.setDateFinished(new Date());
				jms.updateJob(ui, jr);
			}

			jobRecordMap.put(key, jr);
		} catch (Throwable t) {
			t.printStackTrace();
			throw new BaseException(t);
		}
	}

	String getOutputRootDir() {
		return outputRootDir;
	}

	// 8/2/07
	public void removeJob(UserInfo ui, String dbID, String user, String jobID)
			throws BaseException {
		System.out.println("**** removeJob");
		String key = jobID + "_" + user;
		IJobManagementService jms = ServiceFactory
				.getJobManagementService(dbID);

		synchronized (this) {
			JobRecord jrLocal = jobRecordMap.get(key);
			if (jrLocal != null) {
				jobRecordMap.remove(key);
			}
		}
		JobRecord jr = jms.findJob(ui, user, jobID);
		if (jr != null) {
			System.out.println("Removing job:" + jr.toString());
			jr.setStatus(JobInfo.REMOVED);
			jms.updateJob(ui, jr);
			cleanupCache(jobID);
		}
	}

	public synchronized void cleanupCache(String jobID) {
		File dir = new File(outputRootDir, jobID);
		FileUtils.deleteSubdirs(outputRootDir, dir.getAbsolutePath());
		FileUtils.deleteRecursively(dir);
	}

	public void cancelJob(UserInfo ui, String dbID, String user, String jobID)
			throws BaseException {
		String key = jobID + "_" + user;
		JobRecord jrLocal = jobRecordMap.get(key);
		if (jrLocal == null)
			return;
		IJobManagementService jms = ServiceFactory
				.getJobManagementService(dbID);

		synchronized (this) {
			for (Iterator<JobInfo> iter = jobQueue.iterator(); iter.hasNext();) {
				JobInfo jobInfo = iter.next();
				if (jobInfo.getJob().getID().equals(jobID)
						&& jobInfo.getJob().getUser().equals(user)) {
					jobInfo.getJob().cancel();
					iter.remove();
					break;
				}
			}
			jobRecordMap.remove(key);
		}

		JobRecord jr = jms.findJob(ui, user, jobID);
		if (jr != null) {
			jr.setStatus(JobInfo.CANCELED);
			jms.updateJob(ui, jr);
		}
	}

	protected boolean hasEnoughSpace() {
		return !policy.isBelowLowMark(new File(outputRootDir).getUsableSpace());
	}

}
