package clinical.web.services;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts.util.LabelValueBean;

import clinical.server.dao.ConfParamsDAO;
import clinical.server.dao.ConfParamsGroupDAO;
import clinical.server.dao.postgres.VisittypeDAO;
import clinical.server.vo.ConfParams;
import clinical.server.vo.ConfParamsGroup;
import clinical.server.vo.Databaseuser;
import clinical.server.vo.Researchgroup;
import clinical.server.vo.Researchgrouptype;
import clinical.server.vo.Tableid;
import clinical.server.vo.Visittype;
import clinical.utils.Assertion;
import clinical.utils.GenUtils;
import clinical.web.Constants;
import clinical.web.DAOFactory;
import clinical.web.DBUtils;
import clinical.web.IAppConfigService;
import clinical.web.ISQLDialect;
import clinical.web.ISequenceHelper;
import clinical.web.MinimalServiceFactory;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.IDBPoolService;
import clinical.web.common.UserInfo;
import clinical.web.common.query.TSQLProcessor;
import clinical.web.exception.BaseException;
import clinical.web.vo.Visit;

/**
 * @author I. Burak Ozyurt
 * @version $Id: AppConfigService.java 763 2013-02-01 18:36:31Z jinranc $
 */
public class AppConfigService implements IAppConfigService {
	private Map<String, ConfParamsGroup> paramGroupMap = Collections
			.synchronizedMap(new HashMap<String, ConfParamsGroup>(11));
	private Map<BigDecimal, ConfParamsGroup> pgByIdMap = Collections
			.synchronizedMap(new HashMap<BigDecimal, ConfParamsGroup>(11));
	private Map<String, ConfParams> paramMap = Collections
			.synchronizedMap(new HashMap<String, ConfParams>());

	private String dbID;
	private IDBPoolService poolService;
	private static AppConfigService instance = null;
	private Log log = LogFactory.getLog("app-config");

	private AppConfigService(IDBPoolService poolService, String dbID)
			throws BaseException {
		this.poolService = poolService;
		this.dbID = dbID;
		Connection con = null;
		try {
			con = poolService.getConnection(Constants.ADMIN_USER);
			loadAllParams(con);
		} catch (Exception e) {
			if (e instanceof BaseException) {
				throw (BaseException) e;
			}
			throw new BaseException(e);
		} finally {
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}

	public static synchronized AppConfigService getInstance(
			IDBPoolService poolService, String dbID) throws BaseException {
		if (instance == null) {
			instance = new AppConfigService(poolService, dbID);
		}
		return instance;
	}

	/**
	 * 
	 * @return the singleton AppConfigService
	 * @throws BaseException
	 *             if not created before
	 */
	public static synchronized AppConfigService getInstance()
			throws BaseException {
		if (instance == null)
			throw new BaseException(
					"Application Configuration Service is not initialized!");
		return instance;
	}

	public ConfParamsGroup addParameterGroup(String groupName,
			String description) throws Exception {
		Connection con = null;
		try {
			con = poolService.getConnection(Constants.ADMIN_USER);
			con.setAutoCommit(false);
			ConfParamsGroup pg = addParameterGroup(con, groupName, description);
			con.commit();
			synchronized (this) {
				paramGroupMap.put(pg.getGroupName(), pg);
				pgByIdMap.put(pg.getUniqueId(), pg);
			}
			return pg;
		} catch (Exception x) {
			handleErrorAndRollBack(con, "", x);
			return null;
		} finally {
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}

	public void removeParameterGroup(String groupName) throws Exception {
		Connection con = null;
		try {
			con = poolService.getConnection(Constants.ADMIN_USER);
			con.setAutoCommit(false);
			removeParameterGroup(groupName);
			con.commit();
			synchronized (this) {
				paramGroupMap.remove(groupName);
				ConfParamsGroup thePG = null;
				for (ConfParamsGroup pg : pgByIdMap.values()) {
					if (pg.getGroupName().equals(groupName)) {
						thePG = pg;
						break;
					}
				}
				pgByIdMap.remove(thePG.getUniqueId());
			}
		} catch (Exception x) {
			handleErrorAndRollBack(con, "", x);
		} finally {
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}

	public void addParameter(String name, String value, String type,
			String description, String groupName) throws Exception {
		Connection con = null;
		try {
			con = poolService.getConnection(Constants.ADMIN_USER);
			con.setAutoCommit(false);
			ConfParams p = addParameter(con, name, value, type, description,
					groupName);
			con.commit();
			paramMap.put(p.getName(), p);
		} catch (Exception x) {
			handleErrorAndRollBack(con, "", x);
		} finally {
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}

	public void updateParameter(String name, String value) throws Exception {
		Connection con = null;
		try {
			con = poolService.getConnection(Constants.ADMIN_USER);
			con.setAutoCommit(false);
			updateParameter(con, name, value);
			con.commit();
			synchronized (this) {
				ConfParams p = paramMap.get(name);
				p.setValue(value);
			}
		} catch (Exception x) {
			handleErrorAndRollBack(con, "", x);
		} finally {
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}

	public void removeParameter(String name) throws Exception {
		Connection con = null;
		try {
			con = poolService.getConnection(Constants.ADMIN_USER);
			con.setAutoCommit(false);
			removeParameter(con, name);
			con.commit();
			paramMap.remove(name);
		} catch (Exception x) {
			handleErrorAndRollBack(con, "", x);
		} finally {
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}

	public ConfParams getParam(String name) {
		return paramMap.get(name);
	}

	public String getParamValue(String name) {
		ConfParams cp = paramMap.get(name);
		if (cp == null || cp.getValue() == null)
			return null;
		return cp.getValue().trim();
	}

	public String[] getParamValues(String name) {
		
		List<ParamOrder> poList = new ArrayList<ParamOrder>(10);
		for(String pn : paramMap.keySet()) {
			if ( pn.startsWith(name)) {
				ConfParams cp = paramMap.get(pn);
		        String s = pn.substring(name.length() + 1);
		        int idx = GenUtils.toInt(s, -1);
		        if (idx != -1) {
		        	ParamOrder po = new ParamOrder(idx, cp.getValue());
		        	poList.add(po);
		        }		        
			}
		}
		Collections.sort(poList, new Comparator<ParamOrder>() {
			public int compare(ParamOrder o1, ParamOrder o2) {
				return o1.idx - o2.idx;
			}			
		});
		String[] values = new String[poList.size()];
		int i = 0;
		for(ParamOrder po : poList) {
			values[i++] = po.value;
		}
		return values;
	}
	
	class ParamOrder {
		int idx;
		String value;
		public ParamOrder(int idx, String value) {
			this.idx = idx;
			this.value = value;
		}
	}//;
	


	public synchronized ConfParamsGroup getGroupForParam(String name) {
		ConfParams p = paramMap.get(name);
		if (p == null)
			return null;
		return pgByIdMap.get(p.getParamGroupId());
	}

	public List<ConfParamsGroup> getAllParamGroups() {
		return new ArrayList<ConfParamsGroup>(pgByIdMap.values());
	}

	public List<ConfParams> getAllParams() {
		return new ArrayList<ConfParams>(paramMap.values());
	}

	public ConfParamsGroup getParamGroup(String groupName) {
		return paramGroupMap.get(groupName);
	}

	protected ConfParamsGroup addParameterGroup(Connection con,
			String groupName, String description) throws Exception {
		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(dbID);
		ConfParamsGroupDAO pgDAO = DAOFactory.createConfParamsGroupDAO(dbID);
		Databaseuser d = getDatabaseUser(con);
		ConfParamsGroup pg = new ConfParamsGroup();
		pg.setGroupName(groupName);
		pg.setDescription(description);
		pg.setOwner(d.getUniqueid());
		pg.setModUser(d.getUniqueid());
		pg.setModTime(new Date());
		BigDecimal uniqueId = sequenceHelper.getNextUID(con,
				"nc_conf_params_group", "uniqueId");
		pg.setUniqueId(uniqueId);
		pgDAO.insert(con, pg);
		return pg;
	}

	protected void removeParameterGroup(Connection con, String groupName)
			throws Exception {
		ConfParamsGroupDAO pgDAO = DAOFactory.createConfParamsGroupDAO(dbID);
		ConfParamsDAO pDAO = DAOFactory.createConfParamsDAO(dbID);

		ConfParamsGroup cr = new ConfParamsGroup();
		cr.setGroupName(groupName);
		List<ConfParamsGroup> pgList = pgDAO.find(con, cr);
		if (pgList.isEmpty()) {
			return;
		}
		ConfParamsGroup pg = pgList.get(0);
		ConfParams pcr = new ConfParams();
		pcr.setParamGroupId(pg.getUniqueId());
		pDAO.delete(con, pcr);
		pgDAO.delete(con, cr);
	}

	protected ConfParams addParameter(Connection con, String name,
			String value, String type, String description, String groupName)
			throws Exception {
		ConfParamsGroupDAO pgDAO = DAOFactory.createConfParamsGroupDAO(dbID);
		ConfParamsDAO pDAO = DAOFactory.createConfParamsDAO(dbID);

		ConfParamsGroup pgCR = new ConfParamsGroup();
		pgCR.setGroupName(groupName);
		List<ConfParamsGroup> pgList = pgDAO.find(con, pgCR);
		if (pgList.isEmpty() || pgList.size() != 1) {
			throw new Exception(
					"Nonexistent or not unique parameter group name:"
							+ groupName);
		}
		ConfParamsGroup pg = pgList.get(0);
		ConfParams p = new ConfParams();
		p.setName(name);
		p.setValue(value);
		p.setType(type);
		p.setDescription(description);
		p.setParamGroupId(pg.getUniqueId());
		Databaseuser d = getDatabaseUser(con);
		p.setOwner(d.getUniqueid());
		p.setModUser(d.getUniqueid());
		p.setModTime(new Date());
		pDAO.insert(con, p);
		return p;
	}

	public void updateParameter(Connection con, String name, String value)
			throws Exception {
		ConfParamsDAO pDAO = DAOFactory.createConfParamsDAO(dbID);
		ConfParams cr = new ConfParams();
		cr.setName(name);
		List<ConfParams> pList = pDAO.find(con, cr);
		if (!pList.isEmpty()) {
			ConfParams p = new ConfParams();
			p.setValue(value);
			pDAO.update(con, p, cr);
		}
	}

	protected void removeParameter(Connection con, String name)
			throws Exception {
		ConfParamsDAO pDAO = DAOFactory.createConfParamsDAO(dbID);
		ConfParams cr = new ConfParams();
		cr.setName(name);
		pDAO.delete(con, cr);
	}

	protected synchronized void loadAllParams(Connection con) throws Exception {
		ConfParamsDAO pDAO = DAOFactory.createConfParamsDAO(dbID);
		ConfParamsGroupDAO pgDAO = DAOFactory.createConfParamsGroupDAO(dbID);

		List<ConfParamsGroup> pgList = pgDAO.find(con, new ConfParamsGroup());
		List<ConfParams> pList = pDAO.find(con, new ConfParams());
		for (ConfParams p : pList) {
			paramMap.put(p.getName(), p);
		}

		for (ConfParamsGroup pg : pgList) {
			paramGroupMap.put(pg.getGroupName(), pg);
			pgByIdMap.put(pg.getUniqueId(), pg);
		}
	}

	protected Databaseuser getDatabaseUser(Connection con) throws Exception {
		ISQLDialect sqlDialect = MinimalServiceFactory.getSQLDialect(dbID);
		TSQLProcessor tsp = new TSQLProcessor(sqlDialect);
		List<?> results = tsp
				.executeQuery(con,
						"select u.* from Databaseuser as u where u.name ='ADMIN' and u.isgroup = false");
		Assertion.assertFalse(results.isEmpty());
		Databaseuser dbUser = (Databaseuser) results.get(0);
		return dbUser;
	}

	protected void handleErrorAndRollBack(Connection con, String msg,
			Exception x) throws BaseException {
		if (con != null) {
			try {
				con.rollback();
			} catch (SQLException se) {
				log.error("", se);
				throw new BaseException(se);
			}
		}
		log.error(msg, x);
		if (x instanceof BaseException)
			throw (BaseException) x;
		else
			throw new BaseException(x);
	}
	
	
	/*
	 * Retrieve all visit types from table nc_visittype
	 */
	public Collection<LabelValueBean> getAllVisitTypes() throws Exception{
		Connection con = null;
		try {
			con = poolService.getConnection(Constants.ADMIN_USER);
			con.setAutoCommit(false);			
			clinical.server.dao.VisittypeDAO visitDAO = DAOFactory.createVisittypeDAO(dbID);			
			Visittype visitType = new Visittype();
			List<Visittype> listVisitTypes = new ArrayList<Visittype>();
			listVisitTypes = visitDAO.find(con, visitType);			
			
			Collection<LabelValueBean> colVisittypes = new ArrayList<LabelValueBean>();			
			if(listVisitTypes!=null){				
				for(Visittype vType : listVisitTypes){
					LabelValueBean lvb = new LabelValueBean();
					lvb.setLabel(vType.getVisittype());
					lvb.setValue(vType.getUniqueid().toString());
					colVisittypes.add(lvb);
				}
			}
			
			return colVisittypes;
			
		} catch (Exception x) {
			handleErrorAndRollBack(con, "", x);
			return null;
		} finally {
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}
	
	/*
	 * Add Data to VisitType table 
	 */
	public void addVisitType(String visitTypeName, String description, UserInfo ui) 
			throws Exception {
		Connection con = null;
		try {
			if(visitTypeName.length()==0){
				return;
			}
			con = poolService.getConnection(Constants.ADMIN_USER);
			con.setAutoCommit(false);
			ISequenceHelper sequenceHelper = MinimalServiceFactory.getSequenceHelper(dbID);
			clinical.server.dao.VisittypeDAO visitDAO = DAOFactory.createVisittypeDAO(dbID);
			Databaseuser d = getDatabaseUser(con);
			Visittype visitType = new Visittype();
			visitType.setDescription(description);
			visitType.setModtime(new Timestamp(new Date().getTime()));
			visitType.setModuser(d.getUniqueid());
			visitType.setOwner(d.getUniqueid());
			BigDecimal uniqueId = sequenceHelper.getNextUID(con,
					"nc_conf_params_group", "uniqueId");
			visitType.setUniqueid(uniqueId);
			visitType.setVisittype(visitTypeName);
			BigDecimal tableId = getTableID(ui, Constants.VISITTYPE_TABLE, dbID);
			visitType.setTableid(tableId);
			visitDAO.insert(con, visitType);
			
			con.commit();
			
		} catch (Exception x) {
			handleErrorAndRollBack(con, "", x);			
		} finally {
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}
	
	public void deleteVisitType(BigDecimal visitTypeName, UserInfo ui) 
		throws Exception {
		Connection con = null;
		try {
			con = poolService.getConnection(Constants.ADMIN_USER);
			con.setAutoCommit(false);

			clinical.server.dao.VisittypeDAO visitDAO = DAOFactory.createVisittypeDAO(dbID);
			
			Visittype visitType = new Visittype();			
			visitType.setUniqueid(visitTypeName);		
			
			visitDAO.delete(con, visitType);
			
			con.commit();
			
		} catch (Exception x) {
			handleErrorAndRollBack(con, "", x);			
		} finally {
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}
	
	/*
	 * Retrieve all visit types from table nc_visittype
	 */
	public Collection<LabelValueBean> getAllResearchGroupTypes() throws Exception{
		Connection con = null;
		try {
			con = poolService.getConnection(Constants.ADMIN_USER);
			con.setAutoCommit(false);			
			clinical.server.dao.ResearchgrouptypeDAO researchGroupTypeDAO = DAOFactory.createResearchgrouptypeDAO(dbID);			
			Researchgrouptype researchgrouptype = new Researchgrouptype();
			List<Researchgrouptype> listResearchGroupTypes = new ArrayList<Researchgrouptype>();
			listResearchGroupTypes = researchGroupTypeDAO.find(con, researchgrouptype);
			
			Collection<LabelValueBean> colResearchGrouptypes = new ArrayList<LabelValueBean>();			
			if(listResearchGroupTypes!=null){				
				for(Researchgrouptype rType : listResearchGroupTypes){
					LabelValueBean lvb = new LabelValueBean();
					lvb.setLabel(rType.getName());
					lvb.setValue(rType.getUniqueid().toString());
					colResearchGrouptypes.add(lvb);
				}
			}
			
			return colResearchGrouptypes;
			
		} catch (Exception x) {
			handleErrorAndRollBack(con, "", x);
			return null;
		} finally {
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}
	
	/*
	 * Add Data to VisitType table 
	 */
	public void addResearchGroupType(String researchGroupTypeName, String description, UserInfo ui) 
			throws Exception {
		Connection con = null;
		try {
			if(researchGroupTypeName.length()==0){
				return;
			}
			con = poolService.getConnection(Constants.ADMIN_USER);
			con.setAutoCommit(false);
			ISequenceHelper sequenceHelper = MinimalServiceFactory.getSequenceHelper(dbID);
			clinical.server.dao.ResearchgrouptypeDAO researchGroupTypeDAO = DAOFactory.createResearchgrouptypeDAO(dbID);
			Databaseuser d = getDatabaseUser(con);
			Researchgrouptype researchGrpType = new Researchgrouptype();
			researchGrpType.setName(researchGroupTypeName);
			researchGrpType.setDescription(description);
			researchGrpType.setModtime(new Timestamp(new Date().getTime()));
			researchGrpType.setModuser(d.getUniqueid());
			researchGrpType.setOwner(d.getUniqueid());
			BigDecimal uniqueId = sequenceHelper.getNextUID(con,
					"nc_conf_params_group", "uniqueId");
			researchGrpType.setUniqueid(uniqueId);
			BigDecimal tableId = getTableID(ui, Constants.RESEARCHGROUPTYPE_TABLE, dbID);
			researchGrpType.setTableid(tableId);
			researchGroupTypeDAO.insert(con, researchGrpType);
			
			con.commit();
			
		} catch (Exception x) {
			handleErrorAndRollBack(con, "", x);			
		} finally {
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}

	public void deleteResearchGrpType(BigDecimal researchGrpTypeId, UserInfo ui) 
		throws Exception {
		Connection con = null;
		try {
			con = poolService.getConnection(Constants.ADMIN_USER);
			con.setAutoCommit(false);
	
			clinical.server.dao.ResearchgrouptypeDAO researchGrpTypeDAO = DAOFactory.createResearchgrouptypeDAO(dbID);
			
			Researchgrouptype researchGrpType = new Researchgrouptype();			
			researchGrpType.setUniqueid(researchGrpTypeId);		
			
			researchGrpTypeDAO.delete(con, researchGrpType);
			
			con.commit();
			
		} catch (Exception x) {
			handleErrorAndRollBack(con, "", x);			
		} finally {
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}
	
	public boolean researchGrpTypeUsed(BigDecimal researchGrpTypeId, UserInfo ui)
		throws Exception{
		Connection con = null;
		Statement st = null;
		ResultSet rs = null;
		
		try{
			boolean bExist = false;
			con = poolService.getConnection(Constants.ADMIN_USER);			
			st = con.createStatement();
			rs = st.executeQuery("Select * from nc_researchgroup WHERE nc_researchgrouptype_uniqueid=" + researchGrpTypeId);
			if(rs!=null){
				if(rs.next()) bExist= true;
			}
			return bExist;
			
		} catch(Exception x){
			handleErrorAndRollBack(con, "", x);
			return false;
		}finally{
			DBUtils.close(st, rs);
			poolService.releaseConnection(Constants.ADMIN_USER, con);
			
		}
		
	}

	public ArrayList<BigDecimal> getResearchGroups()
		throws Exception{
		ArrayList<BigDecimal> researchGrps = new ArrayList<BigDecimal>();
		Connection con = null;
		Statement st = null;
		ResultSet rs = null;		
		try{
			con = poolService.getConnection(Constants.ADMIN_USER);
			st = con.createStatement();
			rs = st.executeQuery("Select distinct nc_researchgrouptype_uniqueid from nc_researchgroup");
			if(rs!=null){
				while(rs.next()){
					researchGrps.add(rs.getBigDecimal(1));
				}
			}
			return researchGrps;	
		}catch(Exception x){
			handleErrorAndRollBack(con, "", x);
			return null;
		}finally{
			DBUtils.close(st, rs);
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}		
	}
	
	protected BigDecimal getTableID(UserInfo ui, String tableName, String theDBID)
		throws Exception {
		IDBCache dbCache = ServiceFactory.getDBCache(theDBID);
		Tableid tid = dbCache.getTableID(ui, tableName);
		return tid.getTableid();
	}
}
