package clinical.web.services;

import java.math.BigDecimal;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.cache.DBChangeNotifySupport;
import clinical.server.dao.ExpcomponentDAO;
import clinical.server.dao.ExperimentDAO;
import clinical.server.dao.HumansubjectDAO;
import clinical.server.dao.PersonDAO;
import clinical.server.dao.ResearchgroupDAO;
import clinical.server.dao.SubjexperimentDAO;
import clinical.server.vo.Databaseuser;
import clinical.server.vo.Expcomponent;
import clinical.server.vo.Experiment;
import clinical.server.vo.Humansubject;
import clinical.server.vo.Person;
import clinical.server.vo.Researchgroup;
import clinical.server.vo.Subjexperiment;
import clinical.utils.Assertion;
import clinical.utils.GenUtils;
import clinical.web.Constants;
import clinical.web.DAOFactory;
import clinical.web.IExperimentManagement;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.ISecurityService;
import clinical.web.common.UserInfo;
import clinical.web.common.security.User;
import clinical.web.exception.BaseException;
import clinical.web.exception.ExperimentManagementException;
import clinical.web.vo.ExperimentInfo;
import clinical.web.vo.StudyGroupInfo;

/**
 * Implements <code>IExperimentManagement</code> interface.
 * 
 * @author I. Burak Ozyurt
 * @version $Id: ExperimentManagementImpl.java,v 1.14 2006/05/02 01:04:58
 *          bozyurt Exp $
 */
public class ExperimentManagementImpl extends AbstractServiceImpl implements
		IExperimentManagement {
	protected IDBCache dbCache;
	protected String theSiteID;

	private Log log = LogFactory.getLog(ExperimentManagementImpl.class);

	public ExperimentManagementImpl(String dbID) throws BaseException {
		super(dbID);
		dbCache = ServiceFactory.getDBCache(dbID);
		theSiteID = ServiceFactory.getSecurityService().findSiteIDByDbID(dbID);
	}

	public Experiment getExperiment(UserInfo ui, String expName)
			throws ExperimentManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			Experiment criteria = new Experiment();
			criteria.setName(expName);
			ExperimentDAO dao = DAOFactory.createExperimentDAO(this.theDBID);
			List<Experiment> exps = dao.find(con, criteria);
			if (exps.isEmpty())
				return null;
			return exps.get(0);
		} catch (Exception x) {
			log.error("getExperiment", x);
			throw new ExperimentManagementException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public Experiment getExperiment(UserInfo ui, int experimentID)
			throws ExperimentManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			Experiment criteria = new Experiment();
			criteria.setUniqueid(new BigDecimal(String.valueOf(experimentID)));
			ExperimentDAO dao = DAOFactory.createExperimentDAO(this.theDBID);
			List<Experiment> exps = dao.find(con, criteria);
			if (exps.isEmpty())
				return null;
			return exps.get(0);
		} catch (Exception x) {
			log.error("getExperiment", x);
			throw new ExperimentManagementException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public Person getContactPerson(UserInfo ui, int personID)
			throws ExperimentManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			Person criteria = new Person();
			criteria.setUniqueid(new BigDecimal(String.valueOf(personID)));
			PersonDAO dao = DAOFactory.createPersonDAO(this.theDBID);
			List<Person> persons = dao.find(con, criteria);
			if (persons.isEmpty())
				return null;
			return (Person) persons.get(0);
		} catch (Exception x) {
			log.error("getContactPerson", x);
			throw new ExperimentManagementException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public List<Subjexperiment> getEnrolledSubjects(UserInfo ui, int experimentID)
			throws ExperimentManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			Subjexperiment criteria = new Subjexperiment();
			criteria.setNcExperimentUniqueid(new BigDecimal(String
					.valueOf(experimentID)));
			SubjexperimentDAO dao = DAOFactory
					.createSubjexperimentDAO(this.theDBID);
			List<?> list = dao.find(con, criteria);

			// find the unique subjects

			Set<String> subjectIDSet = new HashSet<String>();
			List<Subjexperiment> uniqueList = new LinkedList<Subjexperiment>();
			for (Iterator<?> iter = list.iterator(); iter.hasNext();) {
				Subjexperiment se = (Subjexperiment) iter.next();
				if (!subjectIDSet.contains(se.getSubjectid())) {
					subjectIDSet.add(se.getSubjectid());
					uniqueList.add(se);
				}
			}
			return uniqueList;
		} catch (Exception x) {
			log.error("getEnrolledSubjects", x);
			throw new ExperimentManagementException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public List<Humansubject> getRemainingSubjects(UserInfo ui, int experimentID)
			throws ExperimentManagementException {
		Connection con = null;
		try {

			List<Subjexperiment> enrolledSubjects = getEnrolledSubjects(ui,
					experimentID);
			List<Humansubject> allSubjects = getAllSubjects(ui);
			Map<String, String> subjectIDMap = new HashMap<String, String>();
			for (Iterator<Subjexperiment> iter = enrolledSubjects.iterator(); iter
					.hasNext();) {
				Subjexperiment se = iter.next();
				subjectIDMap.put(se.getSubjectid(), se.getSubjectid());
			}
			List<Humansubject> remSubjects = new LinkedList<Humansubject>();
			for (Iterator<Humansubject> iter = allSubjects.iterator(); iter
					.hasNext();) {
				Humansubject hs = iter.next();
				if (subjectIDMap.get(hs.getSubjectid()) == null)
					remSubjects.add(hs);
			}
			return remSubjects;
		} catch (Exception x) {
			log.error("getRemainingSubjects", x);
			throw new ExperimentManagementException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public List<Humansubject> getAllSubjects(UserInfo ui)
			throws ExperimentManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			Humansubject criteria = new Humansubject();
			HumansubjectDAO dao = DAOFactory.createHumansubjectDAO(this.theDBID);
			return dao.find(con, criteria);
		} catch (Exception x) {
			log.error("getAllSubjects", x);
			throw new ExperimentManagementException(x);
		} finally {
			releaseConnection(con, ui);
		}

	}

	public Map<Researchgroup, Integer> getResearchGroupSubjectCount(UserInfo ui,
			int experimentID) throws ExperimentManagementException {
		Map<Researchgroup, Integer> rgSubjectMap = new HashMap<Researchgroup, Integer>();
		Connection con = null;
		class IntWrapper {
			int count = 0;
			Map<String, String> subjectIds = new HashMap<String, String>();

			public IntWrapper() {}

			public void incr(String subjectID) {
				if (subjectIds.get(subjectID) == null) {
					subjectIds.put(subjectID, subjectID);
					count++;
				}
			}

			public int getCount() {
				return count;
			}
		}
		try {
			con = pool.getConnection(ui.getName());
			Subjexperiment criteria = new Subjexperiment();
			criteria.setNcExperimentUniqueid(new BigDecimal(String
					.valueOf(experimentID)));
			SubjexperimentDAO dao = DAOFactory
					.createSubjexperimentDAO(this.theDBID);
			List<?> sexps = dao.find(con, criteria);
			Map<BigDecimal, IntWrapper> map = new HashMap<BigDecimal, IntWrapper>();
			for (Iterator<?> iter = sexps.iterator(); iter.hasNext();) {
				Subjexperiment se = (Subjexperiment) iter.next();
				IntWrapper iw = map.get(se.getNcResearchgroupUniqueid());
				if (iw == null) {
					map.put(se.getNcResearchgroupUniqueid(), iw = new IntWrapper());
				}
				iw.incr(se.getSubjectid());
			}
			//
			IDBCache dbCache = ServiceFactory.getDBCache(this.theDBID);
			// use cache if exists
			List<Researchgroup> rgs = dbCache.getResearchGroups(ui, false);
			for (Researchgroup rg : rgs) {
				IntWrapper iw = map.get(rg.getUniqueid());
				if (iw != null) {
					rgSubjectMap.put(rg, new Integer(iw.getCount()));
				}
			}
			return rgSubjectMap;
		} catch (Exception x) {
			log.error("getResearchGroupSubjectCount", x);
			throw new ExperimentManagementException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public List<Person> getAllContactPersons(UserInfo ui)
			throws ExperimentManagementException {
		Connection con = null;
		List<?> persons = null;
		try {
			con = pool.getConnection(ui.getName());
			Person criteria = new Person();
			PersonDAO dao = DAOFactory.createPersonDAO(this.theDBID);
			persons = dao.find(con, criteria);
			List<Person> personList = new ArrayList<Person>(persons.size());
			for (Iterator<?> it = persons.iterator(); it.hasNext();) {
				Person person = (Person) it.next();
				personList.add(person);
			}
			return personList;
		} catch (Exception x) {
			log.error("getAllContactPersons", x);
			throw new ExperimentManagementException(x);
		} finally {
			releaseConnection(con, ui);
		}

	}

	public Subjexperiment enrollSubject(Connection con, UserInfo ui,
			int experimentID, String subjectID, int researchGroupID)
			throws ExperimentManagementException {
		Subjexperiment criteria = new Subjexperiment();
		try {
			criteria.setNcResearchgroupUniqueid(new BigDecimal(String
					.valueOf(researchGroupID)));
			criteria.setSubjectid(subjectID);
			criteria.setNcExperimentUniqueid(new BigDecimal(String
					.valueOf(experimentID)));
			SubjexperimentDAO dao = DAOFactory
					.createSubjexperimentDAO(this.theDBID);
			List<Subjexperiment> seList = dao.find(con, criteria);
			if (!seList.isEmpty()) {
				throw new ExperimentManagementException("Subject  " + subjectID
						+ " has already been enrolled!");
			}
			Subjexperiment se = new Subjexperiment();
			se.setNcResearchgroupUniqueid(new BigDecimal(String
					.valueOf(researchGroupID)));
			se.setSubjectid(subjectID);
			se
					.setNcExperimentUniqueid(new BigDecimal(String
							.valueOf(experimentID)));

			Databaseuser databaseUser = getDatabaseUser(ui,
					Constants.USERCLASS_ADMIN);
			Assertion.assertNotNull(databaseUser);

			se.setOwner(databaseUser.getUniqueid());
			se.setModuser(databaseUser.getUniqueid());
			se.setModtime(new java.util.Date());
			BigDecimal tid = getTableID(ui, Constants.SUBJEXPERIMENT_DB_TABLE);
			se.setTableid(tid);

			BigDecimal uniqueid = ServiceFactory.getSequenceHelper(this.theDBID)
					.getNextUID(ui, Constants.SUBJEXPERIMENT_DB_TABLE, "uniqueid");
			se.setUniqueid(uniqueid);
			if (log.isDebugEnabled()) {
				log.debug(se.toString());
			}
			dao.insert(con, se);
			return se;
		} catch (Throwable t) {
			throw new ExperimentManagementException(t);
		}
	}

	public Subjexperiment enrollSubject(UserInfo ui, int experimentID,
			String subjectID, int researchGroupID)
			throws ExperimentManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);

			Subjexperiment se = enrollSubject(con, ui, experimentID, subjectID,
					researchGroupID);

			con.commit();
			return se;
		} catch (Exception x) {
			handleErrorAndRollBack(con, "addExperiment", x, true);
			return null;
		} finally {
			releaseConnection(con, ui);
		}
	}

	public int unenrollSubject(UserInfo ui, int experimentID, String subjectID)
			throws ExperimentManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);

			// first check if any visits are associated with the subject for
			// this experiment
			Expcomponent visitCrit = new Expcomponent();
			visitCrit.setNcExperimentUniqueid(GenUtils.toBigDecimal(experimentID));
			visitCrit.setSubjectid(subjectID);
			ExpcomponentDAO dao = DAOFactory.createExpcomponentDAO(this.theDBID);
			List<Expcomponent> visits = dao.find(con, visitCrit);
			if (visits.isEmpty()) {
				Subjexperiment se = new Subjexperiment();
				se.setSubjectid(subjectID);
				se.setNcExperimentUniqueid(GenUtils.toBigDecimal(experimentID));
				SubjexperimentDAO seDAO = DAOFactory
						.createSubjexperimentDAO(this.theDBID);
				seDAO.delete(con, se);
			}
			con.commit();

			return (visits.isEmpty()) ? SUCCESS : HAS_VISITS;
		} catch (Exception x) {
			handleErrorAndRollBack(con, "addExperiment", x, true);
			return HAS_ERROR;
		} finally {
			releaseConnection(con, ui);
		}
	}

	public int addExperiment(UserInfo ui, ExperimentInfo expInfo)
			throws ExperimentManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);

			Experiment criteria = new Experiment();
			criteria.setName(expInfo.getName());
			ExperimentDAO dao = DAOFactory.createExperimentDAO(this.theDBID);
			List<?> exps = dao.find(con, criteria);
			if (!exps.isEmpty())
				throw new ExperimentManagementException("Experiment "
						+ expInfo.getName() + " already exists!");
			Experiment exp = new Experiment();
			exp.setName(expInfo.getName());
			exp.setContactperson(new BigDecimal(expInfo.getContactPersonValue()));
			if (expInfo.getDescription() != null)
				exp.setDescription(expInfo.getDescription());
			else
				exp.setDescription("");

			if (expInfo.getBaseURI() != null)
				exp.setBaseuri(expInfo.getBaseURI());

			if (expInfo.getStorageType() != null) 
				exp.setStoragetype(expInfo.getStorageType());
			
			// set the owner and moduser
			Databaseuser databaseUser = getDatabaseUser(ui,
					Constants.USERCLASS_ADMIN);
			Assertion.assertNotNull(databaseUser);

			exp.setModuser(databaseUser.getUniqueid());
			exp.setOwner(databaseUser.getUniqueid());
			exp.setModtime(new java.util.Date());
			BigDecimal tid = getTableID(ui, Constants.EXPERIMENT_DB_TABLE);
			exp.setTableid(tid);

			BigDecimal uniqueid = ServiceFactory.getSequenceHelper(this.theDBID)
					.getNextUID(ui, Constants.EXPERIMENT_DB_TABLE, "uniqueid");
			exp.setUniqueid(uniqueid);

			dao.insert(con, exp);
			con.commit();

			DBChangeNotifySupport.getInstance().fireDBChangeNotifyEvent(theSiteID,
					"experiment");
			return uniqueid.intValue();
		} catch (Exception x) {
			handleErrorAndRollBack(con, "addExperiment", x, true);
			return -1;
		} finally {
			releaseConnection(con, ui);
		}
	}

	public int addStudyGroup(UserInfo ui, StudyGroupInfo sgi)
			throws ExperimentManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			Researchgroup criterion = new Researchgroup();
			criterion.setName(sgi.getName());
			criterion.setNcExperimentUniqueid(new BigDecimal(String.valueOf(sgi
					.getExperimentID())));
			criterion.setNcResearchgrouptypeUniqueid(new BigDecimal(String
					.valueOf(sgi.getResearchGroupTypeID())));
			ResearchgroupDAO dao = DAOFactory.createResearchgroupDAO(this.theDBID);
			List<?> rgs = dao.find(con, criterion);
			if (!rgs.isEmpty())
				throw new ExperimentManagementException("Study Group "
						+ sgi.getName() + " already exists!");
			Researchgroup rg = new Researchgroup();
			rg.setName(sgi.getName());
			rg.setDescription(sgi.getDescription());
			rg.setNcExperimentUniqueid(new BigDecimal(String.valueOf(sgi
					.getExperimentID())));
			rg.setNcResearchgrouptypeUniqueid(new BigDecimal(String.valueOf(sgi
					.getResearchGroupTypeID())));

			// set the owner and moduser
			Databaseuser databaseUser = getDatabaseUser(ui,
					Constants.USERCLASS_ADMIN);
			Assertion.assertNotNull(databaseUser);

			rg.setModtime(new java.util.Date());

			BigDecimal tid = getTableID(ui, Constants.RESEARCHGROUP_DB_TABLE);
			rg.setTableid(tid);

			rg.setModuser(databaseUser.getUniqueid());
			rg.setOwner(databaseUser.getUniqueid());

			BigDecimal uniqueid = ServiceFactory.getSequenceHelper(this.theDBID)
					.getNextUID(ui, Constants.USERCLASS_ADMIN, "uniqueid");
			rg.setUniqueid(uniqueid);

			dao.insert(con, rg);
			con.commit();
			DBChangeNotifySupport.getInstance().fireDBChangeNotifyEvent(theSiteID,
					"researchgroup");
			return uniqueid.intValue();
		} catch (Exception x) {
			handleErrorAndRollBack(con, "addStudyGroup", x, true);
			return -1;
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void updateExperiment(UserInfo ui, ExperimentInfo expInfo)
			throws ExperimentManagementException {
		Connection con = null;
		ISecurityService securityService = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			securityService = ServiceFactory.getSecurityService();

			Experiment criteria = new Experiment();
			criteria.setUniqueid(new BigDecimal(String.valueOf(expInfo.getId())));
			ExperimentDAO dao = DAOFactory.createExperimentDAO(this.theDBID);
			Experiment exp = new Experiment();
			exp.setName(expInfo.getName());
			exp.setContactperson(new BigDecimal(expInfo.getContactPersonValue()));
			if (expInfo.getDescription() != null)
				exp.setDescription(expInfo.getDescription());
			else
				exp.setDescription("");
			
			if (expInfo.getStorageType() != null) {
				exp.setStoragetype(expInfo.getStorageType());
			}

			// set the moduser and modificiation time
			Map<String, User> userMap = securityService.getAllUsers(this.theDBID);
			User u = (User) userMap.get(ui.getName());

			Databaseuser databaseUser = dbCache.getDatabaseUser(ui, u.getDbUser()
					.getName(), Constants.USERCLASS_ADMIN);
			exp.setModuser(databaseUser.getUniqueid());
			exp.setModtime(new java.util.Date());

			dao.update(con, exp, criteria);
			con.commit();
			DBChangeNotifySupport.getInstance().fireDBChangeNotifyEvent(theSiteID,
					"experiment");
		} catch (Exception x) {
			handleErrorAndRollBack(con, "updateExperiment", x, true);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void deleteExperiment(UserInfo ui, int experimentID)
			throws ExperimentManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			// remove subjexperiments also
			Subjexperiment sc = new Subjexperiment();
			sc
					.setNcExperimentUniqueid(new BigDecimal(String
							.valueOf(experimentID)));

			SubjexperimentDAO seDAO = DAOFactory
					.createSubjexperimentDAO(this.theDBID);
			seDAO.delete(con, sc);

			// now delete the experiment
			Experiment criteria = new Experiment();
			criteria.setUniqueid(new BigDecimal(String.valueOf(experimentID)));
			ExperimentDAO dao = DAOFactory.createExperimentDAO(this.theDBID);
			dao.delete(con, criteria);

			con.commit();
			DBChangeNotifySupport.getInstance().fireDBChangeNotifyEvent(theSiteID,
					"experiment");
		} catch (Exception x) {
			handleErrorAndRollBack(con, "deleteExperiment", x, true);
		} finally {
			releaseConnection(con, ui);
		}

	}

	public void changeSubjectStudyGroup(UserInfo ui, int experimentID,
			int oldStudyGroupID, int newStudyGroupID, String subjectID)
			throws ExperimentManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			Subjexperiment criteria = new Subjexperiment();
			criteria.setSubjectid(subjectID);
			criteria.setNcExperimentUniqueid(new BigDecimal(String
					.valueOf(experimentID)));
			criteria.setNcResearchgroupUniqueid(new BigDecimal(String
					.valueOf(oldStudyGroupID)));
			Subjexperiment bean = new Subjexperiment();
			bean.setNcResearchgroupUniqueid(new BigDecimal(String
					.valueOf(newStudyGroupID)));
			SubjexperimentDAO dao = DAOFactory
					.createSubjexperimentDAO(this.theDBID);
			dao.update(con, bean, criteria);
			con.commit();
			DBChangeNotifySupport.getInstance().fireDBChangeNotifyEvent(theSiteID,
					"researchgroup");
		} catch (Exception x) {
			handleErrorAndRollBack(con, "changeSubjectStudyGroup", x, true);
		} finally {
			releaseConnection(con, ui);
		}

	}

	public List<Experiment> getAllExperiments(UserInfo ui)
			throws ExperimentManagementException {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			Experiment criteria = new Experiment();
			ExperimentDAO dao = DAOFactory.createExperimentDAO(this.theDBID);
			List<Experiment> expList = dao.find(con, criteria);
			return expList;
		} catch (Exception x) {
			log.error("getEnrolledSubjects", x);
			throw new ExperimentManagementException(x);
		} finally {
			releaseConnection(con, ui);
		}
	}

	protected void handleErrorAndRollBack(Connection con, String msg,
			Exception x, boolean doRollback) throws ExperimentManagementException {
		try {
			super.handleErrorAndRollBack(con, msg, x, doRollback);
		} catch (BaseException be) {
			throw new ExperimentManagementException(be);
		}
	}

}
