package clinical.test;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;

import junit.framework.Test;
import junit.framework.TestSuite;
import clinical.web.SRBUtils;
import edu.sdsc.grid.io.local.LocalFile;
import edu.sdsc.grid.io.srb.SRBAccount;
import edu.sdsc.grid.io.srb.SRBContainer;
import edu.sdsc.grid.io.srb.SRBFile;
import edu.sdsc.grid.io.srb.SRBFileSystem;
import edu.sdsc.grid.io.srb.SRBRandomAccessFile;

/**
 * @author I. Burak Ozyurt
 * @version $Id: SRBTest.java 91 2009-08-17 23:38:26Z bozyurt $
 */
public class SRBTest extends MyTestCase {
	String userInfoFile;
	String srbSeriesPath;
	String localSeries;

	public SRBTest(String testName, String propsFile) throws IOException {
		super(testName, propsFile);
	}

	protected void setUp() throws Exception {
		userInfoFile = getProperty("srb.user_info_file");
		srbSeriesPath = getProperty("srb.series_path");
		localSeries = getProperty("srb.local_series");
		File localSeriesDir = new File(localSeries);
		if (localSeriesDir.exists())
			localSeriesDir.delete();
		if (!localSeriesDir.exists())
			localSeriesDir.mkdirs();

	}

	protected void tearDown() throws Exception {
	}

	public void loadFile(SRBFile srbFile, String outFile) throws IOException {
		SRBRandomAccessFile in = null;
		BufferedOutputStream out = null;
		try {
			in = new SRBRandomAccessFile(srbFile, "r");
			out = new BufferedOutputStream(new FileOutputStream(outFile), 4096);

			int bytesRead;
			byte[] buffer = new byte[4096];
			int totBytes = 0;
			while ((bytesRead = in.read(buffer, 0, buffer.length)) > 0) {
				out.write(buffer, 0, bytesRead);
				totBytes += bytesRead;
				// System.out.println("bytes loaded: "+ totBytes + " bytes
				// Read="+ bytesRead);
			}
		} finally {
			if (out != null)
				try {
					out.close();
				} catch (Exception x) {
				}
			if (in != null)
				try {
					in.close();
				} catch (Exception x) {
				}
		}
	}

	public void testSls() throws Exception {

		SRBAccount account = null;
		SRBFileSystem sfs = null;
		SRBFile srbFile = null;

		account = new SRBAccount(userInfoFile);

		sfs = new SRBFileSystem(account);
		srbFile = new SRBFile(sfs, srbSeriesPath);
		// this should return a list of files
		String[] flist = srbFile.list();
		assertTrue(flist.length > 0);
		for (int i = 0; i < flist.length; i++) {
			System.out.println(flist[i]);
		}
		sfs.close();
	}

	public void testSget() throws Exception {
		// String userInfoFile = "/home/bozyurt/.srb";
		// String srbSeriesPath =
		// "/home/BIRN/Human/Clinical/UCSD/000800156805/Visit_001/Study_0001/RawData/005.ser";

		// String localSeries = "/tmp/000800156805";

		SRBAccount account = null;
		SRBFileSystem sfs = null;
		SRBFile srbFile = null;

		account = new SRBAccount(userInfoFile);
		try {
			sfs = new SRBFileSystem(account);
			srbFile = new SRBFile(sfs, srbSeriesPath);
			String[] flist = srbFile.list();
			for (int i = 0; i < flist.length; i++) {
				System.out.println(flist[i]);
				String srbFilePath = srbSeriesPath + "/" + flist[i];
				String outPath = localSeries + "/" + flist[i];
				System.out.println(">> " + srbFilePath);
				loadFile(new SRBFile(srbFile, flist[i]), outPath);
				File localFile = new File(outPath);
				assertTrue(localFile.exists() && localFile.length() > 0);
			}
		} finally {
			sfs.close();
		}
	}

	public void testSput() throws Exception {
		String srbCollection = "/home/bozyurt.ucsd-fmri";
		String localPath = "/tmp/temp_1077912632829.dat";

		SRBUtils.srbPut(localPath, userInfoFile, srbCollection);
	}

	public void testParalellLoad() throws Exception {
		SRBAccount account = null;
		SRBFileSystem sfs = null;
		SRBFile srbFile = null;

		File localSeriesDir = new File(localSeries);
		account = new SRBAccount(userInfoFile);

		sfs = new SRBFileSystem(account);
		srbFile = new SRBFile(sfs, srbSeriesPath);
		String[] flist = srbFile.list();
		Thread[] threads = new Thread[flist.length];
		for (int i = 0; i < flist.length; i++) {
			System.out.println(flist[i]);
			String srbFilePath = srbSeriesPath + "/" + flist[i];
			String outPath = localSeries + "/" + flist[i];
			System.out.println(">> " + srbFilePath);
			threads[i] = new Thread(new SRBWorker(
					new SRBFile(srbFile, flist[i]), outPath));
			threads[i].setDaemon(true);
			threads[i].start();

		}
		for (int i = 0; i < threads.length; i++) {
			threads[i].join();
		}

		String[] localFiles = localSeriesDir.list();
		assertTrue(localFiles.length == flist.length);

		sfs.close();
	}

	public void testContainer() throws Exception {
		// String userInfoFile = "/home/bozyurt/.srb";
		// String srbContainerPath =
		// "000899127551_Visit_001_Study_0001_005_ser";
		String srbContainerPath = getProperty("srb.container_path");
		// String localSeries = "/tmp/000899127551";
		// File localSeriesDir = new File(localSeries);

		SRBAccount account = null;
		SRBFileSystem sfs = null;

		account = new SRBAccount(userInfoFile);

		sfs = new SRBFileSystem(account);
		SRBContainer container = new SRBContainer(sfs, srbContainerPath);

		// force overwrite
		container.copyTo(new LocalFile(localSeries), true);
		sfs.close();
	}

	public static class SRBWorker implements Runnable {
		SRBFile srbFile;
		String outFile;

		public SRBWorker(SRBFile srbFile, String outFile) {
			this.srbFile = srbFile;
			this.outFile = outFile;
		}

		public void loadFile() throws IOException {
			SRBRandomAccessFile in = null;
			BufferedOutputStream out = null;
			try {
				in = new SRBRandomAccessFile(srbFile, "r");
				out = new BufferedOutputStream(new FileOutputStream(outFile),
						4096);

				int bytesRead;
				byte[] buffer = new byte[4096];
				int totBytes = 0;
				while ((bytesRead = in.read(buffer, 0, buffer.length)) > 0) {
					out.write(buffer, 0, bytesRead);
					totBytes += bytesRead;
					// System.out.println("bytes loaded: "+ totBytes + " bytes
					// Read="+ bytesRead);
				}
			} finally {
				if (out != null)
					try {
						out.close();
					} catch (Exception x) {
					}
				if (in != null)
					try {
						in.close();
					} catch (Exception x) {
					}
			}
		}

		public void run() {
			try {
				loadFile();
			} catch (Exception x) {
				x.printStackTrace();
			}
		}
	}

	public static Test suite() throws IOException {
		TestSuite suite = new TestSuite();
		suite.addTest(new SRBTest("testSls", "test.properties"));
		// suite.addTest(new SRBTest("testSput", "test.properties") );

		// suite.addTest(new SRBTest("testSget", "test.properties") );

		// suite.addTest(new SRBTest("testContainer", "test.properties") );
		// suite.addTest(new SRBTest("testParalellLoad", "test.properties"));
		return suite;
	}

	public static void main(String[] args) throws IOException {
		junit.textui.TestRunner.run(suite());
	}

}