package clinical.web.actions;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;

import oracle.jdbc.Const;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.json.JSONObject;

import clinical.server.vo.Assessment;
import clinical.server.vo.Experiment;
import clinical.server.vo.Extendedtuple;
import clinical.utils.Assertion;
import clinical.utils.GenUtils;
import clinical.utils.ThreadPoolMan;
import clinical.web.Constants;
import clinical.web.IAnalysisResultService;
import clinical.web.IAssessmentService;
import clinical.web.IRemoteDBServices;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.ISecurityService;
import clinical.web.common.UserInfo;
import clinical.web.common.query.MGQueryPartInfo;
import clinical.web.common.query.Operator;
import clinical.web.common.query.QueryPartInfo;
import clinical.web.common.vo.AnalysisResultSummary;
import clinical.web.common.vo.AssessmentSelectionInfo;
import clinical.web.exception.BaseException;
import clinical.web.forms.AsQueryBuilderForm;
import clinical.web.helpers.ExperimentSelector;
import clinical.web.vo.ASValueKey;
import clinical.web.vo.AnalysisResultQueryInfo;
import clinical.web.vo.AssessmentQueryInfo;
import clinical.web.vo.AssessmentResultSummary;
import clinical.web.vo.QuerySummary;
import clinical.web.vo.SubjectAsScoreValueSummary;
import clinical.web.vo.VisitSegAsResultValues;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: QueryAggregator.java 873 2015-01-07 20:55:20Z jinranc $
 */
public class QueryAggregator {
   protected AsQueryBuilderForm queryForm;
   protected String dbID;
   protected UserInfo ui;
   protected IDBCache dbCache = null;
   protected IRemoteDBServices rds;
   protected ExecutorService executor;
   protected Map<String, Experiment> expMap;

   private Log log = LogFactory.getLog(QueryAggregator.class);

   public QueryAggregator(String dbID, UserInfo ui, AsQueryBuilderForm queryForm)
         throws BaseException {
      super();
      this.dbID = dbID;
      this.ui = ui;
      this.queryForm = queryForm;
      dbCache = ServiceFactory.getDBCache(dbID);
      rds = ServiceFactory.getRemoteDBServices();
      executor = ThreadPoolMan.getInstance().getExecutorService();
      List<Experiment> experiments = dbCache.getExperiments(ui, false);
      this.expMap = toMap(experiments);

   }

   protected List<Integer> prepareExperimentIDs(String siteID,
         String primarySiteID) throws Exception {
      ExperimentSelector expSelector = queryForm.getExpSelector();
      List<Integer> experimentIDs = null;
      if (expSelector != null) {
         int seid = expSelector.getSelectedExpID();
         if (seid == ExperimentSelector.ALL_EXPERIMENTS) {
            // do nothing
         } else if (seid == ExperimentSelector.ALL_EXPERIMENTS_MINUS_REGRESSION) {
            /** @todo IMPLEMENT REGRESSION finding logic */
         } else {
            experimentIDs = new ArrayList<Integer>(1);
            if (siteID == null || siteID.equals(primarySiteID)) {
               experimentIDs.add(new Integer(seid));
            } else {
               Map<String, Experiment> map = rds.getAllExperiments(siteID);
               Experiment localExp = expMap.get(String.valueOf(seid));
               String expName = localExp.getName().trim();
               Experiment theExp = null;
               for (Experiment remoteExp : map.values()) {
                  if (remoteExp.getName().trim().equals(expName)) {
                     theExp = remoteExp;
                     break;
                  }
               }
               if (theExp != null) {
                  experimentIDs.add(theExp.getUniqueid().intValue());
               }
            }
         }
      }
      return experimentIDs;
   }

   public void handleQuery(String caQueryString, String ddQueryString,
         String primarySiteID) throws Exception {

      CAQueryWorker caqTask = null;
      DDQueryWorker ddTask = null;
      if(caQueryString.equals("null")){
    	  caQueryString = null;
      }
      if (caQueryString != null && caQueryString.trim().length() > 0) {
         JSONObject js = new JSONObject(caQueryString);
         AssessmentQueryInfo aqi = AssessmentQueryInfo.initializeFromJSON(js);
         caqTask = new CAQueryWorker(aqi, dbID, primarySiteID);
      }
      if (ddQueryString != null && ddQueryString.trim().length() > 0) {
         JSONObject js = new JSONObject(ddQueryString);
         AnalysisResultQueryInfo arqi = AnalysisResultQueryInfo
               .initializeFromJSON(js);
         ddTask = new DDQueryWorker(arqi, dbID, primarySiteID);
      }
      List<SubjectAsScoreValueSummary> summaryList = null;
      List<AnalysisResultSummary> arsList = null;

      if (caqTask != null && ddTask != null) {

         Future<List<SubjectAsScoreValueSummary>> caFuture = executor
               .submit(caqTask);

         Future<List<AnalysisResultSummary>> ddFuture = executor.submit(ddTask);
         summaryList = caFuture.get();
         arsList = ddFuture.get();
      } else if (caqTask != null) {
         summaryList = caqTask.call();
      } else if (ddTask != null) {
         arsList = ddTask.call();
      }

      if (summaryList != null) {
         prepareQSList(primarySiteID, expMap, summaryList);
      }
      if (arsList != null) {
         boolean hasCAQuery = summaryList != null;
         summaryList = null;
         prepQsList(arsList, hasCAQuery);
      }
   }

   public void handleQueryNew(String caQueryString, String ddQueryString, String primarySiteID, 
		   String[] allSegs, String[] selSegs, String[] selDtypes, String[] selVers, 
		   Integer selectedExpId, String[] selProtocols, String queryType) throws Exception {	   
	      CAQueryWorker caqTask = null;
	      DDQueryWorker ddTask = null;
	      if(queryType.equals(Constants.GENETICS_ONLY) || 
	    		  queryType.equals(Constants.IMAGING_ONLY)||
	    		  queryType.equals(Constants.IMAGING_AND_GENETICS)||
	    		  queryType.equals(Constants.IMAGING_OR_GENETICS)||
	    		  queryType.equals(Constants.ASSESS_OR_GENETICS)){
	    	  caqTask = new CAQueryWorker(dbID, primarySiteID, queryType, selectedExpId, selProtocols, selDtypes, selVers);
	      }else{
	    	  if (caQueryString != null && caQueryString.trim().length() > 0) {
	 	         JSONObject js = new JSONObject(caQueryString);
	 	         AssessmentQueryInfo aqi = AssessmentQueryInfo.initializeFromJSON(js);
	 	         caqTask = new CAQueryWorker(aqi, dbID, primarySiteID, queryType);
	 	      }else if(caQueryString.trim().length()==0){
	 	    	  //change queryType to GeneticsOnly from sth.else
	 	    	  queryType = Constants.GENETICS_ONLY;
	 	    	  caqTask = new CAQueryWorker(dbID, primarySiteID, queryType, selectedExpId, selProtocols, selDtypes, selVers);
	 	      }
	      }
	      
	      if (ddQueryString != null && ddQueryString.trim().length() > 0) {
	         JSONObject js = new JSONObject(ddQueryString);
	         AnalysisResultQueryInfo arqi = AnalysisResultQueryInfo
	               .initializeFromJSON(js);
	         ddTask = new DDQueryWorker(arqi, dbID, primarySiteID);
	      }
	      List<SubjectAsScoreValueSummary> summaryList = null;
	      List<AnalysisResultSummary> arsList = null;

	      if (caqTask != null && ddTask != null) {

	         Future<List<SubjectAsScoreValueSummary>> caFuture = executor
	               .submit(caqTask);

	         Future<List<AnalysisResultSummary>> ddFuture = executor.submit(ddTask);
	         summaryList = caFuture.get();
	         arsList = ddFuture.get();
	      } else if (caqTask != null) {
	         summaryList = caqTask.call();
	      } else if (ddTask != null) {
	         arsList = ddTask.call();
	      } 

	      if (summaryList != null) {
	         prepareQSListNew(primarySiteID, expMap, summaryList, selSegs, allSegs, selDtypes, selVers, queryType);
	      }
	      if (arsList != null) {
	         boolean hasCAQuery = summaryList != null;
	         summaryList = null;
	         prepQsList(arsList, hasCAQuery);
	      }
	   }

   public void handleQueryNewer(String caQueryString, String ddQueryString, String primarySiteID, 
		   String[] allSegs, String[] selSegs, String[] selDtypes, String[] selVers, 
		   Integer selectedExpId, String[] selProtocols, String queryType, String isValidatedText) throws Exception {	   
	      CAQueryWorker caqTask = null;
	      DDQueryWorker ddTask = null;
	      if(queryType.equals(Constants.GENETICS_ONLY) || 
	    		  queryType.equals(Constants.IMAGING_ONLY)||
	    		  queryType.equals(Constants.IMAGING_AND_GENETICS)||
	    		  queryType.equals(Constants.IMAGING_OR_GENETICS)||
	    		  queryType.equals(Constants.ASSESS_OR_GENETICS)){
	    	  caqTask = new CAQueryWorker(dbID, primarySiteID, queryType, selectedExpId, selProtocols, 
	    			  selDtypes, selVers, isValidatedText);
	      }else{
	    	  if (caQueryString != null && caQueryString.trim().length() > 0) {
	 	         JSONObject js = new JSONObject(caQueryString);
	 	         AssessmentQueryInfo aqi = AssessmentQueryInfo.initializeFromJSON(js);
	 	         caqTask = new CAQueryWorker(aqi, dbID, primarySiteID, queryType);
	 	         caqTask.isValidated = isValidatedText;
	 	      }else if(caQueryString.trim().length()==0){
	 	    	  //change queryType to GeneticsOnly from sth.else
	 	    	  queryType = Constants.GENETICS_ONLY;
	 	    	  caqTask = new CAQueryWorker(dbID, primarySiteID, queryType, selectedExpId, selProtocols, 
	 	    			  selDtypes, selVers, isValidatedText);
	 	      }
	      }
	      
	      if (ddQueryString != null && ddQueryString.trim().length() > 0) {
	         JSONObject js = new JSONObject(ddQueryString);
	         AnalysisResultQueryInfo arqi = AnalysisResultQueryInfo
	               .initializeFromJSON(js);
	         ddTask = new DDQueryWorker(arqi, dbID, primarySiteID);
	      }
	      List<SubjectAsScoreValueSummary> summaryList = null;
	      List<AnalysisResultSummary> arsList = null;

	      if (caqTask != null && ddTask != null) {

	         Future<List<SubjectAsScoreValueSummary>> caFuture = executor
	               .submit(caqTask);

	         Future<List<AnalysisResultSummary>> ddFuture = executor.submit(ddTask);
	         summaryList = caFuture.get();
	         arsList = ddFuture.get();
	      } else if (caqTask != null) {
	         summaryList = caqTask.call();
	      } else if (ddTask != null) {
	         arsList = ddTask.call();
	      } 

	      if (summaryList != null) {
	         prepareQSListNew(primarySiteID, expMap, summaryList, selSegs, allSegs, selDtypes, selVers, queryType);
	      }
	      if (arsList != null) {
	         boolean hasCAQuery = summaryList != null;
	         summaryList = null;
	         prepQsList(arsList, hasCAQuery);
	      }
	   }

   
   public void handleMultisiteQuery(String caQueryString, String ddQueryString,
         String primarySiteID) throws Exception {
      List<CAQueryWorker> caqTaskList = new LinkedList<CAQueryWorker>();
      List<DDQueryWorker> ddTaskList = new LinkedList<DDQueryWorker>();

      ISecurityService isec = ServiceFactory.getSecurityService();
      String[] allDBIDs = isec.getAllDBIDs();
      String primaryDbID = isec.findDBForSiteID(primarySiteID);

      AssessmentQueryInfo aqi = null;
      AnalysisResultQueryInfo arqi = null;
      if (caQueryString != null && caQueryString.trim().length() > 0) {
         JSONObject js = new JSONObject(caQueryString);
         aqi = AssessmentQueryInfo.initializeFromJSON(js);
         CAQueryWorker caqTask = new CAQueryWorker(aqi, dbID, primarySiteID);

         caqTaskList.add(caqTask);
      }
      if (ddQueryString != null && ddQueryString.trim().length() > 0) {
         JSONObject js = new JSONObject(ddQueryString);
         arqi = AnalysisResultQueryInfo.initializeFromJSON(js);
         DDQueryWorker ddTask = new DDQueryWorker(arqi, dbID, primarySiteID);
         ddTaskList.add(ddTask);
      }

      if (!caqTaskList.isEmpty()) {
         // add other sites also
         for (String aDbID : allDBIDs) {
            if (aDbID.equals(primaryDbID))
               continue;
            CAQueryWorker caqTask = new CAQueryWorker(aqi, aDbID, primarySiteID);
            caqTaskList.add(caqTask);
         }
      }

      if (!ddTaskList.isEmpty()) {
         for (String aDbID : allDBIDs) {
            if (aDbID.equals(primaryDbID))
               continue;
            DDQueryWorker ddTask = new DDQueryWorker(arqi, aDbID, primarySiteID);
            ddTaskList.add(ddTask);
         }
      }

      List<SubjectAsScoreValueSummary> summaryList = new LinkedList<SubjectAsScoreValueSummary>();
      List<AnalysisResultSummary> arsList = new LinkedList<AnalysisResultSummary>();
      if (!caqTaskList.isEmpty()) {
         for (CAQueryWorker worker : caqTaskList) {
            try {
               List<SubjectAsScoreValueSummary> siteSummaryList = worker.call();
               if (siteSummaryList != null && !siteSummaryList.isEmpty())
                  summaryList.addAll(siteSummaryList);
            } catch (Throwable t) {
               log.error("handleMultisiteQuery", t);
            }
         }
         prepareQSList(primarySiteID, expMap, summaryList);
      }
      if (!ddTaskList.isEmpty()) {
         boolean hasCAQuery = !summaryList.isEmpty();
         summaryList = null;
         for (DDQueryWorker worker : ddTaskList) {
            try {
               List<AnalysisResultSummary> siteArsList = worker.call();
               if (siteArsList != null && !siteArsList.isEmpty()) {
                  arsList.addAll(siteArsList);
               }
            } catch (ExecutionException e) {
               log.error("handleMultisiteQuery", e);
            }
         }
         prepQsList(arsList, hasCAQuery);
      }
   }

   public void handleMultisiteQueryNew(String caQueryString, String ddQueryString, String primarySiteID,
		   String[] allExpSegs, String[] selExpSegs, String[] selDataTypes, String[] selVersions) throws Exception {
	      List<CAQueryWorker> caqTaskList = new LinkedList<CAQueryWorker>();
	      List<DDQueryWorker> ddTaskList = new LinkedList<DDQueryWorker>();

	      ISecurityService isec = ServiceFactory.getSecurityService();
	      String[] allDBIDs = isec.getAllDBIDs();
	      String primaryDbID = isec.findDBForSiteID(primarySiteID);

	      AssessmentQueryInfo aqi = null;
	      AnalysisResultQueryInfo arqi = null;
	      if (caQueryString != null && caQueryString.trim().length() > 0) {
	         JSONObject js = new JSONObject(caQueryString);
	         aqi = AssessmentQueryInfo.initializeFromJSON(js);
	         CAQueryWorker caqTask = new CAQueryWorker(aqi, dbID, primarySiteID);

	         caqTaskList.add(caqTask);
	      }
	      if (ddQueryString != null && ddQueryString.trim().length() > 0) {
	         JSONObject js = new JSONObject(ddQueryString);
	         arqi = AnalysisResultQueryInfo.initializeFromJSON(js);
	         DDQueryWorker ddTask = new DDQueryWorker(arqi, dbID, primarySiteID);
	         ddTaskList.add(ddTask);
	      }

	      if (!caqTaskList.isEmpty()) {
	         // add other sites also
	         for (String aDbID : allDBIDs) {
	            if (aDbID.equals(primaryDbID))
	               continue;
	            CAQueryWorker caqTask = new CAQueryWorker(aqi, aDbID, primarySiteID);
	            caqTaskList.add(caqTask);
	         }
	      }

	      if (!ddTaskList.isEmpty()) {
	         for (String aDbID : allDBIDs) {
	            if (aDbID.equals(primaryDbID))
	               continue;
	            DDQueryWorker ddTask = new DDQueryWorker(arqi, aDbID, primarySiteID);
	            ddTaskList.add(ddTask);
	         }
	      }

	      List<SubjectAsScoreValueSummary> summaryList = new LinkedList<SubjectAsScoreValueSummary>();
	      List<AnalysisResultSummary> arsList = new LinkedList<AnalysisResultSummary>();
	      if (!caqTaskList.isEmpty()) {
	         for (CAQueryWorker worker : caqTaskList) {
	            try {
	               List<SubjectAsScoreValueSummary> siteSummaryList = worker.call();
	               if (siteSummaryList != null && !siteSummaryList.isEmpty())
	                  summaryList.addAll(siteSummaryList);
	            } catch (Throwable t) {
	               log.error("handleMultisiteQuery", t);
	            }
	         }
	         prepareQSListNew(primarySiteID, expMap, summaryList, selExpSegs, allExpSegs, selDataTypes, selVersions, "");
	      }
	      if (!ddTaskList.isEmpty()) {
	         boolean hasCAQuery = !summaryList.isEmpty();
	         summaryList = null;
	         for (DDQueryWorker worker : ddTaskList) {
	            try {
	               List<AnalysisResultSummary> siteArsList = worker.call();
	               if (siteArsList != null && !siteArsList.isEmpty()) {
	                  arsList.addAll(siteArsList);
	               }
	            } catch (ExecutionException e) {
	               log.error("handleMultisiteQuery", e);
	            }
	         }
	         prepQsList(arsList, hasCAQuery);
	      }
	   }

   public void handleMultisiteQueryParallel(String caQueryString,
         String ddQueryString, String primarySiteID) throws Exception {
      List<Future<List<SubjectAsScoreValueSummary>>> caqTaskList = new LinkedList<Future<List<SubjectAsScoreValueSummary>>>();
      List<Future<List<AnalysisResultSummary>>> ddTaskList = new LinkedList<Future<List<AnalysisResultSummary>>>();

      ISecurityService isec = ServiceFactory.getSecurityService();
      String[] allDBIDs = isec.getAllDBIDs();
      String primaryDbID = isec.findDBForSiteID(primarySiteID);

      AssessmentQueryInfo aqi = null;
      AnalysisResultQueryInfo arqi = null;
      if (caQueryString != null && caQueryString.trim().length() > 0) {
         JSONObject js = new JSONObject(caQueryString);
         aqi = AssessmentQueryInfo.initializeFromJSON(js);
         CAQueryWorker caqTask = new CAQueryWorker(aqi, dbID, primarySiteID);
         Future<List<SubjectAsScoreValueSummary>> future = executor
               .submit(caqTask);
         caqTaskList.add(future);
      }
      if (ddQueryString != null && ddQueryString.trim().length() > 0) {
         JSONObject js = new JSONObject(ddQueryString);
         arqi = AnalysisResultQueryInfo.initializeFromJSON(js);
         DDQueryWorker ddTask = new DDQueryWorker(arqi, dbID, primarySiteID);
         Future<List<AnalysisResultSummary>> future = executor.submit(ddTask);
         ddTaskList.add(future);
      }

      if (!caqTaskList.isEmpty()) {
         // add other sites also
         for (String aDbID : allDBIDs) {
            if (aDbID.equals(primaryDbID))
               continue;
            CAQueryWorker caqTask = new CAQueryWorker(aqi, aDbID, primarySiteID);
            Future<List<SubjectAsScoreValueSummary>> future = executor
                  .submit(caqTask);
            caqTaskList.add(future);
         }
      }

      if (!ddTaskList.isEmpty()) {
         for (String aDbID : allDBIDs) {
            if (aDbID.equals(primaryDbID))
               continue;
            DDQueryWorker ddTask = new DDQueryWorker(arqi, aDbID, primarySiteID);
            Future<List<AnalysisResultSummary>> future = executor
                  .submit(ddTask);
            ddTaskList.add(future);
         }
      }

      List<SubjectAsScoreValueSummary> summaryList = new LinkedList<SubjectAsScoreValueSummary>();
      List<AnalysisResultSummary> arsList = new LinkedList<AnalysisResultSummary>();
      if (!caqTaskList.isEmpty()) {

      // wait till all tasks are finished in one way or another
         while (true) {
            boolean hasUnfinished = false;
            for (Future<List<SubjectAsScoreValueSummary>> future : caqTaskList) {
               if (!future.isDone()) {
                  hasUnfinished = true;
                  break;
               }
            }
            if (!hasUnfinished)
               break;
            synchronized (this) {
               this.wait(200L);
            }
         }

         for (Future<List<SubjectAsScoreValueSummary>> future : caqTaskList) {
            try {
               List<SubjectAsScoreValueSummary> siteSummaryList = future.get();
               if (siteSummaryList != null && !siteSummaryList.isEmpty())
                  summaryList.addAll(siteSummaryList);
            } catch (Throwable t) {
               log.error("handleMultisiteQuery", t);
            }
         }
         prepareQSList(primarySiteID, expMap, summaryList);
      }
      if (!ddTaskList.isEmpty()) {
         boolean hasCAQuery = !summaryList.isEmpty();

         // wait till all tasks are finished in one way or another
         while (true) {
            boolean hasUnfinished = false;
            for (Future<List<AnalysisResultSummary>> future : ddTaskList) {
               if (!future.isDone()) {
                  hasUnfinished = true;
                  break;
               }
            }
            if (!hasUnfinished)
               break;
            synchronized (this) {
               this.wait(200L);
            }
         }

         summaryList = null;
         for (Future<List<AnalysisResultSummary>> future : ddTaskList) {
            try {
               List<AnalysisResultSummary> siteArsList = future.get();
               if (siteArsList != null && !siteArsList.isEmpty()) {
                  arsList.addAll(siteArsList);
               }
            } catch (ExecutionException e) {
               log.error("handleMultisiteQuery", e);
            }
         }
         prepQsList(arsList, hasCAQuery);
      }
   }

   protected void prepQsList(List<AnalysisResultSummary> arsList,
         boolean hasCAQuery) {
      if (hasCAQuery) {
         Map<String, QuerySummary> map = new HashMap<String, QuerySummary>();
         List<QuerySummary> qsList = queryForm.getQsList();
         for (QuerySummary qs : qsList) {
            AssessmentResultSummary arso = qs.getArs();
            String key = arso.getSubjectID() + ":" + arso.getExpID();
            map.put(key, qs);
         }
         for (AnalysisResultSummary ddrs : arsList) {
            String key = ddrs.getSubjectID() + ":" + ddrs.getExpID();
            QuerySummary qs = map.get(key);
            if (qs != null) {
               qs.setDdrs(ddrs);
            }
         }
      } else {
         List<QuerySummary> qsList = new ArrayList<QuerySummary>(arsList.size());
         for (AnalysisResultSummary arso : arsList) {
            QuerySummary qs = new QuerySummary(arso.getSubjectID(), arso
                  .getExpID());
            qs.setDdrs(arso);
            qsList.add(qs);
         }
         queryForm.setQsList(qsList);
      }
   }

   protected void prepareQSList(String primarySiteID,
         Map<String, Experiment> expMap,
         List<SubjectAsScoreValueSummary> summaryList) throws Exception {
      Map<String, AssessmentResultSummary> arsMap = new LinkedHashMap<String, AssessmentResultSummary>(
            17);
      Map<String, String> siteAsID2NameMap = new HashMap<String, String>(7);
      for (SubjectAsScoreValueSummary sasvs : summaryList) {
         String key = AssessmentResultSummary.getKey(sasvs.getSubjectID(),
               sasvs.getExperimentID(), sasvs.getSiteID());
         AssessmentResultSummary ars = arsMap.get(key);
         int expID = sasvs.getExperimentID();
         int visitID = sasvs.getVisitID();
         int segmentID = sasvs.getSegmentID();
         String siteID = sasvs.getSiteID();
         int asID = GenUtils.toInt(sasvs.getAssessmentID(), -1);
         Assertion.assertTrue(asID != -1);
         if (ars == null) {
            String expName = getExperimentName(sasvs.getExperimentID(), sasvs
                  .getSiteID(), primarySiteID, rds, expMap);
            ars = new AssessmentResultSummary(sasvs.getSubjectID(), String
                  .valueOf(sasvs.getExperimentID()), expName,
                  sasvs.getSiteID(), sasvs.getTimeStamp());
            arsMap.put(key, ars);
         }
         VisitSegAsResultValues vsarv = ars.getOrAdd(visitID, segmentID, expID,
               siteID);
         vsarv.setTimeStamp(sasvs.getTimeStamp());
         String saKey = asID + "_" + siteID;
         String asName = siteAsID2NameMap.get(saKey);
         if (asName == null) {
            if (siteID == null)
               siteID = primarySiteID;
            asName = getAssessmentName(asID, siteID, rds);
            siteAsID2NameMap.put(saKey, asName);
         }

         ASValueKey avKey = new ASValueKey(asID, asName, sasvs.getScoreName());

         vsarv.add(avKey, sasvs);
      }

      List<QuerySummary> qsList = new ArrayList<QuerySummary>(arsMap.size());
      for (AssessmentResultSummary ars : arsMap.values()) {
         QuerySummary qs = new QuerySummary(ars.getSubjectID(), ars.getExpID());
         qs.setArs(ars);
         qsList.add(qs);
      }
      queryForm.setQsList(qsList);
   }

   protected void prepareQSListNew(String primarySiteID,
	         Map<String, Experiment> expMap,
	         List<SubjectAsScoreValueSummary> summaryList, 
	         String[] selSegs, 
	         String[] allSegs, 
	         String[] selDataTypes, 
	         String[] selVers,
	         String queryType) throws Exception {
	      Map<String, AssessmentResultSummary> arsMap = new LinkedHashMap<String, AssessmentResultSummary>(
	            17);
	      Map<String, String> siteAsID2NameMap = new HashMap<String, String>(7);
	      for (SubjectAsScoreValueSummary sasvs : summaryList) {
	         String key = AssessmentResultSummary.getKey(sasvs.getSubjectID(),
	               sasvs.getExperimentID(), sasvs.getSiteID());
	         AssessmentResultSummary ars = arsMap.get(key);
	         int expID = sasvs.getExperimentID();
	         int visitID = sasvs.getVisitID();
	         int segmentID = sasvs.getSegmentID();
	         String siteID = sasvs.getSiteID();
	         int asID = GenUtils.toInt(sasvs.getAssessmentID(), -1);
	         String visitName = sasvs.getVisitName();
	         String visitDate = sasvs.getVisitDate();
	         String studyGroup = sasvs.getStudyGroup();
	         String segName = sasvs.getSegmentName();
	         String segDate = sasvs.getSegmentDate();
	         String studyName = sasvs.getStudyName();
	         String studyDate = sasvs.getStudyDate();
	         String studyId = sasvs.getStudyId();
	         
	         if(!queryType.equals(Constants.IMAGING_ONLY) && 
	        		 !queryType.equals(Constants.ASSESSMENT_OR_IMAGING) &&
	        		 !queryType.equals(Constants.GENETICS_ONLY) &&
	        		 !queryType.equals(Constants.IMAGING_AND_GENETICS) &&
	        		 !queryType.equals(Constants.IMAGING_OR_GENETICS) &&
	        		 !queryType.equals(Constants.ASSESS_OR_GENETICS)) Assertion.assertTrue(asID != -1);
	         if (ars == null) {
	            String expName = getExperimentName(sasvs.getExperimentID(), sasvs
	                  .getSiteID(), primarySiteID, rds, expMap);
	            ars = new AssessmentResultSummary(sasvs.getSubjectID(), String
	                  .valueOf(sasvs.getExperimentID()), expName,
	                  sasvs.getSiteID(), sasvs.getTimeStamp());
	            arsMap.put(key, ars);
	         }
	         VisitSegAsResultValues vsarv = ars.getOrAdd(visitID, segmentID, expID,
	               siteID, visitName, visitDate, studyGroup, segName, segDate, studyName, studyDate, studyId);
	         vsarv.setTimeStamp(sasvs.getTimeStamp());
	         String saKey = asID + "_" + siteID;
	         String asName = siteAsID2NameMap.get(saKey);
	         if (asName == null) {
	            if (siteID == null)
	               siteID = primarySiteID;
	            asName = getAssessmentName(asID, siteID, rds);
	            siteAsID2NameMap.put(saKey, asName);
	         }

	         ASValueKey avKey = new ASValueKey(asID, asName, sasvs.getScoreName());

	         vsarv.add(avKey, sasvs);
	      }

	      List<QuerySummary> qsList = new ArrayList<QuerySummary>(arsMap.size());
	      for (AssessmentResultSummary ars : arsMap.values()) {
	         QuerySummary qs = new QuerySummary(ars.getSubjectID(), ars.getExpID(), selSegs, allSegs, selDataTypes, selVers);
	         qs.setArs(ars);
	         qsList.add(qs);
	      }
	      queryForm.setQsList(qsList);
	   }

   /*
    * This function is created for BatchQueryAll purpose. So user can do a batch query to get all the subjects' query summary
    */
   protected void prepareQSListAllSubj(List<SubjectAsScoreValueSummary> summaryList) throws Exception {
	      List<QuerySummary> qsList = new ArrayList<QuerySummary>();
	      for (SubjectAsScoreValueSummary ss: summaryList) {
	         QuerySummary qs = new QuerySummary(ss.getSubjectID(), String.valueOf(ss.getExperimentID()));	         
	         qsList.add(qs);
	      }
	      queryForm.setQsList(qsList);
	   }

   public static Map<String, Experiment> toMap(List<Experiment> experiments) {
      Map<String, Experiment> map = new HashMap<String, Experiment>();
      for (Iterator<Experiment> iter = experiments.iterator(); iter.hasNext();) {
         Experiment exp = iter.next();
         map.put(exp.getUniqueid().toString(), exp);
      }
      return map;
   }

   /**
    * if site is primary site uses <code>expMap</code> for the primary site
    * otherwise uses {@link IRemoteDBServices} to get the experiment name.
    *
    * @param expID
    * @param siteID
    * @param primarySiteID
    * @param rds
    * @param expMap
    * @return
    * @throws Exception
    */
   public static String getExperimentName(int expID, String siteID,
         String primarySiteID, IRemoteDBServices rds,
         Map<String, Experiment> expMap) throws Exception {
      if (siteID == null || siteID.equals(primarySiteID)) {
         Experiment exp = expMap.get(String.valueOf(expID));
         return exp.getName();
      } else {
         Map<String, Experiment> map = rds.getAllExperiments(siteID);
         Experiment exp = map.get(String.valueOf(expID));
         return exp.getName();
      }
   }

   public static String getAssessmentName(int asID, String siteID,
         IRemoteDBServices rds) throws Exception {
      List<Assessment> asList = rds.getAllAssessments(siteID);
      for (Assessment as : asList) {
         if (as.getAssessmentid().intValue() == asID) {
            return as.getName();
         }
      }
      return null;
   }

   class CAQueryWorker implements Callable<List<SubjectAsScoreValueSummary>> {
      AssessmentQueryInfo aqi;
      String aDBID;
      List<SubjectAsScoreValueSummary> summaryList;
      String primarySiteID;

      Integer selectedExperimentId;
      String protocolIDs;
      String dataObjectTypes = "";
      String protocolVersions;
      String queryType;
      String isValidated = Constants.IsValidated;

      public CAQueryWorker(AssessmentQueryInfo aqi, String aDBID,
            String primarySiteID) {
         this.aqi = aqi;
         this.aDBID = aDBID;
         this.primarySiteID = primarySiteID;
      }
      
      public CAQueryWorker(AssessmentQueryInfo aqi, String aDBID,
              String primarySiteID, String queryType) {
           this.aqi = aqi;
           this.aDBID = aDBID;
           this.primarySiteID = primarySiteID;
           this.queryType = queryType;
      }        
      
      public CAQueryWorker(String aDBID, String primarySiteID, String queryType, 
    		  Integer selectedExpId, String[] selProtocols, String[] selDtypes, 
    		  String[] selVers, String isValidateText){
    	  this.aDBID = aDBID;
    	  this.primarySiteID = primarySiteID;
    	  this.selectedExperimentId = selectedExpId;
    	  this.queryType = queryType;
    	  this.isValidated = isValidateText;
    	  
    	  StringBuilder sb = new StringBuilder();
    	  for(int i=0; i<selProtocols.length; i++){
    		  if(selProtocols[i].equals(Constants.ALL_PROTOCOLS)){
    			  sb = new StringBuilder();
    			  sb.append(selProtocols[i]);
    			  break;
    		  }
    		  if(i>0) sb.append(",");    		  
    		  sb.append("'");    		  
    		  sb.append(selProtocols[i]);
    		  sb.append("'");    		  
    	  }
    	  protocolIDs = sb.toString();
    	  
    	  sb = new StringBuilder();
    	  for(int i=0; i<selVers.length; i++){
    		  if(selVers[i].equals(Constants.ALL_VERSIONS)){
    			  sb = new StringBuilder();
    			  sb.append(selVers[i]);
    			  break;
    		  }
    		  if(i>0) sb.append(",");
    		  sb.append(selVers[i]);    		  
    	  }
    	  protocolVersions= sb.toString();
    	  
    	  String[] dicomTypes = new String[] { "3_SRB COL DICOM","3_SRB FILE DICOM",
					"3_LOC COL DICOM", "3_LOC FILE DICOM",
					"3_GFTP COL DICOM", "3_GFTP FILE DICOM" };
    	  String[] niftiTypes = new String[] {"3_SRB COL NIFTI","3_SRB FILE NIFTI",
					"3_LOC COL NIFTI", "3_LOC FILE NIFTI",
					"3_GFTP COL NIFTI", "3_GFTP FILE NIFTI"	};
	  	  sb = new StringBuilder();
		  for(int i=0; i<selDtypes.length; i++){
			  if(selDtypes[i].equals(Constants.ALL_TYPES)){
				  sb = new StringBuilder();
				  sb.append(selDtypes[i]);
				  dataObjectTypes = sb.toString();
				  break;
			  }else if(selDtypes[i].equals(Constants.DICOM)){
				  sb = new StringBuilder();
				  for(int j=0; j<dicomTypes.length; j++){
					  if(j>0) sb.append(",");
					  sb.append("'");
					  sb.append(dicomTypes[j]);
					  sb.append("'");
				  }	
				  dataObjectTypes = sb.toString();
			  }else if(selDtypes[i].equals(Constants.NIFTI)){
				  sb = new StringBuilder();
				  for(int j=0; j<niftiTypes.length; j++){					  
					  if(j>0) sb.append(",");
					  sb.append("'");
					  sb.append(niftiTypes[j]);
					  sb.append("'");
				  }
				  if(dataObjectTypes.length()>0) {
					  dataObjectTypes = dataObjectTypes + "," + sb.toString();					  
				  }else{
					  dataObjectTypes = sb.toString();
				  }
			  }
		  }
      }

      public CAQueryWorker(String aDBID, String primarySiteID, String queryType, 
    		  Integer selectedExpId, String[] selProtocols, String[] selDtypes, 
    		  String[] selVers){
    	  this.aDBID = aDBID;
    	  this.primarySiteID = primarySiteID;
    	  this.selectedExperimentId = selectedExpId;
    	  this.queryType = queryType;
    	  
    	  StringBuilder sb = new StringBuilder();
    	  for(int i=0; i<selProtocols.length; i++){
    		  if(selProtocols[i].equals(Constants.ALL_PROTOCOLS)){
    			  sb = new StringBuilder();
    			  sb.append(selProtocols[i]);
    			  break;
    		  }
    		  if(i>0) sb.append(",");    		  
    		  sb.append("'");    		  
    		  sb.append(selProtocols[i]);
    		  sb.append("'");    		  
    	  }
    	  protocolIDs = sb.toString();
    	  
    	  sb = new StringBuilder();
    	  for(int i=0; i<selVers.length; i++){
    		  if(selVers[i].equals(Constants.ALL_VERSIONS)){
    			  sb = new StringBuilder();
    			  sb.append(selVers[i]);
    			  break;
    		  }
    		  if(i>0) sb.append(",");
    		  sb.append(selVers[i]);    		  
    	  }
    	  protocolVersions= sb.toString();
    	  
    	  String[] dicomTypes = new String[] { "3_SRB COL DICOM","3_SRB FILE DICOM",
					"3_LOC COL DICOM", "3_LOC FILE DICOM",
					"3_GFTP COL DICOM", "3_GFTP FILE DICOM" };
    	  String[] niftiTypes = new String[] {"3_SRB COL NIFTI","3_SRB FILE NIFTI",
					"3_LOC COL NIFTI", "3_LOC FILE NIFTI",
					"3_GFTP COL NIFTI", "3_GFTP FILE NIFTI"	};
	  	  sb = new StringBuilder();
		  for(int i=0; i<selDtypes.length; i++){
			  if(selDtypes[i].equals(Constants.ALL_TYPES)){
				  sb = new StringBuilder();
				  sb.append(selDtypes[i]);
				  dataObjectTypes = sb.toString();
				  break;
			  }else if(selDtypes[i].equals(Constants.DICOM)){
				  sb = new StringBuilder();
				  for(int j=0; j<dicomTypes.length; j++){
					  if(j>0) sb.append(",");
					  sb.append("'");
					  sb.append(dicomTypes[j]);
					  sb.append("'");
				  }	
				  dataObjectTypes = sb.toString();
			  }else if(selDtypes[i].equals(Constants.NIFTI)){
				  sb = new StringBuilder();
				  for(int j=0; j<niftiTypes.length; j++){					  
					  if(j>0) sb.append(",");
					  sb.append("'");
					  sb.append(niftiTypes[j]);
					  sb.append("'");
				  }
				  if(dataObjectTypes.length()>0) {
					  dataObjectTypes = dataObjectTypes + "," + sb.toString();					  
				  }else{
					  dataObjectTypes = sb.toString();
				  }
			  }
		  }
      }

// Jinran modified to accommodate stand-alone imaging query  
//      @Override
//      public List<SubjectAsScoreValueSummary> call() throws Exception {
//         try {
//            ISecurityService secService = ServiceFactory.getSecurityService();
//            IRemoteDBServices rds = ServiceFactory.getRemoteDBServices();
//            String siteID = secService.findSiteIDByDbID(aDBID);
//            IAssessmentService asService = ServiceFactory
//                  .getAssessmentService(aDBID);
//            UserInfo userInfo = new UserInfo(Constants.ADMIN_USER, null, null);
//            if (!siteID.equals(primarySiteID)) {
//               List<AssessmentSelectionInfo> asiList = rds
//                     .getAllAssessmentsWithScores(siteID);
//               List<QueryPartInfo> qpiList = aqi.getOPIList4Site(asiList);
//               if (qpiList == null) {
//                  System.out
//                        .println("+++++++++++++++++++++++++++++++++++++++++++++++++ qpiList == null");
//                  return new ArrayList<SubjectAsScoreValueSummary>(0);
//               } else {
//                  this.aqi = new AssessmentQueryInfo(asiList);
//                  this.aqi.setQpiList(qpiList);
//               }
//            }
//
//            Operator root = aqi.prepareOpTree();
//            String queryScope = queryForm.getQueryScopeSelector()
//                  .getQueryScope();
//
//            List<Integer> experimentIDs = prepareExperimentIDs(siteID,
//                  primarySiteID);
//            if (experimentIDs != null && experimentIDs.isEmpty()) {
//               // no matching experiment for this site don't bother querying
//               return new ArrayList<SubjectAsScoreValueSummary>(0);
//            }
//
//            summaryList = asService.queryForScores(userInfo, root, aqi
//                  .getAsiList(), experimentIDs, queryScope, primarySiteID);
//         } catch (Exception x) {
//            log.error("CAQueryWorker", x);
//            summaryList = null;
//            throw x;
//         }
//         return summaryList;
//      }

      @Override
      public List<SubjectAsScoreValueSummary> call() throws Exception {
         try {
            ISecurityService secService = ServiceFactory.getSecurityService();
            IRemoteDBServices rds = ServiceFactory.getRemoteDBServices();
            String siteID = secService.findSiteIDByDbID(aDBID);
            IAssessmentService asService = ServiceFactory
                  .getAssessmentService(aDBID);
            UserInfo userInfo = new UserInfo(Constants.ADMIN_USER, null, null);
            
            if(queryType.equals(Constants.IMAGING_ONLY) || 
            		queryType.equals(Constants.GENETICS_ONLY) ||
            		queryType.equals(Constants.IMAGING_AND_GENETICS)||
            		queryType.equals(Constants.IMAGING_OR_GENETICS) ||
            		queryType.equals(Constants.ASSESS_OR_GENETICS)){
            	String queryScope = queryForm.getQueryScopeSelector().getQueryScope();
                summaryList = asService.queryForImagesOnlyOrGeneOnly(userInfo, this.selectedExperimentId, queryScope, 
               		 primarySiteID, this.protocolIDs, this.dataObjectTypes, this.protocolVersions);
                
            }else{
            	if (!siteID.equals(primarySiteID)) {
                    List<AssessmentSelectionInfo> asiList = rds
                          .getAllAssessmentsWithScores(siteID);
                    List<QueryPartInfo> qpiList = aqi.getOPIList4Site(asiList);
                    if (qpiList == null) {
                       System.out
                             .println("+++++++++++++++++++++++++++++++++++++++++++++++++ qpiList == null");
                       return new ArrayList<SubjectAsScoreValueSummary>(0);
                    } else {
                       this.aqi = new AssessmentQueryInfo(asiList);
                       this.aqi.setQpiList(qpiList);
                    }
                }
            	
            	Operator root = aqi.prepareOpTree();
            	
            	List<Integer> experimentIDs = prepareExperimentIDs(siteID,
                         primarySiteID);
            	if (experimentIDs != null && experimentIDs.isEmpty()) {
                      // no matching experiment for this site don't bother querying
            		return new ArrayList<SubjectAsScoreValueSummary>(0);
            	}

            	String queryScope = queryForm.getQueryScopeSelector().getQueryScope();           

            	summaryList = asService.queryForScores(userInfo, root, aqi.getAsiList(), experimentIDs, 
            			queryScope, primarySiteID, this.queryType, this.isValidated);
                   
            }
            
         } catch (Exception x) {
        	 log.error("CAQueryWorker", x);
        	 summaryList = null;
        	 throw x;
         }
         
         return summaryList;
         
      }

      
   }// ;

   class DDQueryWorker implements Callable<List<AnalysisResultSummary>> {
      AnalysisResultQueryInfo arqi;
      String aDBID;
      List<AnalysisResultSummary> arsList;
      String primarySiteID;

      public DDQueryWorker(AnalysisResultQueryInfo arqi, String aDBID,
            String primarySiteID) {
         this.arqi = arqi;
         this.aDBID = aDBID;
         this.primarySiteID = primarySiteID;
      }

      @Override
      public List<AnalysisResultSummary> call() throws Exception {
         ISecurityService secService = ServiceFactory.getSecurityService();
         IRemoteDBServices rds = ServiceFactory.getRemoteDBServices();
         String siteID = secService.findSiteIDByDbID(aDBID);
         if (!siteID.equals(primarySiteID)) {
            List<Extendedtuple> etList = rds.getExtendedTuples(siteID);
            List<MGQueryPartInfo> qpiList = arqi.getQPIListForSite(etList);
            if (qpiList == null) {
               return new ArrayList<AnalysisResultSummary>(0);
            } else {
               arqi = new AnalysisResultQueryInfo();
               arqi.setMgQPIList(qpiList);
            }
         }
         UserInfo userInfo = new UserInfo(Constants.ADMIN_USER, null, null);
         Operator root = arqi.prepareOpTree();

         IAnalysisResultService ars = ServiceFactory
               .getAnalysisResultService(aDBID);

         List<Integer> experimentIDs = prepareExperimentIDs(siteID,
               primarySiteID);
         if (experimentIDs != null && experimentIDs.isEmpty()) {
            // no matching experiment for this site, thus don't bother querying
            return new ArrayList<AnalysisResultSummary>(0);
         }

         arsList = ars
               .queryForAnalysisResults(userInfo, root, experimentIDs,
                     queryForm.getQueryScopeSelector().getQueryScope(),
                     primarySiteID);
         return arsList;
      }

   }
}
