package clinical.web.common;

import java.util.List;
import java.util.Map;

import clinical.server.vo.Animalspecies;
import clinical.server.vo.Argumenttype;
import clinical.server.vo.Assessment;
import clinical.server.vo.Assessmentscorecode;
import clinical.server.vo.Collectionequipment;
import clinical.server.vo.Databaseuser;
import clinical.server.vo.Dataclassification;
import clinical.server.vo.Experiment;
import clinical.server.vo.Expsegment;
import clinical.server.vo.Measurementsystem;
import clinical.server.vo.Measurementtype;
import clinical.server.vo.Measurementunit;
import clinical.server.vo.Ontologyconcept;
import clinical.server.vo.Ontologysource;
import clinical.server.vo.Protocol;
import clinical.server.vo.Researchgroup;
import clinical.server.vo.Researchgrouptype;
import clinical.server.vo.Securityclassification;
import clinical.server.vo.Site;
import clinical.server.vo.Tableid;
import clinical.server.vo.Userclass;
import clinical.server.vo.Userstatus;
import clinical.server.vo.Visittype;
import clinical.web.exception.DBCacheException;
import clinical.web.helpers.ExpSubjInfo;

/**
 * An interface for retrieving and caching nearly static data from the database.
 *
 * @author I. Burak Ozyurt
 * @version $Id: IDBCache.java 873 2015-01-07 20:55:20Z jinranc $
 */
public interface IDBCache {
	/**
	 * looks for an entry in <code>NC_DATABASEUSER</code> table with userName
	 * (in CAPITAL letters!!) with the user classification
	 * <code>userClass</code> which is not a group.
	 *
	 * @param ui
	 *            User Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param userName
	 *            the database user name as stored in the database ( case
	 *            insensitive )
	 * @param userClass
	 *            the userclass that the user belongs (name column from
	 *            <code>NC_USERCLASS</code> table
	 * @return A clinical.server.vo.Databaseuser object corresponding to the
	 *         userName
	 * @throws DBCacheException
	 */
	public Databaseuser getDatabaseUser(UserInfo ui, String userName,
			String userClass) throws DBCacheException;


	/**
	 *
	 * @param ui
	 *            User Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param tableName
	 *            the database table name to get the table id for
	 * @return a <code>clinical.server.vo.Tableid</code> object
	 * @throws DBCacheException
	 */
	public Tableid getTableID(UserInfo ui, String tableName)
			throws DBCacheException;

	/**
	 * Returns all collection equipment records from the database or from cache
	 * if already loaded.
	 *
	 * @param ui
	 *            User Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param forceRecache
	 *            if true clears the cache and reloads from the database
	 * @return a list of <code>clinical.server.vo.Collectionequipment</code>
	 *         objects
	 * @throws DBCacheException
	 */
	public List<Collectionequipment> getCollectionEquipments(UserInfo ui,
			boolean forceRecache) throws DBCacheException;

	/**
	 * Returns all research group type records from the database or from cache
	 * if already loaded.
	 *
	 * @param ui
	 *            User Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param forceRecache
	 *            if true clears the cache and reloads from the database
	 * @return a list of <code>clinical.server.vo.Researchgrouptype</code>
	 *         objects
	 * @throws DBCacheException
	 */
	public List<Researchgrouptype> getResearchGroupTypes(UserInfo ui,
			boolean forceRecache) throws DBCacheException;

	/**
	 * Returns all research group records from the database or from cache if
	 * already loaded.
	 *
	 * @param ui
	 *            User Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param forceRecache
	 *            if true clears the cache and reloads from the database
	 * @return a list of <code>clinical.server.vo.Researchgroup</code> objects
	 * @throws DBCacheException
	 */
	public List<Researchgroup> getResearchGroups(UserInfo ui,
			boolean forceRecache) throws DBCacheException;

	/**
	 * Returns all visit type records from the database or from cache if already
	 * loaded.
	 *
	 * @param ui
	 *            User Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param forceRecache
	 *            if true clears the cache and reloads from the database
	 * @return A list of <code>clinical.server.vo.Visittype</code> objects
	 * @throws DBCacheException
	 */
	public List<Visittype> getVisitTypes(UserInfo ui, boolean forceRecache)
			throws DBCacheException;

	/**
	 * Returns all user class records from the database or from cache if already
	 * loaded.
	 *
	 * @param ui
	 *            User Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param forceRecache
	 *            if true clears the cache and reloads from the database
	 * @return a list of <code>clinical.server.vo.Userclass</code> objects
	 * @throws DBCacheException
	 */
	public List<Userclass> getUserClasses(UserInfo ui, boolean forceRecache)
			throws DBCacheException;

	/**
	 * Returns all security classification records from the database or from
	 * cache if already loaded.
	 *
	 * @param ui
	 *            User Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param forceRecache
	 *            if true clears the cache and reloads from the database
	 * @return a list of <code>clinical.server.vo.Securityclassification</code>
	 *         objects
	 * @throws DBCacheException
	 */
	public List<Securityclassification> getSecurityClassifications(UserInfo ui,
			boolean forceRecache) throws DBCacheException;

	/**
	 * Returns all protocol records from the database or from cache if already
	 * loaded.
	 *
	 * @param ui
	 *            User Information used to use the correct named database user
	 *            connection from the con nection pool
	 * @param forceRecache
	 *            if true clears the cache and reloads from the database
	 * @return a list of <code>clinical.server.vo.Protocol</code> objects
	 * @throws DBCacheException
	 */
	public List<Protocol> getProtocols(UserInfo ui, boolean forceRecache)
			throws DBCacheException;

	/**
	 * Returns all experiment records from the database or from cache if already
	 * loaded.
	 *
	 * @param ui
	 *            User Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param forceRecache
	 *            if true clears the cache and reloads from the database
	 * @return A list of <code>clinical.server.vo.Experiment</code> objects
	 * @throws DBCacheException
	 */
	public List<Experiment> getExperiments(UserInfo ui, boolean forceRecache)
			throws DBCacheException;

	public Map<String,Experiment> getExperimentMap(UserInfo ui, boolean forceRecache)
	throws DBCacheException;

	/**
	 * Returns all score codes keyed by assessment ID and score name.
	 *
	 * @param ui
	 *            User Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param forceRecache
	 *            if true clears the cache and reloads from the database
	 * @return A hash table of <code>Assessmentscorecode</code> objects keyed
	 *         by assessment ID + '_' + score name
	 * @throws DBCacheException
	 */
	public Map<String, List<Assessmentscorecode>> getScoreCodeMap(UserInfo ui,
			boolean forceRecache) throws DBCacheException;

	/**
	 *
	 * @param ui
	 *            User Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param forceRecache
	 *            if true clears the cache and reloads from the database
	 * @return A list of <code>clinical.server.vo.Dataclassification</code>
	 *         objects
	 * @throws DBCacheException
	 */
	public List<Dataclassification> getDataClassications(UserInfo ui,
			boolean forceRecache) throws DBCacheException;

	/**
	 *
	 * @param ui
	 *            User Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param forceRecache
	 *            if true clears the cache and reloads from the database
	 * @throws DBCacheException
	 * @return List A list of <code>clinical.server.vo.Userstatus</code>
	 *         objects
	 */
	public List<Userstatus> getUserStatusList(UserInfo ui, boolean forceRecache)
			throws DBCacheException;

	/**
	 *
	 * @param ui
	 *            User Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param forceRecache
	 *            if true clears the cache and reloads from the database
	 * @return A list of <code>clinical.server.vo.Assessment</code> objects
	 * @throws DBCacheException
	 */
	public List<Assessment> getAssessments(UserInfo ui, boolean forceRecache)
			throws DBCacheException;

	/**
	 *
	 * @param ui
	 *            User Information used to use the correct named database user
	 *            connection from the connection pool
	 * @param forceRecache
	 *            forceRecache if true clears the cache and reloads from the
	 *            database
	 * @return A list of {@link clinical.server.vo.Animalspecies} objects
	 * @throws DBCacheException
	 */
	public List<Animalspecies> getAnimalSpecies(UserInfo ui,
			boolean forceRecache) throws DBCacheException;

	public Map<String, Measurementunit> getMeasurementUnitMap(UserInfo ui,
			boolean forceRecache) throws DBCacheException;

	public Map<String, Measurementtype> getMeasurementTypeMap(UserInfo ui,
			boolean forceRecache) throws DBCacheException;

	public Map<String, Measurementsystem> getMeasurementSystemMap(UserInfo ui,
			boolean forceRecache) throws DBCacheException;

	public Map<String, Ontologyconcept> getOntologyConceptMap(UserInfo ui,
			boolean forceRecache) throws DBCacheException;

	public Map<String, Ontologysource> getOntologySourceMap(UserInfo ui,
			boolean forceRecache) throws DBCacheException;

	public Map<String, Argumenttype> getArgumentTypeMap(UserInfo ui,
			boolean forceRecache) throws DBCacheException;

	public List<Site> getSites(UserInfo ui, boolean forceRecache)
			throws DBCacheException;
	
	/***
	 * Return all nc_expsegment records from db or cache
	 * @param ui
	 * 				User Information used to use the correct named database user
	 *            	connection from the con nection pool
	 * @param forceRecache
	 * 			if true clears the cache and reloads from the database
	 * @return a list of <code>clinical.server.vo.Expsegment</code> objects
	 * @throws DBCacheException
	 */
	public List<Expsegment> getExpsegments(UserInfo ui, boolean forceRecache) throws DBCacheException;
	
	/***
	 * Return all nc_subjexperiment and visit info
	 * @param ui
	 * @param forceRecache
	 * @return a list of ExpSubjInfo 
	 * @throws DBCacheException
	 */
	public List<ExpSubjInfo> getAllExpSubjInfo(UserInfo ui, boolean forceRecache) throws DBCacheException;

}
