package clinical.web.common.query;

import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.Statement;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.utils.GenUtils;
import clinical.utils.NamedUserPool;
import clinical.web.Constants;
import clinical.web.ISQLDialect;
import clinical.web.ServiceFactory;
import clinical.web.common.vo.AsScoreInfo;
import clinical.web.common.vo.AssessmentSelectionInfo;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: AssessmentQueryBuilder.java,v 1.20 2006/06/19 22:42:27 bozyurt
 *          Exp $
 */

public class AssessmentQueryBuilder extends AbstractQueryBuilder implements
		ISearchCriteriaVisitor {
	protected Map<AsScoreInfo, AssessmentSelectionInfo> scoreMap = new HashMap<AsScoreInfo, AssessmentSelectionInfo>();
	protected List<Integer> experimentIDs;
	protected StringBuffer query;
	protected String dbID;
	protected ISQLDialect sqlDialect;
	/** if true only queries for validated assessments (for double entry) */
	protected static boolean doValidationCheck = true;
	protected Log log = LogFactory.getLog(AssessmentQueryBuilder.class);

	private String isValidated;
	/**
	 *
	 * @param assessments
	 */
	public AssessmentQueryBuilder(ISQLDialect sqlDialect,
			List<AssessmentSelectionInfo> assessments) {
		this(sqlDialect, assessments, null);
	}

	public AssessmentQueryBuilder(ISQLDialect sqlDialect,
			List<AssessmentSelectionInfo> assessments,
			List<Integer> experimentIDs) {
		this.sqlDialect = sqlDialect;
		for (AssessmentSelectionInfo asi : assessments) {
			for (AsScoreInfo si : asi.getScores()) {
				scoreMap.put(si, asi);
			}
			this.experimentIDs = experimentIDs;
		}
	}

	public AssessmentQueryBuilder(ISQLDialect sqlDialect,
				List<AssessmentSelectionInfo> assessments,
				List<Integer> experimentIDs, 
				String isValidated) {
		this.sqlDialect = sqlDialect;
		for (AssessmentSelectionInfo asi : assessments) {
			for (AsScoreInfo si : asi.getScores()) {
				scoreMap.put(si, asi);
			}
			this.experimentIDs = experimentIDs;
		}
		this.isValidated = isValidated;
	}
	
	/**
	 * Returns the generated SQL query.
	 *
	 * @return the generated SQL query
	 */
	public String getQuery() {
		return query.toString();
	}

	/**
	 *
	 * @param op
	 */
	public void visit(Operator op) throws Exception {
		StringBuffer inQuery = new StringBuffer(512);
		prepareInnerQuery(inQuery, op);
		query = new StringBuffer(1024);
		prepareOuterQuery(query, inQuery.toString(), op);
	}

	/**
	 * Prepares the sub select query traversing the operator tree rooted at the
	 * <code>op</code>. This process is inherently recursive.
	 *
	 * @param outq
	 *            buffer used to store the outer query
	 * @param innerQuery
	 *            the inner query (mainly a sub select query)
	 * @param op
	 *            root node in the operator tree to be traversed to build the
	 *            outer query
	 */
	protected void prepareOuterQuery(StringBuffer outq, String innerQuery,
			Operator op) throws Exception {
		if (op instanceof UnaryOperator) {
			UnaryOperator uop = (UnaryOperator) op;
			SearchPredicate sp = uop.getPredicate();
			Object o = sp.getAttribute();
			if (!(o instanceof AsScoreInfo))
				throw new Exception(
						"Attribute needs to be of type AsScoreInfo!");
			AsScoreInfo si = (AsScoreInfo) o;
			AssessmentSelectionInfo asi = (AssessmentSelectionInfo) scoreMap
					.get(si);
			outq.append(getOuterStaticPart(si.getType(), asi.getAssessmentID()
					.intValue(), si.getName()));
			outq.append(innerQuery);
			outq.append(")");
			outq.append(prepareExperimentQueryPart());
		} else if (op instanceof BinaryOperator) {
			BinaryOperator bop = (BinaryOperator) op;
			Operator lhs = bop.getLhs();
			prepareOuterQuery(outq, innerQuery, lhs);
			Operator rhs = bop.getRhs();
			/** @todo check why rhs can be null sometimes */
			if (rhs != null) {
				outq.append("\nUNION ALL\n");
				log.debug("rhs=" + rhs);
				prepareOuterQuery(outq, innerQuery, rhs);
			}
		}
	}

	/**
	 * Prepares the static part (mainly the first part) of the outer query.
	 *
	 * @param type
	 *            the type of the assessment data (can be 'integer',
	 *            'varchar','float' 'boolean', or 'timestamp').
	 * @param assessmentID
	 *            the unique ID of the assessment in the database
	 * @param scoreName
	 *            the name of the assessment score in the database
	 * @return the (more and less) fixed part of the outer query
	 */
	protected String getOuterStaticPart(String type, int assessmentID,
			String scoreName) {
		StringBuffer buf = new StringBuffer(256);
		buf
				.append(" SELECT b.subjectid, a.textvalue, a.scoreorder, a.scorename, a.assessmentid,");
		buf
				.append("b.componentid, b.segmentid, b.NC_EXPERIMENT_UNIQUEID, b.time_stamp, a.scoretype FROM ");
		if (type.equalsIgnoreCase("integer")) {
			buf.append(" nc_assessmentinteger a, ");
		} else if (type.equalsIgnoreCase("varchar")) {
			buf.append(" nc_assessmentvarchar a, ");
		} else if (type.equalsIgnoreCase("float")) {
			buf.append(" nc_assessmentfloat a, ");
		} else if (type.equalsIgnoreCase("boolean")) {
			buf.append(" nc_assessmentboolean a, ");
		} else if (type.equalsIgnoreCase("timestamp")) {
			buf.append(" nc_assessmenttimestamp a, ");
		}
		buf
				.append(" nc_storedassessment b WHERE a.STOREDASSESSMENTID = b.UNIQUEID AND ");
		buf.append("a.ASSESSMENTID=").append(assessmentID).append(
				" AND a.SCORENAME=");
		buf.append(sqlDialect.prepareString(scoreName));
		
		/**** Jinran modified: filter assessments by IsValidate  ***/ 
//		if (doValidationCheck) {
//			buf.append(" AND ").append(
//					sqlDialect
//							.prepareBooleanPredicate("a.isvalidated = ", true));
//			// buf.append(" AND a.isvalidated = 1");
//
//		}
		if (isValidated.equals(Constants.IsValidated)) {
			buf.append(" AND ").append(
					sqlDialect.prepareBooleanPredicate("a.isvalidated = ", true));
		}else if(isValidated.equals(Constants.NotValidated)){
			buf.append(" AND ").append(
					sqlDialect.prepareBooleanPredicate("a.isvalidated = ", false));
		}else if(isValidated.equals(Constants.BothValidAndInvalid)){
			
		}

		buf.append(" AND b.subjectid IN (");
		return buf.toString();
	}

	/**
	 * Prepares the static (the first part) of the inner query ( an uncorrelated
	 * sub select).
	 *
	 * @param type
	 *            the type of the assessment data (can be 'integer',
	 *            'varchar','float' or 'boolean'.
	 * @param assessmentID
	 *            the unique ID of the assessment in the database
	 * @return the (more and less) fixed part of the inner query
	 */
	protected String getInnerStaticPart(String type, int assessmentID)
			throws Exception {
		StringBuffer buf = new StringBuffer(256);
		buf.append("SELECT b.SUBJECTID FROM ");
		if (type.equalsIgnoreCase("integer")) {
			buf.append(" nc_assessmentinteger a, ");
		} else if (type.equalsIgnoreCase("varchar")) {
			buf.append(" nc_assessmentvarchar a, ");
		} else if (type.equalsIgnoreCase("float")) {
			buf.append(" nc_assessmentfloat a, ");
		} else if (type.equalsIgnoreCase("boolean")) {
			buf.append(" nc_assessmentboolean a, ");
		} else if (type.equalsIgnoreCase("timestamp")) {
			buf.append(" nc_assessmenttimestamp a, ");
		} else {
			throw new Exception("Not a supported type: " + type);
		}
		buf
				.append(" nc_storedassessment b WHERE a.STOREDASSESSMENTID = b.UNIQUEID AND a.ASSESSMENTID = ");
		buf.append(assessmentID).append(' ');
		buf.append(prepareExperimentQueryPart());
		return buf.toString();
	}

	protected String prepareExperimentQueryPart() {
		if (experimentIDs == null || experimentIDs.isEmpty()) {
			return "";
		}
		StringBuilder buf = new StringBuilder();
		buf.append(" AND ");
		if (experimentIDs.size() == 1) {
			buf.append("b.NC_EXPERIMENT_UNIQUEID = ").append(
					(Integer) experimentIDs.get(0));
			buf.append(' ');
		} else {
			buf.append("b.NC_EXPERIMENT_UNIQUEID IN (");
			for (Iterator<Integer> iter = experimentIDs.iterator(); iter
					.hasNext();) {
				Integer experimentID = iter.next();
				buf.append(experimentID);
				if (iter.hasNext())
					buf.append(',');
			}
			buf.append(") ");
		}
		return buf.toString();
	}

	/**
	 *
	 * @param inq
	 * @param op
	 */
	void prepareInnerQuery(StringBuffer inq, Operator op) throws Exception {
		if (op instanceof UnaryOperator) {
			UnaryOperator uop = (UnaryOperator) op;
			SearchPredicate sp = uop.getPredicate();
			prepareQueryPart(sp, inq);
		} else if (op instanceof BinaryOperator) {
			BinaryOperator bop = (BinaryOperator) op;
			Operator lhs = bop.getLhs();
			prepareInnerQuery(inq, lhs);
			int connective = bop.getLogicalOp();
			if (connective == Operator.AND) {
				inq.append("\nINTERSECT\n");
			} else if (connective == Operator.OR) {
				inq.append("\nUNION\n");
			}
			Operator rhs = bop.getRhs();
			prepareInnerQuery(inq, rhs);
		}
	}

	/**
	 *
	 * @param sp
	 * @param inq
	 *            string buffer for the the inner SQL query (contains the query
	 *            generated so far)
	 */
	protected void prepareQueryPart(SearchPredicate sp, StringBuffer inq)
			throws Exception {
		Object o = sp.getAttribute();
		if (!(o instanceof AsScoreInfo))
			throw new Exception("Attribute needs to be of type AsScoreInfo!");
		AsScoreInfo si = (AsScoreInfo) o;
		AssessmentSelectionInfo asi = (AssessmentSelectionInfo) scoreMap
				.get(si);
		inq.append(getInnerStaticPart(si.getType(), asi.getAssessmentID()
				.intValue()));
		inq.append(" AND ( a.SCORENAME=");
		inq.append(sqlDialect.prepareString(si.getName())).append(" AND ");

		if (sp.getValue() instanceof SearchPredicate.Range) {
			SearchPredicate.Range range = (SearchPredicate.Range) sp.getValue();
			inq.append("a.DATAVALUE BETWEEN ").append(range.getLowBound());
			inq.append(" AND ").append(range.getUppBound());
		} else if (sp.getValue() instanceof SearchPredicate.DateRange) {
			SearchPredicate.DateRange dateRange = (SearchPredicate.DateRange) sp
					.getValue();
			inq.append("a.DATAVALUE BETWEEN ").append(
					toTimeStamp(dateRange.getLowBound(), sqlDialect));
			inq.append(" AND ").append(toTimeStamp(dateRange.getUppBound(), sqlDialect));
		} else {
			inq.append(createPredicate("a.datavalue", sp.getValue(), sp.getOperator(), sp
					.getType()));
		}
		inq.append(')');
	}


	/**
	 * Given a SQL query, connects to the database.
	 *
	 * @param query
	 *            any SQL query
	 */
	public static void doQuery(String query) {
		NamedUserPool pool = null;
		Connection con = null;
		Statement st = null;
		try {
			Properties props = GenUtils
					.loadProperties("assessments.properties");
			String dbURL = props.getProperty("dburl");
			String user = props.getProperty("user");
			String pwd = props.getProperty("pwd");

			pool = NamedUserPool.getInstance("oracle.jdbc.driver.OracleDriver",
					dbURL);
			con = pool.getConnection(user, pwd);
			st = con.createStatement();
			ResultSet rs = st.executeQuery(query);
			ResultSetMetaData rsmd = rs.getMetaData();
			int numCols = rsmd.getColumnCount();
			for (int i = 0; i < numCols; ++i) {
				System.out.print(rsmd.getColumnLabel(i + 1) + " ");
			}
			System.out.println();
			int count = 0;
			while (rs.next()) {
				for (int i = 0; i < numCols; ++i) {
					System.out.print(rs.getObject(i + 1) + " ");
				}
				System.out.println();
				++count;
			}
			System.out.println("number of records returned:" + count);
			rs.close();
			st.close();
		} catch (Exception ex) {
			ex.printStackTrace();
		} finally {
			if (con != null)
				try {
					pool.releaseConnection("ucsd_fmri", con);
				} catch (Exception x) {
				}

			if (pool != null)
				pool.shutdown();
		}

	}

	public static void main(String[] args) {
		String dbID = "ucsd_fbirn";
		List<AssessmentSelectionInfo> assessments = new LinkedList<AssessmentSelectionInfo>();
		AssessmentSelectionInfo asi = new AssessmentSelectionInfo();
		// will change every time data is reuploaded
		String MMSEAssessmentID = "1268";
		String DemographicsAssessmentID = "1269";

		asi.setAssessmentID(new BigDecimal(MMSEAssessmentID));
		asi.setName("MMSE");
		AsScoreInfo si1 = new AsScoreInfo(asi.getAssessmentID());
		si1.setType("integer");
		si1.setName("MMSE Score");
		asi.addScore(si1);
		assessments.add(asi);

		/*
		 * asi = new AssessmentSelectionInfo(); asi.setAssessmentID(new
		 * BigDecimal("74")); asi.setName("Demographics"); AsScoreInfo si2 = new
		 * AsScoreInfo(); si2.setType("integer"); si2.setName("Age");
		 * asi.addScore(si2); assessments.add(asi);
		 */
		asi = new AssessmentSelectionInfo();
		asi.setAssessmentID(new BigDecimal(DemographicsAssessmentID));
		asi.setName("Demographics");
		AsScoreInfo si2 = new AsScoreInfo(asi.getAssessmentID());
		si2.setType("varchar");
		si2.setName("Gender");
		asi.addScore(si2);
		assessments.add(asi);

		SearchPredicate.Range range = new SearchPredicate.Range(
				new Integer(15), new Integer(25));

		// prepare Operator tree
		UnaryOperator lhs = new UnaryOperator(new SearchPredicate(si1,
				range /* new Integer(21) */, SearchPredicate.LESS,
				SearchPredicate.INTEGER), Operator.NONE);
		/*
		 * UnaryOperator rhs = new UnaryOperator( new SearchPredicate(si2, new
		 * Integer(50), SearchPredicate.GREATER, SearchPredicate.INTEGER),
		 * Operator.NONE);
		 */
		UnaryOperator rhs = new UnaryOperator(new SearchPredicate(si2, "M",
				SearchPredicate.EQUAL, SearchPredicate.STRING), Operator.NONE);

		BinaryOperator bop = new BinaryOperator(lhs, rhs, Operator.AND);
		try {
			ISQLDialect sqlDialect = ServiceFactory.getSQLDialect(dbID);
			AssessmentQueryBuilder aqb = new AssessmentQueryBuilder(sqlDialect,
					assessments);

			aqb.visit(bop);

			System.out.println("query=\n" + aqb.getQuery());

			AssessmentQueryBuilder.doQuery(aqb.getQuery());
		} catch (Exception x) {
			x.printStackTrace();
		}
	}
}
