package clinical.web.common.vo;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.jdom.Element;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import clinical.server.vo.Humansubject;
import clinical.utils.GenUtils;
import clinical.web.helpers.AssessmentSelectionInfoHelper;

/**
 * 
 * @version $Id: AssessmentSelectionInfo.java,v 1.3 2006/01/23 20:06:58 bozyurt
 *          Exp $
 * @author I. Burak Ozyurt
 */

public class AssessmentSelectionInfo implements java.io.Serializable {
	private static final long serialVersionUID = 7711040846525465283L;
	private boolean selected;
	private String name;
	private String description;
	private BigDecimal assessmentID;
	private List<AsScoreInfo> scores = new ArrayList<AsScoreInfo>(5);
	private Map<String, AsScoreInfo> scoreMap = new HashMap<String, AsScoreInfo>(
			23);
	private List<AsScoreInfo> topLevelScores = null;
	
	public AssessmentSelectionInfo() {
	}

	public AssessmentSelectionInfo(String name, BigDecimal assessmentID) {
		this.name = name;
		this.assessmentID = assessmentID;
	}

	public AssessmentSelectionInfo(AssessmentSelectionInfo other,
			boolean copyScores, boolean deepCopy) {
		this.name = other.name;
		this.assessmentID = other.assessmentID;
		this.description = other.description;
		this.selected = other.selected;		
		if (copyScores) {
			for (Iterator<AsScoreInfo> iter = other.getScores().iterator(); iter
					.hasNext();) {
				AsScoreInfo si = iter.next();
				if (deepCopy) {
					addScore(new AsScoreInfo(si, deepCopy));
				} else {
					addScore(si);
				}
			}
		}
	}

	// ---------------------- setters --------------
	public void setSelected(boolean newSelected) {
		this.selected = newSelected;
	}

	public void setName(String newName) {
		this.name = newName;
	}

	public void setDescription(String newDescription) {
		this.description = newDescription;
	}

	public void setAssessmentID(BigDecimal newAssessmentID) {
		this.assessmentID = newAssessmentID;
	}

	// ---------------------- getters --------------
	public boolean isSelected() {
		return this.selected;
	}

	public String getName() {
		return this.name;
	}

	public String getDescription() {
		return this.description;
	}

	public BigDecimal getAssessmentID() {
		return this.assessmentID;
	}

	public void addScore(AsScoreInfo asi) {
		if (scoreMap != null) {
			if (scoreMap.get(asi.getName()) == null) {
				scores.add(asi);
			}
			scoreMap.put(asi.getName(), asi);
		} else {
			if (findScore(asi.getName()) == null) {
				scores.add(asi);
			}
		}
	}

	protected AsScoreInfo findScore(String scoreName) {
		for (Iterator<AsScoreInfo> it = scores.iterator(); it.hasNext();) {
			AsScoreInfo asi = it.next();
			if (asi.getName().equals(scoreName)) {
				return asi;
			}
		}
		return null;
	}

	public List<AsScoreInfo> getScores() {
		return scores;
	}

	public void sortScoresBySequence() {
		Collections.sort(scores, new Comparator<AsScoreInfo>() {
			public int compare(AsScoreInfo s1, AsScoreInfo s2) {
				return s2.getSequence() - s1.getSequence();
			}
		});

	}

	public List<AsScoreInfo> getTopLevelScores() {
		return topLevelScores;
	}

	public void setTopLevelScores(List<AsScoreInfo> topLevelScores) {
		this.topLevelScores = topLevelScores;
	}

	public String getScoreIndex(AsScoreInfo si) {
		return "" + scores.indexOf(si);
	}

	public boolean isParentCollapsed(AsScoreInfo si) {
		if (scoreMap == null) {
			scoreMap = new HashMap<String, AsScoreInfo>();
			for (Iterator<AsScoreInfo> iter = scores.iterator(); iter.hasNext();) {
				AsScoreInfo aSi = iter.next();
				scoreMap.put(aSi.getName(), aSi);
			}
		}
		if (si.getParentScore() == null) {
			return false;
		}
		AsScoreInfo parent = scoreMap.get(si.getParentScore());
		if (parent == null) {
			return false;
		}
		return parent.getCollapsed();
	}

	public List<AsScoreInfo> getPreOrderTraverseList(AsScoreInfo parent) {
		List<AsScoreInfo> orderList = new LinkedList<AsScoreInfo>();
		preOrderTraversal(parent, orderList);
		return orderList;
	}

	protected List<AsScoreInfo> prepareLevel(List<AsScoreInfo> parentLevel) {
		List<AsScoreInfo> levelSis = new LinkedList<AsScoreInfo>();
		for (AsScoreInfo parent : parentLevel) {
			if (!parent.getChildren().isEmpty()) {
				levelSis.addAll(parent.getChildren());
			}
		}
		return levelSis;
	}

	protected void preOrderTraversal(AsScoreInfo parent,
			List<AsScoreInfo> orderList) {
		if (parent == null) {
			return;
		}
		for (AsScoreInfo child : parent.getChildren()) {
			orderList.add(child);
			preOrderTraversal(child, orderList);
		}
	}

	class Counter {
		int count = 0;

		public Counter(int initValue) {
			this.count = initValue;
		}

		public void incr() {
			count++;
		}
	}

	public int getMaxTreeLevelCount(AsScoreInfo parent) {
		Counter count = new Counter(0);
		countLevels(parent, count);
		return count.count;
	}

	protected void countLevels(AsScoreInfo parent, Counter counter) {
		if (parent == null || parent.getChildren().isEmpty()) {
			return;
		}
		int maxLevel = counter.count;
		for (AsScoreInfo child : parent.getChildren()) {
			Counter newCounter = new Counter(counter.count + 1);
			countLevels(child, newCounter);
			if (maxLevel < newCounter.count) {
				maxLevel = newCounter.count;
			}
		}
		counter.count = maxLevel;
	}

	public AsScoreInfo findScoreInfo(String scoreName) {
		return scoreMap.get(scoreName);
	}

	// indexed property
	public AsScoreInfo getScore(int idx) {
		return scores.get(idx);
	}

	public Element toXML(boolean skipScores) {
		return toXML(null, skipScores);
	}

	public JSONObject toJSON(JSONObject js) throws JSONException {
		js.put("name", name);		
		js.put("asId", assessmentID);
		if (selected)
			js.put("selected", selected);
		if (!scores.isEmpty()) {
			//sort scores by its sequence #
			Collections.sort(scores, new Comparator<Object>() {
				public int compare(Object o1, Object o2) {
					AsScoreInfo asi1 = (AsScoreInfo) o1;
					AsScoreInfo asi2 = (AsScoreInfo) o2;
					return (asi1.getSequence() < asi2.getSequence() ? -1 : (asi1.getSequence()==asi2.getSequence() ? 0 : 1));
				}
			});
			JSONArray jsArr = new JSONArray();
			for (AsScoreInfo si : scores) {
				jsArr.put(si.toJSON(new JSONObject()));
			}
			js.put("scores", jsArr);
		}
		return js;
	}

	public static AssessmentSelectionInfo initializeFromJSON(JSONObject js)
			throws JSONException {
		AssessmentSelectionInfo asi = new AssessmentSelectionInfo();
		asi.name = js.getString("name");
		asi.assessmentID = new BigDecimal(js.getString("asId"));
		if (js.has("selected"))
			asi.selected = js.getBoolean("selected");
		// TODO also scores
		return asi;
	}

	public Element toXML(AsScoreInfo si, boolean skipScores) {
		Element elem = new Element("as-select-info");
		elem.setAttribute("name", name); // optional
		elem.setAttribute("selected", String.valueOf(selected));
		elem.setAttribute("assessmentID", assessmentID.toString());

		if (skipScores) {
			return elem;
		}

		if (!scores.isEmpty()) {
			Element se = new Element("scores");
			elem.addContent(se);
			for (Iterator<AsScoreInfo> iter = scores.iterator(); iter.hasNext();) {
				AsScoreInfo asi = iter.next();
				if (asi.isSelected()) {
					if (si != null) {
						if (si.getName().equals(asi.getName())) {
							se.addContent(asi.toXML());
						}
					} else {
						se.addContent(asi.toXML());
					}
				}
			}
		}
		return elem;
	}

	public static AssessmentSelectionInfo initializeFromXML(Element elem,
			AssessmentSelectionInfoHelper helper) throws Exception {
		BigDecimal asID = new BigDecimal(elem.getAttributeValue("assessmentID"));
		AssessmentSelectionInfo asi = helper.findAssessmentSelectionInfo(asID,
				true);

		Element se = elem.getChild("scores");
		if (se != null) {
			List<?> children = se.getChildren("score-info");
			for (Iterator<?> iter = children.iterator(); iter.hasNext();) {
				Element e = (Element) iter.next();
				// mainly sets the selected flag
				AsScoreInfo.initializeFromXML(e, helper, asi);
			}
		}
		return asi;
	}

	public static AssessmentSelectionInfo initializeFromXML(Element elem) {
		AssessmentSelectionInfo asi = new AssessmentSelectionInfo();
		asi.setName(elem.getAttributeValue("name"));
		asi.setAssessmentID(new BigDecimal(elem
				.getAttributeValue("assessmentID")));
		asi.setSelected(GenUtils.toBoolean(elem.getAttributeValue("selected"),
				false));
		Element de = elem.getChild("description");
		if (de != null) {
			asi.setDescription(de.getTextTrim());
		}

		Element se = elem.getChild("scores");
		if (se != null) {
			List<?> children = se.getChildren("score-info");
			for (Iterator<?> iter = children.iterator(); iter.hasNext();) {
				Element e = (Element) iter.next();
				AsScoreInfo si = AsScoreInfo.initializeFromXML(e, asi
						.getAssessmentID());
				asi.addScore(si);
			}
		}
		return asi;
	}

	public String toString() {
		StringBuffer buf = new StringBuffer(128);
		buf.append("AssessmentSelectionInfo::[");
		buf.append("selected=").append(selected);
		buf.append(", name=").append(name);
		buf.append(", description=").append(description);
		buf.append(", assessmentID=").append(assessmentID);
		if (!scores.isEmpty()) {
			buf.append("\nScores\n");
		}
		for (Iterator<AsScoreInfo> iter = scores.iterator(); iter.hasNext();) {
			AsScoreInfo item = iter.next();
			buf.append(item.toString());
			if (iter.hasNext()) {
				buf.append(",");
			}
		}
		buf.append(", topLevelScores=").append(topLevelScores);
		buf.append(']');

		return buf.toString();
	}

	public boolean equals(Object other) {
		if (other == null) {
			return false;
		}
		if (other == this) {
			return true;
		}
		if (!(other instanceof AssessmentSelectionInfo)) {
			return false;
		}
		AssessmentSelectionInfo that = (AssessmentSelectionInfo) other;
		if ((name != null && that.name != null && name.equals(that.name))) {
			return true;
		}
		return false;
	}

	public int hashCode() {
		return name.hashCode();
	}

}
