package clinical.web.download;

import java.io.File;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.utils.BIRNURIException;
import clinical.utils.BIRNURIUtils;
import clinical.utils.FileUtils;
import clinical.web.Constants;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: FileInfo.java 873 2015-01-07 20:55:20Z jinranc $
 */
public class FileInfo implements Serializable {
	private Log log = LogFactory.getLog(FileInfo.class);
	private static final long serialVersionUID = 1L;
	protected String path;
	protected int fileType;
	protected int storageType;
	protected String gridFtpHost;
	protected boolean isDir = false;
	/**
	 * any directory structure below the root path is maintained if the file
	 * represented by this object is copied.
	 */
	private String rootPath;
	/** file size in bytes (proxy) */
	protected long fileSize;

	protected String sourceRootDir;
	protected Map<String, Object> metaDataMap;

	public final static int NIFTI = 1;
	public final static int GZIPPED_NIFTI = 2;
	public final static int DICOM = 3;
	public static final int COLLECTION = 4;
	public final static int ANALYZE_7_5 = 5;
	public final static int HTML = 6;
	public final static int EPRIME = 7;
	public final static int AFNI = 8;

	/** tarred gzipped file */
	public static final int TGZ = 50;
	public static final int TAR_GZ = 51;
	public static final int TAR = 52;

	public final static int UNKNOWN = -100;

	public final static int LOCAL = 100;
	public final static int SRB = 200;
	public final static int GRIDFTP = 300;

	public final static String LOCAL_STR = "local";
	public final static String SRB_STR = "srb";
	public final static String GRIDFTP_STR = "gridftp";

	public FileInfo(String path, int fileType, int storageType, String rootPath,
			boolean isDir) throws BIRNURIException {
		super();
		log.info("DYW 1 path=" + path + ", rootPath=" + rootPath);
		this.path = path;
		if (Constants.USE_BIRN_URIS) {
			if (storageType == LOCAL || storageType == SRB) {
				if (BIRNURIUtils.hasSchemePart(path)) {
					this.path = BIRNURIUtils.getPath(path);
				}
			} else {
				this.path = BIRNURIUtils.getPath(path);
			}
		}
		if ( storageType == GRIDFTP ) {
			this.gridFtpHost = "gsiftp://"+ getHost(path);
			this.path = getPath(path);
		log.info("DYW 2 path=" + this.path + ",host=" + this.gridFtpHost);
		}
		this.fileType = fileType;
		this.storageType = storageType;
		this.setRootPath(rootPath);
		this.isDir = isDir;
	}

        protected String getHost(String physicalLoc) {
                String s = physicalLoc.replaceFirst("^gsiftp:\\/\\/", "");
                int idx = s.indexOf('/');
                if (idx == -1)
                        throw new RuntimeException("Not a valid GridFTP URL!:" + physicalLoc);
                String gridFtpHost = s.substring(0, idx);
                return gridFtpHost;
        }

        protected String getPath(String physicalLoc) {
                String s = physicalLoc.replaceFirst("^gsiftp:\\/\\/", "");
                int idx = s.indexOf('/');
                if (idx == -1)
                        throw new RuntimeException("Not a valid GridFTP URL!:" + physicalLoc);
                return s.substring(idx);
        }

	public long getFileSize() {
		return fileSize;
	}

	public String getGridFtpHost() {
		return gridFtpHost;
	}

	public String getPath() {
		return path;
	}

	public int getStorageType() {
		return storageType;
	}

	public String getStorageTypeAsStr() {
		switch (storageType) {
		case LOCAL:
			return LOCAL_STR;
		case SRB:
			return SRB_STR;
		case GRIDFTP:
			return GRIDFTP_STR;
		default:
			throw new RuntimeException("Unsupported Storage type:" + storageType);
		}
	}

	public String getRootPath() {
		// ASSUMPTION: file/ srb paths use both UNIX style path seperators
		if (rootPath == null) {
			String p = getPath();
			int idx = p.lastIndexOf("/");
			assert (idx != -1);
			rootPath = p.substring(0, idx);
		}
		return rootPath;
	}

	public String getRelativePath() {
		String rp = getRootPath();
		String p = getPath();
		int idx = p.indexOf(rp);
		assert (idx == 0);
		if (idx != -1) {
			String relPath = p.substring(rp.length());
			relPath = relPath.replaceFirst("^\\/+", "");
			return relPath;
		} else {
			return p;
		}
	}

	/**
	 * returns the relative directory path stripping the file name,extension and
	 * last path separator part if exists
	 * 
	 * @return
	 */
	public String getRelativeDirPath() {
		String rp = getRootPath();
		String p = getPath();
		int idx = p.indexOf(rp);
		assert (idx == 0);
		if (idx != -1) {
			String relPath = p.substring(rp.length());
			relPath = relPath.replaceFirst("^\\/+", "");
			relPath = relPath.replaceFirst("\\/+[^\\/]+\\.[^\\.]+$", "");
			return relPath;
		} else {
			p = p.replaceFirst("\\/+[^\\/]+\\.[^\\.]+$", "");
			return p;
		}
	}

	public boolean isDir() {
		return isDir;
	}

	public int getFileType() {
		return fileType;
	}

	public String getFileTypeAsString() {
		switch (fileType) {
		case DICOM:
			return "DICOM";
		case NIFTI:
			return "NIFTI";
		case GZIPPED_NIFTI:
			return "GZIPPED_NIFTI";
		case ANALYZE_7_5:
			return "ANALYZE_7_5";
		case EPRIME:
			return "EPRIME";
		case AFNI:
			return "AFNI";
		case HTML:
			return "HTML";
		default:
			return null;
		}
	}

	public String getSourceRootDir() {
		return sourceRootDir;
	}

	public void setSourceRootDir(String sourceRootDir) {
		this.sourceRootDir = sourceRootDir;
	}

	public void setFileSize(long fileSize) {
		this.fileSize = fileSize;
	}

	public boolean hasMetaData() {
		return metaDataMap != null && !metaDataMap.isEmpty();
	}

	public void addMetaData(String key, Object value) {
		if (metaDataMap == null) {
			metaDataMap = new HashMap<String, Object>(3);
		}
		metaDataMap.put(key, value);
	}

	public Object getMetaData(String key) {
		if (metaDataMap == null)
			return null;
		return metaDataMap.get(key);
	}

	public static File getFullPathFile(FileInfo fi, String srcRootDir) {
		String path = fi.getRelativePath();
		String relPath = path;
		if (fi.getStorageType() == FileInfo.LOCAL) {
			// local paths are always absolute
			return new File(fi.getPath());
		}

		if (!relPath.startsWith("/") && relPath.indexOf(':') == -1) {
			relPath = "/" + relPath;
		}
		if (relPath.startsWith(srcRootDir)) {
			// means fi is an absolute path
			return new File(fi.getPath());
		}

		if (path.startsWith("/")) {
			path = path.replaceFirst("^\\/+", "");
		}
		FileUtils.createFullPath(srcRootDir, path);
		return new File(srcRootDir, path);
	}

	public static String findCommonPath(List<FileInfo> fileInfoList) {
		List<String[]> paths = new ArrayList<String[]>(fileInfoList.size());
		int min = Integer.MAX_VALUE;
		for (FileInfo fi : fileInfoList) {
			if (fi.getStorageType() == FileInfo.SRB) {
				continue;
			}
			if (fi.isDir()) {
				String[] pathFragments = fi.getPath().split("\\/");
				paths.add(pathFragments);
				if (pathFragments.length < min) {
					min = pathFragments.length;
				}
			} else {
				String[] pathFragments = new File(fi.getPath()).getParent().split(
						"\\/");
				paths.add(pathFragments);
				if (pathFragments.length < min) {
					min = pathFragments.length;
				}
			}
		}
		StringBuilder buf = new StringBuilder();

		for (int i = 0; i < min; i++) {
			String cf = getCommonFragment(paths, i);
			if (cf != null) {
				// TODO: assumption upto the BIRN ID
				if (cf.matches("0\\d{11,11}")) {
					break;
				}
				buf.append('/');
				buf.append(cf);
			} else {
				break;
			}
		}
		String cp = buf.toString();
		cp = cp.replaceAll("\\/\\/+", "\\/");
		return cp;
	}

	public static String getCommonFragment(List<String[]> paths, int idx) {
		if (paths.isEmpty())
			return null;
		Iterator<String[]> it = paths.iterator();
		String[] fragments = it.next();
		String fragment = fragments[idx];
		while (it.hasNext()) {
			fragments = it.next();
			if (!fragments[idx].equals(fragment)) {
				return null;
			}
		}
		return fragment;
	}

	public void setRootPath(String rootPath) {
		this.rootPath = rootPath;
	}

}
